/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2000 Kevin P. Lawton
 *
 *  bcd.c:  BCD oriented x86 instruction emulation
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "monitor.h"


  void
AAD(vm_t *vm)
{
  Bit32u eflags;
  Bit16u ax;
  Bit8u  imm8;

  ax = G_AX(vm);
  imm8 = vm->i.Ib;

  asm volatile (
    "movb %b3, 1f     \n\t"
    "jmp 0f           \n\t"
    "0:               \n\t"
    ".byte 0xD5       \n\t" /* AAD base(normally 10) */
    "1:               \n\t"
    ".byte 0x0A       \n\t"
    "pushfl           \n\t"
    "popl %0"
    : "=g" (eflags), "=a" (ax)
    : "1" (ax), "q" (imm8)
    : "memory", "cc"
    );

  /* +++ OAC flags are actually undefined for ADD */
  SetOSZAPC(vm, eflags);
  G_AX(vm) = ax;
}

  void
AAA(vm_t *vm)
{
  Bit32u arithmetic_eflags, eflags;
  Bit16u ax;

  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  ax = G_AX(vm);
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %4, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "aaa                   \n\t" /* Do operation */
    "pushfl                \n\t" /* Get resultant flags */
    "popl   %0"
    : "=r" (eflags), "=a" (ax)
    : "1" (ax), "0" (eflags), "g" (arithmetic_eflags)
    : "memory", "cc"
    );
 
  G_AX(vm) = ax;
  SetOSZAPC(vm, eflags);
}

  void
AAM(vm_t *vm)
{
  Bit32u arithmetic_eflags, eflags;
  Bit16u ax;
  Bit8u  imm8;
 
  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  ax = G_AX(vm);
  imm8 = vm->i.Ib;

  asm volatile (
    "movb %b5, 1f          \n\t"
    "jmp 0f                \n\t"
    "0:                    \n\t"
    /* copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %4, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    ".byte 0xD4            \n\t" /* AAM base(normally 10) */
    "1:                    \n\t"
    ".byte 0x0A            \n\t"
    "pushfl                \n\t" /* Get resultant flags */
    "popl   %0"
    : "=r" (eflags), "=a" (ax)
    : "1" (ax), "0" (eflags), "g" (arithmetic_eflags), "q" (imm8)
    : "memory", "cc"
    );
 
  G_AX(vm) = ax;
  SetOSZAPC(vm, eflags);
}

  void
AAS(vm_t *vm)
{
  Bit32u arithmetic_eflags, eflags;
  Bit16u ax;
 
  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  ax = G_AX(vm);
 
  asm volatile (
    /* copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %4, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "aas                   \n\t"
    "pushfl                \n\t" /* Get resultant flags */
    "popl   %0"
    : "=r" (eflags), "=a" (ax)
    : "1" (ax), "0" (eflags), "g" (arithmetic_eflags)
    : "memory", "cc"
    );
 
  G_AX(vm) = ax;
  SetOSZAPC(vm, eflags);
}

  void
DAA(vm_t *vm)
{
  Bit32u arithmetic_eflags, eflags;
  Bit16u ax;
 
  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  ax = G_AX(vm);
 
  asm volatile (
    /* copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %4, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "daa                   \n\t"
    "pushfl                \n\t" /* Get resultant flags */
    "popl   %0"
    : "=r" (eflags), "=a" (ax)
    : "1" (ax), "0" (eflags), "g" (arithmetic_eflags)
    : "memory", "cc"
    );
 
  G_AX(vm) = ax;
  SetOSZAPC(vm, eflags);
}

  void
DAS(vm_t *vm)
{
  Bit32u arithmetic_eflags, eflags;
  Bit16u ax;
 
  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  ax = G_AX(vm);
 
  asm volatile (
    /* copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %4, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "das                   \n\t"
    "pushfl                \n\t" /* Get resultant flags */
    "popl   %0"
    : "=r" (eflags), "=a" (ax)
    : "1" (ax), "0" (eflags), "g" (arithmetic_eflags)
    : "memory", "cc"
    );
 
  G_AX(vm) = ax;
  SetOSZAPC(vm, eflags);
}
