/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2000 Kevin P. Lawton
 *
 *  bit.c:  bit oriented instructions
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "monitor.h"



  void
SETCC_Eb(vm_t *vm)
{
  Bit8u result;
 
  switch (vm->i.b1) {
    case 0x190: /* SETO */
      if (G_GetOF(vm))
        result = 1;
      else
        result = 0;
      break;
    case 0x191: /* SETNO */
      if (G_GetOF(vm)==0)
        result = 1;
      else
        result = 0;
      break;
    case 0x192: /* SETB */
      if (G_GetCF(vm))
        result = 1;
      else
        result = 0;
      break;
    case 0x193: /* SETNB */
      if (G_GetCF(vm)==0)
        result = 1;
      else
        result = 0;
      break;
    case 0x194: /* SETZ */
      if (G_GetZF(vm))
        result = 1;
      else
        result = 0;
      break;
    case 0x195: /* SETNZ */
      if (G_GetZF(vm)==0)
        result = 1;
      else
        result = 0;
      break;
    case 0x196: /* SETBE */
      if (G_GetCF(vm) || G_GetZF(vm))
        result = 1;
      else
        result = 0;
      break;
    case 0x197: /* SETNBE */
      if ((G_GetCF(vm)==0) && (G_GetZF(vm)==0))
        result = 1;
      else
        result = 0;
      break;
    case 0x198: /* SETS */
      if (G_GetSF(vm))
        result = 1;
      else
        result = 0;
      break;
    case 0x199: /* SETNS */
      if (G_GetSF(vm)==0)
        result = 1;
      else
        result = 0;
      break;
    case 0x19A: /* SETP */
      if (G_GetPF(vm))
        result = 1;
      else
        result = 0;
      break;
    case 0x19B: /* SETNP */
      if (G_GetPF(vm) == 0)
        result = 1;
      else
        result = 0;
      break;
    case 0x19C: /* SETL */
      if (G_GetSF(vm) != G_GetOF(vm))
        result = 1;
      else
        result = 0;
      break;
    case 0x19D: /* SETNL */
      if (G_GetSF(vm) == G_GetOF(vm))
        result = 1;
      else
        result = 0;
      break;
    case 0x19E: /* SETLE */
      if (G_GetZF(vm) || (G_GetSF(vm)!=G_GetOF(vm)))
        result = 1;
      else
        result = 0;
      break;
    case 0x19F: /* SETNLE */
      if ((G_GetZF(vm)==0) && (G_GetSF(vm)==G_GetOF(vm)))
        result = 1;
      else
        result = 0;
      break;
    default:
      monpanic(vm, "SETCC_Eb:\n");
    }
 
  if (vm->i.mod == 0xc0) {
    WriteReg8(vm, vm->i.rm, result);
    }
  else {
    write_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &result);
    }
}


  void
BSF_GvEv(vm_t *vm)
{
  Bit32u eflags;

  if (vm->i.os_32) {
    Bit32u op1, op2;
 
    if (vm->i.mod == 0xc0) {
      op2 = ReadReg32(vm, vm->i.rm);
      }
    else {
      read_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op2);
      }
 
    asm volatile (
      "bsfl %2, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "g" (op2)
      : "memory", "cc"
      );

    WriteReg32(vm, vm->i.nnn, op1);
    SetFlagsFromImageZ(vm, eflags);
    }
  else {
    Bit16u op1, op2;
 
    if (vm->i.mod == 0xc0) {
      op2 = ReadReg16(vm, vm->i.rm);
      }
    else {
      read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op2);
      }
 
    asm volatile (
      "bsfw %2, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "g" (op2)
      : "memory", "cc"
      );

    WriteReg16(vm, vm->i.nnn, op1);
    SetFlagsFromImageZ(vm, eflags);
    }
}

  void
BSR_GvEv(vm_t *vm)
{
  Bit32u eflags;

  if (vm->i.os_32) {
    Bit32u op1, op2;
 
    if (vm->i.mod == 0xc0) {
      op2 = ReadReg32(vm, vm->i.rm);
      }
    else {
      read_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op2);
      }
 
    asm volatile (
      "bsrl %2, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "g" (op2)
      : "memory", "cc"
      );

    WriteReg32(vm, vm->i.nnn, op1);
    SetFlagsFromImageZ(vm, eflags);
    }
  else {
    Bit16u op1, op2;
 
    if (vm->i.mod == 0xc0) {
      op2 = ReadReg16(vm, vm->i.rm);
      }
    else {
      read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op2);
      }
 
    asm volatile (
      "bsrw %2, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "g" (op2)
      : "memory", "cc"
      );

    WriteReg16(vm, vm->i.nnn, op1);
    SetFlagsFromImageZ(vm, eflags);
    }
}


  void
BT_EvGv(vm_t *vm)
{
  Bit32u op1_addr;

  if (vm->i.os_32) {
    Bit32u op1_32, op2_32, index;
    Bit32s displacement32;

    op2_32 = ReadReg32(vm, vm->i.nnn);

    if (vm->i.mod == 0xc0) {
      op1_32 = ReadReg32(vm, vm->i.rm);
      op2_32 &= 0x1f;
      G_SetCF(vm, (op1_32 >> op2_32) & 0x01);
      return;
      }

    index = op2_32 & 0x1f;
    displacement32 = ((Bit32s) (op2_32&0xffffffe0)) / 32;
    op1_addr = vm->i.rm_addr + 4 * displacement32;

    read_virtual_dword(vm, vm->i.seg, op1_addr, &op1_32);

    G_SetCF(vm, (op1_32 >> index) & 0x01);
    }
  else {
    Bit16u op1_16, op2_16, index;
    Bit32s displacement32;

    op2_16 = ReadReg16(vm, vm->i.nnn);

    if (vm->i.mod == 0xc0) {
      op1_16 = ReadReg16(vm, vm->i.rm);
      op2_16 &= 0x0f;
      G_SetCF(vm, (op1_16 >> op2_16) & 0x01);
      return;
      }

    index = op2_16 & 0x0f;
    displacement32 = ((Bit16s) (op2_16&0xfff0)) / 16;
    op1_addr = vm->i.rm_addr + 2 * displacement32;

    read_virtual_word(vm, vm->i.seg, op1_addr, &op1_16);

    G_SetCF(vm, (op1_16 >> index) & 0x01);
    }
}

  void
BTS_EvGv(vm_t *vm)
{
  Bit32u op1_addr;

  if (vm->i.os_32) {
    Bit32u op1_32, op2_32, bit_i, index;
    Bit32s displacement32;

    op2_32 = ReadReg32(vm, vm->i.nnn);

    if (vm->i.mod == 0xc0) {
      op1_32 = ReadReg32(vm, vm->i.rm);
      op2_32 &= 0x1f;
      G_SetCF(vm, (op1_32 >> op2_32) & 0x01);
      op1_32 |= (((Bit32u) 1) << op2_32);

      WriteReg32(vm, vm->i.rm, op1_32);
      return;
      }

    index = op2_32 & 0x1f;
    displacement32 = ((Bit32s) (op2_32&0xffffffe0)) / 32;
    op1_addr = vm->i.rm_addr + 4 * displacement32;

    read_RMW_virtual_dword(vm, vm->i.seg, op1_addr, &op1_32);

    bit_i = (op1_32 >> index) & 0x01;
    op1_32 |= (((Bit32u) 1) << index);

    write_RMW_virtual_dword(vm, &op1_32);

    G_SetCF(vm, bit_i);
    }
  else {
    Bit16u op1_16, op2_16, bit_i, index;
    Bit32s displacement32;

    op2_16 = ReadReg16(vm, vm->i.nnn);

    if (vm->i.mod == 0xc0) {
      op1_16 = ReadReg16(vm, vm->i.rm);
      op2_16 &= 0x0f;
      G_SetCF(vm, (op1_16 >> op2_16) & 0x01);
      op1_16 |= (((Bit16u) 1) << op2_16);

      WriteReg16(vm, vm->i.rm, op1_16);
      return;
      }

    index = op2_16 & 0x0f;
    displacement32 = ((Bit16s) (op2_16&0xfff0)) / 16;
    op1_addr = vm->i.rm_addr + 2 * displacement32;

    read_RMW_virtual_word(vm, vm->i.seg, op1_addr, &op1_16);

    bit_i = (op1_16 >> index) & 0x01;
    op1_16 |= (((Bit16u) 1) << index);

    write_RMW_virtual_word(vm, &op1_16);

    G_SetCF(vm, bit_i);
    }
}

  void
BTR_EvGv(vm_t *vm)
{
  Bit32u op1_addr;


  if (vm->i.os_32) {
    Bit32u op1_32, op2_32, index, temp_cf;
    Bit32s displacement32;

    op2_32 = ReadReg32(vm, vm->i.nnn);

    if (vm->i.mod == 0xc0) {
      op1_32 = ReadReg32(vm, vm->i.rm);
      op2_32 &= 0x1f;
      G_SetCF(vm, (op1_32 >> op2_32) & 0x01);
      op1_32 &= ~(((Bit32u) 1) << op2_32);

      WriteReg32(vm, vm->i.rm, op1_32);
      return;
      }

    index = op2_32 & 0x1f;
    displacement32 = ((Bit32s) (op2_32&0xffffffe0)) / 32;
    op1_addr = vm->i.rm_addr + 4 * displacement32;

    read_RMW_virtual_dword(vm, vm->i.seg, op1_addr, &op1_32);

    temp_cf = (op1_32 >> index) & 0x01;
    op1_32 &= ~(((Bit32u) 1) << index);

    write_RMW_virtual_dword(vm, &op1_32);

    G_SetCF(vm, temp_cf);
    }
  else {
    Bit16u op1_16, op2_16, index, temp_cf;
    Bit32s displacement32;

    op2_16 = ReadReg16(vm, vm->i.nnn);

    if (vm->i.mod == 0xc0) {
      op1_16 = ReadReg16(vm, vm->i.rm);
      op2_16 &= 0x0f;
      G_SetCF(vm, (op1_16 >> op2_16) & 0x01);
      op1_16 &= ~(((Bit16u) 1) << op2_16);

      WriteReg16(vm, vm->i.rm, op1_16);
      return;
      }

    index = op2_16 & 0x0f;
    displacement32 = ((Bit16s) (op2_16&0xfff0)) / 16;
    op1_addr = vm->i.rm_addr + 2 * displacement32;

    read_RMW_virtual_word(vm, vm->i.seg, op1_addr, &op1_16);

    temp_cf = (op1_16 >> index) & 0x01;
    op1_16 &= ~(((Bit16u) 1) << index);

    write_RMW_virtual_word(vm, &op1_16);

    G_SetCF(vm, temp_cf);
    }
}

  void
BTC_EvGv(vm_t *vm)
{
  Bit32u op1_addr;

  if (vm->i.os_32) {
    Bit32u op1_32, op2_32, index_32, temp_CF;
    Bit32s displacement32;

    op2_32 = ReadReg32(vm, vm->i.nnn);
    index_32 = op2_32 & 0x1f;

    if (vm->i.mod == 0xc0) {
      op1_32 = ReadReg32(vm, vm->i.rm);
      op1_addr = 0; /* keep compiler happy */
      }
    else {
      displacement32 = ((Bit32s) (op2_32 & 0xffffffe0)) / 32;
      op1_addr = vm->i.rm_addr + 4 * displacement32;
      read_RMW_virtual_dword(vm, vm->i.seg, op1_addr, &op1_32);
      }

    temp_CF = (op1_32 >> index_32) & 0x01;
    op1_32 &= ~(((Bit32u) 1) << index_32);  /* clear out bit */
    op1_32 |= (((Bit32u) !temp_CF) << index_32); /* set to complement */

    if (vm->i.mod == 0xc0) {
      WriteReg32(vm, vm->i.rm, op1_32);
      }
    else {
      write_RMW_virtual_dword(vm, &op1_32);
      }
    G_SetCF(vm, temp_CF);
    }
  else {
    Bit16u op1_16, op2_16, index_16, temp_CF;
    Bit16s displacement16;

    op2_16 = ReadReg16(vm, vm->i.nnn);
    index_16 = op2_16 & 0x0f;

    if (vm->i.mod == 0xc0) {
      op1_16 = ReadReg16(vm, vm->i.rm);
      op1_addr = 0; /* keep compiler happy */
      }
    else {
      displacement16 = ((Bit16s) (op2_16 & 0xfff0)) / 16;
      op1_addr = vm->i.rm_addr + 2 * displacement16;
      read_RMW_virtual_word(vm, vm->i.seg, op1_addr, &op1_16);
      }

    temp_CF = (op1_16 >> index_16) & 0x01;
    op1_16 &= ~(((Bit16u) 1) << index_16);  /* clear out bit */
    op1_16 |= (((Bit16u) !temp_CF) << index_16); /* set to complement */

    if (vm->i.mod == 0xc0) {
      WriteReg16(vm, vm->i.rm, op1_16);
      }
    else {
      write_RMW_virtual_word(vm, &op1_16);
      }
    G_SetCF(vm, temp_CF);
    }
}

  void
BT_EvIb(vm_t *vm)
{
  if (vm->i.os_32) {
    Bit32u op1_32;
    Bit8u  op2_8;

    op2_8 = vm->i.Ib;
    op2_8 %= 32;

    if (vm->i.mod == 0xc0) {
      op1_32 = ReadReg32(vm, vm->i.rm);
      }
    else {
      read_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1_32);
      }

    G_SetCF(vm, (op1_32 >> op2_8) & 0x01);
    }
  else {
    Bit16u op1_16;
    Bit8u  op2_8;


    op2_8 = vm->i.Ib;
    op2_8 %= 16;

    if (vm->i.mod == 0xc0) {
      op1_16 = ReadReg16(vm, vm->i.rm);
      }
    else {
      read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1_16);
      }

    G_SetCF(vm, (op1_16 >> op2_8) & 0x01);
    }
}

  void
BTS_EvIb(vm_t *vm)
{
  if (vm->i.os_32) {
    Bit32u op1_32, temp_CF;
    Bit8u  op2_8;

    op2_8 = vm->i.Ib;
    op2_8 %= 32;

    if (vm->i.mod == 0xc0) {
      op1_32 = ReadReg32(vm, vm->i.rm);
      }
    else {
      read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1_32);
      }

    temp_CF = (op1_32 >> op2_8) & 0x01;
    op1_32 |= (((Bit32u) 1) << op2_8);

    if (vm->i.mod == 0xc0) {
      WriteReg32(vm, vm->i.rm, op1_32);
      }
    else {
      write_RMW_virtual_dword(vm, &op1_32);
      }
    G_SetCF(vm, temp_CF);
    }
  else {
    Bit16u op1_16, temp_CF;
    Bit8u  op2_8;


    op2_8 = vm->i.Ib;
    op2_8 %= 16;

    if (vm->i.mod == 0xc0) {
      op1_16 = ReadReg16(vm, vm->i.rm);
      }
    else {
      read_RMW_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1_16);
      }

    temp_CF = (op1_16 >> op2_8) & 0x01;
    op1_16 |= (((Bit16u) 1) << op2_8);

    if (vm->i.mod == 0xc0) {
      WriteReg16(vm, vm->i.rm, op1_16);
      }
    else {
      write_RMW_virtual_word(vm, &op1_16);
      }
    G_SetCF(vm, temp_CF);
    }
}

  void
BTC_EvIb(vm_t *vm)
{
  if (vm->i.os_32) {
    Bit32u op1_32, temp_CF;
    Bit8u  op2_8;

    op2_8 = vm->i.Ib;
    op2_8 %= 32;

    if (vm->i.mod == 0xc0) {
      op1_32 = ReadReg32(vm, vm->i.rm);
      }
    else {
      read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1_32);
      }

    temp_CF = (op1_32 >> op2_8) & 0x01;

    op1_32 &= ~(((Bit32u) 1) << op2_8);  /* clear out bit */
    op1_32 |= (((Bit32u) !temp_CF) << op2_8); /* set to complement */

    if (vm->i.mod == 0xc0) {
      WriteReg32(vm, vm->i.rm, op1_32);
      }
    else {
      write_RMW_virtual_dword(vm, &op1_32);
      }
    G_SetCF(vm, temp_CF);
    }
  else {
    Bit16u op1_16, temp_CF;
    Bit8u  op2_8;


    op2_8 = vm->i.Ib;
    op2_8 %= 16;

    if (vm->i.mod == 0xc0) {
      op1_16 = ReadReg16(vm, vm->i.rm);
      }
    else {
      read_RMW_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1_16);
      }

    temp_CF = (op1_16 >> op2_8) & 0x01;
    op1_16 &= ~(((Bit16u) 1) << op2_8);  /* clear out bit */
    op1_16 |= (((Bit16u) !temp_CF) << op2_8); /* set to complement */

    if (vm->i.mod == 0xc0) {
      WriteReg16(vm, vm->i.rm, op1_16);
      }
    else {
      write_RMW_virtual_word(vm, &op1_16);
      }
    G_SetCF(vm, temp_CF);
    }
}

  void
BTR_EvIb(vm_t *vm)
{
  if (vm->i.os_32) {
    Bit32u op1_32, temp_CF;
    Bit8u  op2_8;

    op2_8 = vm->i.Ib;
    op2_8 %= 32;

    if (vm->i.mod == 0xc0) {
      op1_32 = ReadReg32(vm, vm->i.rm);
      }
    else {
      read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1_32);
      }

    temp_CF = (op1_32 >> op2_8) & 0x01;
    op1_32 &= ~(((Bit32u) 1) << op2_8);

    if (vm->i.mod == 0xc0) {
      WriteReg32(vm, vm->i.rm, op1_32);
      }
    else {
      write_RMW_virtual_dword(vm, &op1_32);
      }
    G_SetCF(vm, temp_CF);
    }
  else {
    Bit16u op1_16, temp_CF;
    Bit8u  op2_8;


    op2_8 = vm->i.Ib;
    op2_8 %= 16;

    if (vm->i.mod == 0xc0) {
      op1_16 = ReadReg16(vm, vm->i.rm);
      }
    else {
      read_RMW_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1_16);
      }

    temp_CF = (op1_16 >> op2_8) & 0x01;
    op1_16 &= ~(((Bit16u) 1) << op2_8);

    if (vm->i.mod == 0xc0) {
      WriteReg16(vm, vm->i.rm, op1_16);
      }
    else {
      write_RMW_virtual_word(vm, &op1_16);
      }
    G_SetCF(vm, temp_CF);
    }
}

  void
BSWAP_ERX(vm_t *vm)
{
  Bit32u erx;
  unsigned reg;
 
  reg = vm->i.b1 & 0x07;
  erx = ReadReg32(vm, reg);
  asm volatile (
    "bswap %%eax"
      : "=a" (erx)
      : "0" (erx)
    );
  WriteReg32(vm, reg, erx);
}
