/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2000 Kevin P. Lawton
 *
 *  ctrl_xfer32.c:  emulation of 32-bit control transfer instructions
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "monitor.h"



  void
CALL32_Ep(vm_t *vm)
{
  Bit16u cs_raw;
  Bit32u op1_32;
 
  invalidate_prefetch_q();

  /* op1_32 is a register or memory reference */
  if (vm->i.mod == 0xc0) {
    monpanic(vm, "CALL_Ep: op1 is a register\n");
    }
 
  /* pointer, segment address pair */
  read_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1_32);
  read_virtual_word(vm, vm->i.seg, vm->i.rm_addr+4, &cs_raw);
 
  if ( ProtectedMode(vm) ) {
    call_protected(vm, cs_raw, op1_32);
    goto done;
    }
 
  cache_selector(vm, SRegCS);
  push32(vm, vm->guest_cpu.selector[SRegCS].raw);
  push32(vm, G_EIP(vm));
 
  load_seg_reg(vm, SRegCS, cs_raw);
  G_EIP(vm) = op1_32;
 
done:
  /* Instrumentation was here */
  return;
}

  void
JMP32_Ep(vm_t *vm)
{
  Bit16u cs_raw;
  Bit32u op1_32;
 
  invalidate_prefetch_q();
  if (vm->i.mod == 0xc0) {
    /* far indirect must specify a memory address */
    monpanic(vm, "JMP_Ep: op1 is a reg.\n");
    }
 
  read_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1_32);
  read_virtual_word(vm, vm->i.seg, vm->i.rm_addr+4, &cs_raw);
 
  if ( ProtectedMode(vm) ) {
    jump_protected(vm, cs_raw, op1_32);
    goto done;
    }
 
  load_seg_reg(vm, SRegCS, cs_raw);
  G_EIP(vm) = op1_32;
 
done:
  /* instrumentation was here */
  return;
}

  void
RETfar32_Iw(vm_t *vm)
{
  Bit32u eip, ecs_raw;
  Bit16s imm16;
 
  imm16 = vm->i.Iw;
 
  invalidate_prefetch_q();
 
  if ( ProtectedMode(vm) ) {
    return_protected(vm, imm16);
    goto done;
    }
 
  pop32(vm, &eip);
  pop32(vm, &ecs_raw);
  load_seg_reg(vm, SRegCS, (Bit16u) ecs_raw);
  G_EIP(vm) = eip;
  cache_sreg(vm, SRegSS);
  if (vm->guest_cpu.desc_cache[SRegSS].desc.d_b)
    G_ESP(vm) += imm16;
  else
    G_SP(vm)  += imm16;
 
done:
  /* Instrumentation code was here */
  return;
}

  void
RETfar32(vm_t *vm)
{
  Bit32u eip, ecs_raw;
 
  invalidate_prefetch_q();
 
  if ( ProtectedMode(vm) ) {
    return_protected(vm, 0);
    goto done;
    }
 
  pop32(vm, &eip);
  pop32(vm, &ecs_raw); /* 32bit pop, MSW discarded */
  load_seg_reg(vm, SRegCS, (Bit16u) ecs_raw);
  G_EIP(vm) = eip;
 
done:
  /* Instrumentation code was here */
  return;
}

  void
CALL_Ed(vm_t *vm)
{
  Bit32u op1;

  if (vm->i.mod == 0xc0) { /* reg op */
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else { /* memory op */
    read_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }

  if (ProtectedMode(vm)) {
    cache_sreg(vm, SRegCS);
    if (op1 > vm->guest_cpu.desc_cache[SRegCS].limit_scaled) {
      monpanic(vm, "CALL_Ed: EIP out of CS limits!\n");
      exception(vm, ExceptionGP, 0);
      }
    }

  push32(vm, vm->guest.addr.guest_context->eip);
  G_EIP(vm) = op1;
}

  void
CALL_Ad(vm_t *vm)
{
  Bit32u new_EIP;
  Bit32s disp32;

  disp32 = vm->i.Id;

  new_EIP = G_EIP(vm) + disp32;

  if ( ProtectedMode(vm) ) {
    cache_sreg(vm, SRegCS);
    if ( new_EIP > vm->guest_cpu.desc_cache[SRegCS].limit_scaled ) {
      monpanic(vm, "call_ad: offset outside of CS limits\n");
      exception(vm, ExceptionGP, 0);
      }
    }

  /* push 32 bit EA of next instruction */
  push32(vm, G_EIP(vm));
  G_EIP(vm) = new_EIP;
}

  void
CALL32_Ap(vm_t *vm)
{
  Bit16u cs_raw;
  Bit32u disp32;
 
  disp32 = vm->i.Id;
  cs_raw = vm->i.Iw2;
  invalidate_prefetch_q();
 
  if (ProtectedMode(vm)) {
    call_protected(vm, cs_raw, disp32);
    goto done;
    }
  cache_selector(vm, SRegCS);
  push32(vm, vm->guest_cpu.selector[SRegCS].raw);
  push32(vm, G_EIP(vm));
  G_EIP(vm) = disp32;
  load_seg_reg(vm, SRegCS, cs_raw);
 
done:
  /* instrumentation was here */
  return;
}

  void
JMP_Ap(vm_t *vm)
{
  Bit32u disp32;
  Bit16u cs_raw;

  if (vm->i.os_32) {
    disp32 = vm->i.Id;
    }
  else {
    disp32 = vm->i.Iw;
    }
  cs_raw = vm->i.Iw2;

  if (ProtectedMode(vm)) {
    jump_protected(vm, cs_raw, disp32);
    goto done;
    }

  load_seg_reg(vm, SRegCS, cs_raw);
  G_EIP(vm) = disp32;

done:
  /* instrumentation call was here */
  return;
}

  void
JMP_Jd(vm_t *vm)
{
  Bit32u new_EIP;
 
  invalidate_prefetch_q();
 
  new_EIP = G_EIP(vm) + (Bit32s) vm->i.Id;
 
  if (ProtectedMode(vm)) {
    cache_sreg(vm, SRegCS);
    if (new_EIP > vm->guest_cpu.desc_cache[SRegCS].limit_scaled) {
      monpanic(vm, "jmp_jd: offset outside of CS limits\n");
      exception(vm, ExceptionGP, 0);
      }
    }
 
  G_EIP(vm) = new_EIP;
}

  void
JCC_Jd(vm_t *vm)
{
  Boolean condition = 0;
 
  switch (vm->i.b1 & 0x0f) {
    case 0x00: /* JO */ condition = G_GetOF(vm); break;
    case 0x01: /* JNO */ condition = !G_GetOF(vm); break;
    case 0x02: /* JB */ condition = G_GetCF(vm); break;
    case 0x03: /* JNB */ condition = !G_GetCF(vm); break;
    case 0x04: /* JZ */ condition = G_GetZF(vm); break;
    case 0x05: /* JNZ */ condition = !G_GetZF(vm); break;
    case 0x06: /* JBE */ condition = G_GetCF(vm) || G_GetZF(vm); break;
    case 0x07: /* JNBE */ condition = !G_GetCF(vm) && !G_GetZF(vm); break;
    case 0x08: /* JS */ condition = G_GetSF(vm); break;
    case 0x09: /* JNS */ condition = !G_GetSF(vm); break;
    case 0x0A: /* JP */ condition = G_GetPF(vm); break;
    case 0x0B: /* JNP */ condition = !G_GetPF(vm); break;
    case 0x0C: /* JL */ condition = G_GetSF(vm) != G_GetOF(vm); break;
    case 0x0D: /* JNL */ condition = G_GetSF(vm) == G_GetOF(vm); break;
    case 0x0E: /* JLE */
      condition = G_GetZF(vm) || (G_GetSF(vm) != G_GetOF(vm));
      break;
    case 0x0F: /* JNLE */
      condition = (G_GetSF(vm) == G_GetOF(vm)) && !G_GetZF(vm);
      break;
    }
 
  if (condition) {
    Bit32u new_EIP;
 
    new_EIP = G_EIP(vm) + (Bit32s) vm->i.Id;
    if (ProtectedMode(vm)) {
      cache_sreg(vm, SRegCS);
      if (new_EIP > vm->guest_cpu.desc_cache[SRegCS].limit_scaled) {
        monpanic(vm, "jo_routine: offset outside of CS limits\n");
        exception(vm, ExceptionGP, 0);
        }
      }
    G_EIP(vm) = new_EIP;
    revalidate_prefetch_q();
    }
}

  void
JMP_Ed(vm_t *vm)
{
  Bit32u new_EIP;
  Bit32u op1_32;
 
  if (vm->i.mod == 0xc0) {
    op1_32 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1_32);
    }
 
  invalidate_prefetch_q();
  new_EIP = op1_32;
 
  if (ProtectedMode(vm)) {
    cache_sreg(vm, SRegCS);
    if (new_EIP > vm->guest_cpu.desc_cache[SRegCS].limit_scaled) {
      monpanic(vm, "jmp_ev: IP out of CS limits!\n");
      exception(vm, ExceptionGP, 0);
      }
    }
 
  G_EIP(vm) = new_EIP;
}

  void
RETnear32(vm_t *vm)
{
  Bit32u temp_ESP;
  Bit32u return_EIP;
  unsigned d_b;

  cache_sreg(vm, SRegCS);
  cache_sreg(vm, SRegSS);
  d_b = vm->guest_cpu.desc_cache[SRegSS].desc.d_b;
  if (d_b)
    temp_ESP = G_ESP(vm);
  else
    temp_ESP = G_SP(vm);


  if (ProtectedMode(vm)) {
    if ( !can_pop(vm, 4) ) {
      monpanic(vm, "retnear: can't pop EIP\n");
      /* ??? #SS(0) -or #GP(0) */
      }

    access_linear(vm, vm->guest_cpu.desc_cache[SRegSS].base + temp_ESP,
      4, G_CPL(vm)==3, OP_READ, &return_EIP);

    if ( return_EIP > vm->guest_cpu.desc_cache[SRegCS].limit_scaled ) {
      monpanic(vm, "retnear: EIP > limit\n");
      /*exception(vm, ExceptionGP, 0); */
      }
    G_EIP(vm) = return_EIP;
    if (d_b) /* 32bit stack */
      G_ESP(vm) += 4;
    else
      G_SP(vm)  += 4;
    }
  else {
    /* +++ should do the same as above? */
    pop32(vm, &return_EIP);
    G_EIP(vm) = return_EIP;
    }
}

  void
RETnear32_Iw(vm_t *vm)
{
  Bit16u imm16;
  Bit32u temp_ESP;
  Bit32u return_EIP;
  unsigned d_b;

  cache_sreg(vm, SRegCS);
  cache_sreg(vm, SRegSS);
  d_b = vm->guest_cpu.desc_cache[SRegSS].desc.d_b;
  if (d_b)
    temp_ESP = G_ESP(vm);
  else
    temp_ESP = G_SP(vm);

  imm16 = vm->i.Iw;

  invalidate_prefetch_q();


  if (ProtectedMode(vm)) {
    if ( !can_pop(vm, 4) ) {
      monpanic(vm, "retnear_iw: can't pop EIP\n");
      /* ??? #SS(0) -or #GP(0) */
      }

    access_linear(vm, vm->guest_cpu.desc_cache[SRegSS].base + temp_ESP,
      4, G_CPL(vm)==3, OP_READ, &return_EIP);

    if ( return_EIP > vm->guest_cpu.desc_cache[SRegCS].limit_scaled ) {
      monpanic(vm, "retnear_iw: EIP > limit\n");
      }

    /* Pentium book says imm16 is number of words ??? */
    if ( !can_pop(vm, 4 + imm16) ) {
      monpanic(vm, "retnear_iw: can't release bytes from stack\n");
      /*exception(vm, ExceptionGP, 0); */
      /* #GP(0) -or #SS(0) ??? */
      }

    G_EIP(vm) = return_EIP;
    if (d_b) /* 32bit stack */
      G_ESP(vm) += 4 + imm16; /* ??? should it be 2*imm16 ? */
    else
      G_SP(vm)  += 4 + imm16;
    }
  else {
    pop32(vm, &return_EIP);
    G_EIP(vm) = return_EIP;
    if (d_b) /* 32bit stack */
      G_ESP(vm) += imm16; /* ??? should it be 2*imm16 ? */
    else
      G_SP(vm)  += imm16;
    }
}

  void
IRET32(vm_t *vm)
{
  Bit32u eip, ecs_raw, eflags, change_mask;

  if (V8086Mode(vm)) {
    /* IOPL check in stack_return_from_v86() */
    monpanic(vm, "IRET32: v8086\n");
/*stack_return_from_v86(i); */
    goto done;
    }

  if (vm->guest_cpu.cr0.fields.pe) {
    iret_protected(vm);
    goto done;
    }


  pop32(vm, &eip);
  pop32(vm, &ecs_raw);
  pop32(vm, &eflags);

  load_seg_reg(vm, SRegCS, (Bit16u) ecs_raw);
  G_EIP(vm) = eip;

  /* modify: ID,AC,RF,NT,IOPL,OF,DF,IF,TF,SF,ZF,AF,PF,CF */
  /* ignore: (VIP,VIF),VM */
  change_mask = 0x00257fd5;
monprint(vm, "iret32: eflags=0x%x\n", eflags);
  write_eflags(vm, eflags, change_mask);

done:
  /* instrumentation call was here */
}
