/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2000  Kevin P. Lawton
 *
 *  fetchdecode.c: This file contains the logic to fetch/decode
 *    one IA32 instruction, and return instruction attributes.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "guest_context.h"
#include "fetchdecode.h"

#ifndef NULL
#define NULL ((void *) 0)
#endif

typedef void (*ResolveModRM_t)(instruction_t *, guest_context_t *);


static void Resolve16Mod0Rm0(instruction_t *, guest_context_t *);
static void Resolve16Mod0Rm1(instruction_t *, guest_context_t *);
static void Resolve16Mod0Rm2(instruction_t *, guest_context_t *);
static void Resolve16Mod0Rm3(instruction_t *, guest_context_t *);
static void Resolve16Mod0Rm4(instruction_t *, guest_context_t *);
static void Resolve16Mod0Rm5(instruction_t *, guest_context_t *);
static void Resolve16Mod0Rm7(instruction_t *, guest_context_t *);

static void Resolve16Mod1or2Rm0(instruction_t *, guest_context_t *);
static void Resolve16Mod1or2Rm1(instruction_t *, guest_context_t *);
static void Resolve16Mod1or2Rm2(instruction_t *, guest_context_t *);
static void Resolve16Mod1or2Rm3(instruction_t *, guest_context_t *);
static void Resolve16Mod1or2Rm4(instruction_t *, guest_context_t *);
static void Resolve16Mod1or2Rm5(instruction_t *, guest_context_t *);
static void Resolve16Mod1or2Rm6(instruction_t *, guest_context_t *);
static void Resolve16Mod1or2Rm7(instruction_t *, guest_context_t *);

static void Resolve32Mod0Rm0(instruction_t *, guest_context_t *);
static void Resolve32Mod0Rm1(instruction_t *, guest_context_t *);
static void Resolve32Mod0Rm2(instruction_t *, guest_context_t *);
static void Resolve32Mod0Rm3(instruction_t *, guest_context_t *);
static void Resolve32Mod0Rm6(instruction_t *, guest_context_t *);
static void Resolve32Mod0Rm7(instruction_t *, guest_context_t *);

static void Resolve32Mod1or2Rm0(instruction_t *, guest_context_t *);
static void Resolve32Mod1or2Rm1(instruction_t *, guest_context_t *);
static void Resolve32Mod1or2Rm2(instruction_t *, guest_context_t *);
static void Resolve32Mod1or2Rm3(instruction_t *, guest_context_t *);
static void Resolve32Mod1or2Rm5(instruction_t *, guest_context_t *);
static void Resolve32Mod1or2Rm6(instruction_t *, guest_context_t *);
static void Resolve32Mod1or2Rm7(instruction_t *, guest_context_t *);

static void Resolve32Mod0Base0(instruction_t *, guest_context_t *);
static void Resolve32Mod0Base1(instruction_t *, guest_context_t *);
static void Resolve32Mod0Base2(instruction_t *, guest_context_t *);
static void Resolve32Mod0Base3(instruction_t *, guest_context_t *);
static void Resolve32Mod0Base4(instruction_t *, guest_context_t *);
static void Resolve32Mod0Base5(instruction_t *, guest_context_t *);
static void Resolve32Mod0Base6(instruction_t *, guest_context_t *);
static void Resolve32Mod0Base7(instruction_t *, guest_context_t *);

static void Resolve32Mod1or2Base0(instruction_t *, guest_context_t *);
static void Resolve32Mod1or2Base1(instruction_t *, guest_context_t *);
static void Resolve32Mod1or2Base2(instruction_t *, guest_context_t *);
static void Resolve32Mod1or2Base3(instruction_t *, guest_context_t *);
static void Resolve32Mod1or2Base4(instruction_t *, guest_context_t *);
static void Resolve32Mod1or2Base5(instruction_t *, guest_context_t *);
static void Resolve32Mod1or2Base6(instruction_t *, guest_context_t *);
static void Resolve32Mod1or2Base7(instruction_t *, guest_context_t *);


static ResolveModRM_t Resolve16Mod0[8] = {
  Resolve16Mod0Rm0,
  Resolve16Mod0Rm1,
  Resolve16Mod0Rm2,
  Resolve16Mod0Rm3,
  Resolve16Mod0Rm4,
  Resolve16Mod0Rm5,
  NULL, /* d16, no registers used */
  Resolve16Mod0Rm7
  };

static ResolveModRM_t Resolve16Mod1or2[8] = {
  Resolve16Mod1or2Rm0,
  Resolve16Mod1or2Rm1,
  Resolve16Mod1or2Rm2,
  Resolve16Mod1or2Rm3,
  Resolve16Mod1or2Rm4,
  Resolve16Mod1or2Rm5,
  Resolve16Mod1or2Rm6,
  Resolve16Mod1or2Rm7
  };

static ResolveModRM_t Resolve32Mod0[8] = {
  Resolve32Mod0Rm0,
  Resolve32Mod0Rm1,
  Resolve32Mod0Rm2,
  Resolve32Mod0Rm3,
  NULL, /* escape to 2-byte */
  NULL, /* d32, no registers used */
  Resolve32Mod0Rm6,
  Resolve32Mod0Rm7
  };

static ResolveModRM_t Resolve32Mod1or2[8] = {
  Resolve32Mod1or2Rm0,
  Resolve32Mod1or2Rm1,
  Resolve32Mod1or2Rm2,
  Resolve32Mod1or2Rm3,
  NULL, /* escape to 2-byte */
  Resolve32Mod1or2Rm5,
  Resolve32Mod1or2Rm6,
  Resolve32Mod1or2Rm7
  };

static ResolveModRM_t Resolve32Mod0Base[8] = {
  Resolve32Mod0Base0,
  Resolve32Mod0Base1,
  Resolve32Mod0Base2,
  Resolve32Mod0Base3,
  Resolve32Mod0Base4,
  Resolve32Mod0Base5,
  Resolve32Mod0Base6,
  Resolve32Mod0Base7,
  };

static ResolveModRM_t Resolve32Mod1or2Base[8] = {
  Resolve32Mod1or2Base0,
  Resolve32Mod1or2Base1,
  Resolve32Mod1or2Base2,
  Resolve32Mod1or2Base3,
  Resolve32Mod1or2Base4,
  Resolve32Mod1or2Base5,
  Resolve32Mod1or2Base6,
  Resolve32Mod1or2Base7,
  };


static const unsigned sreg_mod00_rm16[8] =
{
    SEG_REG_DS,
    SEG_REG_DS,
    SEG_REG_SS,
    SEG_REG_SS,
    SEG_REG_DS,
    SEG_REG_DS,
    SEG_REG_DS,
    SEG_REG_DS
};


static const unsigned sreg_mod01_rm16[8] =
{
    SEG_REG_DS,
    SEG_REG_DS,
    SEG_REG_SS,
    SEG_REG_SS,
    SEG_REG_DS,
    SEG_REG_DS,
    SEG_REG_SS,
    SEG_REG_DS
};


static const unsigned sreg_mod10_rm16[8] =
{
    SEG_REG_DS,
    SEG_REG_DS,
    SEG_REG_SS,
    SEG_REG_SS,
    SEG_REG_DS,
    SEG_REG_DS,
    SEG_REG_SS,
    SEG_REG_DS
};


/*
 *  the default segment to use for a one-byte modrm with mod==01b
 *  and rm==i
 */

static const unsigned sreg_mod01_rm32[8] =
{
    SEG_REG_DS,
    SEG_REG_DS,
    SEG_REG_DS,
    SEG_REG_DS,
    SEG_REG_NULL,  /* this entry should never be accessed (escape to 2-byte) */
    SEG_REG_SS,
    SEG_REG_DS,
    SEG_REG_DS
};


/*
 *  the default segment to use for a one-byte modrm with mod==10b
 *  and rm==i
 */

static const unsigned sreg_mod10_rm32[8] =
{
    SEG_REG_DS,
    SEG_REG_DS,
    SEG_REG_DS,
    SEG_REG_DS,
    SEG_REG_NULL,  /* this entry should never be accessed (escape to 2-byte) */
    SEG_REG_SS,
    SEG_REG_DS,
    SEG_REG_DS
};


/*
 *  the default segment to use for a two-byte modrm with mod==00b
 *  and base==i
 */

static const unsigned sreg_mod0_base32[8] =
{
    SEG_REG_DS,
    SEG_REG_DS,
    SEG_REG_DS,
    SEG_REG_DS,
    SEG_REG_SS,
    SEG_REG_DS,
    SEG_REG_DS,
    SEG_REG_DS
};


/*
 *  the default segment to use for a two-byte modrm with
 *  mod==01b or mod==10b and base==i
 */

static const unsigned sreg_mod1or2_base32[8] =
{
    SEG_REG_DS,
    SEG_REG_DS,
    SEG_REG_DS,
    SEG_REG_DS,
    SEG_REG_SS,
    SEG_REG_SS,
    SEG_REG_DS,
    SEG_REG_DS
};


static Bit16u groupNattr[NumGroups][8] =
{
/* G1EbIb */
    {
        /* 0 */ Immediate_Ib,        /* ADD_EbIb   */
        /* 1 */ Immediate_Ib,        /* OR_EbIb    */
        /* 2 */ Immediate_Ib,        /* ADC_EbIb   */
        /* 3 */ Immediate_Ib,        /* SBB_EbIb   */
        /* 4 */ Immediate_Ib,        /* AND_EbIb   */
        /* 5 */ Immediate_Ib,        /* SUB_EbIb   */
        /* 6 */ Immediate_Ib,        /* XOR_EbIb   */
        /* 7 */ Immediate_Ib,        /* CMP_EbIb   */
    },

/* G1Ew (attributes defined in main area) */
    {
        /* 0 */ 0,                /* ADD_EwIw   */
        /* 1 */ 0,                /* OR_EwIw    */
        /* 2 */ 0,                /* ADC_EwIw   */
        /* 3 */ 0,                /* SBB_EwIw   */
        /* 4 */ 0,                /* AND_EwIw   */
        /* 5 */ 0,                /* SUB_EwIw   */
        /* 6 */ 0,                /* XOR_EwIw   */
        /* 7 */ 0,                /* CMP_EwIw   */
    },

/* G1Ed (attributes defined in main area) */
    {
        /* 0 */ 0,                /* ADD_EdId   */
        /* 1 */ 0,                /* OR_EdId    */
        /* 2 */ 0,                /* ADC_EdId   */
        /* 3 */ 0,                /* SBB_EdId   */
        /* 4 */ 0,                /* AND_EdId   */
        /* 5 */ 0,                /* SUB_EdId   */
        /* 6 */ 0,                /* XOR_EdId   */
        /* 7 */ 0,                /* CMP_EdId   */
    },

/* G2Eb (attributes defined in main area) */
    {
        /* 0 */ 0,                /* ROL_Eb     */
        /* 1 */ 0,                /* ROR_Eb     */
        /* 2 */ 0,                /* RCL_Eb     */
        /* 3 */ 0,                /* RCR_Eb     */
        /* 4 */ 0,                /* SHL_Eb     */
        /* 5 */ 0,                /* SHR_Eb     */
        /* 6 */ 0,                /* SHL_Eb     */
        /* 7 */ 0,                /* SAR_Eb     */
    },

/* G2Ew (attributes defined in main area) */
    {
        /* 0 */ 0,                /* ROL_Ew     */
        /* 1 */ 0,                /* ROR_Ew     */
        /* 2 */ 0,                /* RCL_Ew     */
        /* 3 */ 0,                /* RCR_Ew     */
        /* 4 */ 0,                /* SHL_Ew     */
        /* 5 */ 0,                /* SHR_Ew     */
        /* 6 */ 0,                /* SHL_Ew     */
        /* 7 */ 0,                /* SAR_Ew     */
    },

/* G2Ed (attributes defined in main area) */
    {
        /* 0 */ 0,                /* ROL_Ed     */
        /* 1 */ 0,                /* ROR_Ed     */
        /* 2 */ 0,                /* RCL_Ed     */
        /* 3 */ 0,                /* RCR_Ed     */
        /* 4 */ 0,                /* SHL_Ed     */
        /* 5 */ 0,                /* SHR_Ed     */
        /* 6 */ 0,                /* SHL_Ed     */
        /* 7 */ 0,                /* SAR_Ed     */
    },

/* G3Eb */
    {
        /* 0 */ Immediate_Ib,        /* TEST_EbIb  */
        /* 1 */ Immediate_Ib,        /* TEST_EbIb  */
        /* 2 */ 0,                /* NOT_Eb     */
        /* 3 */ 0,                /* NEG_Eb     */
        /* 4 */ 0,                /* MUL_ALEb   */
        /* 5 */ 0,                /* IMUL_ALEb  */
        /* 6 */ 0,                /* DIV_ALEb   */
        /* 7 */ 0,                /* IDIV_ALEb  */
    },

/* G3Ew */
    {
        /* 0 */ Immediate_Iw,        /* TEST_EwIw  */
        /* 1 */ Immediate_Iw,        /* TEST_EwIw  */
        /* 2 */ 0,                /* NOT_Ew     */
        /* 3 */ 0,                /* NEG_Ew     */
        /* 4 */ 0,                /* MUL_AXEw   */
        /* 5 */ 0,                /* IMUL_AXEw  */
        /* 6 */ 0,                /* DIV_AXEw   */
        /* 7 */ 0,                /* IDIV_AXEw  */
    },

/* G3Ed */
    {
        /* 0 */ Immediate_Iv,        /* TEST_EdId  */
        /* 1 */ Immediate_Iv,        /* TEST_EdId  */
        /* 2 */ 0,                /* NOT_Ed     */
        /* 3 */ 0,                /* NEG_Ed     */
        /* 4 */ 0,                /* MUL_EAXEd  */
        /* 5 */ 0,                /* IMUL_EAXEd */
        /* 6 */ 0,                /* DIV_EAXEd  */
        /* 7 */ 0,                /* IDIV_EAXEd */
    },

/* G4 */
    {
        /* 0 */ 0,                /* INC_Eb     */
        /* 1 */ 0,                /* DEC_Eb     */
        /* 2 */ InvalidOpcode,
        /* 3 */ InvalidOpcode,
        /* 4 */ InvalidOpcode,
        /* 5 */ InvalidOpcode,
        /* 6 */ InvalidOpcode,
        /* 7 */ InvalidOpcode,
    },

/* G5w (attributes defined in main area) */
    {
        /* 0 */ 0,                /* INC_Ew     */
        /* 1 */ 0,                /* DEC_Ew     */
        /* 2 */ 0,                /* CALL_Ew    */
        /* 3 */ 0,                /* CALL16_Ep  */
        /* 4 */ 0 | UCBranch,        /* JMP_Ew     */
        /* 5 */ 0 | UCBranch,        /* JMP16_Ep   */
        /* 6 */ 0,                /* PUSH_Ew    */
        /* 7 */ InvalidOpcode,
    },

/* G5d (attributes defined in main area) */
    {
        /* 0 */ 0,                /* INC_Ed     */
        /* 1 */ 0,                /* DEC_Ed     */
        /* 2 */ 0,                /* CALL_Ed    */
        /* 3 */ 0,                /* CALL32_Ep  */
        /* 4 */ 0 | UCBranch,        /* JMP_Ed     */
        /* 5 */ 0 | UCBranch,        /* JMP32_Ep   */
        /* 6 */ 0,                /* PUSH_Ed    */
        /* 7 */ InvalidOpcode,
    },

/* G6 (attributes defined in main area) */
    {
        /* 0 */ 0,                /* SLDT_Ew    */
        /* 1 */ 0,                /* STR_Ew     */
        /* 2 */ 0,                /* LLDT_Ew    */
        /* 3 */ 0,                /* LTR_Ew     */
        /* 4 */ 0,                /* VERR_Ew    */
        /* 5 */ 0,                /* VERW_Ew    */
        /* 6 */ InvalidOpcode,
        /* 7 */ InvalidOpcode,
    },

/* G7 */
    {
        /* 0 */ 0,                /* SGDT_Ms    */
        /* 1 */ 0,                /* SIDT_Ms    */
        /* 2 */ 0,                /* LGDT_Ms    */
        /* 3 */ 0,                /* LIDT_Ms    */
        /* 4 */ 0,                /* SMSW_Ew    */
        /* 5 */ InvalidOpcode,
        /* 6 */ 0,                /* LMSW_Ew    */
        /* 7 */ 0,                /* INVLPG     */
    },

/* G8EvIb */
    {
        /* 0 */ InvalidOpcode,
        /* 1 */ InvalidOpcode,
        /* 2 */ InvalidOpcode,
        /* 3 */ InvalidOpcode,
        /* 4 */ Immediate_Ib,        /* BT_EvIb    */
        /* 5 */ Immediate_Ib,        /* BTS_EvIb   */
        /* 6 */ Immediate_Ib,        /* BTR_EvIb   */
        /* 7 */ Immediate_Ib,        /* BTC_EvIb   */
    },

/* G9 */
    {
        /* 0 */ InvalidOpcode,
        /* 1 */ 0,                /* CMPXCHG8B  */
        /* 2 */ InvalidOpcode,
        /* 3 */ InvalidOpcode,
        /* 4 */ InvalidOpcode,
        /* 5 */ InvalidOpcode,
        /* 6 */ InvalidOpcode,
        /* 7 */ InvalidOpcode,
    },
};


/*
 *  512 entries for 16bit mode
 *  512 entries for 32bit mode
 */

static Bit16u OpcodeInfo[512 * 2] =
{
/* 512 entries for 16bit mode */
    /* 00 */ Another,                        /* ADD_EbGb                */
    /* 01 */ Another,                        /* ADD_EwGw                */
    /* 02 */ Another,                        /* ADD_GbEb                */
    /* 03 */ Another,                        /* ADD_GwEw                */
    /* 04 */ Immediate_Ib,                /* ADD_ALIb                */
    /* 05 */ Immediate_Iv,                /* ADD_AXIw                */
    /* 06 */ 0,                                /* PUSH_ES                */
    /* 07 */ 0,                                /* POP_ES                */
    /* 08 */ Another,                        /* OR_EbGb                */
    /* 09 */ Another,                        /* OR_EwGw                */
    /* 0A */ Another,                        /* OR_GbEb                */
    /* 0B */ Another,                        /* OR_GwEw                */
    /* 0C */ Immediate_Ib,                /* OR_ALIb                */
    /* 0D */ Immediate_Iv,                /* OR_AXIw                */
    /* 0E */ 0,                                /* PUSH_CS                */
    /* 0F */ Another,                        /* 2-byte escape        */
    /* 10 */ Another,                        /* ADC_EbGb                */
    /* 11 */ Another,                        /* ADC_EwGw                */
    /* 12 */ Another,                        /* ADC_GbEb                */
    /* 13 */ Another,                        /* ADC_GwEw                */
    /* 14 */ Immediate_Ib,                /* ADC_ALIb                */
    /* 15 */ Immediate_Iv,                /* ADC_AXIw                */
    /* 16 */ 0,                                /* PUSH_SS                */
    /* 17 */ 0,                                /* POP_SS                */
    /* 18 */ Another,                        /* SBB_EbGb                */
    /* 19 */ Another,                        /* SBB_EwGw                */
    /* 1A */ Another,                        /* SBB_GbEb                */
    /* 1B */ Another,                        /* SBB_GwEw                */
    /* 1C */ Immediate_Ib,                /* SBB_ALIb                */
    /* 1D */ Immediate_Iv,                /* SBB_AXIw                */
    /* 1E */ 0,                                /* PUSH_DS                */
    /* 1F */ 0,                                /* POP_DS                */
    /* 20 */ Another,                        /* AND_EbGb                */
    /* 21 */ Another,                        /* AND_EwGw                */
    /* 22 */ Another,                        /* AND_GbEb                */
    /* 23 */ Another,                        /* AND_GwEw                */
    /* 24 */ Immediate_Ib,                /* AND_ALIb                */
    /* 25 */ Immediate_Iv,                /* AND_AXIw                */
    /* 26 */ Prefix | Another,                /* ES:                        */
    /* 27 */ 0,                                /* DAA                        */
    /* 28 */ Another,                        /* SUB_EbGb                */
    /* 29 */ Another,                        /* SUB_EwGw                */
    /* 2A */ Another,                        /* SUB_GbEb                */
    /* 2B */ Another,                        /* SUB_GwEw                */
    /* 2C */ Immediate_Ib,                /* SUB_ALIb                */
    /* 2D */ Immediate_Iv,                /* SUB_AXIw                */
    /* 2E */ Prefix | Another,                /* CS:                        */
    /* 2F */ 0,                                /* DAS                        */
    /* 30 */ Another,                        /* XOR_EbGb                */
    /* 31 */ Another,                        /* XOR_EwGw                */
    /* 32 */ Another,                        /* XOR_GbEb                */
    /* 33 */ Another,                        /* XOR_GwEw                */
    /* 34 */ Immediate_Ib,                /* XOR_ALIb                */
    /* 35 */ Immediate_Iv,                /* XOR_AXIw                */
    /* 36 */ Prefix | Another,                /* SS:                        */
    /* 37 */ 0,                                /* AAA                        */
    /* 38 */ Another,                        /* CMP_EbGb                */
    /* 39 */ Another,                        /* CMP_EwGw                */
    /* 3A */ Another,                        /* CMP_GbEb                */
    /* 3B */ Another,                        /* CMP_GwEw                */
    /* 3C */ Immediate_Ib,                /* CMP_ALIb                */
    /* 3D */ Immediate_Iv,                /* CMP_AXIw                */
    /* 3E */ Prefix | Another,                /* DS:                        */
    /* 3F */ 0,                                /* AAS                        */
    /* 40 */ 0,                                /* INC_RX                */
    /* 41 */ 0,                                /* INC_RX                */
    /* 42 */ 0,                                /* INC_RX                */
    /* 43 */ 0,                                /* INC_RX                */
    /* 44 */ 0,                                /* INC_RX                */
    /* 45 */ 0,                                /* INC_RX                */
    /* 46 */ 0,                                /* INC_RX                */
    /* 47 */ 0,                                /* INC_RX                */
    /* 48 */ 0,                                /* DEC_RX                */
    /* 49 */ 0,                                /* DEC_RX                */
    /* 4A */ 0,                                /* DEC_RX                */
    /* 4B */ 0,                                /* DEC_RX                */
    /* 4C */ 0,                                /* DEC_RX                */
    /* 4D */ 0,                                /* DEC_RX                */
    /* 4E */ 0,                                /* DEC_RX                */
    /* 4F */ 0,                                /* DEC_RX                */
    /* 50 */ 0,                                /* PUSH_RX                */
    /* 51 */ 0,                                /* PUSH_RX                */
    /* 52 */ 0,                                /* PUSH_RX                */
    /* 53 */ 0,                                /* PUSH_RX                */
    /* 54 */ 0,                                /* PUSH_RX                */
    /* 55 */ 0,                                /* PUSH_RX                */
    /* 56 */ 0,                                /* PUSH_RX                */
    /* 57 */ 0,                                /* PUSH_RX                */
    /* 58 */ 0,                                /* POP_RX                */
    /* 59 */ 0,                                /* POP_RX                */
    /* 5A */ 0,                                /* POP_RX                */
    /* 5B */ 0,                                /* POP_RX                */
    /* 5C */ 0,                                /* POP_RX                */
    /* 5D */ 0,                                /* POP_RX                */
    /* 5E */ 0,                                /* POP_RX                */
    /* 5F */ 0,                                /* POP_RX                */
    /* 60 */ 0,                                /* PUSHAD16                */
    /* 61 */ 0,                                /* POPAD16                */
    /* 62 */ Another,                        /* BOUND_GvMa                */
    /* 63 */ Another,                        /* ARPL_EwGw                */
    /* 64 */ Prefix | Another,                /* FS:                        */
    /* 65 */ Prefix | Another,                /* GS:                        */
    /* 66 */ Prefix | Another,                /* OS:                        */
    /* 67 */ Prefix | Another,                /* AS:                        */
    /* 68 */ Immediate_Iv,                /* PUSH_Iw                */
    /* 69 */ Another | Immediate_Iv,        /* IMUL_GwEwIw                */
    /* 6A */ Immediate_Ib_SE,                /* PUSH_Iw                */
    /* 6B */ Another | Immediate_Ib_SE, /* IMUL_GwEwIw                */
    /* 6C */ Repeatable,                /* INSB_YbDX                */
    /* 6D */ Repeatable,                /* INSW_YvDX                */
    /* 6E */ Repeatable,                /* OUTSB_DXXb                */
    /* 6F */ Repeatable,                /* OUTSW_DXXv                */
    /* 70 */ Immediate_BrOff8,                /* JCC_Jw                */
    /* 71 */ Immediate_BrOff8,                /* JCC_Jw                */
    /* 72 */ Immediate_BrOff8,                /* JCC_Jw                */
    /* 73 */ Immediate_BrOff8,                /* JCC_Jw                */
    /* 74 */ Immediate_BrOff8,                /* JCC_Jw                */
    /* 75 */ Immediate_BrOff8,                /* JCC_Jw                */
    /* 76 */ Immediate_BrOff8,                /* JCC_Jw                */
    /* 77 */ Immediate_BrOff8,                /* JCC_Jw                */
    /* 78 */ Immediate_BrOff8,                /* JCC_Jw                */
    /* 79 */ Immediate_BrOff8,                /* JCC_Jw                */
    /* 7A */ Immediate_BrOff8,                /* JCC_Jw                */
    /* 7B */ Immediate_BrOff8,                /* JCC_Jw                */
    /* 7C */ Immediate_BrOff8,                /* JCC_Jw                */
    /* 7D */ Immediate_BrOff8,                /* JCC_Jw                */
    /* 7E */ Immediate_BrOff8,                /* JCC_Jw                */
    /* 7F */ Immediate_BrOff8,                /* JCC_Jw                */
    /* 80 */ Another | GroupN | PutGroupIndex (G1EbIb),
    /* 81 */ Another | GroupN | Immediate_Iv | PutGroupIndex (G1Ew),
    /* 82 */ Another | GroupN | PutGroupIndex (G1EbIb),
    /* 83 */ Another | GroupN | Immediate_Ib_SE | PutGroupIndex (G1Ew),
    /* 84 */ Another,                        /* TEST_EbGb                */
    /* 85 */ Another,                        /* TEST_EwGw                */
    /* 86 */ Another,                        /* XCHG_EbGb                */
    /* 87 */ Another,                        /* XCHG_EwGw                */
    /* 88 */ Another,                        /* MOV_EbGb                */
    /* 89 */ Another,                        /* MOV_EwGw                */
    /* 8A */ Another,                        /* MOV_GbEb                */
    /* 8B */ Another,                        /* MOV_GwEw                */
    /* 8C */ Another,                        /* MOV_EwSw                */
    /* 8D */ Another,                        /* LEA_GwM                */
    /* 8E */ Another,                        /* MOV_SwEw                */
    /* 8F */ Another,                        /* POP_Ew                */
    /* 90 */ 0,                                /* NOP                        */
    /* 91 */ 0,                                /* XCHG_RXAX                */
    /* 92 */ 0,                                /* XCHG_RXAX                */
    /* 93 */ 0,                                /* XCHG_RXAX                */
    /* 94 */ 0,                                /* XCHG_RXAX                */
    /* 95 */ 0,                                /* XCHG_RXAX                */
    /* 96 */ 0,                                /* XCHG_RXAX                */
    /* 97 */ 0,                                /* XCHG_RXAX                */
    /* 98 */ 0,                                /* CBW                        */
    /* 99 */ 0,                                /* CWD                        */
    /* 9A */ Immediate_IvIw,                /* CALL16_Ap                */
    /* 9B */ 0,                                /* FWAIT                */
    /* 9C */ 0,                                /* PUSHF_Fv                */
    /* 9D */ 0,                                /* POPF_Fv                */
    /* 9E */ 0,                                /* SAHF                        */
    /* 9F */ 0,                                /* LAHF                        */
    /* A0 */ Immediate_O,                /* MOV_ALOb                */
    /* A1 */ Immediate_O,                /* MOV_AXOw                */
    /* A2 */ Immediate_O,                /* MOV_ObAL                */
    /* A3 */ Immediate_O,                /* MOV_OwAX                */
    /* A4 */ Repeatable,                /* MOVSB_XbYb                */
    /* A5 */ Repeatable,                /* MOVSW_XvYv                */
    /* A6 */ Repeatable | RepeatableZF,        /* CMPSB_XbYb                */
    /* A7 */ Repeatable | RepeatableZF, /* CMPSW_XvYv                */
    /* A8 */ Immediate_Ib,                /* TEST_ALIb                */
    /* A9 */ Immediate_Iv,                /* TEST_AXIw                */
    /* AA */ Repeatable,                /* STOSB_YbAL                */
    /* AB */ Repeatable,                /* STOSW_YveAX                */
    /* AC */ Repeatable,                /* LODSB_ALXb                */
    /* AD */ Repeatable,                /* LODSW_eAXXv                */
    /* AE */ Repeatable | RepeatableZF,        /* SCASB_ALXb                */
    /* AF */ Repeatable | RepeatableZF,        /* SCASW_eAXXv                */
    /* B0 */ Immediate_Ib,                /* MOV_RLIb                */
    /* B1 */ Immediate_Ib,                /* MOV_RLIb                */
    /* B2 */ Immediate_Ib,                /* MOV_RLIb                */
    /* B3 */ Immediate_Ib,                /* MOV_RLIb                */
    /* B4 */ Immediate_Ib,                /* MOV_RHIb                */
    /* B5 */ Immediate_Ib,                /* MOV_RHIb                */
    /* B6 */ Immediate_Ib,                /* MOV_RHIb                */
    /* B7 */ Immediate_Ib,                /* MOV_RHIb                */
    /* B8 */ Immediate_Iv,                /* MOV_RXIw                */
    /* B9 */ Immediate_Iv,                /* MOV_RXIw                */
    /* BA */ Immediate_Iv,                /* MOV_RXIw                */
    /* BB */ Immediate_Iv,                /* MOV_RXIw                */
    /* BC */ Immediate_Iv,                /* MOV_RXIw                */
    /* BD */ Immediate_Iv,                /* MOV_RXIw                */
    /* BE */ Immediate_Iv,                /* MOV_RXIw                */
    /* BF */ Immediate_Iv,                /* MOV_RXIw                */
    /* C0 */ Another | GroupN | Immediate_Ib | PutGroupIndex (G2Eb),
    /* C1 */ Another | GroupN | Immediate_Ib | PutGroupIndex (G2Ew),
    /* C2 */ Immediate_Iw | UCBranch,        /* RETnear16_Iw                */
    /* C3 */ 0 | UCBranch,                /* RETnear16                */
    /* C4 */ Another,                        /* LES_GvMp                */
    /* C5 */ Another,                        /* LDS_GvMp                */
    /* C6 */ Another | Immediate_Ib,        /* MOV_EbIb                */
    /* C7 */ Another | Immediate_Iv,        /* MOV_EwIw                */
    /* C8 */ Immediate_IwIb,                /* ENTER_IwIb                */
    /* C9 */ 0,                                /* LEAVE                */
    /* CA */ Immediate_Iw | UCBranch,        /* RETfar16_Iw                */
    /* CB */ 0 | UCBranch,                /* RETfar16                */
    /* CC */ 0,                                /* INT3                        */
    /* CD */ Immediate_Ib,                /* INT_Ib                */
    /* CE */ 0,                                /* INTO                        */
    /* CF */ 0 | UCBranch,                /* IRET16                */
    /* D0 */ Another | GroupN | PutGroupIndex (G2Eb),
    /* D1 */ Another | GroupN | PutGroupIndex (G2Ew),
    /* D2 */ Another | GroupN | PutGroupIndex (G2Eb),
    /* D3 */ Another | GroupN | PutGroupIndex (G2Ew),
    /* D4 */ Immediate_Ib,                /* AAM                        */
    /* D5 */ Immediate_Ib,                /* AAD                        */
    /* D6 */ 0,                                /* SALC                        */
    /* D7 */ 0,                                /* XLAT                        */
    /* D8 */ Another,                        /* ESC0                        */
    /* D9 */ Another,                        /* ESC1                        */
    /* DA */ Another,                        /* ESC2                        */
    /* DB */ Another,                        /* ESC3                        */
    /* DC */ Another,                        /* ESC4                        */
    /* DD */ Another,                        /* ESC5                        */
    /* DE */ Another,                        /* ESC6                        */
    /* DF */ Another,                        /* ESC7                        */
    /* E0 */ Immediate_BrOff8,                /* LOOPNE_Jb                */
    /* E1 */ Immediate_BrOff8,                /* LOOPE_Jb                */
    /* E2 */ Immediate_BrOff8,                /* LOOP_Jb                */
    /* E3 */ Immediate_BrOff8,                /* JCXZ_Jb                */
    /* E4 */ Immediate_Ib,                /* IN_ALIb                */
    /* E5 */ Immediate_Ib,                /* IN_eAXIb                */
    /* E6 */ Immediate_Ib,                /* OUT_IbAL                */
    /* E7 */ Immediate_Ib,                /* OUT_IbeAX                */
    /* E8 */ Immediate_BrOff16,                /* CALL_Aw                */
    /* E9 */ Immediate_BrOff16 | UCBranch,        /* JMP_Jw        */
    /* EA */ Immediate_IvIw | UCBranch,                /* JMP_Ap        */
    /* EB */ Immediate_BrOff8 | UCBranch,        /* JMP_Jb        */
    /* EC */ 0,                                /* IN_ALDX                */
    /* ED */ 0,                                /* IN_eAXDX                */
    /* EE */ 0,                                /* OUT_DXAL                */
    /* EF */ 0,                                /* OUT_DXeAX                */
    /* F0 */ Prefix | Another,                /* LOCK:                */
    /* F1 */ 0,                                /* INT1                        */
    /* F2 */ Prefix | Another,                /* REPNE/REPNZ:                */
    /* F3 */ Prefix | Another,                /* REP, REPE/REPZ:        */
    /* F4 */ 0,                                /* HLT                        */
    /* F5 */ 0,                                /* CMC                        */
    /* F6 */ Another | GroupN | PutGroupIndex (G3Eb),
    /* F7 */ Another | GroupN | PutGroupIndex (G3Ew),
    /* F8 */ 0,                                /* CLC                        */
    /* F9 */ 0,                                /* STC                        */
    /* FA */ 0,                                /* CLI                        */
    /* FB */ 0,                                /* STI                        */
    /* FC */ 0,                                /* CLD                        */
    /* FD */ 0,                                /* STD                        */
    /* FE */ Another | GroupN | PutGroupIndex (G4),
    /* FF */ Another | GroupN | PutGroupIndex (G5w),

    /* 0F 00 */ Another | GroupN | PutGroupIndex (G6),
    /* 0F 01 */ Another | GroupN | PutGroupIndex (G7),
    /* 0F 02 */ Another,                /* LAR_GvEw                */
    /* 0F 03 */ Another,                /* LSL_GvEw                */
    /* 0F 04 */ InvalidOpcode,
    /* 0F 05 */ 0,                        /* LOADALL                */
    /* 0F 06 */ 0,                        /* CLTS                        */
    /* 0F 07 */ InvalidOpcode,
    /* 0F 08 */ 0,                        /* INVD                        */
    /* 0F 09 */ 0,                        /* WBINVD                */
    /* 0F 0A */ InvalidOpcode,
    /* 0F 0B */ InvalidOpcode,
    /* 0F 0C */ InvalidOpcode,
    /* 0F 0D */ InvalidOpcode,
    /* 0F 0E */ InvalidOpcode,
    /* 0F 0F */ InvalidOpcode,
    /* 0F 10 */ InvalidOpcode,
    /* 0F 11 */ InvalidOpcode,
    /* 0F 12 */ InvalidOpcode,
    /* 0F 13 */ InvalidOpcode,
    /* 0F 14 */ InvalidOpcode,
    /* 0F 15 */ InvalidOpcode,
    /* 0F 16 */ InvalidOpcode,
    /* 0F 17 */ InvalidOpcode,
    /* 0F 18 */ InvalidOpcode,
    /* 0F 19 */ InvalidOpcode,
    /* 0F 1A */ InvalidOpcode,
    /* 0F 1B */ InvalidOpcode,
    /* 0F 1C */ InvalidOpcode,
    /* 0F 1D */ InvalidOpcode,
    /* 0F 1E */ InvalidOpcode,
    /* 0F 1F */ InvalidOpcode,
    /* 0F 20 */ Another,                /* MOV_RdCd                */
    /* 0F 21 */ Another,                /* MOV_RdDd                */
    /* 0F 22 */ Another,                /* MOV_CdRd                */
    /* 0F 23 */ Another,                /* MOV_DdRd                */
    /* 0F 24 */ Another,                /* MOV_RdTd                */
    /* 0F 25 */ InvalidOpcode,
    /* 0F 26 */ Another,                /* MOV_TdRd                */
    /* 0F 27 */ InvalidOpcode,
    /* 0F 28 */ InvalidOpcode,
    /* 0F 29 */ InvalidOpcode,
    /* 0F 2A */ InvalidOpcode,
    /* 0F 2B */ InvalidOpcode,
    /* 0F 2C */ InvalidOpcode,
    /* 0F 2D */ InvalidOpcode,
    /* 0F 2E */ InvalidOpcode,
    /* 0F 2F */ InvalidOpcode,
    /* 0F 30 */ 0,                        /* WRMSR                */
    /* 0F 31 */ 0,                        /* RDTSC                */
    /* 0F 32 */ 0,                        /* RDMSR                */
    /* 0F 33 */ InvalidOpcode,
    /* 0F 34 */ InvalidOpcode,
    /* 0F 35 */ InvalidOpcode,
    /* 0F 36 */ InvalidOpcode,
    /* 0F 37 */ InvalidOpcode,
    /* 0F 38 */ InvalidOpcode,
    /* 0F 39 */ InvalidOpcode,
    /* 0F 3A */ InvalidOpcode,
    /* 0F 3B */ InvalidOpcode,
    /* 0F 3C */ InvalidOpcode,
    /* 0F 3D */ InvalidOpcode,
    /* 0F 3E */ InvalidOpcode,
    /* 0F 3F */ InvalidOpcode,
    /* 0F 40 */ Another,                /* CMOV_GwEw                */
    /* 0F 41 */ Another,                /* CMOV_GwEw                */
    /* 0F 42 */ Another,                /* CMOV_GwEw                */
    /* 0F 43 */ Another,                /* CMOV_GwEw                */
    /* 0F 44 */ Another,                /* CMOV_GwEw                */
    /* 0F 45 */ Another,                /* CMOV_GwEw                */
    /* 0F 46 */ Another,                /* CMOV_GwEw                */
    /* 0F 47 */ Another,                /* CMOV_GwEw                */
    /* 0F 48 */ Another,                /* CMOV_GwEw                */
    /* 0F 49 */ Another,                /* CMOV_GwEw                */
    /* 0F 4A */ Another,                /* CMOV_GwEw                */
    /* 0F 4B */ Another,                /* CMOV_GwEw                */
    /* 0F 4C */ Another,                /* CMOV_GwEw                */
    /* 0F 4D */ Another,                /* CMOV_GwEw                */
    /* 0F 4E */ Another,                /* CMOV_GwEw                */
    /* 0F 4F */ Another,                /* CMOV_GwEw                */
    /* 0F 50 */ InvalidOpcode,
    /* 0F 51 */ InvalidOpcode,
    /* 0F 52 */ InvalidOpcode,
    /* 0F 53 */ InvalidOpcode,
    /* 0F 54 */ InvalidOpcode,
    /* 0F 55 */ InvalidOpcode,
    /* 0F 56 */ InvalidOpcode,
    /* 0F 57 */ InvalidOpcode,
    /* 0F 58 */ InvalidOpcode,
    /* 0F 59 */ InvalidOpcode,
    /* 0F 5A */ InvalidOpcode,
    /* 0F 5B */ InvalidOpcode,
    /* 0F 5C */ InvalidOpcode,
    /* 0F 5D */ InvalidOpcode,
    /* 0F 5E */ InvalidOpcode,
    /* 0F 5F */ InvalidOpcode,
    /* 0F 60 */ InvalidOpcode,
    /* 0F 61 */ InvalidOpcode,
    /* 0F 62 */ InvalidOpcode,
    /* 0F 63 */ InvalidOpcode,
    /* 0F 64 */ InvalidOpcode,
    /* 0F 65 */ InvalidOpcode,
    /* 0F 66 */ InvalidOpcode,
    /* 0F 67 */ InvalidOpcode,
    /* 0F 68 */ InvalidOpcode,
    /* 0F 69 */ InvalidOpcode,
    /* 0F 6A */ InvalidOpcode,
    /* 0F 6B */ InvalidOpcode,
    /* 0F 6C */ InvalidOpcode,
    /* 0F 6D */ InvalidOpcode,
    /* 0F 6E */ InvalidOpcode,
    /* 0F 6F */ InvalidOpcode,
    /* 0F 70 */ InvalidOpcode,
    /* 0F 71 */ InvalidOpcode,
    /* 0F 72 */ InvalidOpcode,
    /* 0F 73 */ InvalidOpcode,
    /* 0F 74 */ InvalidOpcode,
    /* 0F 75 */ InvalidOpcode,
    /* 0F 76 */ InvalidOpcode,
    /* 0F 77 */ InvalidOpcode,
    /* 0F 78 */ InvalidOpcode,
    /* 0F 79 */ InvalidOpcode,
    /* 0F 7A */ InvalidOpcode,
    /* 0F 7B */ InvalidOpcode,
    /* 0F 7C */ InvalidOpcode,
    /* 0F 7D */ InvalidOpcode,
    /* 0F 7E */ InvalidOpcode,
    /* 0F 7F */ InvalidOpcode,
    /* 0F 80 */ Immediate_BrOff16,        /* JCC_Jw                */
    /* 0F 81 */ Immediate_BrOff16,        /* JCC_Jw                */
    /* 0F 82 */ Immediate_BrOff16,        /* JCC_Jw                */
    /* 0F 83 */ Immediate_BrOff16,        /* JCC_Jw                */
    /* 0F 84 */ Immediate_BrOff16,        /* JCC_Jw                */
    /* 0F 85 */ Immediate_BrOff16,        /* JCC_Jw                */
    /* 0F 86 */ Immediate_BrOff16,        /* JCC_Jw                */
    /* 0F 87 */ Immediate_BrOff16,        /* JCC_Jw                */
    /* 0F 88 */ Immediate_BrOff16,        /* JCC_Jw                */
    /* 0F 89 */ Immediate_BrOff16,        /* JCC_Jw                */
    /* 0F 8A */ Immediate_BrOff16,        /* JCC_Jw                */
    /* 0F 8B */ Immediate_BrOff16,        /* JCC_Jw                */
    /* 0F 8C */ Immediate_BrOff16,        /* JCC_Jw                */
    /* 0F 8D */ Immediate_BrOff16,        /* JCC_Jw                */
    /* 0F 8E */ Immediate_BrOff16,        /* JCC_Jw                */
    /* 0F 8F */ Immediate_BrOff16,        /* JCC_Jw                */
    /* 0F 90 */ Another,                /* SETO_Eb                */
    /* 0F 91 */ Another,                /* SETNO_Eb                */
    /* 0F 92 */ Another,                /* SETB_Eb                */
    /* 0F 93 */ Another,                /* SETNB_Eb                */
    /* 0F 94 */ Another,                /* SETZ_Eb                */
    /* 0F 95 */ Another,                /* SETNZ_Eb                */
    /* 0F 96 */ Another,                /* SETBE_Eb                */
    /* 0F 97 */ Another,                /* SETNBE_Eb                */
    /* 0F 98 */ Another,                /* SETS_Eb                */
    /* 0F 99 */ Another,                /* SETNS_Eb                */
    /* 0F 9A */ Another,                /* SETP_Eb                */
    /* 0F 9B */ Another,                /* SETNP_Eb                */
    /* 0F 9C */ Another,                /* SETL_Eb                */
    /* 0F 9D */ Another,                /* SETNL_Eb                */
    /* 0F 9E */ Another,                /* SETLE_Eb                */
    /* 0F 9F */ Another,                /* SETNLE_Eb                */
    /* 0F A0 */ 0,                        /* PUSH_FS                */
    /* 0F A1 */ 0,                        /* POP_FS                */
    /* 0F A2 */ 0,                        /* CPUID                */
    /* 0F A3 */ Another,                /* BT_EvGv                */
    /* 0F A4 */ Another | Immediate_Ib,        /* SHLD_EwGw                */
    /* 0F A5 */ Another,                /* SHLD_EwGw                */
    /* 0F A6 */ 0,                        /* CMPXCHG_XBTS                */
    /* 0F A7 */ 0,                        /* CMPXCHG_IBTS                */
    /* 0F A8 */ 0,                        /* PUSH_GS                */
    /* 0F A9 */ 0,                        /* POP_GS                */
    /* 0F AA */ 0,                        /* RSM                        */
    /* 0F AB */ Another,                /* BTS_EvGv                */
    /* 0F AC */ Another | Immediate_Ib,        /* SHRD_EwGw                */
    /* 0F AD */ Another,                /* SHRD_EwGw                */
    /* 0F AE */ InvalidOpcode,
    /* 0F AF */ Another,                /* IMUL_GwEw                */
    /* 0F B0 */ Another,                /* CMPXCHG_EbGb                */
    /* 0F B1 */ Another,                /* CMPXCHG_EwGw                */
    /* 0F B2 */ Another,                /* LSS_GvMp                */
    /* 0F B3 */ Another,                /* BTR_EvGv                */
    /* 0F B4 */ Another,                /* LFS_GvMp                */
    /* 0F B5 */ Another,                /* LGS_GvMp                */
    /* 0F B6 */ Another,                /* MOVZX_GwEb                */
    /* 0F B7 */ Another,                /* MOVZX_GwEw                */
    /* 0F B8 */ InvalidOpcode,
    /* 0F B9 */ InvalidOpcode,
    /* 0F BA */ Another | GroupN | PutGroupIndex (G8EvIb),
    /* 0F BB */ Another,                /* BTC_EvGv                */
    /* 0F BC */ Another,                /* BSF_GvEv                */
    /* 0F BD */ Another,                /* BSR_GvEv                */
    /* 0F BE */ Another,                /* MOVSX_GwEb                */
    /* 0F BF */ Another,                /* MOVSX_GwEw                */
    /* 0F C0 */ Another,                /* XADD_EbGb                */
    /* 0F C1 */ Another,                /* XADD_EwGw                */
    /* 0F C2 */ InvalidOpcode,
    /* 0F C3 */ InvalidOpcode,
    /* 0F C4 */ InvalidOpcode,
    /* 0F C5 */ InvalidOpcode,
    /* 0F C6 */ InvalidOpcode,
    /* 0F C7 */ Another | GroupN | PutGroupIndex (G9),
    /* 0F C8 */ 0,                        /* BSWAP_EAX                */
    /* 0F C9 */ 0,                        /* BSWAP_ECX                */
    /* 0F CA */ 0,                        /* BSWAP_EDX                */
    /* 0F CB */ 0,                        /* BSWAP_EBX                */
    /* 0F CC */ 0,                        /* BSWAP_ESP                */
    /* 0F CD */ 0,                        /* BSWAP_EBP                */
    /* 0F CE */ 0,                        /* BSWAP_ESI                */
    /* 0F CF */ 0,                        /* BSWAP_EDI                */
    /* 0F D0 */ InvalidOpcode,
    /* 0F D1 */ InvalidOpcode,
    /* 0F D2 */ InvalidOpcode,
    /* 0F D3 */ InvalidOpcode,
    /* 0F D4 */ InvalidOpcode,
    /* 0F D5 */ InvalidOpcode,
    /* 0F D6 */ InvalidOpcode,
    /* 0F D7 */ InvalidOpcode,
    /* 0F D8 */ InvalidOpcode,
    /* 0F D9 */ InvalidOpcode,
    /* 0F DA */ InvalidOpcode,
    /* 0F DB */ InvalidOpcode,
    /* 0F DC */ InvalidOpcode,
    /* 0F DD */ InvalidOpcode,
    /* 0F DE */ InvalidOpcode,
    /* 0F DF */ InvalidOpcode,
    /* 0F E0 */ InvalidOpcode,
    /* 0F E1 */ InvalidOpcode,
    /* 0F E2 */ InvalidOpcode,
    /* 0F E3 */ InvalidOpcode,
    /* 0F E4 */ InvalidOpcode,
    /* 0F E5 */ InvalidOpcode,
    /* 0F E6 */ InvalidOpcode,
    /* 0F E7 */ InvalidOpcode,
    /* 0F E8 */ InvalidOpcode,
    /* 0F E9 */ InvalidOpcode,
    /* 0F EA */ InvalidOpcode,
    /* 0F EB */ InvalidOpcode,
    /* 0F EC */ InvalidOpcode,
    /* 0F ED */ InvalidOpcode,
    /* 0F EE */ InvalidOpcode,
    /* 0F EF */ InvalidOpcode,
    /* 0F F0 */ InvalidOpcode,
    /* 0F F1 */ InvalidOpcode,
    /* 0F F2 */ InvalidOpcode,
    /* 0F F3 */ InvalidOpcode,
    /* 0F F4 */ InvalidOpcode,
    /* 0F F5 */ InvalidOpcode,
    /* 0F F6 */ InvalidOpcode,
    /* 0F F7 */ InvalidOpcode,
    /* 0F F8 */ InvalidOpcode,
    /* 0F F9 */ InvalidOpcode,
    /* 0F FA */ InvalidOpcode,
    /* 0F FB */ InvalidOpcode,
    /* 0F FC */ InvalidOpcode,
    /* 0F FD */ InvalidOpcode,
    /* 0F FE */ InvalidOpcode,
    /* 0F FF */ InvalidOpcode,

/* 512 entries for 32bit mod */
    /* 00 */ Another,                        /* ADD_EbGb                */
    /* 01 */ Another,                        /* ADD_EdGd                */
    /* 02 */ Another,                        /* ADD_GbEb                */
    /* 03 */ Another,                        /* ADD_GdEd                */
    /* 04 */ Immediate_Ib,                /* ADD_ALIb                */
    /* 05 */ Immediate_Iv,                /* ADD_EAXId                */
    /* 06 */ 0,                                /* PUSH_ES                */
    /* 07 */ 0,                                /* POP_ES                */
    /* 08 */ Another,                        /* OR_EbGb                */
    /* 09 */ Another,                        /* OR_EdGd                */
    /* 0A */ Another,                        /* OR_GbEb                */
    /* 0B */ Another,                        /* OR_GdEd                */
    /* 0C */ Immediate_Ib,                /* OR_ALIb                */
    /* 0D */ Immediate_Iv,                /* OR_EAXId                */
    /* 0E */ 0,                                /* PUSH_CS                */
    /* 0F */ Another,                        /* 2-byte escape        */
    /* 10 */ Another,                        /* ADC_EbGb                */
    /* 11 */ Another,                        /* ADC_EdGd                */
    /* 12 */ Another,                        /* ADC_GbEb                */
    /* 13 */ Another,                        /* ADC_GdEd                */
    /* 14 */ Immediate_Ib,                /* ADC_ALIb                */
    /* 15 */ Immediate_Iv,                /* ADC_EAXId                */
    /* 16 */ 0,                                /* PUSH_SS                */
    /* 17 */ 0,                                /* POP_SS                */
    /* 18 */ Another,                        /* SBB_EbGb                */
    /* 19 */ Another,                        /* SBB_EdGd                */
    /* 1A */ Another,                        /* SBB_GbEb                */
    /* 1B */ Another,                        /* SBB_GdEd                */
    /* 1C */ Immediate_Ib,                /* SBB_ALIb                */
    /* 1D */ Immediate_Iv,                /* SBB_EAXId                */
    /* 1E */ 0,                                /* PUSH_DS                */
    /* 1F */ 0,                                /* POP_DS                */
    /* 20 */ Another,                        /* AND_EbGb                */
    /* 21 */ Another,                        /* AND_EdGd                */
    /* 22 */ Another,                        /* AND_GbEb                */
    /* 23 */ Another,                        /* AND_GdEd                */
    /* 24 */ Immediate_Ib,                /* AND_ALIb                */
    /* 25 */ Immediate_Iv,                /* AND_EAXId                */
    /* 26 */ Prefix | Another,                /* ES:                        */
    /* 27 */ 0,                                /* DAA                        */
    /* 28 */ Another,                        /* SUB_EbGb                */
    /* 29 */ Another,                        /* SUB_EdGd                */
    /* 2A */ Another,                        /* SUB_GbEb                */
    /* 2B */ Another,                        /* SUB_GdEd                */
    /* 2C */ Immediate_Ib,                /* SUB_ALIb                */
    /* 2D */ Immediate_Iv,                /* SUB_EAXId                */
    /* 2E */ Prefix | Another,                /* CS:                        */
    /* 2F */ 0,                                /* DAS                        */
    /* 30 */ Another,                        /* XOR_EbGb                */
    /* 31 */ Another,                        /* XOR_EdGd                */
    /* 32 */ Another,                        /* XOR_GbEb                */
    /* 33 */ Another,                        /* XOR_GdEd                */
    /* 34 */ Immediate_Ib,                /* XOR_ALIb                */
    /* 35 */ Immediate_Iv,                /* XOR_EAXId                */
    /* 36 */ Prefix | Another,                /* SS:                        */
    /* 37 */ 0,                                /* AAA                        */
    /* 38 */ Another,                        /* CMP_EbGb                */
    /* 39 */ Another,                        /* CMP_EdGd                */
    /* 3A */ Another,                        /* CMP_GbEb                */
    /* 3B */ Another,                        /* CMP_GdEd                */
    /* 3C */ Immediate_Ib,                /* CMP_ALIb                */
    /* 3D */ Immediate_Iv,                /* CMP_EAXId                */
    /* 3E */ Prefix | Another,                /* DS:                        */
    /* 3F */ 0,                                /* AAS                        */
    /* 40 */ 0,                                /* INC_ERX                */
    /* 41 */ 0,                                /* INC_ERX                */
    /* 42 */ 0,                                /* INC_ERX                */
    /* 43 */ 0,                                /* INC_ERX                */
    /* 44 */ 0,                                /* INC_ERX                */
    /* 45 */ 0,                                /* INC_ERX                */
    /* 46 */ 0,                                /* INC_ERX                */
    /* 47 */ 0,                                /* INC_ERX                */
    /* 48 */ 0,                                /* DEC_ERX                */
    /* 49 */ 0,                                /* DEC_ERX                */
    /* 4A */ 0,                                /* DEC_ERX                */
    /* 4B */ 0,                                /* DEC_ERX                */
    /* 4C */ 0,                                /* DEC_ERX                */
    /* 4D */ 0,                                /* DEC_ERX                */
    /* 4E */ 0,                                /* DEC_ERX                */
    /* 4F */ 0,                                /* DEC_ERX                */
    /* 50 */ 0,                                /* PUSH_ERX                */
    /* 51 */ 0,                                /* PUSH_ERX                */
    /* 52 */ 0,                                /* PUSH_ERX                */
    /* 53 */ 0,                                /* PUSH_ERX                */
    /* 54 */ 0,                                /* PUSH_ERX                */
    /* 55 */ 0,                                /* PUSH_ERX                */
    /* 56 */ 0,                                /* PUSH_ERX                */
    /* 57 */ 0,                                /* PUSH_ERX                */
    /* 58 */ 0,                                /* POP_ERX                */
    /* 59 */ 0,                                /* POP_ERX                */
    /* 5A */ 0,                                /* POP_ERX                */
    /* 5B */ 0,                                /* POP_ERX                */
    /* 5C */ 0,                                /* POP_ERX                */
    /* 5D */ 0,                                /* POP_ERX                */
    /* 5E */ 0,                                /* POP_ERX                */
    /* 5F */ 0,                                /* POP_ERX                */
    /* 60 */ 0,                                /* PUSHAD32                */
    /* 61 */ 0,                                /* POPAD32                */
    /* 62 */ Another,                        /* BOUND_GvMa                */
    /* 63 */ Another,                        /* ARPL_EwGw                */
    /* 64 */ Prefix | Another,                /* FS:                        */
    /* 65 */ Prefix | Another,                /* GS:                        */
    /* 66 */ Prefix | Another,                /* OS:                        */
    /* 67 */ Prefix | Another,                /* AS:                        */
    /* 68 */ Immediate_Iv,                /* PUSH_Id                */
    /* 69 */ Another | Immediate_Iv,        /* IMUL_GdEdId                */
    /* 6A */ Immediate_Ib_SE,                /* PUSH_Id                */
    /* 6B */ Another | Immediate_Ib_SE,        /* IMUL_GdEdId                */
    /* 6C */ Repeatable,                /* INSB_YbDX                */
    /* 6D */ Repeatable,                /* INSW_YvDX                */
    /* 6E */ Repeatable,                /* OUTSB_DXXb                */
    /* 6F */ Repeatable,                /* OUTSW_DXXv                */
    /* 70 */ Immediate_BrOff8,                /* JCC_Jd                */
    /* 71 */ Immediate_BrOff8,                /* JCC_Jd                */
    /* 72 */ Immediate_BrOff8,                /* JCC_Jd                */
    /* 73 */ Immediate_BrOff8,                /* JCC_Jd                */
    /* 74 */ Immediate_BrOff8,                /* JCC_Jd                */
    /* 75 */ Immediate_BrOff8,                /* JCC_Jd                */
    /* 76 */ Immediate_BrOff8,                /* JCC_Jd                */
    /* 77 */ Immediate_BrOff8,                /* JCC_Jd                */
    /* 78 */ Immediate_BrOff8,                /* JCC_Jd                */
    /* 79 */ Immediate_BrOff8,                /* JCC_Jd                */
    /* 7A */ Immediate_BrOff8,                /* JCC_Jd                */
    /* 7B */ Immediate_BrOff8,                /* JCC_Jd                */
    /* 7C */ Immediate_BrOff8,                /* JCC_Jd                */
    /* 7D */ Immediate_BrOff8,                /* JCC_Jd                */
    /* 7E */ Immediate_BrOff8,                /* JCC_Jd                */
    /* 7F */ Immediate_BrOff8,                /* JCC_Jd                */
    /* 80 */ Another | GroupN | PutGroupIndex (G1EbIb),
    /* 81 */ Another | GroupN | Immediate_Iv | PutGroupIndex (G1Ed),
    /* 82 */ Another | GroupN | PutGroupIndex (G1EbIb),
    /* 83 */ Another | GroupN | Immediate_Ib_SE | PutGroupIndex (G1Ed),
    /* 84 */ Another,                        /* TEST_EbGb                */
    /* 85 */ Another,                        /* TEST_EdGd                */
    /* 86 */ Another,                        /* XCHG_EbGb                */
    /* 87 */ Another,                        /* XCHG_EdGd                */
    /* 88 */ Another,                        /* MOV_EbGb                */
    /* 89 */ Another,                        /* MOV_EdGd                */
    /* 8A */ Another,                        /* MOV_GbEb                */
    /* 8B */ Another,                        /* MOV_GdEd                */
    /* 8C */ Another,                        /* MOV_EwSw                */
    /* 8D */ Another,                        /* LEA_GdM                */
    /* 8E */ Another,                        /* MOV_SwEw                */
    /* 8F */ Another,                        /* POP_Ed                */
    /* 90 */ 0,                                /* NOP                        */
    /* 91 */ 0,                                /* XCHG_ERXEAX                */
    /* 92 */ 0,                                /* XCHG_ERXEAX                */
    /* 93 */ 0,                                /* XCHG_ERXEAX                */
    /* 94 */ 0,                                /* XCHG_ERXEAX                */
    /* 95 */ 0,                                /* XCHG_ERXEAX                */
    /* 96 */ 0,                                /* XCHG_ERXEAX                */
    /* 97 */ 0,                                /* XCHG_ERXEAX                */
    /* 98 */ 0,                                /* CWDE                        */
    /* 99 */ 0,                                /* CDQ                        */
    /* 9A */ Immediate_IvIw,                /* CALL32_Ap                */
    /* 9B */ 0,                                /* FWAIT                */
    /* 9C */ 0,                                /* PUSHF_Fv                */
    /* 9D */ 0,                                /* POPF_Fv                */
    /* 9E */ 0,                                /* SAHF                        */
    /* 9F */ 0,                                /* LAHF                        */
    /* A0 */ Immediate_O,                /* MOV_ALOb                */
    /* A1 */ Immediate_O,                /* MOV_EAXOd                */
    /* A2 */ Immediate_O,                /* MOV_ObAL                */
    /* A3 */ Immediate_O,                /* MOV_OdEAX                */
    /* A4 */ Repeatable,                /* MOVSB_XbYb                */
    /* A5 */ Repeatable,                /* MOVSW_XvYv                */
    /* A6 */ Repeatable | RepeatableZF,        /* CMPSB_XbYb                */
    /* A7 */ Repeatable | RepeatableZF,        /* CMPSW_XvYv                */
    /* A8 */ Immediate_Ib,                /* TEST_ALIb                */
    /* A9 */ Immediate_Iv,                /* TEST_EAXId                */
    /* AA */ Repeatable,                /* STOSB_YbAL                */
    /* AB */ Repeatable,                /* STOSW_YveAX                */
    /* AC */ Repeatable,                /* LODSB_ALXb                */
    /* AD */ Repeatable,                /* LODSW_eAXXv                */
    /* AE */ Repeatable | RepeatableZF,        /* SCASB_ALXb                */
    /* AF */ Repeatable | RepeatableZF,        /* SCASW_eAXXv                */
    /* B0 */ Immediate_Ib,                /* MOV_RLIb                */
    /* B1 */ Immediate_Ib,                /* MOV_RLIb                */
    /* B2 */ Immediate_Ib,                /* MOV_RLIb                */
    /* B3 */ Immediate_Ib,                /* MOV_RLIb                */
    /* B4 */ Immediate_Ib,                /* MOV_RHIb                */
    /* B5 */ Immediate_Ib,                /* MOV_RHIb                */
    /* B6 */ Immediate_Ib,                /* MOV_RHIb                */
    /* B7 */ Immediate_Ib,                /* MOV_RHIb                */
    /* B8 */ Immediate_Iv,                /* MOV_ERXId                */
    /* B9 */ Immediate_Iv,                /* MOV_ERXId                */
    /* BA */ Immediate_Iv,                /* MOV_ERXId                */
    /* BB */ Immediate_Iv,                /* MOV_ERXId                */
    /* BC */ Immediate_Iv,                /* MOV_ERXId                */
    /* BD */ Immediate_Iv,                /* MOV_ERXId                */
    /* BE */ Immediate_Iv,                /* MOV_ERXId                */
    /* BF */ Immediate_Iv,                /* MOV_ERXId                */
    /* C0 */ Another | GroupN | Immediate_Ib | PutGroupIndex (G2Eb),
    /* C1 */ Another | GroupN | Immediate_Ib | PutGroupIndex (G2Ed),
    /* C2 */ Immediate_Iw | UCBranch,        /* RETnear32_Iw                */
    /* C3 */ 0 | UCBranch,                /* RETnear32                */
    /* C4 */ Another,                        /* LES_GvMp                */
    /* C5 */ Another,                        /* LDS_GvMp                */
    /* C6 */ Another | Immediate_Ib,        /* MOV_EbIb                */
    /* C7 */ Another | Immediate_Iv,        /* MOV_EdId                */
    /* C8 */ Immediate_IwIb,                /* ENTER_IwIb                */
    /* C9 */ 0,                                /* LEAVE                */
    /* CA */ Immediate_Iw | UCBranch,        /* RETfar32_Iw                */
    /* CB */ 0 | UCBranch,                /* RETfar32                */
    /* CC */ 0,                                /* INT3                        */
    /* CD */ Immediate_Ib,                /* INT_Ib                */
    /* CE */ 0,                                /* INTO                        */
    /* CF */ 0 | UCBranch,                /* IRET32                */
    /* D0 */ Another | GroupN | PutGroupIndex (G2Eb),
    /* D1 */ Another | GroupN | PutGroupIndex (G2Ed),
    /* D2 */ Another | GroupN | PutGroupIndex (G2Eb),
    /* D3 */ Another | GroupN | PutGroupIndex (G2Ed),
    /* D4 */ Immediate_Ib,                /* AAM                        */
    /* D5 */ Immediate_Ib,                /* AAD                        */
    /* D6 */ 0,                                /* SALC                        */
    /* D7 */ 0,                                /* XLAT                        */
    /* D8 */ Another,                        /* ESC0                        */
    /* D9 */ Another,                        /* ESC1                        */
    /* DA */ Another,                        /* ESC2                        */
    /* DB */ Another,                        /* ESC3                        */
    /* DC */ Another,                        /* ESC4                        */
    /* DD */ Another,                        /* ESC5                        */
    /* DE */ Another,                        /* ESC6                        */
    /* DF */ Another,                        /* ESC7                        */
    /* E0 */ Immediate_BrOff8,                /* LOOPNE_Jb                */
    /* E1 */ Immediate_BrOff8,                /* LOOPE_Jb                */
    /* E2 */ Immediate_BrOff8,                /* LOOP_Jb                */
    /* E3 */ Immediate_BrOff8,                /* JCXZ_Jb                */
    /* E4 */ Immediate_Ib,                /* IN_ALIb                */
    /* E5 */ Immediate_Ib,                /* IN_eAXIb                */
    /* E6 */ Immediate_Ib,                /* OUT_IbAL                */
    /* E7 */ Immediate_Ib,                /* OUT_IbeAX                */
    /* E8 */ Immediate_BrOff32,                /* CALL_Ad                */
    /* E9 */ Immediate_BrOff32 | UCBranch,        /* JMP_Jd        */
    /* EA */ Immediate_IvIw | UCBranch,                /* JMP_Ap        */
    /* EB */ Immediate_BrOff8 | UCBranch,        /* JMP_Jb        */
    /* EC */ 0,                                /* IN_ALDX                */
    /* ED */ 0,                                /* IN_eAXDX                */
    /* EE */ 0,                                /* OUT_DXAL                */
    /* EF */ 0,                                /* OUT_DXeAX                */
    /* F0 */ Prefix | Another,                /* LOCK:                */
    /* F1 */ 0,                                /* INT1                        */
    /* F2 */ Prefix | Another,                /* REPNE/REPNZ:                */
    /* F3 */ Prefix | Another,                /* REP,REPE/REPZ:        */
    /* F4 */ 0,                                /* HLT                        */
    /* F5 */ 0,                                /* CMC                        */
    /* F6 */ Another | GroupN | PutGroupIndex (G3Eb),
    /* F7 */ Another | GroupN | PutGroupIndex (G3Ed),
    /* F8 */ 0,                                /* CLC                        */
    /* F9 */ 0,                                /* STC                        */
    /* FA */ 0,                                /* CLI                        */
    /* FB */ 0,                                /* STI                        */
    /* FC */ 0,                                /* CLD                        */
    /* FD */ 0,                                /* STD                        */
    /* FE */ Another | GroupN | PutGroupIndex (G4),
    /* FF */ Another | GroupN | PutGroupIndex (G5d),

    /* 0F 00 */ Another | GroupN | PutGroupIndex (G6),
    /* 0F 01 */ Another | GroupN | PutGroupIndex (G7),
    /* 0F 02 */ Another,                /* LAR_GvEw                */
    /* 0F 03 */ Another,                /* LSL_GvEw                */
    /* 0F 04 */ InvalidOpcode,                
    /* 0F 05 */ 0,                        /* LOADALL                */
    /* 0F 06 */ 0,                        /* CLTS                        */
    /* 0F 07 */ InvalidOpcode,
    /* 0F 08 */ 0,                        /* INVD                        */
    /* 0F 09 */ 0,                        /* WBINVD                */
    /* 0F 0A */ InvalidOpcode,
    /* 0F 0B */ InvalidOpcode,
    /* 0F 0C */ InvalidOpcode,
    /* 0F 0D */ InvalidOpcode,
    /* 0F 0E */ InvalidOpcode,
    /* 0F 0F */ InvalidOpcode,
    /* 0F 10 */ InvalidOpcode,
    /* 0F 11 */ InvalidOpcode,
    /* 0F 12 */ InvalidOpcode,
    /* 0F 13 */ InvalidOpcode,
    /* 0F 14 */ InvalidOpcode,
    /* 0F 15 */ InvalidOpcode,
    /* 0F 16 */ InvalidOpcode,
    /* 0F 17 */ InvalidOpcode,
    /* 0F 18 */ InvalidOpcode,
    /* 0F 19 */ InvalidOpcode,
    /* 0F 1A */ InvalidOpcode,
    /* 0F 1B */ InvalidOpcode,
    /* 0F 1C */ InvalidOpcode,
    /* 0F 1D */ InvalidOpcode,
    /* 0F 1E */ InvalidOpcode,
    /* 0F 1F */ InvalidOpcode,
    /* 0F 20 */ Another,                /* MOV_RdCd                */
    /* 0F 21 */ Another,                /* MOV_RdDd                */
    /* 0F 22 */ Another,                /* MOV_CdRd                */
    /* 0F 23 */ Another,                /* MOV_DdRd                */
    /* 0F 24 */ Another,                /* MOV_RdTd                */
    /* 0F 25 */ InvalidOpcode,
    /* 0F 26 */ Another,                /* MOV_TdRd                */
    /* 0F 27 */ InvalidOpcode,
    /* 0F 28 */ InvalidOpcode,
    /* 0F 29 */ InvalidOpcode,
    /* 0F 2A */ InvalidOpcode,
    /* 0F 2B */ InvalidOpcode,
    /* 0F 2C */ InvalidOpcode,
    /* 0F 2D */ InvalidOpcode,
    /* 0F 2E */ InvalidOpcode,
    /* 0F 2F */ InvalidOpcode,
    /* 0F 30 */ 0,                        /* WRMSR                */
    /* 0F 31 */ 0,                        /* RDTSC                */
    /* 0F 32 */ 0,                        /* RDMSR                */
    /* 0F 33 */ InvalidOpcode,
    /* 0F 34 */ InvalidOpcode,
    /* 0F 35 */ InvalidOpcode,
    /* 0F 36 */ InvalidOpcode,
    /* 0F 37 */ InvalidOpcode,
    /* 0F 38 */ InvalidOpcode,
    /* 0F 39 */ InvalidOpcode,
    /* 0F 3A */ InvalidOpcode,
    /* 0F 3B */ InvalidOpcode,
    /* 0F 3C */ InvalidOpcode,
    /* 0F 3D */ InvalidOpcode,
    /* 0F 3E */ InvalidOpcode,
    /* 0F 3F */ InvalidOpcode,
    /* 0F 40 */ Another,                /* CMOV_GdEd                */
    /* 0F 41 */ Another,                /* CMOV_GdEd                */
    /* 0F 42 */ Another,                /* CMOV_GdEd                */
    /* 0F 43 */ Another,                /* CMOV_GdEd                */
    /* 0F 44 */ Another,                /* CMOV_GdEd                */
    /* 0F 45 */ Another,                /* CMOV_GdEd                */
    /* 0F 46 */ Another,                /* CMOV_GdEd                */
    /* 0F 47 */ Another,                /* CMOV_GdEd                */
    /* 0F 48 */ Another,                /* CMOV_GdEd                */
    /* 0F 49 */ Another,                /* CMOV_GdEd                */
    /* 0F 4A */ Another,                /* CMOV_GdEd                */
    /* 0F 4B */ Another,                /* CMOV_GdEd                */
    /* 0F 4C */ Another,                /* CMOV_GdEd                */
    /* 0F 4D */ Another,                /* CMOV_GdEd                */
    /* 0F 4E */ Another,                /* CMOV_GdEd                */
    /* 0F 4F */ Another,                /* CMOV_GdEd                */
    /* 0F 50 */ InvalidOpcode,
    /* 0F 51 */ InvalidOpcode,
    /* 0F 52 */ InvalidOpcode,
    /* 0F 53 */ InvalidOpcode,
    /* 0F 54 */ InvalidOpcode,
    /* 0F 55 */ InvalidOpcode,
    /* 0F 56 */ InvalidOpcode,
    /* 0F 57 */ InvalidOpcode,
    /* 0F 58 */ InvalidOpcode,
    /* 0F 59 */ InvalidOpcode,
    /* 0F 5A */ InvalidOpcode,
    /* 0F 5B */ InvalidOpcode,
    /* 0F 5C */ InvalidOpcode,
    /* 0F 5D */ InvalidOpcode,
    /* 0F 5E */ InvalidOpcode,
    /* 0F 5F */ InvalidOpcode,
    /* 0F 60 */ InvalidOpcode,
    /* 0F 61 */ InvalidOpcode,
    /* 0F 62 */ InvalidOpcode,
    /* 0F 63 */ InvalidOpcode,
    /* 0F 64 */ InvalidOpcode,
    /* 0F 65 */ InvalidOpcode,
    /* 0F 66 */ InvalidOpcode,
    /* 0F 67 */ InvalidOpcode,
    /* 0F 68 */ InvalidOpcode,
    /* 0F 69 */ InvalidOpcode,
    /* 0F 6A */ InvalidOpcode,
    /* 0F 6B */ InvalidOpcode,
    /* 0F 6C */ InvalidOpcode,
    /* 0F 6D */ InvalidOpcode,
    /* 0F 6E */ InvalidOpcode,
    /* 0F 6F */ InvalidOpcode,
    /* 0F 70 */ InvalidOpcode,
    /* 0F 71 */ InvalidOpcode,
    /* 0F 72 */ InvalidOpcode,
    /* 0F 73 */ InvalidOpcode,
    /* 0F 74 */ InvalidOpcode,
    /* 0F 75 */ InvalidOpcode,
    /* 0F 76 */ InvalidOpcode,
    /* 0F 77 */ InvalidOpcode,
    /* 0F 78 */ InvalidOpcode,
    /* 0F 79 */ InvalidOpcode,
    /* 0F 7A */ InvalidOpcode,
    /* 0F 7B */ InvalidOpcode,
    /* 0F 7C */ InvalidOpcode,
    /* 0F 7D */ InvalidOpcode,
    /* 0F 7E */ InvalidOpcode,
    /* 0F 7F */ InvalidOpcode,
    /* 0F 80 */ Immediate_BrOff32,        /* JCC_Jd                */
    /* 0F 81 */ Immediate_BrOff32,        /* JCC_Jd                */
    /* 0F 82 */ Immediate_BrOff32,        /* JCC_Jd                */
    /* 0F 83 */ Immediate_BrOff32,        /* JCC_Jd                */
    /* 0F 84 */ Immediate_BrOff32,        /* JCC_Jd                */
    /* 0F 85 */ Immediate_BrOff32,        /* JCC_Jd                */
    /* 0F 86 */ Immediate_BrOff32,        /* JCC_Jd                */
    /* 0F 87 */ Immediate_BrOff32,        /* JCC_Jd                */
    /* 0F 88 */ Immediate_BrOff32,        /* JCC_Jd                */
    /* 0F 89 */ Immediate_BrOff32,        /* JCC_Jd                */
    /* 0F 8A */ Immediate_BrOff32,        /* JCC_Jd                */
    /* 0F 8B */ Immediate_BrOff32,        /* JCC_Jd                */
    /* 0F 8C */ Immediate_BrOff32,        /* JCC_Jd                */
    /* 0F 8D */ Immediate_BrOff32,        /* JCC_Jd                */
    /* 0F 8E */ Immediate_BrOff32,        /* JCC_Jd                */
    /* 0F 8F */ Immediate_BrOff32,        /* JCC_Jd                */
    /* 0F 90 */ Another,                /* SETO_Eb                */
    /* 0F 91 */ Another,                /* SETNO_Eb                */
    /* 0F 92 */ Another,                /* SETB_Eb                */
    /* 0F 93 */ Another,                /* SETNB_Eb                */
    /* 0F 94 */ Another,                /* SETZ_Eb                */
    /* 0F 95 */ Another,                /* SETNZ_Eb                */
    /* 0F 96 */ Another,                /* SETBE_Eb                */
    /* 0F 97 */ Another,                /* SETNBE_Eb                */
    /* 0F 98 */ Another,                /* SETS_Eb                */
    /* 0F 99 */ Another,                /* SETNS_Eb                */
    /* 0F 9A */ Another,                /* SETP_Eb                */
    /* 0F 9B */ Another,                /* SETNP_Eb                */
    /* 0F 9C */ Another,                /* SETL_Eb                */
    /* 0F 9D */ Another,                /* SETNL_Eb                */
    /* 0F 9E */ Another,                /* SETLE_Eb                */
    /* 0F 9F */ Another,                /* SETNLE_Eb                */
    /* 0F A0 */ 0,                        /* PUSH_FS                */
    /* 0F A1 */ 0,                        /* POP_FS                */
    /* 0F A2 */ 0,                        /* CPUID                */
    /* 0F A3 */ Another,                /* BT_EvGv                */
    /* 0F A4 */ Another | Immediate_Ib,        /* SHLD_EdGd                */
    /* 0F A5 */ Another,                /* SHLD_EdGd                */
    /* 0F A6 */ 0,                        /* CMPXCHG_XBTS                */
    /* 0F A7 */ 0,                        /* CMPXCHG_IBTS                */
    /* 0F A8 */ 0,                        /* PUSH_GS                */
    /* 0F A9 */ 0,                        /* POP_GS                */
    /* 0F AA */ 0,                        /* RSM                        */
    /* 0F AB */ Another,                /* BTS_EvGv                */
    /* 0F AC */ Another | Immediate_Ib,        /* SHRD_EdGd                */
    /* 0F AD */ Another,                /* SHRD_EdGd                */
    /* 0F AE */ InvalidOpcode,
    /* 0F AF */ Another,                /* IMUL_GdEd                */
    /* 0F B0 */ Another,                /* CMPXCHG_EbGb                */
    /* 0F B1 */ Another,                /* CMPXCHG_EdGd                */
    /* 0F B2 */ Another,                /* LSS_GvMp                */
    /* 0F B3 */ Another,                /* BTR_EvGv                */
    /* 0F B4 */ Another,                /* LFS_GvMp                */
    /* 0F B5 */ Another,                /* LGS_GvMp                */
    /* 0F B6 */ Another,                /* MOVZX_GdEb                */
    /* 0F B7 */ Another,                /* MOVZX_GdEw                */
    /* 0F B8 */ InvalidOpcode,
    /* 0F B9 */ InvalidOpcode,
    /* 0F BA */ Another | GroupN | PutGroupIndex (G8EvIb),
    /* 0F BB */ Another,                /* BTC_EvGv                */
    /* 0F BC */ Another,                /* BSF_GvEv                */
    /* 0F BD */ Another,                /* BSR_GvEv                */
    /* 0F BE */ Another,                /* MOVSX_GdEb                */
    /* 0F BF */ Another,                /* MOVSX_GdEw                */
    /* 0F C0 */ Another,                /* XADD_EbGb                */
    /* 0F C1 */ Another,                /* XADD_EdGd                */
    /* 0F C2 */ InvalidOpcode,
    /* 0F C3 */ InvalidOpcode,
    /* 0F C4 */ InvalidOpcode,
    /* 0F C5 */ InvalidOpcode,
    /* 0F C6 */ InvalidOpcode,
    /* 0F C7 */ Another | GroupN | PutGroupIndex (G9),
    /* 0F C8 */ 0,                        /* BSWAP_EAX                */
    /* 0F C9 */ 0,                        /* BSWAP_ECX                */
    /* 0F CA */ 0,                        /* BSWAP_EDX                */
    /* 0F CB */ 0,                        /* BSWAP_EBX                */
    /* 0F CC */ 0,                        /* BSWAP_ESP                */
    /* 0F CD */ 0,                        /* BSWAP_EBP                */
    /* 0F CE */ 0,                        /* BSWAP_ESI                */
    /* 0F CF */ 0,                        /* BSWAP_EDI                */
    /* 0F D0 */ InvalidOpcode,
    /* 0F D1 */ InvalidOpcode,
    /* 0F D2 */ InvalidOpcode,
    /* 0F D3 */ InvalidOpcode,
    /* 0F D4 */ InvalidOpcode,
    /* 0F D5 */ InvalidOpcode,
    /* 0F D6 */ InvalidOpcode,
    /* 0F D7 */ InvalidOpcode,
    /* 0F D8 */ InvalidOpcode,
    /* 0F D9 */ InvalidOpcode,
    /* 0F DA */ InvalidOpcode,
    /* 0F DB */ InvalidOpcode,
    /* 0F DC */ InvalidOpcode,
    /* 0F DD */ InvalidOpcode,
    /* 0F DE */ InvalidOpcode,
    /* 0F DF */ InvalidOpcode,
    /* 0F E0 */ InvalidOpcode,
    /* 0F E1 */ InvalidOpcode,
    /* 0F E2 */ InvalidOpcode,
    /* 0F E3 */ InvalidOpcode,
    /* 0F E4 */ InvalidOpcode,
    /* 0F E5 */ InvalidOpcode,
    /* 0F E6 */ InvalidOpcode,
    /* 0F E7 */ InvalidOpcode,
    /* 0F E8 */ InvalidOpcode,
    /* 0F E9 */ InvalidOpcode,
    /* 0F EA */ InvalidOpcode,
    /* 0F EB */ InvalidOpcode,
    /* 0F EC */ InvalidOpcode,
    /* 0F ED */ InvalidOpcode,
    /* 0F EE */ InvalidOpcode,
    /* 0F EF */ InvalidOpcode,
    /* 0F F0 */ InvalidOpcode,
    /* 0F F1 */ InvalidOpcode,
    /* 0F F2 */ InvalidOpcode,
    /* 0F F3 */ InvalidOpcode,
    /* 0F F4 */ InvalidOpcode,
    /* 0F F5 */ InvalidOpcode,
    /* 0F F6 */ InvalidOpcode,
    /* 0F F7 */ InvalidOpcode,
    /* 0F F8 */ InvalidOpcode,
    /* 0F F9 */ InvalidOpcode,
    /* 0F FA */ InvalidOpcode,
    /* 0F FB */ InvalidOpcode,
    /* 0F FC */ InvalidOpcode,
    /* 0F FD */ InvalidOpcode,
    /* 0F FE */ InvalidOpcode,
    /* 0F FF */ InvalidOpcode,
};



unsigned
fetchDecode (Bit8u * iptr, instruction_t * instruction,
             unsigned remain, Boolean is_32)
{
/* remain must be at least 1 */

    unsigned b1, b2, ilen = 1, attr;
    unsigned imm_mode, offset;
    unsigned rm;

    instruction->os_32 = instruction->as_32 = is_32;
    instruction->ResolveModrm = (void *) 0;
    instruction->seg = SEG_REG_NULL;
    instruction->rep_used = 0;

  fetch_b1:
    b1 = *iptr++;

  another_byte:
    offset = instruction->os_32 << 9;        /* multiply by 512 */

    instruction->attr = attr = OpcodeInfo[b1 + offset];

    if (attr & Another)
    {
        if (attr & Prefix)
        {
            switch (b1)
            {
            case 0x66:                /* OpSize */
              instruction->os_32 = !is_32;
              break;
            case 0x67:                /* AddrSize */
              instruction->as_32 = !is_32;
              break;
            case 0xf2:                /* REPNE/REPNZ   */
            case 0xf3:                /* REP/REPE/REPZ */
              instruction->rep_used = b1;
              break;
            case 0x2e:                /* CS: */
              instruction->seg = SEG_REG_CS;
              break;
            case 0x26:                /* ES: */
              instruction->seg = SEG_REG_ES;
              break;
            case 0x36:                /* SS: */
              instruction->seg = SEG_REG_SS;
              break;
            case 0x3e:                /* DS: */
              instruction->seg = SEG_REG_DS;
              break;
            case 0x64:                /* FS: */
              instruction->seg = SEG_REG_FS;
              break;
            case 0x65:                /* GS: */
              instruction->seg = SEG_REG_GS;
              break;
            case 0xf0:                /* LOCK: */
              break;

            default:
                /* panic("fetch_decode: prefix default = 0x%02x\n", b1);*/
                return (0);
            }
            if (ilen < remain) {
              ilen++;
              goto fetch_b1;
              }
            return (0);
        }

        /* opcode requires another byte */

        if (ilen < remain)
        {
            ilen++;
            b2 = *iptr++;
            if (b1 == 0x0f)
            {
                /* 2-byte prefix */
                b1 = 0x100 | b2;
                goto another_byte;
            }
        }
        else
            return (0);

        /* Parse mod-nnn-rm and related bytes */

        instruction->modrm = b2;
        rm = instruction->rm = b2 & 0x07;
        instruction->mod = b2 & 0xc0;        /* leave unshifted         */

        instruction->nnn = (b2 >> 3) & 0x07;

        if (instruction->mod == 0xc0)        /* mod == 11b                */
        {

            goto modrm_done;
        }

        if (instruction->as_32)                /* 32-bit addressing modes; note    */
        {                                /* note that mod==11b handled above */
            if (rm != 4)                /* no s-i-b byte                    */
            {
                if (instruction->mod == 0x00)
                {                        /* mod == 00b                */
                      instruction->ResolveModrm = Resolve32Mod0[rm];

                    if (NULL_SEG_REG (instruction->seg))
                        instruction->seg = SEG_REG_DS;

                    if (rm == 5)
                    {
                        if ((ilen + 3) < remain)
                        {
                            Bit32u imm32u;

                            imm32u = *iptr++;
                            imm32u |= (*iptr++) << 8;
                            imm32u |= (*iptr++) << 16;
                            imm32u |= (*iptr++) << 24;
                            instruction->rm_addr = imm32u;
                            ilen += 4;
                            goto modrm_done;
                        }
                        else
                        {
                            return (0);
                        }
                    }

                    /* mod==00b, rm!=4, rm!=5 */
                    goto modrm_done;
                }

                if (instruction->mod == 0x40)
                {                        /* mod == 01b                */
                      instruction->ResolveModrm = Resolve32Mod1or2[rm];

                    if (NULL_SEG_REG (instruction->seg))
                        instruction->seg = sreg_mod01_rm32[rm];

                  get_8bit_displ:
                    if (ilen < remain)
                    {
                        /* 8 sign extended to 32 */
                        instruction->displ32u = (Bit8s) * iptr++;
                        ilen++;
                        goto modrm_done;
                    }
                    else
                    {
                        return (0);
                    }
                }

                /* (mod == 0x80) mod == 10b */

                  instruction->ResolveModrm = Resolve32Mod1or2[rm];

                if (NULL_SEG_REG (instruction->seg))
                    instruction->seg = sreg_mod10_rm32[rm];

              get_32bit_displ:
                if ((ilen + 3) < remain)
                {
                    Bit32u imm32u;

                    imm32u = *iptr++;
                    imm32u |= (*iptr++) << 8;
                    imm32u |= (*iptr++) << 16;
                    imm32u |= (*iptr++) << 24;
                    instruction->displ32u = imm32u;
                    ilen += 4;
                    goto modrm_done;
                }
                else
                {
                    return (0);
                }
            }
            else
            {                        /* mod!=11b, rm==4, s-i-b byte follows        */
                unsigned sib, base;

                if (ilen < remain)
                {
                    sib = *iptr++;
                    ilen++;
                }
                else
                {
                    return (0);
                }

                instruction->sib = sib;
                base = instruction->base = sib & 0x07;
                sib >>= 3;
                instruction->index = sib & 0x07;
                sib >>= 3;
                instruction->scale = sib;

                if (instruction->mod == 0x00)
                {                /* mod==00b, rm==4        */
                      instruction->ResolveModrm = Resolve32Mod0Base[base];

                    if (NULL_SEG_REG (instruction->seg))
                        instruction->seg = sreg_mod0_base32[base];

                    if (instruction->base == 0x05)
                    {
                        goto get_32bit_displ;
                    }

                    /* mod==00b, rm==4, base!=5 */
                    goto modrm_done;
                }

                if (instruction->mod == 0x40)
                {                /* mod==01b, rm==4        */
                      instruction->ResolveModrm = Resolve32Mod1or2Base[base];

                    if (NULL_SEG_REG (instruction->seg))
                        instruction->seg = sreg_mod1or2_base32[base];

                    goto get_8bit_displ;
                }

                /* (instruction->mod == 0x80),  mod==10b, rm==4 */

                  instruction->ResolveModrm = Resolve32Mod1or2Base[base];

                if (NULL_SEG_REG (instruction->seg))
                    instruction->seg = sreg_mod1or2_base32[base];

                goto get_32bit_displ;
            }
        }
        else        /* 16-bit addressing modes, mod==11b handled above */
        {
            if (instruction->mod == 0x40)
            {                        /* mod == 01b                */
                  instruction->ResolveModrm = Resolve16Mod1or2[rm];

                if (NULL_SEG_REG (instruction->seg))
                    instruction->seg = sreg_mod01_rm16[rm];

                if (ilen < remain)        /* 8 sign extended to 16 */
                {
                    instruction->displ16u = (Bit8s) * iptr++;
                    ilen++;
                    goto modrm_done;
                }
                else
                {
                    return (0);
                }
            }

            if (instruction->mod == 0x80)
            {                        /* mod == 10b                */
                  instruction->ResolveModrm = Resolve16Mod1or2[rm];

                if (NULL_SEG_REG (instruction->seg))
                    instruction->seg = sreg_mod10_rm16[rm];

                if ((ilen + 1) < remain)
                {
                    Bit16u displ16u;

                    displ16u = *iptr++;
                    displ16u |= (*iptr++) << 8;
                    instruction->displ16u = displ16u;
                    ilen += 2;
                    goto modrm_done;
                }
                else
                {
                    return (0);
                }
            }

            /* mod must be 00b at this point */

              instruction->ResolveModrm = Resolve16Mod0[rm];

            if (NULL_SEG_REG (instruction->seg))
                instruction->seg = sreg_mod00_rm16[rm];

            if (rm == 0x06)
            {
                if ((ilen + 1) < remain)
                {
                    Bit16u displ16u;

                    displ16u = *iptr++;
                    displ16u |= (*iptr++) << 8;
                    instruction->rm_addr = displ16u;
                    ilen += 2;
                    goto modrm_done;
                }
                else
                {
                    return (0);
                }
            }

            /* mod=00b rm!=6        */
        }

      modrm_done:
        if (attr & GroupN)
        {
            /* There may be additional attributes in the individual        */
            /* entry for a particular group.  Combine with those.        */

            attr |= groupNattr[GetGroupIndex (attr)][instruction->nnn];
            instruction->attr = attr;
        }
        else
        {
        }
    }
    else
    {
        /*
         *  Opcode does not require a MODRM byte.
         *  Note that a 2-byte opcode (0F XX) will jump to before
         *  the if() above after fetching the 2nd byte, so this path is
         *  taken in all cases if a modrm byte is NOT required.
         */
    }

    imm_mode = attr & Immediate;

    if (imm_mode)
    {
        switch (imm_mode)
        {
        case Immediate_Ib:

            if (ilen < remain)
            {
                instruction->Ib = *iptr;
                ilen++;
            }
            else
            {
                return (0);
            }

            break;

        case Immediate_Ib_SE:        /* Sign extend to OS size        */

            if (ilen < remain)
            {
                Bit8s temp8s;

                temp8s = *iptr;
                if (instruction->os_32)
                    instruction->Id = (Bit32s) temp8s;
                else
                    instruction->Iw = (Bit16s) temp8s;
                ilen++;
            }
            else
            {
                return (0);
            }

            break;

        case Immediate_BrOff32:
        case Immediate_Iv:
        case Immediate_IvIw:        /* CALL_Ap                        */

            if (instruction->os_32)
            {
                if ((ilen + 3) < remain)
                {
                    Bit32u imm32u;

                    imm32u = *iptr++;
                    imm32u |= (*iptr++) << 8;
                    imm32u |= (*iptr++) << 16;
                    imm32u |= (*iptr) << 24;
                    instruction->Id = imm32u;
                    ilen += 4;
                }
                else
                {
                    return (0);
                }
            }
            else
            {
                if ((ilen + 1) < remain)
                {
                    Bit16u imm16u;

                    imm16u = *iptr++;
                    imm16u |= (*iptr) << 8;
                    instruction->Iw = imm16u;
                    ilen += 2;
                }
                else
                {
                    return (0);
                }
            }

            if (imm_mode != Immediate_IvIw)
                break;
            iptr++;

            /* Get Iw for Immediate_IvIw */

            if ((ilen + 1) < remain)
            {
                Bit16u imm16u;

                imm16u = *iptr++;
                imm16u |= (*iptr) << 8;
                instruction->Iw2 = imm16u;
                ilen += 2;
            }
            else
            {
                return (0);
            }

            break;

        case Immediate_O:

            if (instruction->as_32)
            {
                /* fetch 32bit address into Id */

                if ((ilen + 3) < remain)
                {
                    Bit32u imm32u;

                    imm32u = *iptr++;
                    imm32u |= (*iptr++) << 8;
                    imm32u |= (*iptr++) << 16;
                    imm32u |= (*iptr) << 24;
                    instruction->Id = imm32u;
                    ilen += 4;
                }
                else
                {
                    return (0);
                }
            }
            else
            {
                /* fetch 16bit address into Id */

                if ((ilen + 1) < remain)
                {
                    Bit32u imm32u;

                    imm32u = *iptr++;
                    imm32u |= (*iptr) << 8;
                    instruction->Id = imm32u;
                    ilen += 2;
                }
                else
                {
                    return (0);
                }
            }

            break;

        case Immediate_Iw:
        case Immediate_IwIb:

            if ((ilen + 1) < remain)
            {
                Bit16u imm16u;

                imm16u = *iptr++;
                imm16u |= (*iptr) << 8;
                instruction->Iw = imm16u;
                ilen += 2;
            }
            else
            {
                return (0);
            }

            if (imm_mode == Immediate_Iw)
                break;

            iptr++;

            if (ilen < remain)
            {
                instruction->Ib2 = *iptr;
                ilen++;
            }
            else
            {
                return (0);
            }

            break;

        case Immediate_BrOff8:

            if (ilen < remain)
            {
                Bit8s temp8s;

                temp8s = *iptr;
                instruction->Id = temp8s;
                ilen++;
            }
            else
            {
                return (0);
            }

            break;

        case Immediate_BrOff16:

            if ((ilen + 1) < remain)
            {
                Bit16u imm16u;

                imm16u = *iptr++;
                imm16u |= (*iptr) << 8;
                instruction->Id = (Bit16s) imm16u;
                ilen += 2;
            }
            else
            {
                return (0);
            }

            break;

        default:

/*            printf("b1 was %x\n", b1); */
            return (0);

            /* panic("fetchdecode: imm_mode = %u\n", imm_mode); */
        }
    }

    instruction->b1 = b1;
    instruction->ilen = ilen;
    return (1);
}


/* */
/* Some macros to make life easier... */
/* */
#define BX ((Bit16u) context->ebx)
#define SI ((Bit16u) context->esi)
#define BP ((Bit16u) context->ebp)
#define DI ((Bit16u) context->edi)

#define READ_32BIT_REG(index) ({ \
  Bit32u val=0; \
  switch (index) { \
    case 0: val = context->eax; break; \
    case 1: val = context->ecx; break; \
    case 2: val = context->edx; break; \
    case 3: val = context->ebx; break; \
    case 4: val = context->esp; break; \
    case 5: val = context->ebp; break; \
    case 6: val = context->esi; break; \
    case 7: val = context->edi; break; \
    } \
   val; \
  })


/* These functions will resolve a mod-rm address, given the */
/* current instruction and CPU context. */

  void
Resolve32Mod0Rm0(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = context->eax;
}
  void
Resolve32Mod0Rm1(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = context->ecx;
}
  void
Resolve32Mod0Rm2(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = context->edx;
}
  void
Resolve32Mod0Rm3(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = context->ebx;
}
  void
Resolve32Mod0Rm6(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = context->esi;
}
  void
Resolve32Mod0Rm7(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = context->edi;
}


  void
Resolve32Mod1or2Rm0(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = context->eax + i->displ32u;
}
  void
Resolve32Mod1or2Rm1(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = context->ecx + i->displ32u;
}
  void
Resolve32Mod1or2Rm2(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = context->edx + i->displ32u;
}
  void
Resolve32Mod1or2Rm3(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = context->ebx + i->displ32u;
}
  void
Resolve32Mod1or2Rm5(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = context->ebp + i->displ32u;
}
  void
Resolve32Mod1or2Rm6(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = context->esi + i->displ32u;
}
  void
Resolve32Mod1or2Rm7(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = context->edi + i->displ32u;
}

  void
Resolve32Mod0Base0(instruction_t *i, guest_context_t *context)
{
  Bit32u scaled_index;

  if (i->index != 4)
    scaled_index = READ_32BIT_REG(i->index) << i->scale;
  else
    scaled_index = 0;
  i->rm_addr = context->eax + scaled_index;
}
  void
Resolve32Mod0Base1(instruction_t *i, guest_context_t *context)
{
  Bit32u scaled_index;

  if (i->index != 4)
    scaled_index = READ_32BIT_REG(i->index) << i->scale;
  else
    scaled_index = 0;
  i->rm_addr = context->ecx + scaled_index;
}
  void
Resolve32Mod0Base2(instruction_t *i, guest_context_t *context)
{
  Bit32u scaled_index;

  if (i->index != 4)
    scaled_index = READ_32BIT_REG(i->index) << i->scale;
  else
    scaled_index = 0;
  i->rm_addr = context->edx + scaled_index;
}
  void
Resolve32Mod0Base3(instruction_t *i, guest_context_t *context)
{
  Bit32u scaled_index;

  if (i->index != 4)
    scaled_index = READ_32BIT_REG(i->index) << i->scale;
  else
    scaled_index = 0;
  i->rm_addr = context->ebx + scaled_index;
}
  void
Resolve32Mod0Base4(instruction_t *i, guest_context_t *context)
{
  Bit32u scaled_index;

  if (i->index != 4)
    scaled_index = READ_32BIT_REG(i->index) << i->scale;
  else
    scaled_index = 0;
  i->rm_addr = context->esp + scaled_index;
}
  void
Resolve32Mod0Base5(instruction_t *i, guest_context_t *context)
{
  Bit32u scaled_index;

  if (i->index != 4)
    scaled_index = READ_32BIT_REG(i->index) << i->scale;
  else
    scaled_index = 0;
  i->rm_addr = i->displ32u + scaled_index;
}
  void
Resolve32Mod0Base6(instruction_t *i, guest_context_t *context)
{
  Bit32u scaled_index;

  if (i->index != 4)
    scaled_index = READ_32BIT_REG(i->index) << i->scale;
  else
    scaled_index = 0;
  i->rm_addr = context->esi + scaled_index;
}
  void
Resolve32Mod0Base7(instruction_t *i, guest_context_t *context)
{
  Bit32u scaled_index;

  if (i->index != 4)
    scaled_index = READ_32BIT_REG(i->index) << i->scale;
  else
    scaled_index = 0;
  i->rm_addr = context->edi + scaled_index;
}

  void
Resolve32Mod1or2Base0(instruction_t *i, guest_context_t *context)
{
  Bit32u scaled_index;

  if (i->index != 4)
    scaled_index = READ_32BIT_REG(i->index) << i->scale;
  else
    scaled_index = 0;
  i->rm_addr = context->eax + scaled_index + i->displ32u;
}
  void
Resolve32Mod1or2Base1(instruction_t *i, guest_context_t *context)
{
  Bit32u scaled_index;

  if (i->index != 4)
    scaled_index = READ_32BIT_REG(i->index) << i->scale;
  else
    scaled_index = 0;
  i->rm_addr = context->ecx + scaled_index + i->displ32u;
}
  void
Resolve32Mod1or2Base2(instruction_t *i, guest_context_t *context)
{
  Bit32u scaled_index;

  if (i->index != 4)
    scaled_index = READ_32BIT_REG(i->index) << i->scale;
  else
    scaled_index = 0;
  i->rm_addr = context->edx + scaled_index + i->displ32u;
}
  void
Resolve32Mod1or2Base3(instruction_t *i, guest_context_t *context)
{
  Bit32u scaled_index;

  if (i->index != 4)
    scaled_index = READ_32BIT_REG(i->index) << i->scale;
  else
    scaled_index = 0;
  i->rm_addr = context->ebx + scaled_index + i->displ32u;
}
  void
Resolve32Mod1or2Base4(instruction_t *i, guest_context_t *context)
{
  Bit32u scaled_index;

  if (i->index != 4)
    scaled_index = READ_32BIT_REG(i->index) << i->scale;
  else
    scaled_index = 0;
  i->rm_addr = context->esp + scaled_index + i->displ32u;
}
  void
Resolve32Mod1or2Base5(instruction_t *i, guest_context_t *context)
{
  Bit32u scaled_index;

  if (i->index != 4)
    scaled_index = READ_32BIT_REG(i->index) << i->scale;
  else
    scaled_index = 0;
  i->rm_addr = context->ebp + scaled_index + i->displ32u;
}
  void
Resolve32Mod1or2Base6(instruction_t *i, guest_context_t *context)
{
  Bit32u scaled_index;

  if (i->index != 4)
    scaled_index = READ_32BIT_REG(i->index) << i->scale;
  else
    scaled_index = 0;
  i->rm_addr = context->esi + scaled_index + i->displ32u;
}
  void
Resolve32Mod1or2Base7(instruction_t *i, guest_context_t *context)
{
  Bit32u scaled_index;

  if (i->index != 4)
    scaled_index = READ_32BIT_REG(i->index) << i->scale;
  else
    scaled_index = 0;
  i->rm_addr = context->edi + scaled_index + i->displ32u;
}

  void
Resolve16Mod0Rm0(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = (Bit16u) (BX + SI);
}
  void
Resolve16Mod0Rm1(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = (Bit16u) (BX + DI);
}
  void
Resolve16Mod0Rm2(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = (Bit16u) (BP + SI);
}
  void
Resolve16Mod0Rm3(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = (Bit16u) (BP + DI);
}
  void
Resolve16Mod0Rm4(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = (Bit16u) SI;
}
  void
Resolve16Mod0Rm5(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = (Bit16u) DI;
}
  void
Resolve16Mod0Rm7(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = (Bit16u) BX;
}

  void
Resolve16Mod1or2Rm0(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = (Bit16u) (BX + SI + (Bit16s) i->displ16u);
}
  void
Resolve16Mod1or2Rm1(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = (Bit16u) (BX + DI + (Bit16s) i->displ16u);
}
  void
Resolve16Mod1or2Rm2(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = (Bit16u) (BP + SI + (Bit16s) i->displ16u);
}
  void
Resolve16Mod1or2Rm3(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = (Bit16u) (BP + DI + (Bit16s) i->displ16u);
}
  void
Resolve16Mod1or2Rm4(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = (Bit16u) (SI + (Bit16s) i->displ16u);
}
  void
Resolve16Mod1or2Rm5(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = (Bit16u) (DI + (Bit16s) i->displ16u);
}
  void
Resolve16Mod1or2Rm6(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = (Bit16u) (BP + (Bit16s) i->displ16u);
}
  void
Resolve16Mod1or2Rm7(instruction_t *i, guest_context_t *context)
{
  i->rm_addr = (Bit16u) (BX + (Bit16s) i->displ16u);
}
