/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2000  Kevin P. Lawton
 *
 *  io.c: IO oriented instructions.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "monitor.h"


  void
INSB_YbDX(vm_t *vm)
{
  Bit32u edi;
  Bit8u value8=0;

  IOPermCheck(vm, G_DX(vm), 1);

  if (vm->i.as_32)
    edi = G_EDI(vm);
  else
    edi = G_DI(vm);

  /* Write a zero to memory, to trigger any segment or page */
  /* faults before reading from IO port. */
  write_virtual_byte(vm, SRegES, edi, &value8);

  value8 = sysIOIn(vm, G_DX(vm), 1);

  /* no seg override allowed */
  write_virtual_byte(vm, SRegES, edi, &value8);

  if (vm->i.as_32) {
    if (G_GetDF(vm))
      G_EDI(vm) -= 1;
    else
      G_EDI(vm) += 1;
    }
  else {
    if (G_GetDF(vm))
      G_DI(vm) -= 1;
    else
      G_DI(vm) += 1;
    }
}

  void
INSW_YvDX(vm_t *vm)
  /* input word/doubleword from port to string */
{
  Bit32u edi;
  unsigned int incr;

  if (vm->i.as_32)
    edi = G_EDI(vm);
  else
    edi = G_DI(vm);

  if (vm->i.os_32) {
    Bit32u value32=0;

    IOPermCheck(vm, G_DX(vm), 4);

    /* Write a zero to memory, to trigger any segment or page */
    /* faults before reading from IO port. */
    write_virtual_dword(vm, SRegES, edi, &value32);

    value32 = sysIOIn(vm, G_DX(vm), 4);

    /* no seg override allowed */
    write_virtual_dword(vm, SRegES, edi, &value32);
    incr = 4;
    }
  else {
    Bit16u value16=0;

    IOPermCheck(vm, G_DX(vm), 2);

#if COSIMULATE == 0
    /* If conditions are right, we can transfer IO to physical memory
     * in a batch, rather than one instruction at a time.
     */
    if (vm->i.rep_used) {
      Bit32u count_requested, count_actual;
      if (vm->i.as_32)
        count_requested = G_ECX(vm);
      else
        count_requested = G_CX(vm);
      if (count_requested == 256) {
        Bit32u laddr, paddr, ppi, words;
        phy_page_usage_t *pusage;

        cache_sreg(vm, SRegES);
        /* Do guest segment access checks */
#warning "IO batch hack needs segment access checks"
#warning "IO batch hack needs instructions ticks"

        /* Translate guest linear addr to physical addr and do page
         * permissions check if CR0.PG=1.
         */
        laddr = vm->guest_cpu.desc_cache[SRegES].base + edi;
        if (vm->guest_cpu.cr0.fields.pg)
          paddr = dtranslate_linear(vm, laddr, G_CPL(vm)==3, OP_WRITE);
        else
          paddr = A20Addr(vm, laddr);

        /* Get monitor phy page attributes; check attr for RW perms */
        ppi = paddr >> 12;
        if (ppi >= vm->pages.guest_n_pages)
          monpanic(vm, "INSW: ppi>n_pages\n");
        pusage = getPageUsage(vm, ppi);
        /* See if special monitor protections allow for a write to
         * this physical address.  If not, there may be special constructs
         * in memory in this page.
         */
        if ( (pusage->attr.fields.access_perm == PagePermRO) &&
             (pusage->attr.fields.vcode) ) {
          /* This write will invalide the page cache.  We might as well
           * do this now, and fall through to the next check.  If this
           * upgrades the permissions to RW, then we saved some time.
           */
          dumpVCodePage(vm, ppi, pusage);
          removePageAttributes(vm, ppi, PageUsageVCode);
          }
        if (pusage->attr.fields.access_perm == PagePermRW) {
          /* See how many words can fit in the rest of this page */
          words = (0x1000 - (paddr & 0xfff)) >> 1;
          if (words) {
            /* Restrict word count to the number that will fit in this
             * page. */
            if (count_requested > words)
              count_requested = words;
            count_actual = sysIOInBatch(vm, G_DX(vm), 2, count_requested,
                                        paddr);
            if ( (count_actual > count_requested) ||
                 (count_actual == 0) )
              monpanic(vm, "INSW: count req=%u, actual=%u.\n",
                       count_requested, count_actual);
            /* Decrement eCX.  Note, the main loop will decrement 1 */
            if (vm->i.as_32)
              G_ECX(vm) -= (count_actual-1);
            else
              G_CX(vm)  -= (count_actual-1);
            incr = count_actual << 1; /* count * 2 */
            goto do_incr;
            }
          }
        }
      }
#endif

    /* Write a zero to memory, to trigger any segment or page */
    /* faults before reading from IO port. */
    write_virtual_word(vm, SRegES, edi, &value16);

    value16 = sysIOIn(vm, G_DX(vm), 2);

    /* no seg override allowed */
    write_virtual_word(vm, SRegES, edi, &value16);
    incr = 2;
    }

#if COSIMULATE == 0
do_incr:
#endif

  if (vm->i.as_32) {
    if (G_GetDF(vm))
      G_EDI(vm) -= incr;
    else
      G_EDI(vm) += incr;
    }
  else {
    if (G_GetDF(vm))
      G_DI(vm) -= incr;
    else
      G_DI(vm) += incr;
    }
}

  void
OUTSW_DXXv(vm_t *vm)
{
  unsigned seg;
  Bit32u esi;
  unsigned int incr;

  if (!NULL_SEG_REG(vm->i.seg)) {
    seg = vm->i.seg;
    }
  else {
    seg = SRegDS;
    }

  if (vm->i.as_32)
    esi = G_ESI(vm);
  else
    esi = G_SI(vm);

  if (vm->i.os_32) {
    Bit32u value32;

    IOPermCheck(vm, G_DX(vm), 4);

    read_virtual_dword(vm, seg, esi, &value32);

    sysIOOut(vm, G_DX(vm), 4, value32);
    incr = 4;
    }
  else {
    Bit16u value16;

    IOPermCheck(vm, G_DX(vm), 2);

    read_virtual_word(vm, seg, esi, &value16);

    sysIOOut(vm, G_DX(vm), 2, value16);
    incr = 2;
    }

  if (vm->i.as_32) {
    if (G_GetDF(vm))
      G_ESI(vm) -= incr;
    else
      G_ESI(vm) += incr;
    }
  else {
    if (G_GetDF(vm))
      G_SI(vm) -= incr;
    else
      G_SI(vm) += incr;
    }
}

  void
OUT_IbeAX(vm_t *vm)
{
  Bit32u port;
 
  port = vm->i.Ib;
 
  if (vm->i.os_32) {
    IOPermCheck(vm, port, 4);
    sysIOOut(vm, port, 4, G_EAX(vm));
    }
  else {
    IOPermCheck(vm, port, 2);
    sysIOOut(vm, port, 2, G_AX(vm));
    }
}

  void
IN_ALDX(vm_t *vm)
{
  unsigned port;

  port = G_DX(vm);
  IOPermCheck(vm, port, 1);
  G_AL(vm) = sysIOIn(vm, port, 1);
}

  void
IN_eAXDX(vm_t *vm)
{
  unsigned port = G_DX(vm);

  if (vm->i.os_32) {
    IOPermCheck(vm, port, 4);
    G_EAX(vm) = sysIOIn(vm, port, 4);
    }
  else {
    IOPermCheck(vm, port, 2);
    G_AX(vm) = sysIOIn(vm, port, 2);
    }
}

  void
IN_ALIb(vm_t *vm)
{
  Bit8u  al;
  unsigned port;

  port = vm->i.Ib;
  IOPermCheck(vm, port, 1);
  al = sysIOIn(vm, port, 1);
  G_AL(vm) = al;
}

  void
IN_eAXIb(vm_t *vm)
{
  unsigned port = vm->i.Ib;
 
  if (vm->i.os_32) {
    IOPermCheck(vm, port, 4);
    G_EAX(vm) = sysIOIn(vm, port, 4);
    }
  else {
    IOPermCheck(vm, port, 2);
    G_AX(vm) = sysIOIn(vm, port, 2);
    }
}

  void
OUT_IbAL(vm_t *vm)
{
  unsigned port;

  port = vm->i.Ib;
  IOPermCheck(vm, port, 1);
  sysIOOut(vm, port, 1, G_AL(vm));
}
