/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2000 Kevin P. Lawton
 *
 *  shift32.c:  emulation of 32-bit bitshift instructions
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "monitor.h"


  void
SHRD_EdGd(vm_t *vm)
{
  Bit32u op1, op2;
  unsigned count;
  Bit32u eflags, arithmetic_eflags;
 
  /* op1:op2 << count.  result stored in op1 */
  if (vm->i.b1 == 0x1ac)
    count = vm->i.Ib;
  else /* 0x1ad */
    count = G_CL(vm);
 
  count &= 0x1f; /* use only 5 LSB's */
 
  /* SHRD do read before terminating on count=0? +++ */
  if (!count) return; /* NOP */
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
  op2 = ReadReg32(vm, vm->i.nnn);

  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %5, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "shrdl %%cl, %6, %1    \n\t"
    "pushfl                \n\t"
    "popl   %0"
    : "=r" (eflags), "=g" (op1)
    : "1" (op1), "c" (count), "0" (eflags), "g" (arithmetic_eflags),
      "r" (op2)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg32(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_dword(vm, &op1);
    }
  SetOSZAPC(vm, eflags);
}

  void
SHLD_EdGd(vm_t *vm)
{
  Bit32u op1, op2;
  unsigned count;
  Bit32u eflags, arithmetic_eflags;
 
  /* op1:op2 << count.  result stored in op1 */
  if (vm->i.b1 == 0x1a4)
    count = vm->i.Ib;
  else /* 0x1a5 */
    count = G_CL(vm);
 
  count &= 0x1f; /* use only 5 LSB's */
 
  /* SHLD do read before terminating on count=0? +++ */
  if (!count) return; /* NOP */
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
  op2 = ReadReg32(vm, vm->i.nnn);

  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %5, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "shldl %%cl, %6, %1    \n\t"
    "pushfl                \n\t"
    "popl   %0"
    : "=r" (eflags), "=g" (op1)
    : "1" (op1), "c" (count), "0" (eflags), "g" (arithmetic_eflags),
      "r" (op2)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg32(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_dword(vm, &op1);
    }
  SetOSZAPC(vm, eflags);
}

  void
ROL_Ed(vm_t *vm)
{
  Bit32u   op1;
  unsigned count;
  Bit32u   eflags, arithmetic_eflags;
 
  if ( vm->i.b1 == 0xc1 )
    count = vm->i.Ib & 0x1f;
  else if ( vm->i.b1 == 0xd1 )
    count = 1;
  else /* 0xd3 */
    count = G_CL(vm) & 0x1f;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  if (!count) return;
 
  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %5, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "roll %%cl, %1         \n\t" /* Do operation */
    "pushfl                \n\t" /* Get resultant flags */
    "popl   %0"
    : "=r" (eflags), "=g" (op1)
    : "1" (op1), "c" (count), "0" (eflags), "g" (arithmetic_eflags)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg32(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_dword(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
ROR_Ed(vm_t *vm)
{
  Bit32u   op1;
  unsigned count;
  Bit32u   eflags, arithmetic_eflags;
 
  if (vm->i.b1 == 0xc1)
    count = vm->i.Ib & 0x1f;
  else if (vm->i.b1 == 0xd1)
    count = 1;
  else /* 0xd3 */
    count = G_CL(vm) & 0x1f;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  if (!count) return;
  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %5, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "rorl %%cl, %1         \n\t" /* Do operation */
    "pushfl                \n\t" /* Get resultant flags */
    "popl   %0"
    : "=r" (eflags), "=g" (op1)
    : "1" (op1), "c" (count), "0" (eflags), "g" (arithmetic_eflags)
    : "memory", "cc"
    );

  if (vm->i.mod == 0xc0) {
    WriteReg32(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_dword(vm, &op1);
    }

  SetOSZAPC(vm, eflags);
}

  void
RCL_Ed(vm_t *vm)
{
  Bit32u op1;
  unsigned count;
  Bit32u eflags, arithmetic_eflags;
 
  if ( vm->i.b1 == 0xc1 )
    count = vm->i.Ib & 0x1f;
  else if ( vm->i.b1 == 0xd1 )
    count = 1;
  else /* 0xd3 */
    count = G_CL(vm) & 0x1f;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  if (!count) return;
 
  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %5, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "rcll %%cl, %1         \n\t" /* Do operation */
    "pushfl                \n\t" /* Get resultant flags */
    "popl   %0"
    : "=r" (eflags), "=g" (op1)
    : "1" (op1), "c" (count), "0" (eflags), "g" (arithmetic_eflags)
    : "memory", "cc"
    );

  if (vm->i.mod == 0xc0) {
    WriteReg32(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_dword(vm, &op1);
    }

  SetOSZAPC(vm, eflags);
}

  void
RCR_Ed(vm_t *vm)
{
  Bit32u op1;
  unsigned count;
  Bit32u eflags, arithmetic_eflags;
 
  if ( vm->i.b1 == 0xc1 )
    count = vm->i.Ib & 0x1f;
  else if ( vm->i.b1 == 0xd1 )
    count = 1;
  else /* 0xd3 */
    count = G_CL(vm) & 0x1f;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  if (!count) return;
 
  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %5, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "rcrl %%cl, %1         \n\t" /* Do operation */
    "pushfl                \n\t" /* Get resultant flags */
    "popl   %0"
    : "=r" (eflags), "=g" (op1)
    : "1" (op1), "c" (count), "0" (eflags), "g" (arithmetic_eflags)
    : "memory", "cc"
    );

  if (vm->i.mod == 0xc0) {
    WriteReg32(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_dword(vm, &op1);
    }

  SetOSZAPC(vm, eflags);
}

  void
SHL_Ed(vm_t *vm)
{
  Bit32u op1;
  unsigned count;
  Bit32u eflags, arithmetic_eflags;
 
  if ( vm->i.b1 == 0xc1 )
    count = vm->i.Ib & 0x1f;
  else if ( vm->i.b1 == 0xd1 )
    count = 1;
  else /* 0xd3 */
    count = G_CL(vm) & 0x1f;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  if (!count) return;
 
  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %5, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "shll %%cl, %1         \n\t"
    "pushfl                \n\t"
    "popl   %0"
    : "=r" (eflags), "=g" (op1)
    : "1" (op1), "c" (count), "0" (eflags), "g" (arithmetic_eflags)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg32(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_dword(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
SHR_Ed(vm_t *vm)
{
  Bit32u op1;
  unsigned count;
  Bit32u eflags, arithmetic_eflags;
 
  if ( vm->i.b1 == 0xc1 )
    count = vm->i.Ib & 0x1f;
  else if ( vm->i.b1 == 0xd1 )
    count = 1;
  else /* 0xd3 */
    count = G_CL(vm) & 0x1f;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  if (!count) return;
 
  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %5, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "shrl %%cl, %1         \n\t"
    "pushfl                \n\t"
    "popl   %0"
    : "=r" (eflags), "=g" (op1)
    : "1" (op1), "c" (count), "0" (eflags), "g" (arithmetic_eflags)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg32(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_dword(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
SAR_Ed(vm_t *vm)
{
  Bit32u op1;
  Bit32u eflags, arithmetic_eflags;
  unsigned count;
 
  if ( vm->i.b1 == 0xc1 )
    count = vm->i.Ib & 0x1f;
  else if ( vm->i.b1 == 0xd1 )
    count = 1;
  else /* 0xd3 */
    count = G_CL(vm) & 0x1f;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  /* Does this SAR count check occur before op1 read? +++ */
  if (!count) return;
 
  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %5, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "sarl %%cl, %1         \n\t" /* Do operation */
    "pushfl                \n\t" /* Get resultant flags */
    "popl   %0"
    : "=r" (eflags), "=g" (op1)
    : "1" (op1), "c" (count), "0" (eflags), "g" (arithmetic_eflags)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg32(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_dword(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}
