/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2000 Kevin P. Lawton
 *
 *  fault.c:  fault/int handlers for VM monitor
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#define IN_MONITOR_SPACE
#include "monitor.h"


/* The monitor stack frame:
 * 
 * gs 
 * fs        These are pushed by the CPU _only if_ transitioning from
 * ds        guest code running in v86 mode.
 * es 
 *
 * ss
 * esp
 * eflags    Values pushed by the CPU and interrupt stub.  To simplify
 * cs        things, the stub pushes an error of zero for those
 * eip       events which don't naturally cause an error push, and
 * error     also pushes the vector of the exception/interrupt.
 * vector
 *
 * eax
 * ecx
 * edx       General registers, pushed with a PUSHA instruction,
 * ebx       by code below.
 * <esp>
 * ebp
 * esi
 * edi
 *
 * es
 * ds        Segment selectors, pushed by code below.
 * fs
 * gs
 */


Bit32u last_laddr;
unsigned last_event;

static inline Bit32u read_cr2( void )
{
    Bit32u cr2;
    asm volatile("movl %%cr2, %0" : "=r" (cr2));
    return cr2;
}


asm 
(
/*
 * Fault handler stub for all exceptions
 */
".globl __handle_fault  \n" /* Start function __handle_fault() */
"__handle_fault:        \n"
"  pushal               \n" /* Save guest general registers */
"  pushl %es            \n" /* Save guest segment registers */
"  pushl %ds            \n"
"  pushl %fs            \n"
"  pushl %gs            \n"

"  movw  %ss, %ax       \n" /* Copy SS into DS/ES */
"  movw  %ax, %ds       \n"
"  movw  %ax, %es       \n"
"  cld                  \n" /* gcc-compiled code needs this */
"  call handle_fault    \n" /* Call the monitor-side fault handler */
"  cmpl $0x1, %eax      \n" /* Was event generated from monitor code? */
"  je   __ret_to_monitor\n"

".globl __ret_to_guest  \n" /* Event was from generated from guest code */
"__ret_to_guest:        \n"
"  call sbe             \n" /* Prepare fro return to guest */
"  cmpl $0x1, %eax      \n" /* Is guest now in PM? */
"  jb   __ret_to_v86    \n" /* case 0: return to guest in v86 mode */

".globl __ret_to_pm     \n" /* Return to guest in PM */
"__ret_to_pm:           \n"
"  popl  %gs            \n" /* Restore guest segments */
"  popl  %fs            \n"
"  popl  %ds            \n"
"  popl  %es            \n"
"  popal                \n" /* Restore guest general registers */
"  addl  $8, %esp       \n" /* Ignore vector and error dwords */
"  iret                 \n" /* Resume execution of guest */

".globl __ret_to_v86    \n" /* Guest is now in v86 */
"__ret_to_v86:          \n"
"  addl $16, %esp       \n" /* Ignore, segs are in IRET area for v86 */
"  popal                \n" /* Restore guest general registers */
"  addl $8, %esp        \n" /* Ignore vector and error dwords */
"  iret                 \n" /* Resume execution of guest */

".globl __ret_to_monitor\n" /* Return to guest in v86 */
"__ret_to_monitor:      \n"
"  popl  %gs            \n" /* Restore monitor segments */
"  popl  %fs            \n"
"  popl  %ds            \n"
"  popl  %es            \n"
"  popal                \n" /* Restore monitor general registers */
"  addl  $8, %esp       \n" /* ignore vector and error dwords */
"  iret                 \n" /* Resume execution of monitor */


/*
 * Hardware interrupt handler stub
 */
".globl __handle_int    \n" /* Return to monitor code */
"__handle_int:          \n"
"  pushal               \n" /* Save guest general registers */
"  pushl %es            \n" /* Save guest segment registers */
"  pushl %ds            \n"
"  pushl %fs            \n"
"  pushl %gs            \n"

"  movw  %ss, %ax       \n" /* Copy SS into DS/ES */
"  movw  %ax, %ds       \n"
"  movw  %ax, %es       \n"
"  cld                  \n" /* gcc-compiled code needs this */
"  call handle_int      \n" /* monitor interrupt handler */
"  cmpl $0x1, %eax      \n" /* Was interrupt generated from monitor code? */
"  je   __ret_to_monitor\n" /* Yes, so return to monitor code */
"  jmp  __ret_to_guest  \n" /* No, so return to guest code */
);



  unsigned
sbe( guest_context_t context, v86_sregs_t v86_sregs )
{
    nexus_t *nexus = (nexus_t *) (((Bit32u) &context) & 0xfffff000);
    vm_t    *vm    = (vm_t *) nexus->vm;

    Bit32u   guest_laddr, guest_paddr, guest_ppage_index;
    unsigned seg32;
    unsigned icache_index;
    unsigned pdi, pti;
    Bit8u    temp8;
    pageEntry_t pte_new;
    Bit32u private_codepage_laddr, gerror;
    unsigned us, rw;
    page_t *monPTbl;
    unsigned pt_index;
    unsigned m;

#if ANAL_CHECKS
    if ( !(context.eflags & FLG_VM) && !(context.cs & 0x0003) )
      monpanic(vm, "sbe: monitor code!\n");
#endif


#if ANAL_CHECKS
    if ( !(context.eflags & FLG_IF) )
       monpanic(vm, "sbe: guest IF=0\n");

    if ( context.eflags & (
         FLG_IOPL | /* FLG_NT | FLG_RF | FLG_VM | FLG_AC | */
         FLG_VIF | FLG_VIP) )
       monpanic(vm, "sbe: eflags=0x%x\n", context.eflags);
#endif


    STI();

loop:
    guest_cycles_elapsed(vm);

    /* Past execution (emulation) could have created a need for
     * a remap of the guest in the monitor space
     */
    if (vm->modeChange)
      monModeChange(vm);

    /* Check for asynchronous events, inhibits, traps etc before
     * executing guest code
     */
    emulate_async_checks(vm);

    /* +++ Look again here???  Remove check above??? */
    if (vm->modeChange)
      monModeChange(vm);

/* Notes:
 *   if (vm->guest_cpu.inhibit_mask || vm->guest_cpu.debug_trap)
 *     then have to make sure inhibits are honored.
 */

    /* See if cosimulation demands the guest takes an interrupt
     * at this point.  This would be the case if it is being
     * driven as a slave simulator.
     */
    if (vm->dbg_force_int) {
      unsigned vector = vm->dbg_force_int & 0xff;
      if (vm->guest_cpu.inhibit_mask || vm->guest_cpu.debug_trap)
        monpanic(vm, "sbe: force_int w/ IM || DT\n");
      vm->dbg_force_int = 0;
      emulate_interrupt(vm, vector);
      sysReqComplete(vm);
      /* An interrupt may put the VM in a state which requires
       * remapping.  Repeat the process
       */
      goto loop;
      }

    switch (vm->executeMethod) {
      case RunGuestNMethodExecute:
        /* Really, should copy guest TF to monitor TF here.  For
         * now just turn off TF always.  Need to check what case
         * set it in handler when enabled.
         */
        if (vm->guest_cpu.inhibit_mask || vm->guest_cpu.debug_trap)
          monpanic(vm, "sbe: MethodExecute: IM || DT\n");
        context.eflags &= ~FLG_TF;
        break;

      case RunGuestNMethodEmulate:
        emulate_instr(vm, &context, 4);
        /* Above call will decrement executeN if execution completes */
        goto loop;

      case RunGuestNMethodBreakpoint:
        /* Regardless of guest TF, we need to set monitor TF to generate
         * a breakpoint after the next instruction executes
         */
        context.eflags |= FLG_TF;
        break;

      default:
        monpanic(vm, "sbe: Method default\n");
      }

    cache_sreg(vm, SRegCS);
    if (context.eip > vm->guest_cpu.desc_cache[SRegCS].limit_scaled)
      monpanic(vm, "sbe: guest EIP > CS.limit\n");
    guest_laddr = vm->guest_cpu.desc_cache[SRegCS].base + context.eip;
    us = (G_GetCPL(vm)==3);
    rw = 0; /* need at least read priv for code */
    switch ( m = map_guest_laddr(vm, guest_laddr, &guest_ppage_index,
                             us, rw, PageUsageVCode, &gerror) ) {
      case MapLinOK:
      case MapLinAlreadyMapped:
        break;
      case MapLinMonConflict:
        monpanic(vm, "sbe: MapLinMonConflict:\n");
      case MapLinPPageOOB:
        monpanic(vm, "sbe: MapLinPPageOOB:\n");
      case MapLinException:
        /* Executing code at this location would generate a #PF */
        vm->guest_cpu.cr2 = guest_laddr;
        emulate_exception(vm, ExceptionPF, gerror);
        /* An exception means a possible monitor remap request, mode
         * change etc.  We need to repeat this process.
         */
        goto loop;
      case MapLinEmulate:
        monpanic(vm, "sbe: MapLinEmulate:\n");
        emulate_instr(vm, &context, 5);
        goto loop;
      default:
        monpanic(vm, "sbe: MapLin: default case:\n");
      }

    seg32 = vm->guest_cpu.desc_cache[SRegCS].desc.d_b;
    guest_paddr = (guest_ppage_index<<12) | (guest_laddr & 0xfff);

    /* The monitor linear address of the real code page */
    vm->codepage_laddr = Guest2Monitor(vm, guest_laddr & 0xfffff000);

    /* The address of the PTE pointing to the real code page */
    pdi = guest_laddr >> 22;
    pti = (guest_laddr >> 12) & 0x3ff;

    pt_index = getMonPTi(vm, pdi, 15);
    monPTbl = &vm->guest.addr.page_tbl[pt_index];

    vm->codepage_pte_p = &monPTbl->pte[pti];
    vm->codepage_pte_saved = *vm->codepage_pte_p;

    icache_index = prescan(vm, (Bit8u *) vm->codepage_laddr, guest_paddr,
                           seg32, G_GetCPL(vm), 1);
    if (icache_index == (unsigned)-1) {
      /* If the very first instruction is virtualized, there is no
       * sense trying to execute it.  It will only result in emulation
       * of the instruction, so just do that now.
       */
      emulate_instr(vm, &context, 5);
      goto loop;
      }

    /* Update timestamp for latest time of guest execution */
    vm->code_cache[icache_index].ts.guest_executed = vm_rdtsc();


    /* Split I&D TLB trick.  Our scan-before-execute technique */
    /* creates a private modified (virtualized) copy of the current */
    /* code page.  We want to be able to execute from this private */
    /* code page, yet allow memory accesses to see the original */
    /* code/data page so code does not detect our changes.  Since */
    /* the IA32 processor does not offer a native mechanism to */
    /* differentiate between read and code accesses within a page, */
    /* we make use of the split I&D TLB nature.  What we need to */
    /* do is to get the TLB to load a value with user privilege */
    /* into the I TLB cache, then reset the corresponding page */
    /* table entry to contain supervisor privilege.  Thus code */
    /* will continue to execute based on the user privilege level, */
    /* yet data accesses will fetch the entry from the actual */
    /* page tables and cause an exception. */

    /* The linear address of the virtualized code page */
    private_codepage_laddr =
      ((Bit32u)vm->guest.addr.icache) + (icache_index<<12);

    /* The temporary PTE pointing to the virtualized page */
    pte_new.fields.base = vm->pages.icache[icache_index];
    pte_new.fields.avail = 0;
    pte_new.fields.G = 0;      /* not global           */
    pte_new.fields.PS = 0;     /* (unused in pte)      */
    pte_new.fields.D = 0;      /* clean                */
    pte_new.fields.A = 0;      /* not accessed         */
    pte_new.fields.PCD = 0;    /* normal caching       */
    pte_new.fields.PWT = 0;    /* normal write-back    */
    pte_new.fields.US = 1;     /* user                 */
    pte_new.fields.RW = 0;     /* read-only            */
    pte_new.fields.P = 1;      /* present in memory    */

    if (GetMonMode(vm) == MonModeVM) {
      /* If guest code is monitored in V86M, then it is important
       * that we load the context area with the correct selectors,
       * even though they are currently virtualized.
       */

      if ( vm->selectorInEmu & (1 << SRegES) )
        context.es = vm->guest_cpu.selector[SRegES].raw;
      if ( vm->selectorInEmu & (1 << SRegCS) )
        context.cs = vm->guest_cpu.selector[SRegCS].raw;
      if ( vm->selectorInEmu & (1 << SRegSS) )
        context.ss = vm->guest_cpu.selector[SRegSS].raw;
      if ( vm->selectorInEmu & (1 << SRegDS) )
        context.ds = vm->guest_cpu.selector[SRegDS].raw;
      if ( vm->selectorInEmu & (1 << SRegFS) )
        context.fs = vm->guest_cpu.selector[SRegFS].raw;
      if ( vm->selectorInEmu & (1 << SRegGS) )
        context.gs = vm->guest_cpu.selector[SRegGS].raw;

      /* Since we are executing guest code using v86 mode, we will
       * allow natural selector reads, and segment reloads to occur
       * unvirtualized.  Thus, we need to mark the fact that selectors
       * and descriptor cache values are no longer consistent with
       * those stored in the guest_cpu area.  
       */
      vm->selectorInEmu   = 0;
      vm->descriptorInEmu = 0;
      vm->segmentUpdated  = 0;

/* +++ we could eliminate double copy of selectors here and below */
      }
    else if (GetGuestMode(vm) == GuestModeRM) {
if ( (vm->selectorInEmu & 0x3f) != 0x3f )
  monpanic(vm, "sbe: MonModeVM, sInEmu=0x%x\n",
    vm->selectorInEmu);

      /* MonModePMR3, GuestModeRM.  Since the monitor is not running
       * guest code in VM, there must be non-compatible values in
       * the descriptor caches.  In this case we are virtualizing
       * segment registers accesses, and using PM selectors.  Update
       * virtualized descriptors based on updated guest values.  In
       * this case, there is no notion of a null selector.
       */
      if ( vm->segmentUpdated ) {
        unsigned i;
        for (i=0; i<6; i++) {
          if (vm->guest_cpu.desc_cache[i].valid) {
            vm->guest.addr.gdt[4+i] = vm->guest_cpu.desc_cache[i].desc;
            vm->guest.addr.gdt[4+i].dpl = 3;
            }
          else {
            vm->guest.addr.gdt[4+i] = nullDescriptor;
            }
          }
        vm->segmentUpdated = 0;
        }
      }
    else {
if ( (vm->selectorInEmu & 0x3f) != 0x3f )
  monpanic(vm, "sbe: MonModeVM, sInEmu=0x%x\n",
    vm->selectorInEmu);

      if ( vm->segmentUpdated ) {
        unsigned i;

/* +++ similar code in mon-mode.c */
        /* If the descriptor caches for each segment register is
         * invalid, then set the virtualized selector to NULL so
         * an exception is not generated upon return to the guest.
         * Otherwise, point them at virtualized descriptors representing
         * the shadow cache values of each segment register.
         */
        if ( vm->segmentUpdated & (1<<SRegES) ) {
          if ( !vm->guest_cpu.desc_cache[SRegES].valid )
            vm->guest.addr.guest_context->es = nullSelector.raw;
          else
            vm->guest.addr.guest_context->es = Selector(4, 0, RPL3);
          }
     
        if ( vm->segmentUpdated & (1<<SRegCS) ) {
          if ( !vm->guest_cpu.desc_cache[SRegCS].valid )
            vm->guest.addr.guest_context->cs = nullSelector.raw;
          else
            vm->guest.addr.guest_context->cs = Selector(5, 0, RPL3);
          }

        if ( vm->segmentUpdated & (1<<SRegSS) ) {
          if ( !vm->guest_cpu.desc_cache[SRegSS].valid )
            vm->guest.addr.guest_context->ss = nullSelector.raw;
          else
            vm->guest.addr.guest_context->ss = Selector(6, 0, RPL3);
          }

        if ( vm->segmentUpdated & (1<<SRegDS) ) {
          if ( !vm->guest_cpu.desc_cache[SRegDS].valid )
            vm->guest.addr.guest_context->ds = nullSelector.raw;
          else
            vm->guest.addr.guest_context->ds = Selector(7, 0, RPL3);
          }

        if ( vm->segmentUpdated & (1<<SRegFS) ) {
          if ( !vm->guest_cpu.desc_cache[SRegFS].valid )
            vm->guest.addr.guest_context->fs = nullSelector.raw;
          else
            vm->guest.addr.guest_context->fs = Selector(8, 0, RPL3);
          }

        if ( vm->segmentUpdated & (1<<SRegGS) ) {
          if ( !vm->guest_cpu.desc_cache[SRegGS].valid )
            vm->guest.addr.guest_context->gs = nullSelector.raw;
          else
            vm->guest.addr.guest_context->gs = Selector(9, 0, RPL3);
          }

        for (i=0; i<6; i++) {
          if (vm->guest_cpu.desc_cache[i].valid) {
            vm->guest.addr.gdt[4+i] = vm->guest_cpu.desc_cache[i].desc;
            vm->guest.addr.gdt[4+i].dpl = 3;
            }
          else {
            vm->guest.addr.gdt[4+i] = nullDescriptor;
            }
          }
        vm->segmentUpdated = 0;
        }
      }

    if (context.eflags & FLG_VM) {
      /* Copy working sregs back to stack so IRET will use them. */
      v86_sregs.es = context.es;
      v86_sregs.ds = context.ds;
      v86_sregs.fs = context.fs;
      v86_sregs.gs = context.gs;
      vm->guest.addr.tss->esp0 = ((Bit32u)nexus) + PAGESIZE;
      }
    else {
      vm->guest.addr.tss->esp0 =
        ((Bit32u)nexus) + PAGESIZE - sizeof(v86_sregs_t);
      }

    temp8 = 0; /* keep compiler happy */
    CLI();
    asm volatile
    (
        /* Write 'virtual' page table entry, saving the real one. */
        "xchgl %4, (%3)\n\t"

        /* Save byte at beginning of virtualized code page and replace
           it with the opcode for RET, using the alternate address. */
        "movb $0xc3, %5\n\t"
        "xchgb %5, (%1)\n\t"

        /* Invalidate the page entry for the real code page address (I&D). */
        "invlpg (%2)\n\t"

        /* Call the RET instruction (just comes right back) using the actual
           code page address; this loads the TLB (I-only). */
        "call *%2\n\t"

        /* Replace the RET instruction with the original byte, using
           the alternate address. */
        "xchgb %5, (%1)\n\t"

        /* Restore the 'real' page table entry, activating protection. */
        "andl $0xfffffff8, %4\n\t"
        "movl %4, (%3)"

        : "=d" (temp8)
        : "r" (private_codepage_laddr), "r" (vm->codepage_laddr),
          "r" (vm->codepage_pte_p), "r" (pte_new), "0" (temp8)
        : "memory"
    );
    vm->system.t0 = vm_rdtsc();
    if (context.eflags & FLG_VM)
      return 0; /* to guest in V86M */
    else
      return 1; /* to guest in PM */
}


  unsigned
handle_int( guest_context_t context, v86_sregs_t v86_sregs )
/*
 * handle_int(): Redirect a hardware interrupt back to the host
 */
{
    nexus_t *nexus = (nexus_t *) (((Bit32u) &context) & 0xfffff000);
    vm_t    *vm    = (vm_t *) nexus->vm;
    unsigned from_monitor;

    if (context.eflags & FLG_VM) {
      /* Interrupt from guest code executing in v8086 mode. */
      /* The data segment selectors will have been pushed by the */
      /* the CPU for the interrupt.  Copy them down to the area */
      /* where we save them for protected mode. */
#if 0
      vm->guest_cpu.selector[SRegES].raw = context.es = v86_sregs.es;
      vm->guest_cpu.selector[SRegDS].raw = context.ds = v86_sregs.ds;
      vm->guest_cpu.selector[SRegFS].raw = context.fs = v86_sregs.fs;
      vm->guest_cpu.selector[SRegGS].raw = context.gs = v86_sregs.gs;
      vm->guest_cpu.selector[SRegSS].raw = context.ss;
      vm->guest_cpu.selector[SRegCS].raw = context.cs;
      vm->selectorInEmu = 0x3f;
#else
      context.es = v86_sregs.es;
      context.ds = v86_sregs.ds;
      context.fs = v86_sregs.fs;
      context.gs = v86_sregs.gs;
#endif
      from_monitor = 0; /* Event from guest code */
      }
    else if ( (context.cs & 0x0003) == 0x0003 ) {
      from_monitor = 0; /* Event from guest code */
      }
    else {
      from_monitor = 1; /* Event from monitor code */
      }

    if ( !from_monitor ) {
      Bit64u t1;

      if (1) { /* +++ ultimately depend on sbe flag */
        /* Reinstate the PTE for the current code page. */
        *vm->codepage_pte_p = vm->codepage_pte_saved;
        /* +++ the following call not really needed. */
        invlpg_mon_offset(vm->codepage_laddr);
        }

      /* End of elapsed guest execution duration.  Add elapsed */
      /* cycles to time framework. */
      t1 = vm_rdtsc();
      vm->system.elapsed += (t1 - vm->system.t0);
      }

    /* Interrupts are off naturally here. */
    vm->mon_request = MON_REQ_REDIRECT;
    vm->redirect_vector = context.vector;
    vm->guest.__mon2host();
    return(from_monitor);
}


  unsigned
handle_fault( guest_context_t context, v86_sregs_t v86_sregs )
/*
 *  handle_fault():  main monitor-side fault handler
 *
 *  This fault handler is called from the assembly stub
 *  __handle_fault.
 *
 *  INPUTS:
 *     context: guest context at point of exception
 *
 */
{
    nexus_t *nexus = (nexus_t *) (((Bit32u) &context) & 0xfffff000);
    vm_t    *vm    = (vm_t *) nexus->vm;
    Bit32u  cr2    = read_cr2();
    Bit64u t1;
    unsigned from_monitor;

    if (context.eflags & FLG_VM) {
      /* Interrupt from guest code executing in v8086 mode. */
      /* The data segment selectors will have been pushed by the */
      /* the CPU for the interrupt.  Copy them down to the area */
      /* where we save them for protected mode. */
#if 0
      vm->guest_cpu.selector[SRegES].raw = context.es = v86_sregs.es;
      vm->guest_cpu.selector[SRegDS].raw = context.ds = v86_sregs.ds;
      vm->guest_cpu.selector[SRegFS].raw = context.fs = v86_sregs.fs;
      vm->guest_cpu.selector[SRegGS].raw = context.gs = v86_sregs.gs;
      vm->guest_cpu.selector[SRegSS].raw = context.ss;
      vm->guest_cpu.selector[SRegCS].raw = context.cs;
      vm->selectorInEmu = 0x3f;
#else
      context.es = v86_sregs.es;
      context.ds = v86_sregs.ds;
      context.fs = v86_sregs.fs;
      context.gs = v86_sregs.gs;
#endif
      from_monitor = 0; /* Event from guest code */
      }
    else if ( (context.cs & 0x0003) == 0x0003 ) {
      from_monitor = 0; /* Event from guest code */
      }
    else {
      /* Abort if fault happened inside monitor code. */
      vm->abort_code = 1;
      monpanic_nomess(vm);
      from_monitor = 1; /* Event from monitor code */
      }

#if ANAL_CHECKS
    if ( !(context.eflags & FLG_IF) )
      monpanic(vm, "handle_int: guest IF=0\n");
#endif

    /* End of elapsed guest execution duration */
    t1 = vm_rdtsc();
    vm->system.elapsed += (t1 - vm->system.t0);

    /* +++ Eventually, skip INVLPG if sbe==0 */
    if (1) {
      *vm->codepage_pte_p = vm->codepage_pte_saved;
      invlpg_mon_offset(vm->codepage_laddr);
      }

    STI();

    switch ( context.vector )
    {
    case ExceptionDB: /* 1 */

      if (vm->executeMethod == RunGuestNMethodBreakpoint) {
        /* Breakpoint generated because we requested it via TF=1 */
        if (vm->executeN) {
          vm->executeN--;
          if (vm->executeN==0) {
            sysEOICount(vm);
            }
          }
#warning "breakpoint elapsed hack"
        vm->system.elapsed = 400;
        }
      else {
        monpanic(vm, "handle_fault: #DB, method=%u not coded\n",
          vm->executeMethod);
        }
      break;

    case ExceptionBR: /* 5 */
        /* BOUND instruction fault; array index not in bounds */
        emulate_exception(vm, context.vector, 0);
        break;

    case ExceptionDE: /* 0 */
    case ExceptionBP: /* 3 */
    case ExceptionOF: /* 4 */
    case ExceptionNM: /* 7 */
    case ExceptionMF: /* 16 */
        monpanic(vm, "handle_fault: %u\n", context.vector);
        /* emulate_interrupt(vm, context.vector); */
        break;

    case ExceptionNP: /* 11 */
    case ExceptionSS: /* 12 */
    case ExceptionAC: /* 17 */
        /* use emulate_xyz() */
        /*interrupt(vm, context.vector, 0, 1, context.error); */
monprint(vm, "last_laddr = 0x%x event=0x%x\n", last_laddr, last_event);
        monpanic(vm, "handle_fault: %u\n", context.vector);
        break;

    case ExceptionUD: /* 6 */
    case ExceptionGP: /* 13 */
        emulate_instr(vm, &context, 1);
        break;

    case ExceptionPF: /* 14 */
        monPageFault(vm, &context, cr2);
        break;

    default:
        monpanic(vm, "Other Fault: %u\n", context.vector);
        break;
    }

  return(from_monitor);
}
