/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2000  Kevin P. Lawton
 *
 *  mon-mode.c:  Handling of mode to run guest code in.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

/* TODO:
 *   - kick out of RMMonPM when descriptor caches are compat w/ RM
 *     (and switch to differnet virtualization matrix)
 */

#include "plex86.h"
#define IN_MONITOR_SPACE
#include "monitor.h"


/* +++ Deal with desc.valid==0 */
/* +++ Deal with page remapping upon certain transitions */
/* +++ Integrate SBE flag into vm->guestMode */
/* +++ Fix interaction with set_cpu */


unsigned isV86MCompatible(vm_t *);


#warning "need to deal with LGDT for sbe==0"



  void
monModeChange(vm_t *vm)
{
  unsigned virtualizeSegRegs = 0;

  if (vm->modeChange & ModeChangeCheck) {
    /* If the monitor saw an event which may change the mode of
     * the guest, such as a CS register reload, than have a look.
     */
    unsigned newGuestMode;
    if (V8086Mode(vm))
      newGuestMode = GuestModeVM;
    else if (RealMode(vm))
      newGuestMode = GuestModeRM;
    else {
      newGuestMode = GuestModePMR0 | G_GetCPL(vm);
      }
    if (GetGuestMode(vm) != newGuestMode) {
      /* Yep, the guest's mode changed.  Flag this as a transition */
      vm->modeChange |=  ModeChangeTransition;
      }
    vm->modeChange &= ~ModeChangeCheck;
    }

  /* If we don't need remap due to a guest mode change, then
   * skip the following code and at check for the need for a
   * paging remap
   */
  if ( !(vm->modeChange & ModeChangeTransition) )
    goto pageRemapCheck;

  if (!vm->vOpcodeMap)
    monpanic(vm, "monModeChange: sbe=0\n");

  if (RealMode(vm)) {
    /* Guest ... -> RM */
    switch (GetGuestMode(vm)) {
      case GuestModePMR0:
        /* Guest PMR0 -> RM.  We need to look at
         * the guest descriptor shadow caches.  If they were loaded
         * with RM compatible values, then we can run the RM guest
         * code in v86 mode.  Some software does not load all segment
         * registers with RM compatible values before transitioning
         * back to RM.  As the PM legacy values will be used until
         * a segment reload occurs, this is how some RM code accesses
         * code outside the normal 1Meg+ boundary.
         */

        /* Sanity check: because segment register accesses were previously
         * virtualized, all selectors and shadow cache values should
         * already be maintained in vm->guest_cpu.
         */
        if ( (vm->descriptorInEmu & 0x3f) != 0x3f )
          monpanic(vm, "ModeChange: PMR0->RM, dInEmu=0x%x\n",
            vm->descriptorInEmu);
        if ( (vm->selectorInEmu & 0x3f) != 0x3f )
          monpanic(vm, "ModeChange: PMR0->RM, sInEmu=0x%x\n",
            vm->selectorInEmu);

        if ( isV86MCompatible(vm) ) {
          SetMonGuestMode(vm, MonModeVM, GuestModeRM);
          vm->vOpcodeMap = &vOpcodeMapV86;
          }
        else {
          /* The legacy PM values are not compatible with RM.  Thus,
           * we cannot run this code in VM, because there is no way
           * to maintain the shadow caches values as per the legacy
           * PM values.  Rather we have to use PM, and virtualize
           * segment register accesses.
           */
          SetMonGuestMode(vm, MonModePMR3, GuestModeRM);
          vm->vOpcodeMap = &vOpcodeMapStrongest;
          }
        break;
      default:
        monpanic(vm, "monModeChange mode=%u->RM\n", GetGuestMode(vm));
      }
    }

  else if (ProtectedMode(vm)) {
    /* Guest ... -> PM */
    unsigned currGuestMode;
    currGuestMode = GuestModePMR0 | G_GetCPL(vm);

    /* Since we need to virtualize selectors, move all guest
     * values into guest_cpu area, if not already.
     */
    if ( !(vm->selectorInEmu & (1<<SRegES)) )
      vm->guest_cpu.selector[SRegES].raw =
          vm->guest.addr.guest_context->es;
    if ( !(vm->selectorInEmu & (1<<SRegCS)) )
      vm->guest_cpu.selector[SRegCS].raw =
          vm->guest.addr.guest_context->cs;
    if ( !(vm->selectorInEmu & (1<<SRegSS)) )
      vm->guest_cpu.selector[SRegSS].raw =
          vm->guest.addr.guest_context->ss;
    if ( !(vm->selectorInEmu & (1<<SRegDS)) )
      vm->guest_cpu.selector[SRegDS].raw =
          vm->guest.addr.guest_context->ds;
    if ( !(vm->selectorInEmu & (1<<SRegFS)) )
      vm->guest_cpu.selector[SRegFS].raw =
          vm->guest.addr.guest_context->fs;
    if ( !(vm->selectorInEmu & (1<<SRegGS)) )
      vm->guest_cpu.selector[SRegGS].raw =
          vm->guest.addr.guest_context->gs;
    /* All real selector values moved to guest_cpu area */
    vm->selectorInEmu = 0x3f;

    virtualizeSegRegs = 1;

    /* Guest ... -> PM */
    switch (GetGuestMode(vm)) {

      case GuestModeRM:
        /* Guest RM -> PM */
        if (GetMonMode(vm) == MonModePMR3) {
          /* Guest RM(pm) -> PM */
          /* Sanity check: because segment register accesses were previously
           * virtualized, all selectors and shadow cache values should
           * already be maintained in vm->guest_cpu.
           */
          if ( (vm->descriptorInEmu & 0x3f) != 0x3f )
            monpanic(vm, "ModeChange: RM(pm)->PM, dInEmu=0x%x\n",
              vm->descriptorInEmu);
          if ( (vm->selectorInEmu & 0x3f) != 0x3f )
            monpanic(vm, "ModeChange: RM(pm)->PM, sInEmu=0x%x\n",
              vm->selectorInEmu);
          }
        else if (GetMonMode(vm) == MonModeVM) {
          /* Guest RM(vm) -> PM */
          /* Since guest code may have been able to reload segment
           * registers natively, shadow cache values in vm->guest_cpu
           * may not be current.
           */

          /* For now, even in VM, descriptors are virtualized.  So just
           * sanity check here.  But soon, code here should load the
           * vm->guest_cpu shadow registers with values derived from
           * the RM guest selectors
           */
          if ( (vm->descriptorInEmu & 0x3f) != 0x3f )
            monpanic(vm, "ModeChange: RM(vm)->PM, dInEmu=0x%x\n",
              vm->descriptorInEmu);
          if ( (vm->selectorInEmu & 0x3f) != 0x3f )
            monpanic(vm, "ModeChange: RM(vm)->PM, sInEmu=0x%x\n",
              vm->selectorInEmu);
          }
        else
          monpanic(vm, "ModeChange: RM(%u)->PM\n", GetMonMode(vm));
        break;

      case GuestModeVM:
        /* Guest VM -> PM */
        if (GetMonMode(vm) == MonModeVM) {
          /* For now, even in VM, descriptors are virtualized.  So just
           * sanity check here.  But soon, code here should load the
           * vm->guest_cpu shadow registers with values derived from
           * the RM guest selectors
           */
          if ( (vm->descriptorInEmu & 0x3f) != 0x3f )
            monpanic(vm, "ModeChange: VM(vm)->PM, dInEmu=0x%x\n",
              vm->descriptorInEmu);
          if ( (vm->selectorInEmu & 0x3f) != 0x3f )
            monpanic(vm, "ModeChange: VM(vm)->PM, sInEmu=0x%x\n",
              vm->selectorInEmu);
          }
        else
          monpanic(vm, "ModeChange: VM(%u)->PM\n", GetMonMode(vm));
        break;

      case GuestModePMR0:
      case GuestModePMR1:
      case GuestModePMR2:
      case GuestModePMR3:
        /* +++ should have other code here */
        if ( (GetGuestMode(vm) & 0x3) != G_GetCPL(vm) ) {
          /* Previous ring level is not equal to current.  We need
           * to remap paging permissions based on the new level */
          vm->modeChange |= ModeChangePaging;
          }
        else {
          monpanic(vm, "ModeChange: PMR%u->PMR%u\n",
                   GetGuestMode(vm), G_GetCPL(vm));
          }
        break;

      default:
        monpanic(vm, "ModeChange %u->PM\n", GetGuestMode(vm));
      }

    SetMonGuestMode(vm, MonModePMR3, currGuestMode);
    vm->vOpcodeMap = &vOpcodeMapStrongest;
    }

  else { /* Must be VM */
    /* Guest ... -> VM */
    if ( (GetGuestMode(vm) >= GuestModePMR0) &&
         (GetGuestMode(vm) <= GuestModePMR3) ) {
      /* PMR{0,1,2,3} -> VM */

      /* Sanity check: because segment register accesses were previously
       * virtualized, all selectors and shadow cache values should
       * already be maintained in vm->guest_cpu.
       */
      if ( (vm->descriptorInEmu & 0x3f) != 0x3f )
        monpanic(vm, "ModeChange: PMR0->VM, dInEmu=0x%x\n",
          vm->descriptorInEmu);
      if ( (vm->selectorInEmu & 0x3f) != 0x3f )
        monpanic(vm, "ModeChange: PMR0->VM, sInEmu=0x%x\n",
          vm->selectorInEmu);

      SetMonGuestMode(vm, MonModeVM, GuestModeVM);
      vm->vOpcodeMap = &vOpcodeMapV86;
      }
    else {
      monpanic(vm, "monModeChange mode=%u->VM\n", GetGuestMode(vm));
      }
    }

  sysRemapMonitor(vm);

pageRemapCheck:

  /* First set EFLAGS.VM bit according to the mode that the guest
   * will be executed in.
   */
  if (GetMonMode(vm) == MonModeVM)
    vm->guest.addr.guest_context->eflags |= FLG_VM;
  else
    vm->guest.addr.guest_context->eflags &= ~FLG_VM;

  /* If there was a request for a paging remap, handle that now */
  if (vm->modeChange & ModeChangePaging) {
    monPagingRemap(vm);
    }

  /* All mode changes have been handled, reset flag */
  vm->modeChange = 0;
}
