/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2000  Kevin P. Lawton
 *
 *  prescan.c: Logic to scan a code sequence in a given page,
 *    and mark instructions which need to be virtualized.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *  
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *  
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#define IN_MONITOR_SPACE
#include "monitor.h"



static unsigned get_icache_entry(vm_t *vm, Bit8u *actual_page_p,
                    Bit32u page_bits, unsigned seg32, unsigned cpl);



/*
 *  The function get_icache_entry() returns an index into the
 *  instruction cache.  It is responsible for either returning
 *  the cached index for a code page, or returning an index
 *  to an unused entry when there is no cache hit (possibly
 *  freeing an entry if the cache is full).
 */

  unsigned
get_icache_entry(vm_t *vm, Bit8u *actual_page_p, Bit32u page_bits,
                 unsigned seg32, unsigned cpl)
{
    unsigned icache_index, i;
    Bit64u guest_exec;
    code_cache_entry_t *cache_entry;
    page_t *virtualized_page_p;
    page_t *meta_page_p;

    /*
     *  +++  The caching algorithms used here are VERY stupid!!!
     *       This should be changed to something way more efficient
     *       later.
     *       Basically, this is an LRU algorithm with very inefficient
     *       loops.  For now, it'll do.           -- Ramon, 8 nov 2000
     */

    /* Look whether we have a cache hit. */
    for (i = 0; i < ICACHE_PAGES; i ++)
    {
        cache_entry = &vm->code_cache[i];

        if ((page_bits == cache_entry->guest_paddr) &&
            (seg32 == cache_entry->seg32) &&
            (cpl == cache_entry->cpl))
        {
#if 0
/* +++ this code used to be in previous version */
if (!(meta_page_p->bytes[offset] & MetaOpcodeStart)) {
  cache_entry->ts.mon_prescan = vm_rdtsc ();
  }
#endif
            return i;
        }
    }

    /* No hit: choose a page to kick out */
    icache_index = 0;
    guest_exec = vm->code_cache[0].ts.guest_executed;
    for (i = 1; i<ICACHE_PAGES; i++)
    {
        cache_entry = &vm->code_cache[i];

        /* Pick page which was used least recently */
        if (cache_entry->ts.guest_executed < guest_exec)
        {
            icache_index = i;
            guest_exec = cache_entry->ts.guest_executed;
        }
    }

    /* Initialise the cache entry */
    cache_entry = &vm->code_cache[icache_index];
    virtualized_page_p = &vm->guest.addr.icache[icache_index];
    meta_page_p = &vm->guest.addr.meta[icache_index];

    /* Address of actual code page in guest physical address space */
    cache_entry->guest_paddr = page_bits;

    /* Zero out code meta page */
    zero_memory(meta_page_p, 4096);

    /* Start private virtualized code page with copy of initial code page */
    copy_memory(virtualized_page_p, actual_page_p, 4096);

    cache_entry->seg32 = seg32;
    cache_entry->cpl = cpl;
    cache_entry->ts.mon_prescan =
        cache_entry->ts.mon_created = vm_rdtsc();
    cache_entry->ts.guest_read = 0;        /* Hasn't been read from yet  */
    cache_entry->ts.guest_write = 0;        /* Hasn't been written to yet */

    return icache_index;
}


/*
 *  Prescan guest code before it can be executed, starting at a
 *  particular address.
 *
 *  What we do is _iterate_ through a code sequence until a
 *  terminal condition is reached, and _recurse_ to branch
 *  targets where possible.  For most instructions, we can
 *  continue scanning the instruction that follows.  For
 *  conditional branch instructions, we can possibly scan
 *  both the following instruction, and the branch target.
 *  For unconditional branches, we can only potentially scan
 *  the branch target.
 */

  unsigned
prescan(vm_t * vm, Bit8u *actual_page_p, Bit32u paddr, unsigned seg32,
        unsigned cpl, unsigned depth)
{
    unsigned ret;
    instruction_t i;
    unsigned vir_opcode_attr, immediate_type;
    Bit32u page_boundary, max_len;
    unsigned j, padding;
    unsigned recurse_to_branch, iterate_to_next;
    Bit32u offset;
    Bit32u branch_offset;
    code_cache_entry_t *cache_entry;
    Bit32u page_bits;
    unsigned icache_index;
    page_t *virtualized_page_p;
    page_t *meta_page_p;
    vOpcodeMap_t *vOpMap = vm->vOpcodeMap;

/* +++ bounds checks on paddr? */

    page_bits = paddr & 0xfffff000;
    offset = paddr & 0x00000fff;

    icache_index = get_icache_entry(vm, actual_page_p, page_bits,
                                    seg32, cpl);
    cache_entry = &vm->code_cache[icache_index];

    virtualized_page_p = &vm->guest.addr.icache[icache_index];
    meta_page_p = &vm->guest.addr.meta[icache_index];

    if ( (depth==1) && (virtualized_page_p->bytes[offset] == INT3OP) ) {
      /* If the very first instruction is virtualized, then return
       * a flag signaling this.  There is no sense attempting to
       * execute the virtualized code, since it will only end up
       * being emulated anyways.
       */
      return((unsigned) -1);
      }

    branch_offset = 0;

/* +++ segment base, base+limit within page?    */
/* +++ make sure INT3 planted at all terminals  */

  decode_next:

    /*
     *  If the current instruction has already been scanned, and
     *  is marked in the meta cache, then we are done.
     */

    if (meta_page_p->bytes[offset] & MetaOpcodeStart)
    {
          /*monprint(vm, "TERMINAL: instruction already scanned\n"); */
        return (icache_index);
    }

    /* Start out with the most common values. */
    recurse_to_branch = 0;
    iterate_to_next = 1;

    page_boundary = offset | 0x00000fff;
    max_len = (page_boundary - offset) + 1;
    /* shorten to maximum x86 instruction length */
    if (max_len > MaxIA32OpcodeLen)
        max_len = MaxIA32OpcodeLen;

    ret = fetchDecode (&actual_page_p[offset], &i, max_len, seg32);
    /*fetch_count_all++;*/ /* counter for debug purposes */

#if 0
    /*monprint(vm, "ret=%u len=%u b1=0x%x inv=%u\n", ret, i.ilen, i.b1, */
    /*  (i.attr & InvalidOpcode) > 0); */
    /*monprint(vm, "GroupN  = %u\n", (i.attr & GroupN) > 0); */
    /*monprint(vm, "seg   = 0x%x (%s)\n", i.seg, segnames[i.seg]); */
    /*monprint(vm, "modrm   = 0x%x\n", i.modrm); */
    /*monprint(vm, "  mod   = 0x%x\n", i.mod); */
    /*monprint(vm, "  nnn   = 0x%x\n", i.nnn); */
    /*monprint(vm, "   rm   = 0x%x\n", i.rm); */
    /*monprint(vm, "sib     = 0x%x\n", i.sib); */
    /*monprint(vm, "  scale = 0x%x\n", i.scale); */
    /*monprint(vm, "  index = 0x%x\n", i.index); */
    /*monprint(vm, "  base  = 0x%x\n", i.base); */
#endif

    /*
     *  If fetch-decode ran up against a boundary and could not
     *  fetch enough bytes, or if the opcode is invalid, then mark
     *  the instruction as needing virtualization.
     */

    if ((ret == 0) || (i.attr & InvalidOpcode))
    {
        iterate_to_next = 0;
        goto virtualize_instruction;
    }

    /*
     *  Depending on whether opcode is a standard 1 or 2 byte opcode,
     *  or one of the special groupN opcodes, get the virtualization
     *  attributes from the virtualization opcode map.
     */

    if (i.attr & GroupN)
    {
        vir_opcode_attr = vOpMap->groups[GetGroupIndex (i.attr)][i.nnn];
    }
    else
    {
        vir_opcode_attr = vOpMap->standard[i.b1];
    }

    /*
     *  We need to make sure that this instruction fits within the
     *  page boundary, otherwise we have to virtualize it.
     *  For most instructions, non-branch and conditional-branch,
     *  execution of the following instruction is a possibility,
     *  in which case we need to allow for one extra byte, so we
     *  can virtualize the next instruction.  For unconditional
     *  branches, we don't need this extra byte of padding.
     */

    padding = (i.attr & UCBranch) == 0;
    if ((offset + i.ilen + padding) > 4096)
    {
        /* KPRINTF("TERMINAL: instruction does not fit within page\n"); */
        iterate_to_next = 0;
        goto virtualize_instruction;
    }

    /*
     *  If instruction was an unconditional relative branch, then
     *  we do not need to iterate on the next instruction.
     */

    if (padding == 0)
        iterate_to_next = 0;

    /*
     *  For relative branches, we need to look at the branch offset
     *  to determine if this instruction can be run natively.  If it
     *  branches within this page, then it can be run natively, otherwise
     *  we must virtualize it.
     */

    immediate_type = i.attr & Immediate;

    if ((immediate_type >= Immediate_BrOff8) &&
        (immediate_type <= Immediate_BrOff32))
    {
        Bit32s opcode_offset;

        /*KPRINTF("### found Immediate_BrOffx (0x%x), off=%x\n", */
        /*        immediate_type, i.Id); */

        opcode_offset = i.Id;
        branch_offset = (offset + i.ilen) + opcode_offset;

        /*KPRINTF("offset = 0x%x\n", offset); */
        /*KPRINTF("i.ilen = %u\n", i.ilen); */
        /*KPRINTF("opcode_offset = 0x%x\n", opcode_offset); */

        if (branch_offset >= 4096)
        {
            /*
             *  Relative branch to location out of page boundary; branch
             *  needs to be virtualized.  We can, however, still iterate
             *  on the next instruction if not an unconditional branch.
             */

            /* monprint(vm, "RelBranch: OOB (0x%x)\n", branch_offset); */
            goto virtualize_instruction;
        }
        else
        {
            /*
             *  Branch within page boundaries.  We can recurse to the
             *  target branch address.
             */

            /* monprint(vm, "RelBranch: OK (0x%x)\n", branch_offset); */
            recurse_to_branch = 1;
        }
    }

    /* KPRINTF("RunNative = %u\n", (vir_opcode_attr & OpcodeRunNative)>0); */

    if (recurse_to_branch && (depth >= vm->prescanDepth)) {
      /* +++ Make sure that if we have reached a maximum recursion level,
       * +++ that virtualize this instruction.  Otherwise, it may branch
       * +++ to an instruction uncontrolled.  Clean up this code.
       */
#warning "clean this up"
      recurse_to_branch = 0;
      iterate_to_next = 0; /* ??? */
      goto virtualize_instruction;
      }

    /*
     *  We need to see if this instruction overlaps another instruction
     *  which is virtualized.  If so, then we have no choice but to
     *  virtualize this instruction, since the opcode bytes are changed.
     *  Only bother looking if already marked to run native.
     */

    if (vir_opcode_attr & OpcodeRunNative)
    {
        for (j = 1; j < i.ilen; j++)
        {
            if ((meta_page_p->bytes[offset + j] & (MetaOpcodeStart | MetaRunNative))
                == MetaOpcodeStart)
            {
                /*
                 *  Overlap condition found.  In case our scanning has gotten
                 *  thrown off track, stop iterating and recursing.  If this
                 *  path is taken, we can resume scanning later when the
                 *  breakpoint is encountered.
                 */

                recurse_to_branch = 0;
                iterate_to_next = 0;
                goto virtualize_instruction;
            }
        }

        /*
         *  OK, circumstances are right so that we can let this
         *  instruction execute natively.
         */

        /*
         *  Fill in meta cache entry, now that the instruction has been
         *  decoded and we have the desired virtualization attributes set.
         */

#if ANAL_CHECKS
        if (offset > 4095) monpanic(vm, "error in prescan:\n");
#endif
        meta_page_p->bytes[offset] = MetaOpcodeStart | MetaOpcodeByte |
            MetaRunNative | i.ilen;

        /* Mark the rest of the bytes involved in this instruction opcode. */
        for (j = 1; j < i.ilen; j++)
        {
#if ANAL_CHECKS
            if ((offset+j) > 4095) monpanic(vm, "error in prescan:\n");
#endif
            meta_page_p->bytes[offset + j] |= MetaOpcodeByte;
        }
    }
    else
    {
        unsigned n;
        Bit32u o;

      virtualize_instruction:

        /*
         *  This instruction requires virtualization, normally or due
         *  to circumstances above.  We must insert a software breakpoint
         *  at the beginning of the instruction.  If by doing so, we step
         *  on any instructions before this one, which are not virtualized,
         *  we must mark them as requiring virtualization.  This is because
         *  we would be changing their opcode.  So we have to scan backwards
         *  and see if any instructions span this one.
         */

        /* Mark instruction as needing virtualization and with ilen==1 */
        o = offset;
#if ANAL_CHECKS
        if (o > 4095) monpanic(vm, "error in prescan:\n");
#endif
        meta_page_p->bytes[o] = MetaOpcodeStart | MetaOpcodeByte | 1;
        virtualized_page_p->bytes[o] = INT3OP;        /* insert software breakpoint */

        n = 0;
        while ((n < MaxIA32OpcodeLen) && o)
        {
            o--;
            n++;
            if (((meta_page_p->bytes[o] & (MetaOpcodeStart | MetaRunNative)) ==
                 (MetaOpcodeStart | MetaRunNative)) &&
                (MetaGetILen (meta_page_p->bytes[o]) > n))
            {
                /*
                 *  Found the start of an instruction which has been scanned,
                 *  is not virtualized, and spans the last modified instruction.
                 *  Since we have to modify this instruction, we have to reset
                 *  the counter and start the process over from here.
                 */

                /*  Insert software breakpoint */
#if ANAL_CHECKS
                if (o > 4095) monpanic(vm, "error in prescan:\n");
#endif
                virtualized_page_p->bytes[o] = INT3OP;
                meta_page_p->bytes[o] &= ~MetaRunNative;
                n = 0;
            }
        }
    }

    if (recurse_to_branch && (depth < vm->prescanDepth))
    {
        /*
         *  If this is a conditional or unconditional branch to a
         *  static intrapage address, and we have not exceeded our
         *  maximum recursion depth, scan the code sequence at the
         *  branch target address.
         */

        prescan(vm, actual_page_p, page_bits | branch_offset, seg32, cpl,
                depth + 1);
    }

    if (iterate_to_next)
    {
        /* Iterate scanning at next instruction. */
        offset += i.ilen;

        goto decode_next;
    }

    /* KPRINTF("TERMINAL: iteration stops.\n"); */

    return (icache_index);
}

  void
dumpVCodePage(vm_t *vm, Bit32u ppi, phy_page_usage_t *usage)
{
  unsigned i;
  code_cache_entry_t *cache_entry;
  Bit32u paddr = ppi<<12;
#if ANAL_CHECKS
  unsigned found = 0;
#endif

  /* Cycle through all cache entries.  Dump any and all that match
   * requested physical address.  There could be multiple entries
   * for a given physical address, but for different segment sizes etc.
   */
  for (i=0; i<ICACHE_PAGES; i++) {
    cache_entry = &vm->code_cache[i];
    if (cache_entry->guest_paddr == paddr) {

#if ANAL_CHECKS
      found = 1;
#endif
      cache_entry->ts.guest_executed = 0;
      cache_entry->guest_paddr = 0xffffffff; /* impossible value */
      }
    }
}
