/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999  Kevin P. Lawton
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#ifndef __PLEX86_H__
#define __PLEX86_H__

#include "config.h"
#include "descriptor.h"


#define PLEX86_INIT_TEST   10
#define PLEX86_INIT_LINUX  11

/*
 * the eflags field looks like this:
 * bit:  0  1 2  3 4  5 6  7  8  9  A  B  C/D  E  F 10 11 12 13  14  15 16
 * flg:  CF 1 PF 0 AF 0 ZF SF TF IF DF OF IOPL NT 0 RF VM AC VIF VIP ID 0
 */

#define FLG_CF    (1<<0)
#define FLG_PF    (1<<2)
#define FLG_AF    (1<<4)
#define FLG_ZF    (1<<6)
#define FLG_SF    (1<<7)
#define FLG_TF    (1<<8)
#define FLG_IF    (1<<9)
#define FLG_DF    (1<<10)
#define FLG_OF    (1<<11)
#define FLG_IOPL  (3<<12)
#define FLG_NT    (1<<14)
#define FLG_RF    (1<<16)
#define FLG_VM    (1<<17)
#define FLG_AC    (1<<18)
#define FLG_VIF   (1<<19)
#define FLG_VIP   (1<<20)
#define FLG_ID    (1<<21)



typedef Bit64u icount_t;
#define ICOUNT_INDEFINITE  ((icount_t) 0)
#define ICOUNT_CONTINUE   (((icount_t) 0) - 1)


typedef struct {
  selector_t   sel;
  descriptor_t des;
  unsigned     valid;
  } guest_sreg_t;

typedef struct {
  Bit32u eax;
  Bit32u ebx;
  Bit32u ecx;
  Bit32u edx;
  Bit32u ebp;
  Bit32u esi;
  Bit32u edi;
  Bit32u esp;
  Bit32u eflags;
  Bit32u eip;
  guest_sreg_t cs;
  guest_sreg_t ss;
  guest_sreg_t ds;
  guest_sreg_t es;
  guest_sreg_t fs;
  guest_sreg_t gs;
  guest_sreg_t ldtr;
  guest_sreg_t tr;
  struct { Bit32u base, limit; } gdtr;
  struct { Bit32u base, limit; } idtr;
  Bit32u dr0, dr1, dr2, dr3, dr6, dr7;
  Bit32u tr3, tr4, tr5, tr6, tr7;
  Bit32u cr0, cr1, cr2, cr3, cr4;
  unsigned inhibit_mask;
  } guest_cpu_t;




/*
 *  ioctl() names
 */

#define PLEX86_ALLOCVPHYS   0x6b02
#define PLEX86_ALLOCINT     0x6b03
#define PLEX86_RELEASEINT   0x6b04
#define PLEX86_RESET        0x6b05
#define PLEX86_MESSAGEQ     0x6b06
#define PLEX86_TEARDOWN     0x6b07
#define PLEX86_SETINTR      0x6b08
#define PLEX86_PRESCANDEPTH 0x6b09
#define PLEX86_SET_CPU      0x6b0c
#define PLEX86_RESET_CPU    0x6b0d
#define PLEX86_GET_CPU      0x6b0e
#define PLEX86_FORCE_INT    0x6b0f
#define PLEX86_SET_A20      0x6b10
#define PLEX86_PHYMEM_MOD   0x6b11


/* ========================================================== */
/* Messages which are passed between the user program (u) and */
/* the monitor (m) of the VM. */
/* ========================================================== */
#define VMMessageNone           0

#define VMMessageIACRequest     1 /* m->u */
#define VMMessageIACResponse    2 /* u->m */
#define VMMessageIntRequest     3 /* m->u */
#define VMMessageIntResponse    4 /* u->m */

#define VMMessageIOInRequest    5 /* m->u */
#define VMMessageIOInResponse   6 /* u->m */
#define VMMessageIOOutRequest   7 /* m->u */

#define VMMessageMemMapIOReadRequest  8  /* m->u */
#define VMMessageMemMapIOReadResponse 9  /* u->m */
#define VMMessageMemMapIOWriteRequest 10 /* m->u */

#define VMMessagePanic         11 /* m->u */
#define VMMessagePrintBuf      12 /* m->u */
#define VMMessageRunGuestN     13 /* u->m */

#define VMMessageTimeElapsed   14 /* m->u */

#define VMMessageDisasm        15 /* m->u */

#define VMMessageEOICount      16 /* m->u */
#define VMMessageReqComplete   17 /* m->u */

#define VMMessageIOInBatchRequest    18 /* m->u */
#define VMMessageIOInBatchResponse   19 /* u->m */
#define VMMessageIOOutBatchRequest   20 /* m->u */


#define MAX_VM_MESSAGE_PACKET  128

typedef struct {
  struct {
    unsigned msg_type;
    unsigned msg_len;
    } header;
  unsigned char msg[MAX_VM_MESSAGE_PACKET];
  } vm_messages_t;

#define IO_IN  10
#define IO_OUT 11

typedef struct {
  Bit32u   port;
  unsigned len;
  unsigned op;
  Bit32u   data;
  } IO_msg_t;

typedef struct {
  Bit32u   port;
  unsigned len;
  unsigned op;
  unsigned n;
  Bit32u   paddr;
  } IOBatch_msg_t;

typedef struct {
  Bit32u   addr;
  unsigned len;
  unsigned op;
  Bit32u   data;
  } memMapIO_msg_t;

typedef struct {
  unsigned vector;
  } IAC_msg_t;

typedef struct {
  unsigned vector;
  unsigned reflect;
  } INT_msg_t;


/* There are 3 ways to instruct the VM to run the guest.  The first
 * is for running the guest indefinitely in the normal fashion, and
 * must pass an instruction count of ICOUNT_INDEFINITE.
 *
 *   Execute: Run normally.  Most code executes natively, some
 *     code is necessarily virtualized (and emulated) by the VM monitor.
 *
 * The other 2 are for running the guest for exactly N instructions.
 *
 *   Emulate: The VM monitor operates in a pure emulation mode, executing
 *     exactly 'icount' instructions.
 *   Breakpoint: The VM monitor executes the guest natively where
 *     possible, but turns on the Trap Flag, so that it can execute
 *     instructions one-at-a-time.
 */

#define RunGuestNMethodExecute    10
#define RunGuestNMethodEmulate    11
#define RunGuestNMethodBreakpoint 12

typedef struct {
  icount_t icount;
  unsigned method;
  } run_guest_n_t;

typedef struct {
  Bit64u elapsed;
  } time_elapsed_t;

typedef struct {
  unsigned data;
  } event_msg_t;

typedef struct {
  Bit32u cs;
  Bit32u eip;
  Bit32u laddr;
  Bit32u seg32;
  } EOICount_t;

#endif  /* #ifndef __PLEX86_H__ */
