/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2000 Kevin P. Lawton
 *
 *  cosim.c:  Cosimulation interface for plex86 with bochs
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>

#include "plex86.h"
#include "user.h"
#include "plugin.h"



#define RunMethod RunGuestNMethodBreakpoint
/* #define RunMethod RunGuestNMethodEmulate */


#define BX_DBG_ICOUNT_SIZE 64
#define BX_DEBUGGER 1
#define BX_BASIC_TYPES_DEFINED
#define BX_DBG_SUPPORT_VIR_BPOINT 1
#define BX_DBG_SUPPORT_LIN_BPOINT 1
#define BX_DBG_SUPPORT_PHY_BPOINT 1
#define BX_DBG_MAX_VIR_BPOINTS 10
#define BX_DBG_MAX_LIN_BPOINTS 10
#define BX_DBG_MAX_PHY_BPOINTS 10

/* Special loader stuff only I use - KPL */
#define BX_USE_LOADER USE_LOADER

#include "/home/kpl/bochs/bochs-2000_0325a/debug/debug.h"

#define BX_MAX_DIRTY_PAGE_TABLE_MEGS 64

extern int filehdl;
extern char *printBuffer;
extern char *ptr;
static unsigned sim_id = 1;



Boolean  sim1_set_mem(Bit32u addr, unsigned len, Bit8u *buf);
Boolean  sim1_fetch_mem(Bit32u addr, unsigned len, Bit8u *buf);
void     sim1_xlate_linear2phy(Bit32u linear, Bit32u *phy, Boolean *valid);
Boolean  sim1_set_reg(unsigned reg, Bit32u val);
Bit32u   sim1_get_reg(unsigned reg);
Boolean  sim1_get_sreg(bx_dbg_sreg_t *sreg, unsigned sreg_no);
Boolean  sim1_set_cpu(bx_dbg_cpu_t *cpu);
Boolean  sim1_get_cpu(bx_dbg_cpu_t *cpu);
unsigned       dirty_page_tbl_size;
unsigned char sim1_dirty_pages[(BX_MAX_DIRTY_PAGE_TABLE_MEGS * 1024 * 1024) / 4096];
void     sim1_atexit(void);
unsigned sim1_query_pending(void);
void     sim1_cpu_loop(void);
void     sim1_take_irq(void);
void     sim1_take_dma(void);
void     sim1_reset_cpu(unsigned source);
void     bx_dbg_init_cpu_mem_env1(bx_dbg_callback_t *callback, int argc, char *argv[]);
void     sim1_init_mem(int size_in_bytes);
void     sim1_load_ROM(const char *path, Bit32u romaddress);

void     sim1_set_A20(unsigned val);
void     sim1_set_NMI(unsigned val);
void     sim1_set_RESET(unsigned val);
void     sim1_set_INTR(unsigned val);
void     sim1_force_interrupt(unsigned vector);
#if USE_LOADER
void     sim1_special_loader(char *path, bx_loader_misc_t *misc_ptr);
#endif



#if BX_DBG_EXTENSIONS
/* return 0 if command not handled by extensions, bochs will handle */
/* return 1 if command handled by extensions, bochs will ignore */
  int
bx_dbg_extensions(char *command)
{
  UNUSED(command);
  return(0); /* no extensions for now */
}
#endif


  void
bx_dbg_init_cpu_mem_env1(bx_dbg_callback_t *callback, int argc, char *argv[])
{
  callback->setphymem           = sim1_set_mem;
  callback->getphymem           = sim1_fetch_mem;
  callback->xlate_linear2phy    = sim1_xlate_linear2phy;
  callback->set_reg             = sim1_set_reg;
  callback->get_reg             = sim1_get_reg;
  callback->get_sreg            = sim1_get_sreg;
  callback->set_cpu             = sim1_set_cpu;
  callback->get_cpu             = sim1_get_cpu;
  callback->dirty_page_tbl_size = sizeof(sim1_dirty_pages);
  callback->dirty_page_tbl      = sim1_dirty_pages;
  callback->atexit              = sim1_atexit;
  callback->query_pending       = sim1_query_pending;
  callback->execute             = sim1_cpu_loop;
  callback->take_irq            = sim1_take_irq;
  callback->take_dma            = sim1_take_dma;
  callback->reset_cpu           = sim1_reset_cpu;
  callback->init_mem            = sim1_init_mem;
  callback->load_ROM            = sim1_load_ROM;

  /* You may set this to NULL and use values in bx_pc_system as per */
  /* docs-html/cosimulation.html */
  callback->set_A20             = sim1_set_A20;

  callback->set_NMI             = sim1_set_NMI;
  callback->set_RESET           = sim1_set_RESET;
  callback->set_INTR            = sim1_set_INTR;
  callback->force_interrupt     = sim1_force_interrupt;
#if USE_LOADER
  callback->loader              = sim1_special_loader;
#endif
}

  Boolean
sim1_set_mem(Bit32u addr, unsigned len, Bit8u *buf)
{
  return( !vm_write_physical(addr, len, buf) );
}

  Boolean
sim1_fetch_mem(Bit32u addr, unsigned len, Bit8u *buf)
{
  return(!vm_read_physical(addr, len, buf));
}

  void
sim1_xlate_linear2phy(Bit32u linear, Bit32u *phy, Boolean *valid)
{
  guest_cpu_t guest_cpu;

  vm_get_cpu(&guest_cpu);
  if (guest_cpu.cr0 & 0x80000000) {
    /* paging enabled */
    Bit32u pde, pte, pde_addr, pte_addr;
    pde_addr = (guest_cpu.cr3 & 0xfffff000) |
               ((linear>>20)  & 0x00000ffc);
    if ( !vm_read_physical(pde_addr, 4, &pde) ) {
      if (pde & 1) { /* Present */
        pte_addr = (pde & 0xfffff000) |
                   ((linear>>10) & 0x00000ffc);
        if ( !vm_read_physical(pte_addr, 4, &pte) ) {
          if (pte & 1) { /* Present */
            *phy = (pte & 0xfffff000) | (linear & 0x00000fff);
            *valid = 1;
            return;
            }
          }
        }
      }
    *phy = 0xffffffff;
    *valid = 0;
    }
  else {
    /* Paging disabled, no translation needed */
    *phy = linear;
    *valid = 1;
    }
}

  Boolean
sim1_set_reg(unsigned reg, Bit32u val)
{
fprintf(stderr, "sim1_set_reg:\n");
exit(1);
  return(0);
}

  Bit32u
sim1_get_reg(unsigned reg)
{
fprintf(stderr, "sim1_get_reg:\n");
exit(1);
  return(0);
}

  Boolean
sim1_get_sreg(bx_dbg_sreg_t *sreg, unsigned sreg_no)
{
fprintf(stderr, "sim1_get_sreg:\n");
exit(1);
  return(0);
}

  Boolean
sim1_set_cpu(bx_dbg_cpu_t *cpu)
{
fprintf(stderr, "sim1_set_cpu:\n");
exit(1);
  return(0);
}

  Boolean
sim1_get_cpu(bx_dbg_cpu_t *cpu)
{
  guest_cpu_t guest_cpu;

  vm_get_cpu(&guest_cpu);

  cpu->eax = guest_cpu.eax;
  cpu->ebx = guest_cpu.ebx;
  cpu->ecx = guest_cpu.ecx;
  cpu->edx = guest_cpu.edx;
  cpu->ebp = guest_cpu.ebp;
  cpu->esi = guest_cpu.esi;
  cpu->edi = guest_cpu.edi;
  cpu->esp = guest_cpu.esp;
  cpu->eflags = guest_cpu.eflags;
  cpu->eip = guest_cpu.eip;

  cpu->cs.sel = guest_cpu.cs.sel.raw;
  cpu->cs.des_l = * ((Bit32u *) &guest_cpu.cs.des);
  cpu->cs.des_h = * (((Bit32u *) &guest_cpu.cs.des)+1);
  cpu->cs.valid = guest_cpu.cs.valid;

  cpu->ss.sel = guest_cpu.ss.sel.raw;
  cpu->ss.des_l = * ((Bit32u *) &guest_cpu.ss.des);
  cpu->ss.des_h = * (((Bit32u *) &guest_cpu.ss.des)+1);
  cpu->ss.valid = guest_cpu.ss.valid;

  cpu->ds.sel = guest_cpu.ds.sel.raw;
  cpu->ds.des_l = * ((Bit32u *) &guest_cpu.ds.des);
  cpu->ds.des_h = * (((Bit32u *) &guest_cpu.ds.des)+1);
  cpu->ds.valid = guest_cpu.ds.valid;

  cpu->es.sel = guest_cpu.es.sel.raw;
  cpu->es.des_l = * ((Bit32u *) &guest_cpu.es.des);
  cpu->es.des_h = * (((Bit32u *) &guest_cpu.es.des)+1);
  cpu->es.valid = guest_cpu.es.valid;

  cpu->fs.sel = guest_cpu.fs.sel.raw;
  cpu->fs.des_l = * ((Bit32u *) &guest_cpu.fs.des);
  cpu->fs.des_h = * (((Bit32u *) &guest_cpu.fs.des)+1);
  cpu->fs.valid = guest_cpu.fs.valid;

  cpu->gs.sel = guest_cpu.gs.sel.raw;
  cpu->gs.des_l = * ((Bit32u *) &guest_cpu.gs.des);
  cpu->gs.des_h = * (((Bit32u *) &guest_cpu.gs.des)+1);
  cpu->gs.valid = guest_cpu.gs.valid;

  cpu->ldtr.sel = guest_cpu.ldtr.sel.raw;
  cpu->ldtr.des_l = * ((Bit32u *) &guest_cpu.ldtr.des);
  cpu->ldtr.des_h = * (((Bit32u *) &guest_cpu.ldtr.des)+1);
  cpu->ldtr.valid = guest_cpu.ldtr.valid;

  cpu->tr.sel = guest_cpu.tr.sel.raw;
  cpu->tr.des_l = * ((Bit32u *) &guest_cpu.tr.des);
  cpu->tr.des_h = * (((Bit32u *) &guest_cpu.tr.des)+1);
  cpu->tr.valid = guest_cpu.tr.valid;

  cpu->gdtr.base = guest_cpu.gdtr.base;
  cpu->gdtr.limit = guest_cpu.gdtr.limit;

  cpu->idtr.base = guest_cpu.idtr.base;
  cpu->idtr.limit = guest_cpu.idtr.limit;

  cpu->dr0 = guest_cpu.dr0;
  cpu->dr1 = guest_cpu.dr1;
  cpu->dr2 = guest_cpu.dr2;
  cpu->dr3 = guest_cpu.dr3;
  cpu->dr6 = guest_cpu.dr6;
  cpu->dr7 = guest_cpu.dr7;

  cpu->tr3 = guest_cpu.tr3;
  cpu->tr4 = guest_cpu.tr4;
  cpu->tr5 = guest_cpu.tr5;
  cpu->tr6 = guest_cpu.tr6;
  cpu->tr7 = guest_cpu.tr7;

  cpu->cr0 = guest_cpu.cr0;
  cpu->cr1 = guest_cpu.cr1;
  cpu->cr2 = guest_cpu.cr2;
  cpu->cr3 = guest_cpu.cr3;
  cpu->cr4 = guest_cpu.cr4;

  cpu->inhibit_mask = guest_cpu.inhibit_mask;

  return(0);
}

  void
sim1_atexit(void)
{
fprintf(stderr, "sim1_atexit:\n");

exit(1);
}

  unsigned
sim1_query_pending(void)
{
fprintf(stderr, "sim1_query_pending:\n");
exit(1);
  return(0);
}

  void
sim1_cpu_loop(void)
{
  vm_messages_t user_msgs;
  run_guest_n_t *run_guest_n_p;
  EOICount_t *eoic_msg = (EOICount_t *) user_msgs.msg;

  if ( !(bx_guard.guard_for & BX_DBG_GUARD_ICOUNT) ||
     !bx_guard.icount ) {
    fprintf(stderr, "sim1_cpu_loop: not guarding for icount\n");
    exit(1);
    }

  /* Build message to run guest */
  user_msgs.header.msg_type = VMMessageRunGuestN;
  user_msgs.header.msg_len = sizeof(run_guest_n_t);
  run_guest_n_p = (run_guest_n_t *) user_msgs.msg;
  run_guest_n_p->icount = bx_guard.icount;
  run_guest_n_p->method = RunMethod;

  for (;;)
  {
      if (ioctl (filehdl, PLEX86_MESSAGEQ, &user_msgs) == -1)
      {
          perror ("ioctl MESSAGEQ: ");
          vm_abort();
      }

      switch (user_msgs.header.msg_type) 
      {
      case VMMessageEOICount:
          goto done;
          break;

      case VMMessageNone:
          user_msgs.header.msg_type = VMMessageRunGuestN;
          user_msgs.header.msg_len = sizeof(run_guest_n_t);
          run_guest_n_p->icount = ICOUNT_CONTINUE;
          run_guest_n_p->method = RunMethod;
          break;

      case VMMessagePanic:
          fprintf(stderr, "Fatal monitor error caused Panic\n");
          vm_debug_exception();
          vm_abort();
          break;

      case VMMessagePrintBuf:
      {
          fprintf(stderr, "::%s\n", printBuffer);
          /* No response required */
          user_msgs.header.msg_type = VMMessageRunGuestN;
          user_msgs.header.msg_len = sizeof(run_guest_n_t);
          run_guest_n_p->icount = ICOUNT_CONTINUE;
          run_guest_n_p->method = RunMethod;
          break;
      }

      case VMMessageIOInRequest:
      {
          IO_msg_t *io_msg = (IO_msg_t *) user_msgs.msg;
          /*plugin_emulate_inport(io_msg->port, io_msg->len, 1, */
          /*                      &io_msg->data); */
          io_msg->data = bx_dbg_inp(io_msg->port, io_msg->len);

          /* Change message type; most data stays the same */
          user_msgs.header.msg_type = VMMessageIOInResponse;
          break;
      }

      case VMMessageIOOutRequest:
      {
          IO_msg_t *io_msg = (IO_msg_t *) user_msgs.msg;
          /*plugin_emulate_outport(io_msg->port, io_msg->len, 1, */
          /*                       &io_msg->data); */
          bx_dbg_outp(io_msg->port, io_msg->data, io_msg->len);

          /* No response required */
          user_msgs.header.msg_type = VMMessageRunGuestN;
          user_msgs.header.msg_len = sizeof(run_guest_n_t);
          run_guest_n_p->icount = ICOUNT_CONTINUE;
          run_guest_n_p->method = RunMethod;
          break;
      }

      case VMMessageMemMapIOReadRequest:
          {
          memMapIO_msg_t *msg = (memMapIO_msg_t *) user_msgs.msg;
          Bit32u addr;
          Bit8u *p;
          unsigned i;

          /* Zero out data, since it's potentially bigger in size than
           * the IO operation we will request
           */
          msg->data = 0;

          addr = msg->addr;
          p = (Bit8u *) &msg->data;

          for (i=0; i<msg->len; i++) {
            *p++ = bx_dbg_ucmem_read(addr++);
            }
          /* Change message type; most data stays the same */
          user_msgs.header.msg_type = VMMessageMemMapIOReadResponse;
          break;
          }

      case VMMessageMemMapIOWriteRequest:
          {
          memMapIO_msg_t *msg = (memMapIO_msg_t *) user_msgs.msg;
          Bit32u addr;
          Bit8u *p;
          unsigned i;

          addr = msg->addr;
          p = (Bit8u *) &msg->data;

          for (i=0; i<msg->len; i++) {
            bx_dbg_ucmem_write(addr++, *p++);
            }
          user_msgs.header.msg_type = VMMessageRunGuestN;
          user_msgs.header.msg_len = sizeof(run_guest_n_t);
          run_guest_n_p->icount = ICOUNT_CONTINUE;
          run_guest_n_p->method = RunMethod;
          break;
          }

      case VMMessageIACRequest:
      {
          IAC_msg_t *iac_msg = (IAC_msg_t *) user_msgs.msg;
fprintf(stderr, "VMMessageIACRequest found while stepping\n");
vm_abort();
          iac_msg->vector =  plugin_acknowledge_intr();

          user_msgs.header.msg_type = VMMessageIACResponse;
          user_msgs.header.msg_len = sizeof(IAC_msg_t);
          break;
      }

      case VMMessageIntRequest:
      {
          INT_msg_t *int_msg = (INT_msg_t *) user_msgs.msg;
fprintf(stderr, "VMMessageIntRequest found while stepping\n");
vm_abort();
          int_msg->reflect = plugin_emulate_int (int_msg->vector);

          user_msgs.header.msg_type = VMMessageIntResponse;
          break;
      }

      case VMMessageTimeElapsed:
      {
          /* No response required */
          user_msgs.header.msg_type = VMMessageRunGuestN;
          user_msgs.header.msg_len = sizeof(run_guest_n_t);
          run_guest_n_p->icount = ICOUNT_CONTINUE;
          run_guest_n_p->method = RunMethod;
          break;
      }

      default:
          fprintf(stderr, "VMMessage type %u not handled yet\n",
                  user_msgs.header.msg_type);
          vm_abort();
          break;
      }

      plugin_handle_periodic();
  }

done:
  bx_guard_found[sim_id].icount = bx_guard.icount;
  bx_guard_found[sim_id].cs  = eoic_msg->cs;
  bx_guard_found[sim_id].eip = eoic_msg->eip;
  bx_guard_found[sim_id].laddr = eoic_msg->laddr;
  bx_guard_found[sim_id].is_32bit_code = eoic_msg->seg32;
  bx_guard_found[sim_id].guard_found = BX_DBG_GUARD_ICOUNT;
}

  void
sim1_take_irq(void)
{
fprintf(stderr, "sim1_take_irq:\n");
exit(1);
}

  void
sim1_take_dma(void)
{
fprintf(stderr, "sim1_take_dma:\n");
exit(1);
}

  void
sim1_reset_cpu(unsigned source)
{
  if (ioctl(filehdl, PLEX86_RESET_CPU, 0)) {
    perror("ioctl RESET_CPU: ");
    vm_abort();
    }
}

  void
sim1_init_mem(int size_in_bytes)
{
fprintf(stderr, "sim1_init_mem:\n");
}

  void
sim1_load_ROM(const char *path, Bit32u address)
{
  if (vm_load_rom(path, address))
    exit(1);
}

  void
sim1_set_A20(unsigned val)
{
  if (ioctl(filehdl, PLEX86_SET_A20, val)) {
    perror("ioctl SET_A20: ");
    vm_abort();
    }
}

  void
sim1_set_NMI(unsigned val)
{
fprintf(stderr, "sim1_set_NMI:\n");
exit(1);
}

  void
sim1_set_RESET(unsigned val)
{
fprintf(stderr, "sim1_set_RESET:\n");
exit(1);
}

  void
sim1_set_INTR(unsigned val)
{
fprintf(stderr, "sim1_set_INTR:\n");
exit(1);
}

  void
sim1_force_interrupt(unsigned vector)
{
  vm_messages_t user_msgs;
  run_guest_n_t *run_guest_n_p;

  if (ioctl(filehdl, PLEX86_FORCE_INT, vector)) {
    perror("ioctl FORCE_INT: ");
    close(filehdl);
    exit(1);
    }
  /* Build message to run guest */
  user_msgs.header.msg_type = VMMessageRunGuestN;
  user_msgs.header.msg_len = sizeof(run_guest_n_t);
  run_guest_n_p = (run_guest_n_t *) user_msgs.msg;
  run_guest_n_p->icount = bx_guard.icount;
  run_guest_n_p->method = RunMethod;
 
send_message:
  if (ioctl (filehdl, PLEX86_MESSAGEQ, &user_msgs) == -1) {
    perror ("ioctl MESSAGEQ: ");
    vm_abort();
    }
 
  switch (user_msgs.header.msg_type) {
    case VMMessageReqComplete:
      break;

    case VMMessagePanic:
      fprintf(stderr, "Fatal monitor error caused Panic\n");
      vm_debug_exception();
      vm_abort();
      break;
 
    case VMMessagePrintBuf:
      fprintf(stderr, "::%s\n", printBuffer);
      /* No response required */
      user_msgs.header.msg_type = VMMessageRunGuestN;
      user_msgs.header.msg_len = sizeof(run_guest_n_t);
      run_guest_n_p->icount = ICOUNT_CONTINUE;
      run_guest_n_p->method = RunMethod;
      goto send_message;

    case VMMessageTimeElapsed:
      /* Just ignore it, not used. (+++ optimize this) */
      user_msgs.header.msg_type = VMMessageRunGuestN;
      user_msgs.header.msg_len = sizeof(run_guest_n_t);
      run_guest_n_p->icount = ICOUNT_CONTINUE;
      run_guest_n_p->method = RunMethod;
      goto send_message;

    default:
      fprintf(stderr, "sim1_force_interrupt: msg_type=%u\n",
        user_msgs.header.msg_type);
      vm_abort();
    }
}

  Bit8u *
SIM1_GET_PHYS_PTR(Bit32u page_start)
{
fprintf(stderr, "sim1_get_phys_ptr:\n");
exit(1);
  return(0);
}

#if USE_LOADER
  void
sim1_special_loader(char *path, bx_loader_misc_t *misc_ptr)
{
  guest_cpu_t guest_cpu;

  /* Special loader stuff only I use - KPL */
  bx_dbg_loader(path, misc_ptr, ptr);

  vm_get_cpu(&guest_cpu);
  guest_cpu.dr0 = misc_ptr->dr0;
  guest_cpu.dr1 = misc_ptr->dr1;
  guest_cpu.dr2 = misc_ptr->dr2;
  guest_cpu.dr3 = misc_ptr->dr3;
  guest_cpu.dr6 = misc_ptr->dr6;
  guest_cpu.dr7 = misc_ptr->dr7;
  vm_set_cpu(&guest_cpu);
}
#endif
