/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2000  The plex86 developers team
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <libgen.h>

#include "plex86.h"
#include "user.h"
#include "plugin.h"


/************************************************************************/
/* Declarations                                                         */
/************************************************************************/

static void abort_handler (int i);
callback_command_t *callback_command_list;

static char *eat_token(char *s, char c);
static char *eat_whitespace(char *s);
static char *eat_until_token(char *s, char c);
static void  confFileError(char *current, char *msg);
static unsigned get_fname_len(char *s);
static void parse_plex86_option(unsigned linelen);
static void usage(void);



/************************************************************************/
/* Structures / Variables                                               */
/************************************************************************/

/* Global data */

config_info_t vm_conf;
static char conf_fname[256];
static char  line[256];
static int lineno;
static FILE *config_fd;
static char *save_current;

char *argv0 = NULL;


/************************************************************************/
/* Main code                                                            */
/************************************************************************/

  int
main (int argc, char *argv[])
{
    int   i;
    struct sigaction sg_act;


    /* Store name of executable for printing messages */
    argv0 = strdup( basename(argv[0]) );

    /* some inits */

    vm_conf.max_memory    =  0;
    vm_conf.text_address  = -1;
    vm_conf.stack_address = -1;
    vm_conf.verbose       =  0;
    vm_conf.dump_vm       =  0;
    vm_conf.exit_wait     =  0;
    vm_conf.syntax        =  SX_ATT;
    plugin_startup ();



    /* parsing command line for arguments */

    fprintf (stderr, "Processing command-line options\n");

    vm_open();

    for (i = 1; i < argc;) {
      if (argv[i][0] != '-')
        goto unrecognized;

      switch (argv[i][1]) {

        /* Read options from a config file */
        case 'f':
          {
          lineno = 0;
          if (i >= (argc-1)) {
            fprintf(stderr, "\n%s: command line option -f expects "
                            "filename.\n", argv0);
            usage();
            exit(-1);
            }
          strncpy(conf_fname, argv[i+1], sizeof(conf_fname));
          conf_fname[sizeof(conf_fname)-1] = 0; /* Terminate overrun */
          fprintf(stderr, "Processing config file %s...\n",
                  conf_fname);
      
          config_fd = fopen(conf_fname, "r");
          if (!config_fd) {
            fprintf(stderr, "\n%s: error opening config file '%s': ",
                    argv0, conf_fname);
            perror("");
            exit (1);
            }
      
          while (1) {
            unsigned linelen;
      
            if ( !fgets(line, sizeof(line)-1, config_fd) ) {
              fprintf(stderr, "Processed %u lines of config file '%s'\n",
                      lineno, conf_fname);
              break;
              }
            /* Terminate with NULL in case of overrun */
            line[sizeof(line)-1] = 0;
            linelen = strlen(line);
            /* Get rid of newline read in from fgets() */
            if (line[linelen-1] == '\n') {
              line[linelen-1] = 0;
              linelen--;
              }
            parse_plex86_option(linelen);
            }
      
          fclose(config_fd);
          i += 2; /* option and filename processed */
          break;
          }

        /* Read remaining arguments as options (directives), as though
         * they were from a config file.  Only one '-o' should be used,
         * as all remaining arguments are assumed to be options, one per
         * each argument.  Thus quote each option on the command line
         * if it contains spaces etc.
         */
        case 'o':
          {
          /* Let the parser know where this option is coming from */
          strcpy(conf_fname, "(command line option)");
          lineno = -1;

          /* Make sure the user actually passed an option */
          if (i >= (argc-1)) {
            fprintf(stderr, "\n%s: command line option -o expects "
                            "an option.\n", argv0);
            usage();
            exit(-1);
            }
          /* Pass the argument to the configuration line parser, as
           * though it came from a config file.
           */
          strncpy(line, argv[i+1], sizeof(line));
          line[sizeof(line)-1] = 0; /* terminate overrun case */
          parse_plex86_option( strlen(line) );
          i += 2; /* '-o' and option processed from arg stream */
          break;
          }

#if 0
#if USE_LOADER
        case 'l':
          {
          extern char *special_loader_fname;
          special_loader_fname = strdup(argv[i+1]);
          i += 2; /* option and filename processed */
          break;
          }
#endif
#endif

        case 'v':
          vm_conf.verbose = 1;
          i += 1; /* option processed */
          break;

        case 'h': /* -h: help */
          if ( !strcmp(argv[i], "-h") ||
               !strcmp(argv[i], "-help") ) {
help:
            usage();
            exit(0); /* OK */
            }
          else
            goto unrecognized;

        case '-': /* -h: help */
          /* GNU type of help option */
          if ( !strcmp(argv[i], "--help") ) {
            goto help;
            }
          goto unrecognized;


        default:
unrecognized:
          fprintf(stderr, "\n%s: unrecognized command line "
                          "option '%s'.\n\n", argv0, argv[i]);
          usage();
          exit(-1);
        }
      }



    /* set default settings, if not specified */

    if (vm_conf.text_address == -1)
        vm_conf.text_address = 0;
    if (vm_conf.stack_address == -1)
      vm_conf.stack_address = 0;



    /* setup to catch SIGINT, SIGABRT and SIGQUIT signals so we can clean up */

    memset (&sg_act, 0, sizeof (sg_act));
    sg_act.sa_handler = abort_handler;
    sg_act.sa_flags = SA_RESETHAND;
    sigaction (SIGINT, &sg_act, NULL);
    sigaction (SIGABRT, &sg_act, NULL);
    sigaction (SIGQUIT, &sg_act, NULL);



    /* kickstart the VM */

    vm_kickstart ();

    /* Verify if addresses specified are valid */
    
    if (vm_conf.verbose)
    {
        fprintf (stderr, "Memory:\t\t%d\n",      vm_conf.max_memory);
        fprintf (stderr, "Text address:\t0x%x\n",  vm_conf.text_address);
        fprintf (stderr, "Stack address:\t0x%x\n", vm_conf.stack_address);
    }

    if ( vm_conf.max_memory*1024*1024 < vm_conf.text_address ||
         vm_conf.max_memory*1024*1024 < vm_conf.stack_address )
    {
        fprintf (stderr, "Total memory for VM less than some of the \n");
        fprintf (stderr, "  specified segment addresses, Quiting...\n");
        exit (1);
    }

    /* run the VM */

    fprintf (stderr, "Running VM\n");
#if COSIMULATE
    {
    int   bx_dbg_main(int argc, char *argv[]);
    int   dbg_argc = 1;
    char *dbg_argv[] = { "plex86", NULL };

    /* Pass execution to cosimulation controller */
    bx_dbg_main(dbg_argc, dbg_argv);
    }
#else
    vm_event_loop();
#endif



    /* vm_event_loop() should never return */

    fprintf (stderr, "Fatal: vm_event_loop() returned.  This is a bug. ");
    fprintf (stderr, "Please contact " BUGSMAIL "\n");
    return 1;
}


  void
parse_plex86_option(unsigned linelen)
{
        char *current;

        current = line;
        /* If option comming from config file, increment line number */
        if (lineno >= 0)
          lineno++;

        current = eat_whitespace(current); /* Eat initial whitespace */

        /* If continuation character, then keep going. */
        if (line[linelen-1] == '\\') {
          fprintf(stderr, "Continuation character found, not supported.\n");
          exit(0);
          }

        /* Eat blank and comment lines */
        if ( (*current == 0) || (*current == '#') )
          return;

        if (!strncmp(current, "memory", strlen("memory"))) {
          current += strlen("memory");
          current = eat_token(current, '=');
          if (!current)
            confFileError(current, "Expecting '='");
          vm_conf.max_memory = atoi(current);
          vm_init_memory(vm_conf.max_memory);
          }
        else if (!strncmp(current, "db_syntax", strlen("db_syntax"))) {
          char  sxname[10];
          current += strlen("db_syntax");
          current = eat_token(current, '=');
          if (!current)
            confFileError(current, "Expecting '='");
          strcpy (sxname, current);

          if (!strcmp (sxname, "intel"))
              vm_conf.syntax = SX_INTEL;
          else if (!strcmp (sxname, "at&t"))
              vm_conf.syntax = SX_ATT;
          else {
            confFileError(current, "Expecting 'intel' or 'at&t'");
            }
          }
        else if (!strncmp(current, "prescan", strlen("prescan"))) {
          current += strlen("prescan");
          current = eat_token(current, ':');
          if (!current)
            confFileError(current, "Expecting ':'");
          while ( *current ) {
            if ( !strncmp(current, "depth", strlen("depth")) ) {
              current += strlen("depth");
              current = eat_token(current, '=');
              if (!current)
                confFileError(current, "Expecting integer");
              vm_conf.prescanDepth = atoi(current);
              vm_init_prescan_depth(vm_conf.prescanDepth);
              current = eat_until_token(current, ',');
              if (!current)
                return;
              current = eat_token(current, ',');
              }
            else if ( !strncmp(current, "ring3", strlen("ring3")) ) {
              current += strlen("ring3");
              current = eat_token(current, '=');
              if (!current)
                confFileError(current, "Expecting '='");
              if ( !strncmp(current, "on", strlen("on")) ) {
                fprintf(stderr, "prescan on\n");
                current += 2;
                }
              else if ( !strncmp(current, "auto", strlen("auto")) ) {
                fprintf(stderr, "prescan auto\n");
                current += 4;
                }
              else if ( !strncmp(current, "off", strlen("off")) ) {
                fprintf(stderr, "prescan off\n");
                current += 3;
                }
              else
                confFileError(current, "Expecting {on,auto,off}");
              current = eat_token(current, ',');
              if (!current)
                return;
              }
            else {
              fprintf(stderr, "prescan directive malformed.\n");
              exit(0);
              }
            }
        }
        else if (!strncmp(current, "exit_wait", 9)) {
          current += strlen("exit_wait");
          current = eat_token(current, '=');
          if (!current)
            confFileError(current, "Expecting integer");
          vm_conf.exit_wait = atoi(current);
          }
        else if (!strncmp(line, "plugin", 6)) {
          char *plugin_name, *plugin_args;

          /* BTW, plugin code free()s memory, so we need to pass
           * allocated memory */
          current += strlen("plugin");
          current = eat_token(current, '=');
          if (!current)
            confFileError(current, "Expecting '='");
          plugin_name = strdup(current);

          /* Scan forward to find where arguments start, if any */
          plugin_args = plugin_name + 1;
          while ( (*plugin_args!=0) && (*plugin_args!=' ') )
            plugin_args++;
          if (*plugin_args==0) {
            /* No args */
            }
          else { /* must have been a space */
            *plugin_args++ = 0; /* Terminate plugin command */
            /* Eat whitespace between command and args */
            plugin_args = eat_whitespace(plugin_args);
            }
          plugin_args = strdup(plugin_args);
          plugin_load(plugin_name, plugin_args);
          }
        else if (!strncmp (line, "load-rom", strlen("load-rom"))) {
          Bit32u address;
          char *fname_ptr, *fname_terminate_ptr;
          unsigned fname_len;

          current += strlen("load-rom");
          current = eat_token(current, ':');
          if (!current)
            confFileError(current, "Expecting ':'");
          if ( strncmp(current, "file", strlen("file")) )
            confFileError(current, "Expecting 'file'");
          current += strlen("file");

          current = eat_token(current, '=');
          if (!current)
            confFileError(current, "Expecting '='");

          fname_ptr = current;
          fname_len = get_fname_len(current);
          if (!fname_len)
            confFileError(current, "Expecting filename");
          current += fname_len;
          fname_terminate_ptr = current;
          current = eat_token(current, ',');
          if (!current)
            confFileError(current, "Expecting ','");

          if ( strncmp(current, "address", strlen("address")) )
            confFileError(current, "Expecting 'address'");
          current += strlen("address");
          current = eat_token(current, '=');
          if (!current)
            confFileError(current, "Expecting '='");

          address = strtoul(current, NULL, 0);
          *fname_terminate_ptr = 0; /* Terminate filename */
          if (vm_load_rom(fname_ptr, address)) {
            fprintf(stderr, "load-rom '%s' @ 0x%x failed.\n",
                    fname_ptr, address);
            exit(0);
            }
          }
        else {
          /* Command is not one of the predefined ones.  Check */
          /* list of registered commands from the plugin modules. */
          callback_command_t *command = callback_command_list;
          unsigned found = 0;
          while (command) {
            if ( !strncmp(current, command->command,
                          strlen(command->command)) ) {
              found = 1;
              /* Scan past plug callback command */
              while ( (*current!=' ') && (*current!=0) )
                current++;
              current = eat_whitespace(current);
              command->f(current);
              return;
              }
            command = command->next;
            }
          if (!found)
            confFileError(current, "Expecting recognized directive");
          }
}


  char *
eat_token(char *s, char c)
{
  save_current = s;

  /* Eat initial whitespace */
  while (*s == ' ')
    s++;

  /* See if token is the expected one; if not return error */
  if (*s != c) {
    return NULL;
    }
  else
    s++;

  /* Eat trailing whitespace */
  while (*s==' ')
    s++;

  /* Return pointer to string after characters eaten */
  return s;
}

  char *
eat_whitespace(char *s)
{
  while (*s == ' ')
    s++;
  return s;
}

  void
confFileError(char *current, char *msg)
{
  unsigned i, spaces;

  if (!current)
    current = save_current;
  if (lineno == -1)
    fprintf(stderr, "\n%s: error in command line option\n",
            argv0);
  else
    fprintf(stderr, "\n%s: error in config file '%s', line %u\n",
            argv0, conf_fname, lineno);
  fprintf(stderr, "%s\n", line);
  spaces = (current - line);
  if (spaces > sizeof(line))
    spaces = 0;
  for (i=0; i<spaces; i++)
    fputc(' ', stderr);

  fprintf(stderr, "^ %s\n", msg);
  exit(1);
}

  unsigned
get_fname_len(char *s)
{
  unsigned incr = 0;

  while ( (*s!=0) && (*s!=' ') && (*s!=',') ) {
    s++;
    incr++;
    }

  return incr;
}

  char *
eat_until_token(char *s, char c)
{
  while ( (*s!=0) && (*s!=c) )
    s++;

  if (*s==0)
    return 0;
  return s;
}

/************************************************************************/
/* Signal handlers                                                      */
/************************************************************************/

  void
abort_handler (int i)
{
    switch (i)
    {
    case SIGINT:
        printf ("^C pressed. aborting execution\n");
        break;

    case SIGABRT:
    case SIGQUIT:
        printf ("^\\ pressed. dumping VM core and aborting execution\n");
        vm_conf.dump_vm = 1;
        break;

    default:
        printf ("Signal %d caught, aborting execution\n", i);
        break;
    }

    vm_abort ();
    return;
}

  void
usage(void)
{
  fprintf(stderr, "%s " VERSION "\n", argv0);
  fprintf(stderr, "usage:  %s [-f path] [-hv] -o options\n", argv0);
  fprintf(stderr, "Options:\n");
  fprintf(stderr, "  -f path    Config file pathname (multiple -f's "
                  "allowed)\n");
  fprintf(stderr, "  -h         Print this help message\n");
  fprintf(stderr, "  -help      ...\n");
  fprintf(stderr, "  --help     ...\n");
  fprintf(stderr, "  -v         Print debug messages (verbose)\n");
  fprintf(stderr, "  -o ...     Remainder of args are taken as config file "
                  "directives\n");
}
