/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2000  The plex86 developers team
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#include "plex86.h"
#include "user.h"
#include "plugin.h"



/************************************************************************/
/* Plugin initialization / deinitialization                             */
/************************************************************************/

static void plugin_init_one(plugin_t *plugin);

plugin_t *plugins = NULL;      /* Head of the linked list of plugins  */

  void
plugin_load (char *name, char *args)
{
    plugin_t *plugin;
    const char *plug_err;

    plugin = malloc (sizeof (plugin_t));
    if (!plugin)
    {
        perror ("malloc");
        exit (1);
    }

    plugin->name = name;
    plugin->args = args;
    plugin->initialized = 0;

    plugin->handle = dlopen (name, RTLD_LAZY);
    if (!plugin->handle)
    {
        fputs (dlerror (), stderr);
        exit (1);
    }

    plugin->init = dlsym (plugin->handle, PLUGIN_INIT);
    if ((plug_err = dlerror ()) != NULL)
    {
        fputs (plug_err, stderr);
        exit (1);
    }

    plugin->fini = dlsym (plugin->handle, PLUGIN_FINI);
    if ((plug_err = dlerror ()) != NULL)
    {
        fputs (plug_err, stderr);
        exit (1);
    }

    /* Insert plugin at the _end_ of the plugin linked list. */
    plugin->next = NULL;

    if (!plugins)
    {
        /* Empty list, this become the first entry. */
        plugins = plugin;
    }
    else
    {
        /* Non-empty list.  Add to end. */
        plugin_t *temp = plugins;

        while (temp->next)
            temp = temp->next;

        temp->next = plugin;
    }

plugin_init_one(plugin);

    return;
}


#if 0
void
plugin_init_all (void)
{
    plugin_t *plugin;

    for (plugin = plugins; plugin; plugin = plugin->next)
    {
        char *arg_ptr = plugin->args;

        /* process the command line */
        plugin->argc = 0;
        while (plugin->argc < MAX_ARGC)
        {
            while (*arg_ptr && isspace (*arg_ptr))
                arg_ptr++;

            if (!*arg_ptr)
                break;
            plugin->argv[plugin->argc++] = arg_ptr;

            while (*arg_ptr && !isspace (*arg_ptr))
                arg_ptr++;

            if (!*arg_ptr)
                break;
            *arg_ptr++ = '\0';
        }

        /* initialize the plugin */
        if (plugin->init (plugin, plugin->argc, plugin->argv))
        {
            fprintf (stderr, "Plugin initialization failed for %s\n", plugin->name);
            plugin_abort();
        }

        plugin->initialized = 1;
    }

    return;
}
#endif


void
plugin_init_one(plugin_t *plugin)
{
        char *arg_ptr = plugin->args;
 
        /* process the command line */
        plugin->argc = 0;
        while (plugin->argc < MAX_ARGC)
        {
            while (*arg_ptr && isspace (*arg_ptr))
                arg_ptr++;
 
            if (!*arg_ptr)
                break;
            plugin->argv[plugin->argc++] = arg_ptr;
 
            while (*arg_ptr && !isspace (*arg_ptr))
                arg_ptr++;
 
            if (!*arg_ptr)
                break;
            *arg_ptr++ = '\0';
        }
 
        /* initialize the plugin */
        if (plugin->init (plugin, plugin->argc, plugin->argv))
        {
            fprintf (stderr, "Plugin initialization failed for %s\n", plugin->name);
            plugin_abort();
        }
 
        plugin->initialized = 1;
}


  plugin_t *
plugin_unload(plugin_t *plugin)
{
    int i;
    plugin_t *dead_plug;

    if (plugin->initialized)
        plugin->fini ();

    plugin_free_inp  (plugin, 0, 65536);
    plugin_free_outp (plugin, 0, 65536);
    for (i=0; i<256; i++)
        plugin_free_intr (plugin, i);

    dlclose (plugin->handle);
    free (plugin->name);
    free (plugin->args);

    dead_plug = plugin;
    plugin = plugin->next;
    free (dead_plug);

    return plugin;
}


void
plugin_fini_all (void)
{
    plugin_t *plugin;

    for (plugin = plugins; plugin; plugin = plugin_unload (plugin));

    return;
}



/************************************************************************/
/* I/O port handling                                                    */
/************************************************************************/

/*
 *  I/O ports are allocated in a three-layered structure.  There is one
 *  layer for 4-byte allocations, one layer for 2-byte allocations, and
 *  one layer with 1-byte allocations.  When an I/O port is allocated,
 *  the bottom (1-byte) layer is always updated.  It is then checked
 *  whether the plugin has allocated two consecutive I/O ports (and they
 *  do not cross an alignment boundary), in which case the plugin is
 *  also entered into the layer above it.  The top layer is updated in
 *  the case of the allocation of four consecutive ports (with correct
 *  alignment).
 *
 *  When an I/O event comes in, we start at the layer which corresponds
 *  to the opsize of the event (bottom layer for inb/outb, middle layer
 *  for inw/outw, etc.)  If this layer contains an entry, the event is
 *  send to the appropiate handler and we're finished.  If not, we break
 *  up the request into two pieces and go down to the next layer.
 *  Of course, the event is broken up immediately if it is not properly
 *  aligned.
 *
 *  Note: unlike interrupts, only one device can allocate an I/O port
 *  at any time (otherwise we have a conflict!).
 */

#define PORTS 65536

static struct ioport_t
{
    plugin_t  *plugin;
    handler_t  handler;
}
ilayer0[PORTS >> 0],        /* input  Single-byte aligns */
olayer0[PORTS >> 0];        /* output Single-byte aligns */

static handler_t
    ilayer1[PORTS >> 1],    /* input  Word alignment     */
    olayer1[PORTS >> 1],    /* output Word alignment     */
    ilayer2[PORTS >> 2],    /* input  Dword alignment    */
    olayer2[PORTS >> 2];    /* output Dword alignment    */

  static int
io_sift_alloc_inp(plugin_t *plugin, handler_t handler, int port)
{
    if (ilayer0[port].plugin != NULL)
        return -1;

    ilayer0[port].plugin  = plugin;
    ilayer0[port].handler = handler;

    if (ilayer0[port | 0x1].handler == ilayer0[port & ~0x1].handler)
    {
        ilayer1[port >> 1] = handler;
    }

    if (ilayer1[(port >> 1) | 0x1] == ilayer1[(port >> 1) & ~0x1])
    {
        ilayer2[port >> 2] = handler;
    }

    return 0;
}

  static int
io_sift_alloc_outp(plugin_t *plugin, handler_t handler, int port)
{
    if (olayer0[port].plugin != NULL)
        return -1;
 
    olayer0[port].plugin  = plugin;
    olayer0[port].handler = handler;

    if (olayer0[port | 0x1].handler == olayer0[port & ~0x1].handler)
    {
        olayer1[port >> 1] = handler;
    }

    if (olayer1[(port >> 1) | 0x1] == olayer1[(port >> 1) & ~0x1])
    {
        olayer2[port >> 2] = handler;
    }
 
    return 0;
}


  static int
io_sift_free_inp(plugin_t *plugin, int port)
{
    if (ilayer0[port].plugin != plugin)
        return -1;

    ilayer0[port].plugin  = NULL;
    ilayer0[port].handler = NULL;
 
    ilayer1[port >> 1] = NULL;
    ilayer2[port >> 2] = NULL;
    return 0;
}



  static int
io_sift_free_outp(plugin_t *plugin, int port)
{
    if (olayer0[port].plugin != plugin)
        return -1;
 
    olayer0[port].plugin  = NULL;
    olayer0[port].handler = NULL;


    olayer1[port >> 1] = NULL;
    olayer2[port >> 2] = NULL;

    return 0;
}


  static void
io_rearrange_data (int op_size, int count, void *data)
/*
 *  We need io_rearrange_data() in order to split the data stream into
 *  muliple data streams, which we use when we drop down a layer (see
 *  plugin_emulate_inport() for more details).  This is the idea:
 *  initially the data consist of low byte/word (o), high byte/word (x),
 *  in this order:
 *      oxoxoxoxox...
 *  We want to rearrange it to look like
 *      ooooo...xxxxx...
 *  The algorithm used is simple but it work (check this for
 *  yourself by drawing strings like the ones above on a piece of
 *  paper and exchanging the o's and x's).
 */
{
#define EXCHG(dt,a,b) \
    { dt tmp; tmp = *(dt *)a; *(dt *)a = *(dt *)b; *(dt *)b = tmp; }

    int i, j;


    if (count == 1)
        return;

    switch (op_size)
    {
    case 4:
        for (i = 1; i < count; i++)
            for (j = 0; j < i; j++)
                EXCHG (short,
                       & ((short *) data)[2 * i - j],
                       & ((short *) data)[2 * i - 1 - j]);
        break;

    case 2:
        for (i = 1; i < count; i++)
            for (j = 0; j < i; j++)
                EXCHG (char,
                       & ((char *) data)[2 * i - j],
                       & ((char *) data)[2 * i - 1 - j]);
        break;

    default:
    }

    return;

#undef EXCHG
}


  int
plugin_alloc_inp(plugin_t *plugin, handler_t handler, int base, int range)
{
    int port;

    if (base+range > PORTS)
    {
        return -1;
    }
 
    for (port = base; port < base+range; port++)
    {
        if (ilayer0[port].plugin != NULL)
            return -1;
    }


    for (port = base; port < base+range; port++)
    {
        io_sift_alloc_inp (plugin, handler, port);
    }

    return 0;
}


  int
plugin_alloc_outp(plugin_t *plugin, handler_t handler, int base, int range)
{
    int port;

    if (base+range > PORTS)
    {
        return -1;
    }

    for (port = base; port < base+range; port++)
    {
        if (olayer0[port].plugin != NULL)
            return -1;
    }

    for (port = base; port < base+range; port++)
    {
        io_sift_alloc_outp (plugin, handler, port);
    }

    return 0;
}

  void
plugin_free_inp(plugin_t *plugin, int base, int range)
{
    int port;

    if (base+range > PORTS)
    {
        return;
    }

    for (port = base; port < base+range; port++)
    {
        io_sift_free_inp (plugin, port);
    }

    return;
}



  void
plugin_free_outp(plugin_t *plugin, int base, int range)
{
    int port;

    if (base+range > PORTS)
    {
        return;
    }

    for (port = base; port < base+range; port++)
    {
        io_sift_free_outp (plugin, port);
    }

    return;
}


  int
plugin_emulate_inport(int port, int op_size, int count, void *loc)
{
    int i;
    int split = 1, split2 = 0;

    if ((port & (op_size - 1)) == 0)
    {
        /* Aligned I/O, let's sift through the layers. */

        switch (op_size)
        {
        case 4:
            if (ilayer2[port >> 2] != NULL)
            {
                ilayer2[port >> 2] (EVT_INPORT, port, 4, count, loc);
                break;
            }

            /* Rearrange data from 1 * op_size=4 to 2 * op_size=2 */

            io_rearrange_data (4, count, loc);

            /* Fallthrough to next layer */

            split = 2;

        case 2:
            for (i = 0; i < split; i++)
            {
                if (ilayer1[(port >> 1) + i] != NULL)
                {
                    ilayer1[(port >> 1) + i] (EVT_INPORT,
                                              port + i * 2,
                                              2,
                                              count,
                                              loc  + i * count * 2);
                }
                else
                {
                    split2 |= (i + 1);
                    io_rearrange_data (2, count, loc + i * count * 2);
                }
            }

            if (split2 == 0)
                break;

            /* Fallthrough to next layer */

            if (split2 == 3)
                split = 4;
            else if (split2 == 2)
            {
                port += 2;
                loc  += count * 2;
            }
            else
                split = 2;

        case 1:
            for (i = 0; i < split; i++)
            {
                if (ilayer0[port + i].handler != NULL)
                {
                    ilayer0[port + i].handler (EVT_INPORT,
                                               port + i,
                                               1,
                                               count,
                                               loc  + i * count);
                }
            }
        }
    }
    else
    {
        /* Unaligned write, do it slow & simple, who cares */

        for (i = 0; i < count * op_size; i++)
        {
            if (ilayer0[port + (i % op_size)].handler != NULL)
            {
                ilayer0[port + (i % op_size)].handler (EVT_INPORT,
                                                       port + (i % op_size),
                                                       1, 1,
                                                       loc + i);
            }
        }
    }

    return 0;
}


  int
plugin_emulate_outport(int port, int op_size, int count, void *loc)
{
    int i;
    int split = 1, split2 = 0;

    if ((port & (op_size - 1)) == 0)
    {
        /* Aligned I/O, let's sift through the layers. */

        switch (op_size)
        {
        case 4:
            if (olayer2[port >> 2] != NULL)
            {
                olayer2[port >> 2] (EVT_OUTPORT, port, 4, count, loc);
                break;
            }

            /* Rearrange data from 1 * op_size=4 to 2 * op_size=2 */

            io_rearrange_data (4, count, loc);

            /* Fallthrough to next layer */

            split = 2;

        case 2:
            for (i = 0; i < split; i++)
            {
                if (olayer1[(port >> 1) + i] != NULL)
                {
                    olayer1[(port >> 1) + i] (EVT_OUTPORT,
                                              port + i * 2,
                                              2,
                                              count,
                                              loc  + i * count * 2);
                }
                else
                {
                    split2 |= (i + 1);
                    io_rearrange_data (2, count, loc + i * count * 2);
                }
            }

            if (split2 == 0)
                break;

            /* Fallthrough to next layer */

            if (split2 == 3)
                split = 4;
            else if (split2 == 2)
            {
                port += 2;
                loc  += count * 2;
            }
            else
                split = 2;

        case 1:
            for (i = 0; i < split; i++)
            {
                if (olayer0[port + i].handler != NULL)
                {
                    olayer0[port + i].handler (EVT_OUTPORT,
                                               port + i,
                                               1,
                                               count,
                                               loc  + i * count);
                }
            }
        }
    }
    else
    {
        /* Unaligned write, do it slow & simple, who cares */

        for (i = 0; i < count * op_size; i++)
        {
            if (olayer0[port + (i % op_size)].handler != NULL)
            {
                olayer0[port + (i % op_size)].handler (EVT_OUTPORT,
                                                       port + (i % op_size),
                                                       1, 1,
                                                       loc + i);
            }
        }
    }

    return 0;
}



/************************************************************************/
/* Software interrupt handling                                          */
/************************************************************************/

static struct intr_t
{
    plugin_t  *plugin;
    handler_t  handler;
    struct intr_t *next;
}
*ints[256];         /* Heads of linked lists for each int */


int
plugin_alloc_intr (plugin_t *plugin, handler_t handler, int vec)
{
    struct intr_t *intr;

    for (intr = ints[vec]; intr != NULL; intr = intr->next)
    {
        if (intr->plugin == plugin)
            return -1;
    }

    if (ints[vec] == NULL && vm_alloc_intr (vec))
        return -1;

    intr = malloc (sizeof (struct intr_t));
    intr->plugin  = plugin;
    intr->handler = handler;
    intr->next    = ints[vec];
    ints[vec]     = intr;

    return 0;
}


  void
plugin_free_intr (plugin_t *plugin, int vec)
{
    struct intr_t *intr, *dummy;

    if ((intr = ints[vec]) == NULL)
        return;

    if (intr->plugin == plugin)
    {
       dummy = intr;
       ints[vec] = intr->next;
       free (dummy);
    }
    else
        for (; intr != NULL; intr = intr->next)
        {
            if (intr->next == NULL)
                break;

            if (intr->next->plugin == plugin)
            {
                dummy = intr->next;
                intr->next = intr->next->next;
                free (dummy);

                break;
            }
        }

    if (ints[vec] == NULL)
        vm_release_intr (vec);

    return;
}


  int
plugin_emulate_int(int vec)
{
    struct intr_t *intr;
    int reflect = 0;

    for (intr = ints[vec]; intr != NULL; intr = intr->next)
    {
        if (intr->handler (EVT_INT, vec, 0, 0, NULL))
            reflect = 1;
    }

    return reflect;
}



/************************************************************************/
/* Hardware interrupt handling                                          */
/************************************************************************/

static iac_handler_t plugin_iac_handler = NULL;


void
plugin_set_intr (int intr)
{
    vm_set_intr (intr);
}


void
plugin_announce_iac_handler (iac_handler_t handler)
{
    if (plugin_iac_handler)
    {
        fprintf (stderr, "Two interrupt sources announced!\n");
        vm_abort();
    }

    plugin_iac_handler = handler;
}


int
plugin_acknowledge_intr (void)
{
    if (!plugin_iac_handler)
    {
        fprintf (stderr, "No interrupt source was announced!\n");
        vm_abort();
    }

    return plugin_iac_handler();
}



/************************************************************************/
/* VM control                                                           */
/************************************************************************/

void
plugin_abort (void)
{
    vm_abort ();
    return;
}



/************************************************************************/
/* Plugin pending                                                       */
/************************************************************************/

/*
 * NOTE:  This code was explicitly written to be reentrant using atomic
 *        XCHG, because it will usually be called from signal handlers and
 *        the like.  None of the other plugin functions are reentrant, so
 *        a plugin wanting to perform plugin operations after reception
 *        of a signal should always use these functions to pend for CPU
 *        time.
 */


/************************************************************************/
/*                                                                      */
/************************************************************************/

void (*memMapFunct)(Bit32u, unsigned, unsigned, Bit32u *) = NULL;

  void
plugin_register_mem_map_IO( void (*f)(Bit32u, unsigned, unsigned, Bit32u *),
  Bit32u range0, Bit32u range1)
{
  memMapFunct = f;
}


/************************************************************************/
/* Timing control                                                       */
/************************************************************************/

static void (*save_funct)(Bit64u) = NULL;
static void (*periodic_funct)(void) = NULL;


  void
plugin_register_elapsed(void (*funct)(Bit64u))
{
  save_funct = funct;
}

  void
plugin_register_periodic(void (*funct)(void))
{
  periodic_funct = funct;
}


  void
plugin_call_elapsed( Bit64u elapsed )
{
  if (save_funct)
    save_funct(elapsed);
}

  void
plugin_handle_periodic(void)
{
  static Bit64u prev_user_usec = 0;
  Bit64u new_user_usec;
extern Bit64u user_time_usec;

#warning "fix: need exclusive access to variable used by async handler"
  new_user_usec = user_time_usec;
  if ( (new_user_usec - prev_user_usec) >= 500000 ) {
    prev_user_usec = new_user_usec;
    periodic_funct();
    }
}



/************************************************************************/
/* Plugin system: plex86 startup function                               */
/************************************************************************/

#define ZERO_ARRAY(a)  memset (a, 0, sizeof(a))

void
plugin_startup (void)
{
    ZERO_ARRAY(ilayer0);
    ZERO_ARRAY(ilayer1);
    ZERO_ARRAY(ilayer2);

    ZERO_ARRAY(olayer0);
    ZERO_ARRAY(olayer1);
    ZERO_ARRAY(olayer2);

    ZERO_ARRAY(ints);

    return;
}
