/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2000  The plex86 developers team
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#ifndef __PLUGIN_H
#define __PLUGIN_H

#ifdef __cplusplus
extern "C" {
#endif


#include <dlfcn.h>

#define PLUGIN_INIT  "plugin_init"
#define PLUGIN_FINI  "plugin_fini"

#define MAX_ARGC     10

typedef enum
{
    EVT_INPORT, EVT_OUTPORT, EVT_INT
} event_t;

typedef int (*handler_t)
    (event_t event, int data, int op_size, int count, void *loc);

typedef struct _plugin_t
{
    int  initialized;
    void *handle;
    int  argc;
    char *name, *args, *argv[MAX_ARGC];
    int  (*init)(struct _plugin_t *plugin, int argc, char *argv[]);
    void (*fini)(void);
    struct _plugin_t *next;
} plugin_t;

extern plugin_t *plugins;

void plugin_startup (void);
void plugin_load (char *name, char *args);
plugin_t *plugin_unload (plugin_t *plugin);
void plugin_init_all (void);
void plugin_fini_all (void);

int plugin_alloc_inp (plugin_t *plugin, handler_t handler, int base, int  range);
int plugin_alloc_outp (plugin_t *plugin, handler_t handler, int base, int  range);
void plugin_free_inp (plugin_t *plugin, int base, int range);
void plugin_free_outp (plugin_t *plugin, int base, int range);
int plugin_emulate_inport (int port, int op_size, int count, void *loc);
int plugin_emulate_outport (int port, int op_size, int count, void *loc);
int plugin_alloc_intr (plugin_t *plugin, handler_t handler, int vec);
void plugin_free_intr (plugin_t *plugin, int vec);
int plugin_emulate_int (int vec);

typedef int  (*iac_handler_t)(void);
typedef void (*plugin_pending_t)(void);
void plugin_set_intr (int intr);
int  plugin_acknowledge_intr (void);
void plugin_announce_iac_handler (iac_handler_t handler);
void plugin_enqueue (plugin_pending_t func);
void plugin_handle_pending (void);
void plugin_register_elapsed(void (*funct)(Bit64u));
void plugin_register_periodic(void (*funct)(void));
void plugin_register_mem_map_IO(
  void (*f)(Bit32u, unsigned, unsigned, Bit32u *),
  Bit32u, Bit32u);
void plugin_call_elapsed (Bit64u elapsed);
void plugin_handle_periodic(void);
unsigned plugin_get_A20E(void);
void     plugin_set_A20E(unsigned val);

extern void (*memMapFunct)(Bit32u, unsigned, unsigned, Bit32u *);

void plugin_abort (void);

/* Atomic XCHG for reentrant function plugin_enqueue() */
#define XCHG(x,y)  \
    ({ \
        typeof(x) p;  \
        asm ( "xchg %0, %1" : "=r" (p) : "m" (x), "0" (y) : "memory" ); \
        p;  \
    })


#ifdef __cplusplus
};
#endif

#endif /* __PLUGIN_H */
