/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2000  Kevin P. Lawton
 *
 *  hack-linux.c: hacks so we can load Linux straight into memory
 *    without the BIOS.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>

#include "bios.h"




typedef struct
{
  Bit32u low;
  Bit32u high;
} gdt_entry;

typedef struct
{
   /* 0x000 */ Bit8u   orig_x;
   /* 0x001 */ Bit8u   orig_y;
   /* 0x002 */ Bit16u  memory_size_std;
   /* 0x004 */ Bit16u  orig_video_page;
   /* 0x006 */ Bit8u   orig_video_mode;
   /* 0x007 */ Bit8u   orig_video_cols;
   /* 0x008 */ Bit16u  unused1;
   /* 0x00a */ Bit16u  orig_video_ega_bx;
   /* 0x00c */ Bit16u  unused2;
   /* 0x00e */ Bit8u   orig_video_lines;
   /* 0x00f */ Bit8u   orig_video_isVGA;
   /* 0x010 */ Bit16u  orig_video_points;
   /* 0x012 */ Bit8u   pad1[0x40 - 0x12];
   /* 0x040 */ Bit8u   apm_info[0x80 - 0x40];
   /* 0x080 */ Bit8u   hd0_info[16];
   /* 0x090 */ Bit8u   hd1_info[16];
   /* 0x0a0 */ Bit8u   pad2[0x1e0 - 0xa0];
   /* 0x1e0 */ Bit32u  memory_size_ext;
   /* 0x1e4 */ Bit8u   pad3[0x1f1 - 0x1e4];
   /* 0x1f1 */ Bit8u   setup_sects;
   /* 0x1f2 */ Bit16u  mount_root_rdonly;
   /* 0x1f4 */ Bit16u  sys_size;
   /* 0x1f6 */ Bit16u  swap_dev;
   /* 0x1f8 */ Bit16u  ramdisk_flags;
   /* 0x1fa */ Bit16u  vga_mode;
   /* 0x1fc */ Bit16u  orig_root_dev;
   /* 0x1fe */ Bit16u  bootsect_magic;
   /* 0x200 */ Bit8u   pad4[0x210 - 0x200];
   /* 0x210 */ Bit32u  loader_type;
   /* 0x214 */ Bit32u  kernel_start;
   /* 0x218 */ Bit32u  initrd_start;
   /* 0x21c */ Bit32u  initrd_size;
   /* 0x220 */ Bit8u   pad5[0x400 - 0x220];
   /* 0x400 */ gdt_entry gdt[128];
   /* 0x800 */ Bit8u   commandline[2048];
} __attribute__ ((packed)) linux_setup_params;

  unsigned
load_linux_setup_params( Bit32u rd_start, Bit32u rd_size )
{
  linux_setup_params params;

  memset( &params, 0, sizeof(params) );

  /* Video settings (standard VGA) */
  params.orig_x = 0;
  params.orig_y = 0;
  params.orig_video_page = 0;
  params.orig_video_mode = 3;
  params.orig_video_cols = 80;
  params.orig_video_lines = 25;
  params.orig_video_points = 16;
  params.orig_video_isVGA = 1;
  params.orig_video_ega_bx = 3;

  /* Memory size (total mem - 1MB, in KB) */
  params.memory_size_ext = (vm_conf.max_memory - 1) * 1024;

  /* Boot parameters */
  params.loader_type = 1;
  params.bootsect_magic = 0xaa55;
  params.mount_root_rdonly = 0;
  params.orig_root_dev = 0x0100;
  params.initrd_start = rd_start;
  params.initrd_size  = rd_size;

  /* Initial GDT */
  params.gdt[2].high = 0x00cf9a00;
  params.gdt[2].low  = 0x0000ffff;
  params.gdt[3].high = 0x00cf9200;
  params.gdt[3].low  = 0x0000ffff;

  if (vm_write_physical(0x00090000, sizeof(params), &params)) {
    fprintf (stderr, "load_linux_setup_params: trying to load "
                     "beyond available VM memory.\n");
    return 1;
    }
  return 0; // OK
}


  void
get_cpu_linux_values(guest_cpu_t *cpu)
{
  // Set fields in a guest_cpu_t structure to RESET values
  memset(cpu, 0, sizeof(*cpu));

  // General Registers
  cpu->eax = 0; // processor tests passed
  cpu->ebx = 0; // +++
  cpu->ecx = 0;
  cpu->edx = 0;
  cpu->ebp = 0;
  cpu->esi = 0;
  cpu->edi = 0;
  cpu->esp = 0;
  cpu->eflags = 0x00000002; // only reserved bit set
  cpu->eip = 0xfff0;

  // CS
  cpu->cs.sel.raw = Selector(2, 0, RPL0);
  cpu->cs.des.limit_low = 0xffff;
  cpu->cs.des.base_low = 0;
  cpu->cs.des.base_med = 0;
  cpu->cs.des.type = 0x1A; // execute/read
  cpu->cs.des.dpl = 0;
  cpu->cs.des.p = 1;
  cpu->cs.des.limit_high = 0xf;
  cpu->cs.des.avl = 0;
  cpu->cs.des.reserved = 0;
  cpu->cs.des.d_b = 1;
  cpu->cs.des.g = 1;
  cpu->cs.des.base_high = 0;
  cpu->cs.valid = 1;

  // DS
  cpu->ds.sel.raw = Selector(2, 0, RPL0);
  cpu->ds.des.limit_low = 0xffff;
  cpu->ds.des.base_low = 0;
  cpu->ds.des.base_med = 0;
  cpu->ds.des.type = 0x12; // data seg, RW
  cpu->ds.des.dpl = 0;
  cpu->ds.des.p = 1;
  cpu->ds.des.limit_high = 0xf;
  cpu->ds.des.avl = 0;
  cpu->ds.des.reserved = 0;
  cpu->ds.des.d_b = 1;
  cpu->ds.des.g = 1;
  cpu->ds.des.base_high = 0;
  cpu->ds.valid = 1;

  // Copy DS to other data segments
  cpu->ss = cpu->ds;
  cpu->es = cpu->ds;
  cpu->fs = cpu->ds;
  cpu->gs = cpu->ds;

  // LDTR
  cpu->ldtr.sel.raw = 0;
  // +++ check out Table 8-1
  cpu->ldtr.valid = 0;

  // TR
  cpu->tr.sel.raw = 0;
  // +++ check out Table 8-1
  cpu->tr.valid = 0;

  // GDTR
  cpu->gdtr.limit = 0x400;
  cpu->gdtr.base  = 0x00090400;

  // IDTR
  cpu->idtr.base = 0;
  // +++ check out Table 8-1
  cpu->idtr.limit = 0x03ff;

  // Debug Registers
  cpu->dr0 = 0;
  cpu->dr1 = 0;
  cpu->dr2 = 0;
  cpu->dr3 = 0;
#if 0
  if (cpuid_info.procSignature.fields.family <= 4) {
    cpu->dr6 = 0xFFFF1FF0;
    cpu->dr7 = 0x00000000;
    }
  else {
#endif
    cpu->dr6 = 0xFFFF0FF0;
    cpu->dr7 = 0x00000400;
#if 0
    }
#endif

  // Test Registers
  cpu->tr3 = 0;
  cpu->tr4 = 0;
  cpu->tr5 = 0;
  cpu->tr6 = 0;
  cpu->tr7 = 0;

  // Control Registers
  // +++ CR0 reset: CD/NW unchanged, bit4 set, other cleared
  cpu->cr0 = 0x00000011;
  cpu->cr1 = 0;
  cpu->cr2 = 0;
  cpu->cr3 = 0;
  cpu->cr4 = 0;

  cpu->inhibit_mask = 0;
}


  void
get_cpu_test_values(guest_cpu_t *cpu)
{
  // Set fields in a guest_cpu_t structure to RESET values
  memset(cpu, 0, sizeof(*cpu));

  // General Registers
  cpu->eax = 0; // processor tests passed
  cpu->ebx = 0; // +++
  cpu->ecx = 0;
  cpu->edx = 0;
  cpu->ebp = 0;
  cpu->esi = 0;
  cpu->edi = 0;
  cpu->esp = 0;
  cpu->eflags = 0x00000002; // only reserved bit set
  cpu->eip = 0xfff0;

  // CS
  cpu->cs.sel.raw = Selector(1, 0, RPL0);
  cpu->cs.des.limit_low = 0xffff;
  cpu->cs.des.base_low = 0;
  cpu->cs.des.base_med = 0;
  cpu->cs.des.type = 0x1A; // execute/read
  cpu->cs.des.dpl = 0;
  cpu->cs.des.p = 1;
  cpu->cs.des.limit_high = 0xf;
  cpu->cs.des.avl = 0;
  cpu->cs.des.reserved = 0;
  cpu->cs.des.d_b = 1;
  cpu->cs.des.g = 1;
  cpu->cs.des.base_high = 0;
  cpu->cs.valid = 1;

  // DS
  cpu->ds.sel.raw = Selector(2, 0, RPL0);
  cpu->ds.des.limit_low = 0xffff;
  cpu->ds.des.base_low = 0;
  cpu->ds.des.base_med = 0;
  cpu->ds.des.type = 0x12; // data seg, RW
  cpu->ds.des.dpl = 0;
  cpu->ds.des.p = 1;
  cpu->ds.des.limit_high = 0xf;
  cpu->ds.des.avl = 0;
  cpu->ds.des.reserved = 0;
  cpu->ds.des.d_b = 1;
  cpu->ds.des.g = 1;
  cpu->ds.des.base_high = 0;
  cpu->ds.valid = 1;

  // Copy DS to other data segments
  cpu->ss = cpu->ds;
  cpu->es = cpu->ds;
  cpu->fs = cpu->ds;
  cpu->gs = cpu->ds;

  // LDTR
  cpu->ldtr.sel.raw = 0;
  // +++ check out Table 8-1
  cpu->ldtr.valid = 0;

  // TR
  cpu->tr.sel.raw = 0;
  // +++ check out Table 8-1
  cpu->tr.valid = 0;

  // GDTR
  cpu->gdtr.base = 0;
  // +++ check out Table 8-1
  cpu->gdtr.limit = 0;

  // IDTR
  cpu->idtr.base = 0;
  // +++ check out Table 8-1
  cpu->idtr.limit = 0x03ff;

  // Debug Registers
  cpu->dr0 = 0;
  cpu->dr1 = 0;
  cpu->dr2 = 0;
  cpu->dr3 = 0;
#if 0
  if (cpuid_info.procSignature.fields.family <= 4) {
    cpu->dr6 = 0xFFFF1FF0;
    cpu->dr7 = 0x00000000;
    }
  else {
#endif
    cpu->dr6 = 0xFFFF0FF0;
    cpu->dr7 = 0x00000400;
#if 0
    }
#endif

  // Test Registers
  cpu->tr3 = 0;
  cpu->tr4 = 0;
  cpu->tr5 = 0;
  cpu->tr6 = 0;
  cpu->tr7 = 0;

  // Control Registers
  // +++ CR0 reset: CD/NW unchanged, bit4 set, other cleared
  cpu->cr0 = 0x00000011;
  cpu->cr1 = 0;
  cpu->cr2 = 0;
  cpu->cr3 = 0;
  cpu->cr4 = 0;

  cpu->inhibit_mask = 0;
}
