/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2000  The plex86 developers team
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/mman.h>
#include <signal.h>
#include <errno.h>

#include "plex86.h"
#include "user.h"
#include "plugin.h"
#include "decode.h"


/* TODO: vm_read_physical doesn't deal with UCMem (VGA) */

/* #define RunMethod RunGuestNMethodBreakpoint */
/* #define RunMethod RunGuestNMethodEmulate */
#define RunMethod RunGuestNMethodExecute



/************************************************************************/
/* Structures / Variables                                               */
/************************************************************************/

char *ptr = NULL;         /* Pointer to the guest virtual phys mem */
char *printBuffer = NULL; /* mmap to monitor print buffer */

int filehdl;                     /* File handle for /dev/plex86    */

static unsigned char int_usage[256];    /* Interrupt usage count table    */

static void vm_timer_handler(int i);
Bit64u user_time_usec = 0;
static unsigned a20_val = 1;


/************************************************************************/
/* Initialization and abort code                                        */
/************************************************************************/

  void
vm_kickstart (void)
{
  /* open a new VM */

  vm_init();

  memset (int_usage, 0, 256);

  /* initialize all plugins */

  fprintf (stderr, "Initializing plugins\n");
  /*plugin_init_all (); */

#if !COSIMULATE
  {
  struct sigaction sg_act;
 
  memset(&sg_act, 0, sizeof(sg_act));
  sg_act.sa_handler = vm_timer_handler;
  sg_act.sa_flags = SA_RESTART;
  sigaction(SIGALRM,&sg_act,NULL);
  }
#endif

#if !COSIMULATE
  {
  struct itimerval timer;
  memset(&timer, 0, sizeof(timer));
  timer.it_value.tv_sec  = 0;
  timer.it_value.tv_usec = 500000;
  timer.it_interval.tv_sec  = 0;
  timer.it_interval.tv_usec = 500000;
  setitimer(ITIMER_REAL, &timer, NULL);
  }
#endif
}


  void
vm_timer_handler(int i)
{
#warning "fix: need exclusive access to variable in async handler"
  user_time_usec += 500000;
}

  void
vm_set_cpu(guest_cpu_t *guest_cpu)
{
  if (ioctl (filehdl, PLEX86_SET_CPU, guest_cpu)) {
    perror ("ioctl SET_CPU: \n");
    close (filehdl);
    exit (1);
    }
}


  void
vm_abort(void)
{
    static int was_waiting = 0;

    if (! was_waiting && vm_conf.exit_wait > 0) {
        struct timeval tv;

        was_waiting = 1;
        tv.tv_sec = vm_conf.exit_wait;
        tv.tv_usec = 0;

        printf("Delaying exit by %d seconds\n", (int)tv.tv_sec);
        while (1) {
           select(0, NULL, NULL, NULL, &tv);
         if (errno == EINTR && tv.tv_sec > 0)
           continue;
         break;
      }
    }

    /* deinitialize all plugins */

    fprintf (stderr, "Shutting down plugins\n");
    plugin_fini_all ();


    /* shut down the virtual machine */

    vm_fini ();


    /* we're done!! */

#if COSIMULATE
    {
    void bx_dbg_exit(int);
    bx_dbg_exit(1);
    }
#endif
    exit (0);
}

/************************************************************************/
/* VM initialization and deinitialization code                          */
/************************************************************************/

void
vm_open(void)
{
    /* open a new VM */

    fprintf (stderr, "Opening VM (/dev/plex86)\n");
    filehdl = open ("/dev/misc/plex86", O_RDWR);
    if (filehdl < 0)
    {
      /* Try the old name */
       filehdl =  open("/dev/plex86", O_RDWR);
       if (filehdl < 0) {
         fprintf(stderr, "Did you load the kernel module?"
                 "  Read the toplevel README file!\n");
         perror ("open");
         exit (1);
       }
    }
}

  void
vm_init_prescan_depth(unsigned depth)
{
    /* Request any tweaks to the VM environment: */
    if (vm_conf.prescanDepth) {
      fprintf (stderr, "Setting prescan depth to %u\n", vm_conf.prescanDepth);
      if (ioctl (filehdl, PLEX86_PRESCANDEPTH, vm_conf.prescanDepth) == -1) {
        perror ("ioctl PRESCANDEPTH: ");
        close (filehdl);
        exit (1);
        }
      }
}


void
vm_init_memory(unsigned nmegs)
{
    /* allocate memory from the host OS for the virtual physical memory */

    fprintf (stderr, "Allocating %dMB of physical memory in VM\n", nmegs);
    if (ioctl (filehdl, PLEX86_ALLOCVPHYS, nmegs) == -1)
    {
        perror ("ioctl ALLOCVPHYS: ");
        close (filehdl);
        exit (1);
    }

    /* map guest virtual physical memory into user address space and zero it */

    fprintf (stderr, "Mapping virtualized physical memory into monitor\n");
    ptr = mmap (NULL, nmegs * 1024 * 1024, PROT_READ | PROT_WRITE,
                MAP_SHARED, filehdl, 0);
    if (ptr == (void *) -1)
    {
        perror ("mmap of physical pages");
        close (filehdl);
        exit (1);
    }

    fprintf (stderr, "Zeroing virtualized physical memory\n");
    memset (ptr, 0, nmegs * 1024 * 1024);


    /* Create a memory mapping of the monitor's print buffer into
     * user memory.  This is used for efficient printing of info that
     * the monitor prints out.
     */
    fprintf (stderr, "Mapping monitor print buffer into user mem.\n");
    printBuffer = mmap(NULL, 4096, PROT_READ,
                       MAP_SHARED, filehdl, nmegs*1024*1024);
    if (ptr == (void *) -1)
    {
        perror ("mmap of monitor print buffer");
        close (filehdl);
        exit (1);
    }
}

  void
vm_init(void)
{
}

void
vm_fini (void)
{
    /* unmap the guest's virtual physical memory */

    fprintf (stderr, "Unmapping guest physical memory.\n");
    if (munmap (ptr, vm_conf.max_memory * 1024 * 1024) != 0)
    {
        perror ("munmap of guest physical memory");
        exit (1);
    }

    fprintf (stderr, "Unmapping monitor print buffer.\n");
    if (munmap (printBuffer, 4096) != 0)
    {
        perror ("munmap of monitor print buffer.");
        exit (1);
    }

    /* tell kernel module to clean up the VM */

    fprintf (stderr, "Tearing down VM\n");
    if (ioctl (filehdl, PLEX86_TEARDOWN, 0) == -1)
    {
        perror ("ioctl TEARDOWN: ");
        exit (1);
    }


    /* close the connection to the kernel module */

    fprintf (stderr, "Closing VM\n");
    if (close (filehdl) == -1)
    {
        perror ("close\n");
        exit (1);
    }
}


#if 0
#if USE_LOADER
/* Special loader stuff only I use - KPL */
#include "../../loader/loader_misc.h"
special_loader_misc_t loader_misc;
char *special_loader_fname = NULL;
#endif
#endif


/************************************************************************/
/* The main event loop                                                  */
/************************************************************************/

  void
vm_event_loop (void)
{
    vm_messages_t user_msgs;
    run_guest_n_t *run_guest_n_p;

#if 0
#if USE_LOADER
    /* Special loader stuff only I use - KPL */
    special_loader(special_loader_fname, &loader_misc, ptr);
#endif
#endif

    /* Build message to run guest */
    user_msgs.header.msg_type = VMMessageRunGuestN;
    user_msgs.header.msg_len = sizeof(run_guest_n_t);
    run_guest_n_p = (run_guest_n_t *) user_msgs.msg;
    run_guest_n_p->icount = ICOUNT_INDEFINITE;
    run_guest_n_p->method = RunMethod;

    for (;;)
    {
        if (ioctl (filehdl, PLEX86_MESSAGEQ, &user_msgs) == -1)
        {
            perror ("ioctl MESSSAGEQ: ");
            vm_abort();
        }

        switch (user_msgs.header.msg_type) 
        {
        case VMMessageNone:
            user_msgs.header.msg_type = VMMessageRunGuestN;
            user_msgs.header.msg_len = sizeof(run_guest_n_t);
            run_guest_n_p->icount = ICOUNT_INDEFINITE;
            run_guest_n_p->method = RunMethod;
            break;

        case VMMessagePanic:
            fprintf(stderr, "Fatal monitor error caused Panic\n");
            vm_debug_exception();
            vm_abort();
            break;

        case VMMessagePrintBuf:
        {
            fprintf(stderr, "::%s\n", printBuffer);
            /* No response required */
            user_msgs.header.msg_type = VMMessageRunGuestN;
            user_msgs.header.msg_len = sizeof(run_guest_n_t);
            run_guest_n_p->icount = ICOUNT_INDEFINITE;
            run_guest_n_p->method = RunMethod;
            break;
        }

        case VMMessageIOInRequest:
        {
            IO_msg_t *io_msg = (IO_msg_t *) user_msgs.msg;

            /* Zero out data, since it's potentially bigger in size than
             * the IO operation we will request
             */
            io_msg->data = 0;

            plugin_emulate_inport(io_msg->port, io_msg->len, 1, &io_msg->data);
            /* Change message type; most data stays the same */
            user_msgs.header.msg_type = VMMessageIOInResponse;
            break;
        }

        case VMMessageIOInBatchRequest:
        {
            IOBatch_msg_t *msg = (IOBatch_msg_t *) user_msgs.msg;
            unsigned i;

            for (i=0; i<msg->n; i++) {
#warning "put physical mem limit check here"
              plugin_emulate_inport(msg->port, msg->len, 1,
                                    &ptr[msg->paddr + i*msg->len]);
              }
            msg->n = msg->n; /* should be result of call above */
            /* Change message type; most data stays the same */
            user_msgs.header.msg_type = VMMessageIOInBatchResponse;
            break;
        }

        case VMMessageIOOutRequest:
        {
            IO_msg_t *io_msg = (IO_msg_t *) user_msgs.msg;
            plugin_emulate_outport(io_msg->port, io_msg->len, 1, &io_msg->data);
            /* No response required */
            user_msgs.header.msg_type = VMMessageRunGuestN;
            user_msgs.header.msg_len = sizeof(run_guest_n_t);
            run_guest_n_p->icount = ICOUNT_INDEFINITE;
            run_guest_n_p->method = RunMethod;
            break;
        }

        case VMMessageMemMapIOReadRequest:
            {
            memMapIO_msg_t *msg = (memMapIO_msg_t *) user_msgs.msg;

            /* Zero out data, since it's potentially bigger in size than
             * the IO operation we will request
             */
            msg->data = 0;

            if (memMapFunct) {
              memMapFunct(msg->addr, msg->len, IO_IN, &msg->data);
              }
            else {
              fprintf(stderr, "MemMapIOWriteRequest: not handled.\n");
              vm_abort();
              }
            /* Change message type; most data stays the same */
            user_msgs.header.msg_type = VMMessageMemMapIOReadResponse;
            break;
            }

        case VMMessageMemMapIOWriteRequest:
            {
            memMapIO_msg_t *msg = (memMapIO_msg_t *) user_msgs.msg;
            if (memMapFunct) {
              memMapFunct(msg->addr, msg->len, IO_OUT, &msg->data);
              }
            else {
              fprintf(stderr, "MemMapIOWriteRequest: not handled.\n");
              vm_abort();
              }
            user_msgs.header.msg_type = VMMessageRunGuestN;
            user_msgs.header.msg_len = sizeof(run_guest_n_t);
            run_guest_n_p->icount = ICOUNT_INDEFINITE;
            run_guest_n_p->method = RunMethod;
            break;
            }

        case VMMessageIACRequest:
        {
            IAC_msg_t *iac_msg = (IAC_msg_t *) user_msgs.msg;
            iac_msg->vector =  plugin_acknowledge_intr();

            user_msgs.header.msg_type = VMMessageIACResponse;
            user_msgs.header.msg_len = sizeof(IAC_msg_t);
            break;
        }

        case VMMessageIntRequest:
        {
            INT_msg_t *int_msg = (INT_msg_t *) user_msgs.msg;
            int_msg->reflect = plugin_emulate_int (int_msg->vector);

            user_msgs.header.msg_type = VMMessageIntResponse;
            break;
        }

        case VMMessageTimeElapsed:
        {
            void do_timer(Bit64u elapsed);
            time_elapsed_t *time_elapsed = (time_elapsed_t *) user_msgs.msg;
            plugin_call_elapsed(time_elapsed->elapsed);
 
            /* No response required */
            user_msgs.header.msg_type = VMMessageRunGuestN;
            user_msgs.header.msg_len = sizeof(run_guest_n_t);
            run_guest_n_p->icount = ICOUNT_INDEFINITE;
            run_guest_n_p->method = RunMethod;
            break;
        }

        case VMMessageDisasm:
          vm_debug_exception();
          user_msgs.header.msg_type = VMMessageRunGuestN;
          user_msgs.header.msg_len = sizeof(run_guest_n_t);
          run_guest_n_p->icount = ICOUNT_INDEFINITE;
          run_guest_n_p->method = RunMethod;
          fprintf(stderr, "> ");
          getchar();
          break;


        default:
            fprintf(stderr, "VMMessage type %u not handled yet\n",
                    user_msgs.header.msg_type);
            vm_abort();
            break;
        }

        plugin_handle_periodic();
    }
}




/************************************************************************/
/* Interrupt stuff                                                      */
/************************************************************************/

void
vm_set_intr (int intr)
{
    if (ioctl (filehdl, PLEX86_SETINTR, intr) == -1)
    {
        perror ("ioctl SETINTR: ");
        vm_abort();
    }
}

int
vm_alloc_intr (int intr)
{
    if (intr < 0 || intr > 255)
        return 1;

    if (ioctl (filehdl, PLEX86_ALLOCINT, intr) == -1)
        return 1;

    int_usage[intr]++;
    return 0;
}

int
vm_release_intr (int intr)
{
    if (intr < 0 || intr > 255)
        return 1;

    if (--int_usage[intr])
        return 0;

    if (ioctl (filehdl, PLEX86_RELEASEINT, intr) == -1)
        return 1;

    return 0;
}


/************************************************************************/
/* Guest register access                                                */
/************************************************************************/

  void
vm_get_cpu(guest_cpu_t *cpu)
{
  if (ioctl(filehdl, PLEX86_GET_CPU, cpu)) {
    perror("ioctl GET_CPU:");
    close(filehdl);
    exit(1);
    }
}

/************************************************************************/
/* Guest memory access                                                  */
/************************************************************************/

int
vm_read_physical (Bit32u address, unsigned length, void *data)
{
    if ( address+length >= vm_conf.max_memory*1024*1024 )
        return 1;

    memcpy (data, ptr+address, length);
    return 0;
}

int
vm_write_physical (Bit32u address, unsigned length, void *data)
{
    if ( address+length >= vm_conf.max_memory*1024*1024 )
        return 1;

    memcpy(((char *)ptr)+address, data, length);
    if ( ioctl(filehdl, PLEX86_PHYMEM_MOD, 0) ) {
      perror("ioctl PHYMEM_MOD:");
      close(filehdl);
      exit(1);
      }
    return 0;
}


/************************************************************************/
/* Debug event handler                                                  */
/************************************************************************/

int
vm_debug_exception(void)
{
    guest_cpu_t     cpu;

    struct i386_context ctx;
    struct i386_decode instr;
    int   i, len = 8;
    char  prefix[256], text[256];
    Bit8u *seg_base_ptr;


#if COSIMULATE
    bx_print_last_sync_icount();
#endif

    /* get guest context */

    vm_get_cpu(&cpu);

#define LowDWord(d)  ( ((Bit32u*) (&(d)))[0] )
#define HighDWord(d) ( ((Bit32u*) (&(d)))[1] )

    /* Dump state of CPU */
    fprintf(stderr, "eax:0x%x\n", (unsigned) cpu.eax);
    fprintf(stderr, "ebx:0x%x\n", (unsigned) cpu.ebx);
    fprintf(stderr, "ecx:0x%x\n", (unsigned) cpu.ecx);
    fprintf(stderr, "edx:0x%x\n", (unsigned) cpu.edx);

    fprintf(stderr, "ebp:0x%x\n", (unsigned) cpu.ebp);
    fprintf(stderr, "esi:0x%x\n", (unsigned) cpu.esi);
    fprintf(stderr, "edi:0x%x\n", (unsigned) cpu.edi);
    fprintf(stderr, "esp:0x%x\n", (unsigned) cpu.esp);

    fprintf(stderr, "eflags:0x%x\n", (unsigned) cpu.eflags);
    fprintf(stderr, "eip:0x%x\n", (unsigned) cpu.eip);

    fprintf(stderr, "cs:s=0x%x, dl=0x%x, dh=0x%x, valid=%u\n",
      (unsigned) cpu.cs.sel.raw, LowDWord(cpu.cs.des),
      (unsigned) HighDWord(cpu.cs.des), (unsigned) cpu.cs.valid);

    fprintf(stderr, "ss:s=0x%x, dl=0x%x, dh=0x%x, valid=%u\n",
      (unsigned) cpu.ss.sel.raw, (unsigned) LowDWord(cpu.ss.des),
      (unsigned) HighDWord(cpu.ss.des), (unsigned) cpu.ss.valid);

    fprintf(stderr, "ds:s=0x%x, dl=0x%x, dh=0x%x, valid=%u\n",
      (unsigned) cpu.ds.sel.raw, (unsigned) LowDWord(cpu.ds.des),
      (unsigned) HighDWord(cpu.ds.des), (unsigned) cpu.ds.valid);

    fprintf(stderr, "es:s=0x%x, dl=0x%x, dh=0x%x, valid=%u\n",
      (unsigned) cpu.es.sel.raw, (unsigned) LowDWord(cpu.es.des),
      (unsigned) HighDWord(cpu.es.des), (unsigned) cpu.es.valid);

    fprintf(stderr, "fs:s=0x%x, dl=0x%x, dh=0x%x, valid=%u\n",
      (unsigned) cpu.fs.sel.raw, (unsigned) LowDWord(cpu.fs.des),
      (unsigned) HighDWord(cpu.fs.des), (unsigned) cpu.fs.valid);

    fprintf(stderr, "gs:s=0x%x, dl=0x%x, dh=0x%x, valid=%u\n",
      (unsigned) cpu.gs.sel.raw, (unsigned) LowDWord(cpu.gs.des),
      (unsigned) HighDWord(cpu.gs.des), (unsigned) cpu.gs.valid);

    fprintf(stderr, "ldtr:s=0x%x, dl=0x%x, dh=0x%x, valid=%u\n",
      (unsigned) cpu.ldtr.sel.raw, (unsigned) LowDWord(cpu.ldtr.des),
      (unsigned) HighDWord(cpu.ldtr.des), (unsigned) cpu.ldtr.valid);

    fprintf(stderr, "tr:s=0x%x, dl=0x%x, dh=0x%x, valid=%u\n",
      (unsigned) cpu.tr.sel.raw, (unsigned) LowDWord(cpu.tr.des),
      (unsigned) HighDWord(cpu.tr.des), (unsigned) cpu.tr.valid);

    fprintf(stderr, "gdtr:base=0x%x, limit=0x%x\n",
      (unsigned) cpu.gdtr.base, (unsigned) cpu.gdtr.limit);

    fprintf(stderr, "idtr:base=0x%x, limit=0x%x\n",
      (unsigned) cpu.idtr.base, (unsigned) cpu.idtr.limit);

    fprintf(stderr, "dr0:0x%x\n", (unsigned) cpu.dr0);
    fprintf(stderr, "dr1:0x%x\n", (unsigned) cpu.dr1);
    fprintf(stderr, "dr2:0x%x\n", (unsigned) cpu.dr2);
    fprintf(stderr, "dr3:0x%x\n", (unsigned) cpu.dr3);
    fprintf(stderr, "dr6:0x%x\n", (unsigned) cpu.dr6);
    fprintf(stderr, "dr7:0x%x\n", (unsigned) cpu.dr7);

    fprintf(stderr, "tr3:0x%x\n", (unsigned) cpu.tr3);
    fprintf(stderr, "tr4:0x%x\n", (unsigned) cpu.tr4);
    fprintf(stderr, "tr5:0x%x\n", (unsigned) cpu.tr5);
    fprintf(stderr, "tr6:0x%x\n", (unsigned) cpu.tr6);
    fprintf(stderr, "tr7:0x%x\n", (unsigned) cpu.tr7);

    fprintf(stderr, "cr0:0x%x\n", (unsigned) cpu.cr0);
    fprintf(stderr, "cr1:0x%x\n", (unsigned) cpu.cr1);
    fprintf(stderr, "cr2:0x%x\n", (unsigned) cpu.cr2);
    fprintf(stderr, "cr3:0x%x\n", (unsigned) cpu.cr3);
    fprintf(stderr, "cr4:0x%x\n", (unsigned) cpu.cr4);

    fprintf(stderr, "inhibit_mask:%u\n", cpu.inhibit_mask);


    /* dump the guest context */

    memset (&ctx, 0, sizeof (ctx));
    memset (&instr, 0, sizeof (instr));

    if (!cpu.cs.valid) {
      fprintf(stderr, "vm_debug_exception: CS.valid=0\n");
      return 0;
      }
    if (cpu.cr0 & 1) {
      if (cpu.eflags & 0x20000)
        ctx.mode = VM86;
      else if (cpu.cs.des.d_b)
        ctx.mode = CODE32;
      else
        ctx.mode = CODE16;
      }
    else { /* Real Mode */
      ctx.mode = VM86;
      }
    seg_base_ptr = ptr + BaseOfDescriptor(cpu.cs.des);
    ctx.base    = seg_base_ptr;
    ctx.segment = cpu.cs.sel.raw;
    ctx.offset  = cpu.eip;


    if (!i386_decode (&ctx, &instr) || !i386_decode_text (&ctx, &instr, text, vm_conf.syntax))
        strcpy (text, "???");
    else
        len = instr.length;

    sprintf (prefix, "%04X.%08X  ", cpu.cs.sel.raw, cpu.eip);

    for (i = 0; i < len && i < 12; i++)
        sprintf (prefix + strlen (prefix), "%02X",
                 ((unsigned char *) seg_base_ptr)[cpu.eip + i]);
    for (; i < 12; i++)
        strcat (prefix, "  ");


    fprintf (stderr, "Stack dump:\n");
#if 0
    for (i = 0; i < 4; i++)
        fprintf (stderr, " %08x:  %08lx %08lx %08lx %08lx\n",
                 context.esp + 16 * i,
                 *(unsigned long *) (ptr + context.esp + 16 * i),
                 *(unsigned long *) (ptr + context.esp + 16 * i + 4),
                 *(unsigned long *) (ptr + context.esp + 16 * i + 8),
                 *(unsigned long *) (ptr + context.esp + 16 * i + 12));
    fprintf (stderr, "\n");
#endif

    fprintf (stderr, "Current instruction:\n");
    fprintf (stderr, " %s %s\n", prefix, text);
    fprintf (stderr, "\n");

    return 1;
}

  void
vm_register_callback(const char *command, void (*f)(unsigned char *))
{
  callback_command_t *entry;

  entry = malloc( sizeof(callback_command_t) );
  if (!entry) {
    fprintf(stderr, "vm_register_callback: malloc failed\n");
    exit(1);
    }
  entry->next = callback_command_list;
  callback_command_list = entry;
  entry->command = command;
  entry->f = f;
}


  unsigned
vm_load_rom(const char *path, Bit32u address)
{
  struct stat stat_buf;
  int fileno;
  unsigned char *image;


  fileno = open(path, O_RDONLY);
  if (fileno < 0) {
    fprintf(stderr, "rom: open %s: %s\n", path, strerror(errno));
    return 1;
    }
  if (fstat(fileno, &stat_buf) != 0) {
    fprintf(stderr, "rom: fstat %s: %s\n", path, strerror(errno));
    return 1;
    }

  fprintf(stderr, "ROM: loading image '%s' @ 0x%x (%u bytes)\n",
    path, address, (unsigned) stat_buf.st_size);

  if ((image = (void *) malloc (stat_buf.st_size)) == NULL) {
    perror("rom: malloc");
    return 1;
    }
  if (read (fileno, image, stat_buf.st_size) != stat_buf.st_size) {
    perror ("rom: read");
    return 1;
    }
  close(fileno);

  if (vm_write_physical(address, stat_buf.st_size, image)) {
    fprintf (stderr, "rom: trying to load beyond available VM memory.\n");
    return 1;
    }

  free(image);

  return 0;
}

  unsigned
linux_hack(void)
{
  Bit32u data;

  /* +++ Enable A20 line */
  /*BX_SET_ENABLE_A20( 1 ); */

  /* Setup PICs the way Linux likes it */
  data = 0x11; plugin_emulate_outport( 0x20, 1, 1, &data);
  data = 0x11; plugin_emulate_outport( 0xA0, 1, 1, &data);
  data = 0x20; plugin_emulate_outport( 0x21, 1, 1, &data);
  data = 0x28; plugin_emulate_outport( 0xA1, 1, 1, &data);
  data = 0x04; plugin_emulate_outport( 0x21, 1, 1, &data);
  data = 0x02; plugin_emulate_outport( 0xA1, 1, 1, &data);
  data = 0x01; plugin_emulate_outport( 0x21, 1, 1, &data);
  data = 0x01; plugin_emulate_outport( 0xA1, 1, 1, &data);
  data = 0xFF; plugin_emulate_outport( 0x21, 1, 1, &data);
  data = 0xFB; plugin_emulate_outport( 0xA1, 1, 1, &data);

  /* Disable interrupts and NMIs */
  /*BX_CPU_THIS_PTR eflags.if_ = 0; */
  data = 0x80; plugin_emulate_outport( 0x70, 1, 1, &data);

  /* Enter protected mode */
  /*BX_CPU_THIS_PTR cr0.pe = 1; */

  /* Set up initial GDT */
  /*BX_CPU_THIS_PTR gdtr.limit = 0x400; */
  /*BX_CPU_THIS_PTR gdtr.base  = 0x00090400; */

  /* Jump to protected mode entry point */
  /*BX_CPU_THIS_PTR jump_protected( NULL, 0x10, 0x00100000 ); */

  return 0;
}


  unsigned
plugin_get_A20E(void)
{
  return a20_val > 0;
}

  void
plugin_set_A20E(unsigned val)
{
  a20_val = val;
  if (ioctl(filehdl, PLEX86_SET_A20, val)) {
    perror("ioctl SET_A20: ");
    vm_abort();
    }
}
