/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2000 Kevin P. Lawton
 *
 *  arith16.c:  16-bit arithmetic instructions
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "monitor.h"



  void
CMPXCHG_EwGw(vm_t *vm)
{
  Bit16u op2, op1, diff;
  Bit32u eflags;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "subw %2, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags), "=r" (diff)
    : "g" (op1), "1" (G_AX(vm))
    : "memory", "cc"
    );
 
  if (diff == 0) {
    /* dest <-- src */
    op2 = ReadReg16(vm, vm->i.nnn);
 
    /* +++ docs say CMPXCHG causes a write cycle regardless */
    if (vm->i.mod == 0xc0) {
      WriteReg16(vm, vm->i.rm, op2);
      }
    else {
      write_RMW_virtual_word(vm, &op2);
      }
    }
  else {
    /* accumulator <-- dest */
    G_AX(vm) = op1;
    }

  SetOSZAPC(vm, eflags);
}

  void
ADC_AXIw(vm_t *vm)
{
  Bit16u op1, op2;
  Bit32u eflags;
 
  op1 = G_AX(vm);
  op2 = vm->i.Iw;
 
  if (G_GetCF(vm)) {
    asm volatile (
      "stc        \n\t"
      "adcw %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
  else {
    asm volatile (
      "clc        \n\t"
      "adcw %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
 
  G_AX(vm) = op1;
 
  SetOSZAPC(vm, eflags);
}

  void
SBB_EwGw(vm_t *vm)
{
  Bit16u op2, op1;
  Bit32u eflags;
 
  op2 = ReadReg16(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  if (G_GetCF(vm)) {
    asm volatile (
      "stc        \n\t"
      "sbbw %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
  else {
    asm volatile (
      "clc        \n\t"
      "sbbw %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
 
  if (vm->i.mod == 0xc0) {
    WriteReg16(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_word(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
SBB_AXIw(vm_t *vm)
{
  Bit16u op1, op2;
  Bit32u eflags;
 
  op1 = G_AX(vm);
  op2 = vm->i.Iw;

  if (G_GetCF(vm)) {
    asm volatile (
      "stc        \n\t"
      "sbbw %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
  else {
    asm volatile (
      "clc        \n\t"
      "sbbw %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
 
  G_AX(vm) = op1;
 
  SetOSZAPC(vm, eflags);
}

  void
ADC_GwEw(vm_t *vm)
{
  Bit16u op1, op2;
  Bit32u eflags;
 
  op1 = ReadReg16(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
 
  if (G_GetCF(vm)) {
    asm volatile (
      "stc        \n\t"
      "adcw %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
  else {
    asm volatile (
      "clc        \n\t"
      "adcw %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
 
  WriteReg16(vm, vm->i.nnn, op1);
 
  SetOSZAPC(vm, eflags);
}

  void
ADC_EwGw(vm_t *vm)
{
  Bit16u op2, op1;
  Bit32u eflags;
 
  op2 = ReadReg16(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  if (G_GetCF(vm)) {
    asm volatile (
      "stc        \n\t"
      "adcw %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
  else {
    asm volatile (
      "clc        \n\t"
      "adcw %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
 
  if (vm->i.mod == 0xc0) {
    WriteReg16(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_word(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
SUB_AXIw(vm_t *vm)
{
  Bit16u op1, op2;
  Bit32u eflags;
 
  op1 = G_AX(vm);
  op2 = vm->i.Iw;
 
  asm volatile (
    "subw %3, %1 \n\t"
    "pushfl      \n\t"
    "popl   %0"
    : "=g" (eflags), "=r" (op1)
    : "1" (op1), "g" (op2)
    : "memory", "cc"
    );
 
  G_AX(vm) = op1;
 
  SetOSZAPC(vm, eflags);
}

  void
NEG_Ew(vm_t *vm)
{
  Bit16u op1;
  Bit32u eflags;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "negw %1 \n\t"
    "pushfl  \n\t"
    "popl   %0"
    : "=g" (eflags), "=r" (op1)
    : "1" (op1)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg16(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_word(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
DEC_Ew(vm_t *vm)
{
  Bit16u op1;
  Bit32u eflags;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "decw %1 \n\t"
    "pushfl  \n\t"
    "popl   %0"
    : "=g" (eflags), "=r" (op1)
    : "1" (op1)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg16(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_word(vm, &op1);
    }
 
  SetOSZAP(vm, eflags);
}

  void
SBB_GwEw(vm_t *vm)
{
  Bit16u op1, op2;
  Bit32u eflags;

  op1 = ReadReg16(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
 
  if (G_GetCF(vm)) {
    asm volatile (
      "stc         \n\t"
      "sbbw %3, %1 \n\t"
      "pushfl      \n\t"
      "popl   %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
  else {
    asm volatile (
      "clc         \n\t"
      "sbbw %3, %1 \n\t"
      "pushfl      \n\t"
      "popl   %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
 
  WriteReg16(vm, vm->i.nnn, op1);
 
  SetOSZAPC(vm, eflags);
}

  void
CWD(vm_t *vm)
{
  if (G_AX(vm) & 0x8000) {
    G_DX(vm) = 0xFFFF;
    }
  else {
    G_DX(vm) = 0x0000;
    }
}

  void
SUB_EwGw(vm_t *vm)
{
  Bit16u op2, op1;
  Bit32u eflags;
 
  op2 = ReadReg16(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "subw %3, %1 \n\t"
    "pushfl      \n\t"
    "popl   %0"
    : "=g" (eflags), "=r" (op1)
    : "1" (op1), "g" (op2)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg16(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_word(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
ADD_AXIw(vm_t *vm)
{
  Bit16u op1, op2;
  Bit32u eflags;
 
  op1 = G_AX(vm);
  op2 = vm->i.Iw;
 
  asm volatile (
    "addw %3, %1 \n\t"
    "pushfl      \n\t"
    "popl   %0"
    : "=g" (eflags), "=r" (op1)
    : "1" (op1), "g" (op2)
    : "memory", "cc"
    );
 
  G_AX(vm) = op1;
 
  SetOSZAPC(vm, eflags);
}

  void
CMP_GwEw(vm_t *vm)
{
  Bit16u op1, op2;
  Bit32u eflags;
 
  op1 = ReadReg16(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
 
  asm volatile (
    "cmpw %2, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags)
    : "r" (op1), "g" (op2)
    : "memory", "cc"
    );
 
  SetOSZAPC(vm, eflags);
}

  void
ADC_EwIw(vm_t *vm)
{
  Bit16u op2, op1;
  Bit32u eflags;
 
  op2 = vm->i.Iw;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  if (G_GetCF(vm)) {
    asm volatile (
      "stc        \n\t"
      "adcw %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
  else {
    asm volatile (
      "clc        \n\t"
      "adcw %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
 
  if (vm->i.mod == 0xc0) {
    WriteReg16(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_word(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
SBB_EwIw(vm_t *vm)
{
  Bit16u op2, op1;
  Bit32u eflags;
 
  op2 = vm->i.Iw;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  if (G_GetCF(vm)) {
    asm volatile (
      "stc        \n\t"
      "sbbw %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
  else {
    asm volatile (
      "clc        \n\t"
      "sbbw %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
 
  if (vm->i.mod == 0xc0) {
    WriteReg16(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_word(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
SUB_EwIw(vm_t *vm)
{
  Bit16u op2, op1;
  Bit32u eflags;
 
  op2 = vm->i.Iw;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "subw %3, %1 \n\t"
    "pushfl      \n\t"
    "popl   %0"
    : "=g" (eflags), "=r" (op1)
    : "1" (op1), "g" (op2)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg16(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_word(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
CMP_EwIw(vm_t *vm)
{
  Bit16u op2, op1;
  Bit32u eflags;
 
  op2 = vm->i.Iw;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "cmpw %2, %1 \n\t"
    "pushfl      \n\t"
    "popl   %0"
    : "=g" (eflags)
    : "r" (op1), "g" (op2)
    : "memory", "cc"
    );
 
  SetOSZAPC(vm, eflags);
}

  void
INC_RX(vm_t *vm)
{
  Bit16u rx;
  unsigned reg;
  Bit32u eflags;

  reg = vm->i.b1 & 0x07;
  rx = ReadReg16(vm, reg);

  asm volatile (
    "incw %1 \n\t"
    "pushfl  \n\t"
    "popl   %0"
    : "=g" (eflags), "=r" (rx) 
    : "1" (rx)
    : "memory", "cc"
    );

  WriteReg16(vm, reg, rx);
  SetOSZAP(vm, eflags);
}

  void
DEC_RX(vm_t *vm)
{
  Bit16u rx;
  unsigned reg;
  Bit32u eflags;
 
  reg = vm->i.b1 & 0x07;
  rx = ReadReg16(vm, reg);
 
  asm volatile (
    "decw %1 \n\t"
    "pushfl  \n\t"
    "popl   %0"
    : "=g" (eflags), "=r" (rx)
    : "1" (rx)
    : "memory", "cc"
    );
 
  WriteReg16(vm, reg, rx);
  SetOSZAP(vm, eflags);
}

  void
CMP_AXIw(vm_t *vm)
{
  Bit16u op2, op1;
  Bit32u eflags;
 
  op1 = G_AX(vm);
  op2 = vm->i.Iw;
 
  asm volatile (
    "cmpw %2, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags)
    : "r" (op1), "g" (op2)
    : "memory", "cc"
    );
 
  SetOSZAPC(vm, eflags);
}


  void
INC_Ew(vm_t *vm)
{
  Bit16u op1;
  Bit32u eflags;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "incw %1 \n\t"
    "pushfl  \n\t"
    "popl   %0"
    : "=g" (eflags), "=r" (op1) 
    : "1" (op1)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg16(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_word(vm, &op1);
    }
 
  SetOSZAP(vm, eflags);
}

  void
CMP_EwGw(vm_t *vm)
{
  Bit16u op2, op1;
  Bit32u eflags;
 
  op2 = ReadReg16(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "cmpw %2, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags)
    : "r" (op1), "g" (op2)
    : "memory", "cc"
    );
 
  SetOSZAPC(vm, eflags);
}

  void
ADD_EwGw(vm_t *vm)
{
  Bit16u op2, op1;
  Bit32u eflags;
 
  op2 = ReadReg16(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "addw %3, %1 \n\t"
    "pushfl      \n\t"
    "popl   %0"
    : "=g" (eflags), "=r" (op1)
    : "1" (op1), "g" (op2)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg16(vm, vm->i.rm, op1);
    }
  else {
    write_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
ADD_GwEw(vm_t *vm)
{
  Bit16u op1, op2;
  Bit32u eflags;
 
  op1 = ReadReg16(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
 
  asm volatile (
    "addw %3, %1 \n\t"
    "pushfl      \n\t"
    "popl   %0"
    : "=g" (eflags), "=r" (op1)
    : "1" (op1), "g" (op2)
    : "memory", "cc"
    );
 
  WriteReg16(vm, vm->i.nnn, op1);
 
  SetOSZAPC(vm, eflags);
}

  void
ADD_EwIw(vm_t *vm)
{
  Bit16u op2, op1;
  Bit32u eflags;
 
 
  op2 = vm->i.Iw;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "addw %3, %1 \n\t"
    "pushfl      \n\t"
    "popl   %0"
    : "=g" (eflags), "=r" (op1)
    : "1" (op1), "g" (op2)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg16(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_word(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
SUB_GwEw(vm_t *vm)
{
  Bit32u eflags;
  Bit16u op1, op2;
 
  op1 = ReadReg16(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
 
  asm volatile (
    "subw %3, %1 \n\t"
    "pushfl      \n\t"
    "popl   %0"
    : "=g" (eflags), "=r" (op1)
    : "1" (op1), "g" (op2)
    : "memory", "cc"
    );

 
  WriteReg16(vm, vm->i.nnn, op1);
 
  SetOSZAPC(vm, eflags);
}

  void
CBW(vm_t *vm)
{
  G_AX(vm) = (Bit8s) G_AL(vm);
}

  void
XADD_EwGw(vm_t *vm)
{
  Bit16u op2, op1, sum;
  Bit32u eflags;
 
  /* XADD dst(r/m8), src(r8)
   * temp <-- src + dst         | sum = op2 + op1
   * src  <-- dst               | op2 = op1
   * dst  <-- tmp               | op1 = sum
   */
 
  op2 = ReadReg16(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
  sum = op1;
 
  asm volatile (
    "addw %3, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags), "=r" (sum)
    : "1" (sum), "g" (op2)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg16(vm, vm->i.nnn, op1);
    WriteReg16(vm, vm->i.rm, sum);
    }
  else {
    write_RMW_virtual_word(vm, &sum);
    WriteReg16(vm, vm->i.nnn, op1);
    }
 
  SetOSZAPC(vm, eflags);
}
