/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2000 Kevin P. Lawton
 *
 *  ctrl_xfer16.c:  emulation of 16-bit control transfer instructions
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "monitor.h"


  void
RETfar16_Iw(vm_t *vm)
{
  Bit16s imm16;
  Bit16u ip, cs_raw;
 
  invalidate_prefetch_q();

  imm16 = vm->i.Iw;
 
  if (ProtectedMode(vm)) {
    return_protected(vm, imm16);
    goto done;
    }
 
  cache_sreg(vm, SRegSS);

  pop16(vm, &ip);
  pop16(vm, &cs_raw);
  load_seg_reg(vm, SRegCS, cs_raw);
  G_EIP(vm) = (Bit32u) ip;

  if (vm->guest_cpu.desc_cache[SRegSS].desc.d_b)
    G_ESP(vm) += imm16;
  else
    G_SP(vm)  += imm16;
 
done:
  /* instrumentation call was here */
  return;
}

  void
RETfar16(vm_t *vm)
{
  Bit16u ip, cs_raw;
 
  invalidate_prefetch_q();
 
  if ( ProtectedMode(vm) ) {
    return_protected(vm, 0);
    goto done;
    }
 
  pop16(vm, &ip);
  pop16(vm, &cs_raw);
  load_seg_reg(vm, SRegCS, cs_raw);
  G_EIP(vm) = (Bit32u) ip;
 
done:
  /* instrumentation was here */
  return;
}

  void
CALL_Aw(vm_t *vm)
{
  Bit32u new_EIP;

  invalidate_prefetch_q();
 
  new_EIP = G_EIP(vm) + (Bit32s) vm->i.Id;
  new_EIP &= 0x0000ffff;
 
  if ( ProtectedMode(vm) ) {
    cache_sreg(vm, SRegCS);
    if ( new_EIP > vm->guest_cpu.desc_cache[SRegCS].limit_scaled ) {
      monpanic(vm, "call_aw: offset outside of CS limits\n");
      exception(vm, ExceptionGP, 0);
      }
    }
 
  /* push 32 bit EA of next instruction */
  push16(vm, G_IP(vm));
  G_EIP(vm) = new_EIP;
}

  void
CALL16_Ap(vm_t *vm)
{
  Bit16u cs_raw;
  Bit16u disp16;
 
  invalidate_prefetch_q();
  disp16 = vm->i.Iw;
  cs_raw = vm->i.Iw2;
 
  if (ProtectedMode(vm)) {
    call_protected(vm, cs_raw, disp16);
    goto done;
    }
  cache_selector(vm, SRegCS);
  push16(vm, vm->guest_cpu.selector[SRegCS].raw);
  push16(vm, G_IP(vm));
  load_seg_reg(vm, SRegCS, cs_raw);
  G_EIP(vm) = (Bit32u) disp16;
 
done:
  /* instrumentation code was here */
  return;
}

  void
CALL_Ew(vm_t *vm)
{
  Bit32u temp_ESP;
  Bit16u op1_16;
 
  cache_sreg(vm, SRegCS);
  cache_sreg(vm, SRegSS);
  if (vm->guest_cpu.desc_cache[SRegSS].desc.d_b)
    temp_ESP = G_ESP(vm);
  else
    temp_ESP = G_SP(vm);
 
 
  if (vm->i.mod == 0xc0) {
    op1_16 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1_16);
    }
  invalidate_prefetch_q();
 
  if (ProtectedMode(vm)) {
    if (op1_16 > vm->guest_cpu.desc_cache[SRegCS].limit_scaled) {
      monpanic(vm, "call_ev: IP out of CS limits!\n");
      exception(vm, ExceptionGP, 0);
      }
    }
 
  push16(vm, G_IP(vm));
 
  G_EIP(vm) = op1_16;
}

  void
CALL16_Ep(vm_t *vm)
{
  Bit16u cs_raw;
  Bit16u op1_16;
 
  invalidate_prefetch_q();

  if (vm->i.mod == 0xc0) {
    monpanic(vm, "CALL_Ep: op1 is a register");
    }
 
  /* pointer, segment address pair */
  read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1_16);
  read_virtual_word(vm, vm->i.seg, vm->i.rm_addr+2, &cs_raw);
 
  if ( ProtectedMode(vm) ) {
    call_protected(vm, cs_raw, op1_16);
    goto done;
    }
 
  cache_selector(vm, SRegCS);
  push16(vm, vm->guest_cpu.selector[SRegCS].raw);
  push16(vm, G_IP(vm));
 
  load_seg_reg(vm, SRegCS, cs_raw);
  G_EIP(vm) = op1_16;
 
done:
  /* Instrumentation call was here */
  return;
}

  void
JMP_Jw(vm_t *vm)
{
  Bit32u new_EIP;
 
  invalidate_prefetch_q();
 
  new_EIP = G_EIP(vm) + (Bit32s) vm->i.Id;
  new_EIP &= 0x0000ffff;
 
  if (ProtectedMode(vm)) {
    cache_sreg(vm, SRegCS);
    if (new_EIP > vm->guest_cpu.desc_cache[SRegCS].limit_scaled) {
      monpanic(vm, "jmp_jw: offset outside of CS limits\n");
      exception(vm, ExceptionGP, 0);
      }
    }
 
  G_EIP(vm) = new_EIP;
}

  void
JCC_Jw(vm_t *vm)
{
  Boolean condition = 0;
 
  switch (vm->i.b1 & 0x0f) {
    case 0x00: /* JO */ condition = G_GetOF(vm); break;
    case 0x01: /* JNO */ condition = !G_GetOF(vm); break;
    case 0x02: /* JB */ condition = G_GetCF(vm); break;
    case 0x03: /* JNB */ condition = !G_GetCF(vm); break;
    case 0x04: /* JZ */ condition = G_GetZF(vm); break;
    case 0x05: /* JNZ */ condition = !G_GetZF(vm); break;
    case 0x06: /* JBE */ condition = G_GetCF(vm) || G_GetZF(vm); break;
    case 0x07: /* JNBE */ condition = !G_GetCF(vm) && !G_GetZF(vm); break;
    case 0x08: /* JS */ condition = G_GetSF(vm); break;
    case 0x09: /* JNS */ condition = !G_GetSF(vm); break;
    case 0x0A: /* JP */ condition = G_GetPF(vm); break;
    case 0x0B: /* JNP */ condition = !G_GetPF(vm); break;
    case 0x0C: /* JL */ condition = G_GetSF(vm) != G_GetOF(vm); break;
    case 0x0D: /* JNL */ condition = G_GetSF(vm) == G_GetOF(vm); break;
    case 0x0E: /* JLE */
      condition = G_GetZF(vm) || (G_GetSF(vm) != G_GetOF(vm));
      break;
    case 0x0F: /* JNLE */
      condition = (G_GetSF(vm) == G_GetOF(vm)) && !G_GetZF(vm);
      break;
    }
 
  if (condition) {
    Bit32u new_EIP;
 
    new_EIP = G_EIP(vm) + (Bit32s) vm->i.Id;
    new_EIP &= 0x0000ffff;
    if (ProtectedMode(vm)) {
      cache_sreg(vm, SRegCS);
      if (new_EIP > vm->guest_cpu.desc_cache[SRegCS].limit_scaled) {
        monpanic(vm, "jcc_jw: offset outside of CS limits\n");
        exception(vm, ExceptionGP, 0);
        }
      }
    G_EIP(vm) = new_EIP;
    revalidate_prefetch_q();
    }
}

  void
JMP_Ew(vm_t *vm)
{
  Bit32u new_EIP;
  Bit16u op1_16;
 
  if (vm->i.mod == 0xc0) {
    op1_16 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1_16);
    }
 
  invalidate_prefetch_q();
  new_EIP = op1_16;
 
  if (ProtectedMode(vm)) {
    cache_sreg(vm, SRegCS);
    if (new_EIP > vm->guest_cpu.desc_cache[SRegCS].limit_scaled) {
      monpanic(vm, "jmp_ev: IP out of CS limits!\n");
      exception(vm, ExceptionGP, 0);
      }
    }
 
  G_EIP(vm) = new_EIP;
}

  void
RETnear16(vm_t *vm)
{
  Bit32u temp_ESP;
  Bit16u return_IP;
  unsigned d_b;
 
  invalidate_prefetch_q();

  cache_sreg(vm, SRegCS);
  cache_sreg(vm, SRegSS);
  d_b = vm->guest_cpu.desc_cache[SRegSS].desc.d_b;
  if (d_b)
    temp_ESP = G_ESP(vm);
  else
    temp_ESP = G_SP(vm);
 
  if (ProtectedMode(vm)) {
    if ( !can_pop(vm, 2) ) {
      monpanic(vm, "retnear: can't pop EIP\n");
      /* ??? #SS(0) -or #GP(0) */
      }
 
    access_linear(vm, vm->guest_cpu.desc_cache[SRegSS].base + temp_ESP,
      2, G_CPL(vm)==3, OP_READ, &return_IP);
 
    if ( return_IP > vm->guest_cpu.desc_cache[SRegCS].limit_scaled ) {
      monpanic(vm, "retnear: EIP > limit\n");
      /*exception(vm, ExceptionGP, 0); */
      }
    G_EIP(vm) = return_IP;
    if (d_b) /* 32bit stack */
      G_ESP(vm) += 2;
    else
      G_SP(vm)  += 2;
    }
  else {
    /* +++ should do the same as above? */
    pop16(vm, &return_IP);
    G_EIP(vm) = return_IP;
    }
}

  void
RETnear16_Iw(vm_t *vm)
{
  Bit16u imm16;
  Bit32u temp_ESP;
  Bit16u return_IP;
  unsigned d_b;

  invalidate_prefetch_q();

  cache_sreg(vm, SRegCS);
  cache_sreg(vm, SRegSS);
  d_b = vm->guest_cpu.desc_cache[SRegSS].desc.d_b;
  if (d_b)
    temp_ESP = G_ESP(vm);
  else
    temp_ESP = G_SP(vm);

  imm16 = vm->i.Iw;

  if (ProtectedMode(vm)) {
    if ( !can_pop(vm, 2) ) {
      monpanic(vm, "retnear_iw: can't pop IP\n");
      /* ??? #SS(0) -or #GP(0) */
      }

    access_linear(vm, vm->guest_cpu.desc_cache[SRegSS].base + temp_ESP,
      2, G_CPL(vm)==3, OP_READ, &return_IP);


    if ( return_IP > vm->guest_cpu.desc_cache[SRegCS].limit_scaled ) {
      monpanic(vm, "retnear_iw: IP > limit\n");
      }

    if ( !can_pop(vm, 2 + imm16) ) {
      monpanic(vm, "retnear_iw: can't release bytes from stack\n");
      /* #GP(0) -or #SS(0) ??? */
      }

    G_EIP(vm) = return_IP;
    if (d_b) /* 32bit stack */
      G_ESP(vm) += 2 + imm16;
    else
      G_SP(vm)  += 2 + imm16;
    }
  else {
    pop16(vm, &return_IP);
    G_EIP(vm) = return_IP;
    if (d_b) /* 32bit stack */
      G_ESP(vm) += imm16; /* ??? should it be 2*imm16 ? */
    else
      G_SP(vm)  += imm16;
    }
}

  void
JMP16_Ep(vm_t *vm)
{
  Bit16u cs_raw;
  Bit16u op1_16;
 
  invalidate_prefetch_q();
  if (vm->i.mod == 0xc0) {
    /* far indirect must specify a memory address */
    monpanic(vm, "JMP_Ep: op1 is a register\n");
    }
 
  read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1_16);
  read_virtual_word(vm, vm->i.seg, vm->i.rm_addr+2, &cs_raw);
 
  if ( ProtectedMode(vm) ) {
    jump_protected(vm, cs_raw, op1_16);
    goto done;
    }
 
  load_seg_reg(vm, SRegCS, cs_raw);
  G_EIP(vm) = op1_16;
 
done:
  /* instrumentation was here */
  return;
}

  void
IRET16(vm_t *vm)
{
  Bit16u ip, cs_raw, flags;
  Bit32u change_mask;
 
  invalidate_prefetch_q();

  if (V8086Mode(vm)) {
    /* IOPL check in stack_return_from_v86() */
    stack_return_from_v86(vm);
    goto done;
    }
 
  if (vm->guest_cpu.cr0.fields.pe) {
    iret_protected(vm);
    goto done;
    }
 
 
  pop16(vm, &ip);
  pop16(vm, &cs_raw);
  pop16(vm, &flags);
 
  load_seg_reg(vm, SRegCS, (Bit16u) cs_raw);
  G_EIP(vm) = ip;
 
  /* modify: ID,AC,RF,NT,IOPL,OF,DF,IF,TF,SF,ZF,AF,PF,CF */
  /* ignore: (VIP,VIF),VM */
  change_mask = 0x00257fd5; /* +++ */
  write_eflags(vm, flags, change_mask);
 
done:
  /* instrumentation call was here */
}
