/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2000 Kevin P. Lawton
 *
 *  ctrl_xfer_pro.c:  protected mode control transfer mechanisms
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "monitor.h"



  void
jump_protected(vm_t *vm, Bit16u cs_raw, Bit32u disp32)
{
  descriptor_cache_t cache;
  selector_t         selector;

  /* destination selector is not null else #GP(0) */
  if (IsNullSelector(cs_raw)) {
    monpanic(vm, "jump_pro: cs == 0\n");
    exception(vm, ExceptionGP, 0);
    }

  selector.raw = cs_raw;

  /* destination selector index is whithin its descriptor table
     limits else #GP(selector) */
  fetch_raw_descriptor(vm, selector, &cache.desc, ExceptionGP);

  /* examine AR byte of destination selector for legal values: */
  descriptor2cache(vm, &cache);

  if ( cache.desc.type & D_S ) { /* normal data segment */
    if ( !(cache.desc.type & D_EXECUTE) ) { /* executable */
      monprint(vm, "jump_pro: S=1: descriptor not executable\n");
      exception(vm, ExceptionGP, cs_raw & 0xfffc);
      }
    /* CASE: JUMP CONFORMING CODE SEGMENT: */
    if ( cache.desc.type & D_CONFORM ) {
      /* descripor DPL must be <= CPL else #GP(selector) */
      if (cache.desc.dpl > G_CPL(vm)) {
        monprint(vm, "jump_pro: dpl > CPL\n");
        exception(vm, ExceptionGP, cs_raw & 0xfffc);
        }

      /* segment must be PRESENT else #NP(selector) */
      if (cache.desc.p == 0) {
        monprint(vm, "jump_pro: p == 0\n");
        exception(vm, ExceptionNP, cs_raw & 0xfffc);
        }

      /* instruction pointer must be in code segment limit else #GP(0) */
      if (disp32 > cache.limit_scaled) {
        monpanic(vm, "jump_pro: IP > limit\n");
        exception(vm, ExceptionGP, 0);
        }

      /* Load CS:IP from destination pointer */
      /* Load CS-cache with new segment descriptor */
      /* CPL does not change for conforming code segment */
      load_cs_pro(vm, selector, &cache, G_CPL(vm));
      G_EIP(vm) = disp32;
      }

    /* CASE: jump nonconforming code segment: */
    else {
      /* RPL of destination selector must be <= CPL else #GP(selector) */
      if (selector.fields.rpl > G_CPL(vm)) {
        monpanic(vm, "jump_pro: rpl > CPL\n");
        exception(vm, ExceptionGP, cs_raw & 0xfffc);
        }

      /* descriptor DPL must = CPL else #GP(selector) */
      if (cache.desc.dpl != G_CPL(vm)) {
        monprint(vm, "jump_pro: dpl != CPL\n");
        exception(vm, ExceptionGP, cs_raw & 0xfffc);
        }

      /* segment must be PRESENT else #NP(selector) */
      if (cache.desc.p == 0) {
        monprint(vm, "jump_pro: p == 0\n");
        exception(vm, ExceptionNP, cs_raw & 0xfffc);
        }

      /* IP must be in code segment limit else #GP(0) */
      if (disp32 > cache.limit_scaled) {
        monpanic(vm, "jump_pro: IP > limit\n");
        exception(vm, ExceptionGP, 0);
        }

      /* load CS:IP from destination pointer */
      /* load CS-cache with new segment descriptor */
      /* set RPL field of CS register to CPL */
      load_cs_pro(vm, selector, &cache, G_CPL(vm));
      G_EIP(vm) = disp32;
      }
    return;
    }

  else {
    selector_t tss_selector, gate_cs_selector;
    descriptor_cache_t tss_cache, gate_cs_cache;
    Bit32u temp_eIP;
    Bit32u gate286_offset, gate386_offset;


    switch ( cache.desc.type ) {
      case  1: /* 286 available TSS */
      case  9: /* 386 available TSS */
        /* TSS DPL must be >= CPL, else #GP(TSS selector) */
        if (cache.desc.dpl < G_CPL(vm)) {
          monpanic(vm, "jump_pro: TSS.dpl < CPL\n");
          exception(vm, ExceptionGP, cs_raw & 0xfffc);
          }

        /* TSS DPL must be >= TSS selector RPL, else #GP(TSS selector) */
        if (cache.desc.dpl < selector.fields.rpl) {
          monpanic(vm, "jump_pro: TSS.dpl < selector.rpl\n");
          exception(vm, ExceptionGP, cs_raw & 0xfffc);
          }

        /* Descriptor AR byte must specify available TSS,
         *   else #GP(TSS selector)
         * this is taken care of by the 'default' case of switch statement
         *
         * Task State Seg must be present, else #NP(TSS selector)
         * checked in task_switch()
         */

        /* SWITCH_TASKS _without_ nesting to TSS */
        task_switch(vm, selector, &cache, TASK_FROM_JUMP);

        /* EIP must be in code seg limit, else #GP(0) */
        cache_sreg(vm, SRegCS);
        if (G_EIP(vm) > vm->guest_cpu.desc_cache[SRegCS].limit_scaled ) {
          monpanic(vm, "jump_pro: TSS.p == 0\n");
          exception(vm, ExceptionGP, 0);
          }
        return;

      case  3: /* Busy 286 TSS */
        monpanic(vm, "jump_pro: JUMP to busy 286 TSS.\n");
        return;
        break;

      case  4: /* 286 call gate */
        /* descriptor DPL must be >= CPL else #GP(gate selector) */
        if (cache.desc.dpl < G_CPL(vm)) {
          monprint(vm, "jump_pro: gate.dpl < CPL\n");
          exception(vm, ExceptionGP, cs_raw & 0xfffc);
          return;
          }

        /* descriptor DPL must be >= gate selector RPL
         *   else #GP(gate selector) */
        if (cache.desc.dpl < selector.fields.rpl) {
          monprint(vm, "jump_pro: gate.dpl < selector.rpl\n");
          exception(vm, ExceptionGP, cs_raw & 0xfffc);
          }

        /* gate must be present else #NP(gate selector) */
        if (cache.desc.p==0) {
          monpanic(vm, "jump_pro: gate.p == 0\n");
          exception(vm, ExceptionNP, cs_raw & 0xfffc);
          }

        /* examine selector to code segment given in call gate descriptor */
        /* selector must not be null, else #GP(0) */
        gate_cs_selector = ((gate_t *) &cache.desc)->selector;
        if ( IsNullSelector(gate_cs_selector.raw) ) {
          monpanic(vm, "jump_pro: CS selector null\n");
          exception(vm, ExceptionGP, 0x0000);
          }

        /* Selector must be within its descriptor table limits
         *   else #GP(CS selector) */
        fetch_raw_descriptor(vm, gate_cs_selector, &gate_cs_cache.desc,
          ExceptionGP);
        descriptor2cache(vm, &gate_cs_cache);
        /* descriptor AR byte must indicate code segment else #GP(CS selector) */
        if ( (gate_cs_cache.valid==0) ||
             !(gate_cs_cache.desc.type & D_S) ||
             !(gate_cs_cache.desc.type & D_EXECUTE) ) {
          monprint(vm, "jump_pro: AR byte: not code segment.\n");
          exception(vm, ExceptionGP, gate_cs_selector.raw & 0xfffc);
          }

        /* if non-conforming, code segment descriptor DPL must = CPL
         *   else #GP(CS selector) */
        if ( !(gate_cs_cache.desc.type & D_CONFORM) ) {
          if (gate_cs_cache.desc.dpl != G_CPL(vm)) {
            monprint(vm, "jump_pro: non-conform: code des DPL != CPL.\n");
            exception(vm, ExceptionGP, gate_cs_selector.raw & 0xfffc);
            }
          }
        /* if conforming, then code segment descriptor DPL must <= CPL
         *   else #GP(CS selector) */
        else {
          if (gate_cs_cache.desc.dpl > G_CPL(vm)) {
            monprint(vm, "jump_pro: conform: code seg des DPL > CPL.\n");
            exception(vm, ExceptionGP, gate_cs_selector.raw & 0xfffc);
            }
          }

        /* code segment must be present else #NP(CS selector) */
        if (gate_cs_cache.desc.p==0) {
          monprint(vm, "jump_pro: code seg not present.\n");
          exception(vm, ExceptionNP, gate_cs_selector.raw & 0xfffc);
          }

        /* IP must be in code segment limit else #GP(0) */
        gate286_offset = ((gate_t *) &cache.desc)->offset_low;
        if ( gate286_offset > gate_cs_cache.limit_scaled ) {
          monpanic(vm, "jump_pro: IP > limit\n");
          exception(vm, ExceptionGP, 0x0000);
          }

        /* load CS:IP from call gate */
        /* load CS cache with new code segment */
        /* set rpl of CS to CPL */
        load_cs_pro(vm, gate_cs_selector, &gate_cs_cache, G_CPL(vm));
        G_EIP(vm) = gate286_offset;
        return;
        break;

      case  5: /* task gate */

        /* gate descriptor DPL must be >= CPL else #GP(gate selector) */
        if (cache.desc.dpl < G_CPL(vm)) {
          monpanic(vm, "jump_pro: gate.dpl < CPL\n");
          exception(vm, ExceptionGP, cs_raw & 0xfffc);
          }

        /* gate descriptor DPL must be >= gate selector RPL */
        /*   else #GP(gate selector) */
        if (cache.desc.dpl < selector.fields.rpl) {
          monpanic(vm, "jump_pro: gate.dpl < selector.rpl\n");
          exception(vm, ExceptionGP, cs_raw & 0xfffc);
          }

        /* task gate must be present else #NP(gate selector) */
        if (cache.desc.p==0) {
          monpanic(vm, "jump_pro: task gate.p == 0\n");
          exception(vm, ExceptionNP, cs_raw & 0xfffc);
          }

        /* examine selector to TSS, given in Task Gate descriptor
         * must specify global in the local/global bit
         * else #GP(TSS selector)
         */

        tss_selector = ((gate_t *) &cache.desc)->selector;
        if (tss_selector.fields.ti) {
          monpanic(vm, "jump_pro: tss_selector.ti=1\n");
          exception(vm, ExceptionGP, tss_selector.raw & 0xfffc);
          }

        /* index must be within GDT limits else #GP(TSS selector) */
        fetch_raw_descriptor(vm, tss_selector, &tss_cache.desc, ExceptionGP);

        /* descriptor AR byte must specify available TSS */
        /*   else #GP(TSS selector) */
        descriptor2cache(vm, &tss_cache);
        if ( tss_cache.valid==0 ) {
          monprint(vm, "jump_pro: TSS selector points to bad TSS\n");
          exception(vm, ExceptionGP, tss_selector.raw & 0xfffc);
          }
        if ( (tss_cache.desc.type!=9) && (tss_cache.desc.type!=1) ) {
          monprint(vm, "jump_pro: TSS selector points to bad TSS\n");
          exception(vm, ExceptionGP, tss_selector.raw & 0xfffc);
          }


        /* task state segment must be present, else #NP(tss selector) */
        if (tss_cache.desc.p==0) {
          monpanic(vm, "jump_pro: task descriptor.p == 0\n");
          exception(vm, ExceptionNP, tss_selector.raw & 0xfffc);
          }

        /* SWITCH_TASKS _without_ nesting to TSS */
        task_switch(vm, tss_selector, &tss_cache, TASK_FROM_JUMP);

        /* eIP must be within code segment limit, else #GP(0) */
#warning "assuming task_switch makes CS cache current"
        if (vm->guest_cpu.desc_cache[SRegCS].desc.d_b)
          temp_eIP = G_EIP(vm);
        else
          temp_eIP = G_IP(vm);
        if (temp_eIP > vm->guest_cpu.desc_cache[SRegCS].limit_scaled) {
          monpanic(vm, "jump_pro: eIP > cs.limit\n");
          exception(vm, ExceptionGP, 0x0000);
          }
        break;

      case 11: /* Busy 386 TSS */
        monpanic(vm, "jump_pro: JUMP to busy 386 TSS unsupported.\n");
        return;
        break;

      case 12: /* 386 call gate */
        /* descriptor DPL must be >= CPL else #GP(gate selector) */
        if (cache.desc.dpl < G_CPL(vm)) {
          monpanic(vm, "jump_pro: gate.dpl < CPL\n");
          exception(vm, ExceptionGP, cs_raw & 0xfffc);
          }

        /* descriptor DPL must be >= gate selector RPL
         * else #GP(gate selector) */
        if (cache.desc.dpl < selector.fields.rpl) {
          monpanic(vm, "jump_pro: gate.dpl < selector.rpl\n");
          exception(vm, ExceptionGP, cs_raw & 0xfffc);
          }

        /* gate must be present else #NP(gate selector) */
        if (cache.desc.p==0) {
          monpanic(vm, "jump_pro: task gate.p == 0\n");
          exception(vm, ExceptionNP, cs_raw & 0xfffc);
          }

        /* examine selector to code segment given in call gate descriptor */
        /* selector must not be null, else #GP(0) */
        gate_cs_selector = ((gate_t *) &cache.desc)->selector;
        if ( IsNullSelector(gate_cs_selector.raw) ) {
          monpanic(vm, "jump_pro: CS selector null\n");
          exception(vm, ExceptionGP, 0x0000);
          }

        /* selector must be within its descriptor table limits
         *   else #GP(CS selector) */
        fetch_raw_descriptor(vm, gate_cs_selector, &gate_cs_cache.desc,
          ExceptionGP);
        descriptor2cache(vm, &gate_cs_cache);

        /* descriptor AR byte must indicate code segment
         *   else #GP(CS selector) */
        if ( (gate_cs_cache.valid==0) ||
             !(gate_cs_cache.desc.type & D_S) ||
             !(gate_cs_cache.desc.type & D_EXECUTE) ) {
          monpanic(vm, "jump_pro: AR byte: not code segment.\n");
          exception(vm, ExceptionGP, gate_cs_selector.raw & 0xfffc);
          }

        /* if non-conforming, code segment descriptor DPL must = CPL
         *   else #GP(CS selector) */
        if ( !(gate_cs_cache.desc.type & D_CONFORM) ) {
          if (gate_cs_cache.desc.dpl != G_CPL(vm)) {
            monpanic(vm, "jump_pro: non-conform: code des DPL != CPL.\n");
            exception(vm, ExceptionGP, gate_cs_selector.raw & 0xfffc);
            }
          }
        /* if conforming, then code segment descriptor DPL must <= CPL
         *   else #GP(CS selector) */
        else {
          if (gate_cs_cache.desc.dpl > G_CPL(vm)) {
            monpanic(vm, "jump_pro: conform: code seg des DPL > CPL.\n");
            exception(vm, ExceptionGP, gate_cs_selector.raw & 0xfffc);
            }
          }

        /* code segment must be present else #NP(CS selector) */
        if (gate_cs_cache.desc.p==0) {
          monpanic(vm, "jump_pro: code seg not present.\n");
          exception(vm, ExceptionNP, gate_cs_selector.raw & 0xfffc);
          }

        /* IP must be in code segment limit else #GP(0) */
        gate386_offset = ((gate_t *) &cache.desc)->offset_low |
          (((gate_t *) &cache.desc)->offset_high << 16);
        if ( gate386_offset > gate_cs_cache.limit_scaled ) {
          monpanic(vm, "jump_pro: IP > limit\n");
          exception(vm, ExceptionGP, 0x0000);
          }

        /* load CS:IP from call gate */
        /* load CS cache with new code segment */
        /* set rpl of CS to CPL */
        load_cs_pro(vm, gate_cs_selector, &gate_cs_cache, G_CPL(vm));
        G_EIP(vm) = gate386_offset;
        return;
        break;

      default:
        monprint(vm, "jump_pro: gate type %u unsupported\n",
          (unsigned) cache.desc.type);
        exception(vm, ExceptionGP, cs_raw & 0xfffc);
        break;
      }
    }
}


  void
iret_protected(vm_t *vm)
{
  selector_t cs_selector, ss_selector;
  descriptor_cache_t cs_cache, ss_cache;

  if (G_FLG_NT(vm)) { /* NT = 1: RETURN FROM NESTED TASK */
    /* what's the deal with NT & VM ? */
    Bit32u base32;
    selector_t   link_selector;
    descriptor_cache_t tss_cache;

    if (G_GetVM(vm))
      monpanic(vm, "iret_pro: vm set?\n");

    /* TASK_RETURN: */

    if (vm->guest_cpu.tr_cache.valid==0)
      monpanic(vm, "iret_pro: TR not valid\n");
    if (vm->guest_cpu.tr_cache.desc.type == 1)
      base32 = vm->guest_cpu.tr_cache.base;
    else if (vm->guest_cpu.tr_cache.desc.type == 9)
      base32 = vm->guest_cpu.tr_cache.base;
    else {
      monpanic(vm, "iret_pro: TR not valid\n");
      base32 = 0; /* keep compiler happy */
      }

    /* examine back link selector in TSS addressed by current TR: */
    access_linear(vm, base32 + 0, 2, 0, OP_READ, &link_selector.raw);

    /* must specify global, else #TS(new TSS selector) */
    if (link_selector.fields.ti) {
      monpanic(vm, "iret_pro: link selector.ti=1\n");
      exception(vm, ExceptionTS, link_selector.raw & 0xfffc);
      }

    /* index must be within GDT limits, else #TS(new TSS selector) */
    fetch_raw_descriptor(vm, link_selector, &tss_cache.desc, ExceptionTS);

    /* AR byte must specify TSS, else #TS(new TSS selector) */
    /* new TSS must be busy, else #TS(new TSS selector) */
    descriptor2cache(vm, &tss_cache);

    if ( (tss_cache.valid==0) ||
         ((tss_cache.desc.type!=11) && (tss_cache.desc.type!=3)) ) {
      monprint(vm, "iret_pro: TSS selector points to bad TSS\n");
      exception(vm, ExceptionTS, link_selector.raw & 0xfffc);
      }

    /* TSS must be present, else #NP(new TSS selector) */
    if (tss_cache.desc.p==0) {
      monprint(vm, "iret_pro: task descriptor.p == 0\n");
      exception(vm, ExceptionNP, link_selector.raw & 0xfffc);
      }

    /* switch tasks (without nesting) to TSS specified by back link selector */
    task_switch(vm, link_selector, &tss_cache,
                TASK_FROM_IRET);

    /* mark the task just abandoned as not busy */

    /* eIP must be within code seg limit, else #GP(0) */
    if (G_EIP(vm) > vm->guest_cpu.desc_cache[SRegCS].limit_scaled ) {
      monpanic(vm, "iret: eIP > cs.limit\n");
      exception(vm, ExceptionGP, 0);
      }
    return;
    }

  else { /* NT = 0: INTERRUPT RETURN ON STACK -or STACK_RETURN_TO_V86 */
    Bit16u top_nbytes_same, top_nbytes_outer;
    Bit32u cs_offset, ss_offset;
    Bit32u new_eip, new_esp, temp_ESP, new_eflags;
    Bit8u prev_cpl;
    Bit32u change_mask;

    /* 16bit opsize  |   32bit opsize
     * ==============================
     * SS     eSP+8  |   SS     eSP+16
     * SP     eSP+6  |   ESP    eSP+12
     * -------------------------------
     * FLAGS  eSP+4  |   EFLAGS eSP+8
     * CS     eSP+2  |   CS     eSP+4
     * IP     eSP+0  |   EIP    eSP+0
     */

    if (vm->i.os_32) {
      top_nbytes_same    = 12;
      top_nbytes_outer   = 20;
      cs_offset = 4;
      ss_offset = 16;
      }
    else {
      top_nbytes_same    = 6;
      top_nbytes_outer   = 10;
      cs_offset = 2;
      ss_offset = 8;
      }

    /* CS on stack must be within stack limits, else #SS(0) */
    /* can_pop() calls  cache_sreg(vm, SRegSS); */
    if ( !can_pop(vm, top_nbytes_same) ) {
      monpanic(vm, "iret: CS not within stack limits\n");
      exception(vm, ExceptionSS, 0);
      }

    if (vm->guest_cpu.desc_cache[SRegSS].desc.d_b)
      temp_ESP = G_ESP(vm);
    else
      temp_ESP = G_SP(vm);

    access_linear(vm, vm->guest_cpu.desc_cache[SRegSS].base + temp_ESP + cs_offset,
      2, G_CPL(vm)==3, OP_READ, &cs_selector);

    if (vm->i.os_32) {
      access_linear(vm, vm->guest_cpu.desc_cache[SRegSS].base + temp_ESP + 0,
        4, G_CPL(vm)==3, OP_READ, &new_eip);
      access_linear(vm, vm->guest_cpu.desc_cache[SRegSS].base + temp_ESP + 8,
        4, G_CPL(vm)==3, OP_READ, &new_eflags);

      /* if VM=1 in flags image on stack then STACK_RETURN_TO_V86 */
      if (new_eflags & 0x00020000) {
        if (G_CPL(vm) != 0)
          monpanic(vm, "iret: VM set on stack, CPL!=0\n");
        stack_return_to_v86(vm, new_eip, cs_selector, new_eflags);
        return;
        }
      }
    else {
      Bit16u new_ip, new_flags;
      access_linear(vm, vm->guest_cpu.desc_cache[SRegSS].base + temp_ESP + 0,
        2, G_CPL(vm)==3, OP_READ, &new_ip);
      access_linear(vm, vm->guest_cpu.desc_cache[SRegSS].base + temp_ESP + 4,
        2, G_CPL(vm)==3, OP_READ, &new_flags);
      new_eip = new_ip;
      new_eflags = new_flags;
      }

    /* return CS selector must be non-null, else #GP(0) */
    if ( IsNullSelector(cs_selector.raw) ) {
      monpanic(vm, "iret: return CS selector null\n");
      exception(vm, ExceptionGP, 0);
      }

    /* selector index must be within descriptor table limits, */
    /* else #GP(return selector) */
    fetch_raw_descriptor(vm, cs_selector, &cs_cache.desc, ExceptionGP);

    descriptor2cache(vm, &cs_cache);

    /* AR byte must indicate code segment else #GP(return selector) */
    if ( !cs_cache.valid ||
         !(cs_cache.desc.type & D_S)  ||
         !(cs_cache.desc.type & D_EXECUTE) ) {
      monpanic(vm, "iret: AR byte indicated non code segment\n");
      exception(vm, ExceptionGP, cs_selector.raw & 0xfffc);
      }

    /* return CS selector RPL must be >= CPL, else #GP(return selector) */
    if (cs_selector.fields.rpl < G_CPL(vm)) {
      monpanic(vm, "iret: return selector RPL < CPL\n");
      exception(vm, ExceptionGP, cs_selector.raw & 0xfffc);
      }

    /* if return code seg descriptor is conforming */
    /*   and return code seg DPL > return code seg selector RPL */
    /*     then #GP(return selector) */
    if ( (cs_cache.desc.type & D_CONFORM)  &&
         (cs_cache.desc.dpl > cs_selector.fields.rpl) ) {
      monpanic(vm, "iret: conforming, DPL > cs_selector.RPL\n");
      exception(vm, ExceptionGP, cs_selector.raw & 0xfffc);
      }

    /* if return code seg descriptor is non-conforming */
    /*   and return code seg DPL != return code seg selector RPL */
    /*     then #GP(return selector) */
    if ( !(cs_cache.desc.type & D_CONFORM) &&
         (cs_cache.desc.dpl != cs_selector.fields.rpl) ) {
      monpanic(vm, "(mch) iret: Return with DPL != RPL. #GP(selector)\n");
      exception(vm, ExceptionGP, cs_selector.raw & 0xfffc);
      }

    /* segment must be present else #NP(return selector) */
    if ( cs_cache.desc.p==0 ) {
      monpanic(vm, "iret: not present\n");
      exception(vm, ExceptionNP, cs_selector.raw & 0xfffc);
      }

    if (cs_selector.fields.rpl == G_CPL(vm)) { /* IRET TO SAME LEVEL */
      /* top 6/12 bytes on stack must be within limits, else #SS(0) */
      /* satisfied above */

      /* return EIP must be in code segment limit else #GP(0) */
      if ( new_eip > cs_cache.limit_scaled ) {
        monpanic(vm, "iret: EIP > descriptor limit\n");
        exception(vm, ExceptionGP, 0);
        }
      /* load CS:EIP from stack */
      /* load CS-cache with new code segment descriptor */
      load_cs_pro(vm, cs_selector, &cs_cache, G_CPL(vm));
      G_EIP(vm) = new_eip;

      /* always modify: ID,AC,RF,NT,OF,DF,TF,SF,ZF,AF,PF,CF */
      /* unmodified: (VIP,VIF),VM */
      /* modify IOPL if previous cpl==0 */
      /* modify IF   if previous cpl<=IOPL, IF */
      change_mask = 0x00254dd5;
      if (G_CPL(vm)==0)
        change_mask |= 0x00003000; /* IOPL */
      if (G_CPL(vm)<=G_FLG_IOPL(vm))
        change_mask |= 0x00000200; /* IF */

      if (!cs_cache.desc.d_b) {
        /* 16bit IRET */
        /* +++ perhaps should not modify upper word bits in eflags? */
        /* Currently, for 16 bit IRET, upper bits are cleared because */
        /* only 16bits are popped off stack, so upper bits masked */
        /* by change_mask will be cleared.  If they shouldn't be */
        /* cleared, then use the following code: */
        /*   change_mask &= 0x0000ffff; */
        }
/*monprint(vm, "iret_pro: eflags=0x%x\n", new_eflags); */
      write_eflags(vm, new_eflags, change_mask);

      /* increment stack by 6/12 */
      if (vm->guest_cpu.desc_cache[SRegSS].desc.d_b)
        G_ESP(vm) += top_nbytes_same;
      else
        G_SP(vm)  += top_nbytes_same;
      return;
      }
    else { /* INTERRUPT RETURN TO OUTER PRIVILEGE LEVEL */
      /* 16bit opsize  |   32bit opsize
       * ==============================
       * SS     eSP+8  |   SS     eSP+16
       * SP     eSP+6  |   ESP    eSP+12
       * FLAGS  eSP+4  |   EFLAGS eSP+8
       * CS     eSP+2  |   CS     eSP+4
       * IP     eSP+0  |   EIP    eSP+0
       */

      /* top 10/20 bytes on stack must be within limits else #SS(0) */
      if ( !can_pop(vm, top_nbytes_outer) ) {
        monpanic(vm, "iret: top 10/20 bytes not within stack limits\n");
        exception(vm, ExceptionSS, 0);
        }

      /* examine return SS selector and associated descriptor */
      access_linear(vm, vm->guest_cpu.desc_cache[SRegSS].base + temp_ESP + ss_offset,
        2, 0, OP_READ, &ss_selector);

      /* selector must be non-null, else #GP(0) */
      if ( IsNullSelector(ss_selector.raw) ) {
        monpanic(vm, "iret: SS selector null\n");
        exception(vm, ExceptionGP, 0);
        }

      /* selector RPL must = RPL of return CS selector,
       * else #GP(SS selector) */
      if ( ss_selector.fields.rpl != cs_selector.fields.rpl) {
        monpanic(vm, "iret: SS.rpl != CS.rpl\n");
        exception(vm, ExceptionGP, ss_selector.raw & 0xfffc);
        }

      /* selector index must be within its descriptor table limits,
       * else #GP(SS selector) */
      fetch_raw_descriptor(vm, ss_selector, &ss_cache.desc, ExceptionGP);

      descriptor2cache(vm, &ss_cache);

      /* AR byte must indicate a writable data segment,
       * else #GP(SS selector) */
      if ( !ss_cache.valid ||
           !(ss_cache.desc.type & D_S)  ||
           (ss_cache.desc.type & D_EXECUTE)  ||
           !(ss_cache.desc.type & D_WRITE) ) {
        monpanic(vm, "iret: SS AR byte not writable code segment\n");
        exception(vm, ExceptionGP, ss_selector.raw & 0xfffc);
        }

      /* stack segment DPL must equal the RPL of the return CS selector,
       * else #GP(SS selector) */
      if ( ss_cache.desc.dpl != cs_selector.fields.rpl ) {
        monpanic(vm, "iret: SS.dpl != CS selector RPL\n");
        exception(vm, ExceptionGP, ss_selector.raw & 0xfffc);
        }

      /* SS must be present, else #NP(SS selector) */
      if ( ss_cache.desc.p==0 ) {
        monpanic(vm, "iret: SS not present!\n");
        exception(vm, ExceptionNP, ss_selector.raw & 0xfffc);
        }


      if (vm->i.os_32) {
        access_linear(vm, vm->guest_cpu.desc_cache[SRegSS].base + temp_ESP + 0,
          4, 0, OP_READ, &new_eip);
        access_linear(vm, vm->guest_cpu.desc_cache[SRegSS].base + temp_ESP + 8,
          4, 0, OP_READ, &new_eflags);
        access_linear(vm, vm->guest_cpu.desc_cache[SRegSS].base + temp_ESP + 12,
          4, 0, OP_READ, &new_esp);
        }
      else {
        Bit16u new_ip, new_flags, new_sp;
        access_linear(vm, vm->guest_cpu.desc_cache[SRegSS].base + temp_ESP + 0,
          2, 0, OP_READ, &new_ip);
        access_linear(vm, vm->guest_cpu.desc_cache[SRegSS].base + temp_ESP + 4,
          2, 0, OP_READ, &new_flags);
        access_linear(vm, vm->guest_cpu.desc_cache[SRegSS].base + temp_ESP + 6,
          2, 0, OP_READ, &new_sp);
        new_eip = new_ip;
        new_esp = new_sp;
        new_eflags = new_flags;
        }

      /* EIP must be in code segment limit, else #GP(0) */
      if ( new_eip > cs_cache.limit_scaled ) {
        monpanic(vm, "iret: IP > descriptor limit\n");
        exception(vm, ExceptionGP, 0);
        }

      /* load CS:EIP from stack */
      /* load the CS-cache with CS descriptor */
      /* set CPL to the RPL of the return CS selector */
      prev_cpl = G_CPL(vm); /* previous CPL */
      load_cs_pro(vm, cs_selector, &cs_cache, cs_selector.fields.rpl);
      G_EIP(vm) = new_eip;

      /* load flags from stack */

      /* always modify: ID,AC,RF,NT,OF,DF,TF,SF,ZF,AF,PF,CF */
      /* unmodified: (VIP,VIF),VM */
      /* modify IOPL if previous cpl==0 */
      /* modify IF   if previous cpl<=IOPL, IF */
      change_mask = 0x00254dd5;
      if (prev_cpl==0)
        change_mask |= 0x00003000; /* IOPL */
      if (prev_cpl<=G_FLG_IOPL(vm))
        change_mask |= 0x00000200; /* IF */

      if (!cs_cache.desc.d_b) {
        /* 16bit IRET */
        /* +++ perhaps should not modify upper word bits in eflags? */
        /* Currently, for 16 bit IRET, upper bits are cleared because */
        /* only 16bits are popped off stack, so upper bits masked */
        /* by change_mask will be cleared.  If they shouldn't be */
        /* cleared, then use the following code: */
        /*   change_mask &= 0x0000ffff; */
        }
/*monprint(vm, "iret_pro: eflags=0x%x\n", new_eflags); */
      write_eflags(vm, new_eflags, change_mask);

      /* load SS:eSP from stack */
      /* load the SS-cache with SS descriptor */
      load_ss_pro(vm, ss_selector, &ss_cache, cs_selector.fields.rpl);
      if (ss_cache.desc.d_b)
        G_ESP(vm) = new_esp;
      else
        G_SP(vm)  = new_esp;

      validate_seg_regs(vm);

      return;
      }
    }
  monpanic(vm, "IRET: shouldn't get here!\n");
}


  void
call_protected(vm_t *vm, Bit16u cs_raw, Bit32u disp32)
{
  selector_t cs_selector;
  descriptor_cache_t cs_cache;
  
  /* Opsize in effect for CALL is specified by the D bit for the
   * segment containing dest & by any opsize prefix.
   * For gate descriptor, deterermined by type of call gate:
   * 4=16bit, 12=32bit
   * count field: 16bit specifies #words, 32bit specifies #dwords
   */
  cache_sreg(vm, SRegCS);
  cache_sreg(vm, SRegSS);

  /* new cs selector must not be null, else #GP(0) */
  if ( (cs_raw & 0xfffc) == 0 ) {
    monpanic(vm, "call_pro: CS selector null\n");
    exception(vm, ExceptionGP, 0);
    }

  cs_selector.raw = cs_raw;

  /* check new CS selector index within its descriptor limits,
   * else #GP(new CS selector) */
  fetch_raw_descriptor(vm, cs_selector, &cs_cache.desc, ExceptionGP);
  descriptor2cache(vm, &cs_cache);

  /* examine AR byte of selected descriptor for various legal values */
  if (cs_cache.valid==0) {
    monpanic(vm, "call_pro: invalid CS descriptor\n");
    exception(vm, ExceptionGP, cs_raw & 0xfffc);
    }

  if (cs_cache.desc.type & D_S) { /* normal segment */
    Bit32u temp_ESP;

    if ( !(cs_cache.desc.type & D_EXECUTE) ) {
      monpanic(vm, "call_pro: non executable segment\n");
      exception(vm, ExceptionGP, cs_raw & 0xfffc);
      }

    if ( cs_cache.desc.type & D_CONFORM ) { /* Conforming code seg */
      /* DPL must be <= CPL, else #GP(code seg selector) */
      if (cs_cache.desc.dpl > G_CPL(vm)) {
        monpanic(vm, "call_pro: cs.dpl > CPL\n");
        exception(vm, ExceptionGP, cs_raw & 0xfffc);
        }
      }
    else { /* Non-conforming code segment */
      /* RPL must be <= CPL, else #GP(code seg selector) */
      /* DPL must be = CPL, else #GP(code seg selector) */
      if ( (cs_selector.fields.rpl > G_CPL(vm)) ||
           (cs_cache.desc.dpl != G_CPL(vm)) ) {
        monpanic(vm, "call_pro: cs.rpl > CPL\n");
        exception(vm, ExceptionGP, cs_raw & 0xfffc);
        }
      }

    /* segment must be present, else #NP(code seg selector) */
    if (cs_cache.desc.p == 0) {
      monprint(vm, "call_pro: cs.p = 0\n");
      exception(vm, ExceptionNP, cs_raw & 0xfffc);
      }

    if (vm->guest_cpu.desc_cache[SRegSS].desc.d_b)
      temp_ESP = G_ESP(vm);
    else
      temp_ESP = G_SP(vm);

    /* stack must be big enough for return addr, else #SS(0) */
    if (vm->i.os_32) {
      if ( !can_push(vm, &vm->guest_cpu.desc_cache[SRegSS], temp_ESP, 8) ) {
        monpanic(vm, "call_pro: stack doesn't have room for ret addr\n");
        exception(vm, ExceptionSS, 0);
        }

      /* IP must be in code seg limit, else #GP(0) */
      if (disp32 > cs_cache.limit_scaled) {
        monpanic(vm, "call_pro: IP not in code seg limit\n");
        exception(vm, ExceptionGP, 0);
        }

      /* push return address onto stack (CS padded to 32bits) */
      push32(vm, (Bit32u) vm->guest_cpu.selector[SRegCS].raw);
      push32(vm, G_EIP(vm));
      }
    else { /* 16bit opsize */
      if ( !can_push(vm, &vm->guest_cpu.desc_cache[SRegSS], temp_ESP, 4) ) {
        monpanic(vm, "call_pro: stack doesn't have room for ret addr\n");
        exception(vm, ExceptionSS, 0);
        }

      /* IP must be in code seg limit, else #GP(0) */
      if (disp32 > cs_cache.limit_scaled) {
        monpanic(vm, "call_pro: IP not in code seg limit\n");
        exception(vm, ExceptionGP, 0);
        }

      push16(vm, vm->guest_cpu.selector[SRegCS].raw);
      push16(vm, G_IP(vm));
      }

    /* load code segment descriptor into CS cache
     * load CS with new code segment selector
     * set RPL of CS to CPL
     * load eIP with new offset
     */
    load_cs_pro(vm, cs_selector, &cs_cache, G_CPL(vm));
    G_EIP(vm) = disp32;
    if (cs_cache.desc.d_b==0)
      G_EIP(vm) &= 0x0000ffff;
    return;
    }
  else { /* gate & special segment */
    descriptor_cache_t  gate_cache, tss_cache;
    selector_t          gate_selector, tss_selector;
    Bit32u new_EIP;
    Bit16u dest_selector;
    Bit16u raw_tss_selector;
    Bit32u temp_eIP;

    /* 1 level of indirection via gate, switch gate & cs */
    gate_cache    = cs_cache; /* remove this copy +++ */
    gate_selector = cs_selector;

    switch (gate_cache.desc.type) {
      case 1: /* available 16bit TSS */
      case 9: /* available 32bit TSS */
        /* TSS DPL must be >= CPL, else #TS(TSS selector) */
        if (gate_cache.desc.dpl < G_CPL(vm)) {
          monpanic(vm, "call_pro: TSS.dpl < CPL\n");
          exception(vm, ExceptionTS, cs_raw & 0xfffc);
          }

        /* TSS DPL must be >= TSS selector RPL, else #TS(TSS selector) */
        if (gate_cache.desc.dpl < gate_selector.fields.rpl) {
          monpanic(vm, "call_pro: TSS.dpl < selector.rpl\n");
          exception(vm, ExceptionTS, cs_raw & 0xfffc);
          }

        /* descriptor AR byte must specify available TSS,
         *   else #TS(TSS selector)
         * this is taken care of by the 'default' case of switch statement

         * Task State Seg must be present, else #NP(TSS selector)
         * checked in task_switch()
         */

        /* SWITCH_TASKS _without_ nesting to TSS */
        task_switch(vm, gate_selector, &gate_cache, TASK_FROM_CALL_OR_INT);

        /* IP must be in code seg limit, else #TS(0) */
        if (G_EIP(vm) > vm->guest_cpu.desc_cache[SRegCS].limit_scaled ) {
          monprint(vm, "call_pro: EIP>CS.limit\n");
          exception(vm, ExceptionTS, 0);
          }
        return;

#if 0
      case 5: /* TASK GATE */
        /* gate descriptor DPL must be >= CPL else #TS(gate selector) */
        if (gate_descriptor.dpl < CPL) {
          monpanic(vm, "call_pro: gate.dpl < CPL\n");
          exception(vm, ExceptionTS, cs_raw & 0xfffc, 0);
          return;
          }

        /* gate descriptor DPL must be >= gate selector RPL
         *   else #TS(gate selector) */
        if (gate_descriptor.dpl < gate_selector.rpl) {
          monpanic(vm, "call_pro: gate.dpl < selector.rpl\n");
          exception(vm, ExceptionTS, cs_raw & 0xfffc, 0);
          return;
          }

        /* task gate must be present else #NP(gate selector) */
        if (gate_descriptor.p==0) {
          monpanic(vm, "call_pro: task gate.p == 0\n");
          exception(vm, ExceptionNP, cs_raw & 0xfffc, 0);
          return;
          }

        /* examine selector to TSS, given in Task Gate descriptor
         * must specify global in the local/global bit else #TS(TSS selector)
         */

        raw_tss_selector = gate_descriptor.u.taskgate.tss_selector;
        parse_selector(raw_tss_selector, &tss_selector);
        if (tss_selector.ti) {
          monpanic(vm, "call_pro: tss_selector.ti=1\n");
          exception(vm, ExceptionTS, raw_tss_selector & 0xfffc, 0);
          return;
          }

        /* index must be within GDT limits else #TS(TSS selector) */
        fetch_raw_descriptor(&tss_selector, &dword1, &dword2,
          BX_TS_EXCEPTION);

        /* descriptor AR byte must specify available TSS
         *   else #TS(TSS selector)
         */
        parse_descriptor(dword1, dword2, &tss_descriptor);
        if (tss_descriptor.valid==0 || tss_descriptor.segment) {
          monpanic(vm, "call_pro: TSS selector points to bad TSS\n");
          exception(vm, ExceptionTS, raw_tss_selector & 0xfffc, 0);
          }
        if (tss_descriptor.type!=9 && tss_descriptor.type!=1) {
          monpanic(vm, "call_pro: TSS selector points to bad TSS\n");
          exception(vm, ExceptionTS, raw_tss_selector & 0xfffc, 0);
          }


        /* task state segment must be present, else #NP(tss selector) */
        if (tss_descriptor.p==0) {
          monpanic(vm, "call_pro: task descriptor.p == 0\n");
          exception(vm, ExceptionNP, raw_tss_selector & 0xfffc, 0);
          }

        /* SWITCH_TASKS without nesting to TSS */
        task_switch(&tss_selector, &tss_descriptor,
                    BX_TASK_FROM_CALL_OR_INT, dword1, dword2);

        /* eIP must be within code segment limit, else #TS(0) */
        if (BX_CPU_THIS_PTR sregs[BX_SEG_REG_CS].cache.u.segment.d_b)
          temp_eIP = EIP;
        else
          temp_eIP =  IP;
        if (temp_eIP > BX_CPU_THIS_PTR sregs[BX_SEG_REG_CS].cache.u.segment.limit_scaled) {
          monpanic(vm, "call_pro: eIP > cs.limit\n");
          exception(vm, ExceptionTS, 0x0000, 0);
          }

        return;
        break;

      case  4: /* 16bit CALL GATE */
      case 12: /* 32bit CALL GATE */
        /* call gate DPL must be >= CPL, else #GP(call gate selector)
         * call gate DPL must be >= RPL, else #GP(call gate selector)
         */
        if ( (gate_descriptor.dpl < CPL) ||
             (gate_descriptor.dpl < gate_selector.rpl) ) {
          monpanic(vm, "call_pro: DPL < CPL or RPL\n");
          exception(vm, ExceptionGP, gate_selector.value & 0xfffc, 0);
          }

        /* call gate must be present, else #NP(call gate selector) */
        if (gate_descriptor.p==0) {
          monpanic(vm, "call_pro: not present\n");
          exception(vm, ExceptionNP, gate_selector.value & 0xfffc, 0);
          }

        /* examine code segment selector in call gate descriptor */

        if (gate_descriptor.type==4) {
          dest_selector = gate_descriptor.u.gate286.dest_selector;
          new_EIP = gate_descriptor.u.gate286.dest_offset;
          }
        else {
          dest_selector = gate_descriptor.u.gate386.dest_selector;
          new_EIP = gate_descriptor.u.gate386.dest_offset;
          }

        /* selector must not be null else #GP(0) */
        if ( (dest_selector & 0xfffc) == 0 ) {
          monpanic(vm, "call_pro: selector in gate null\n");
          exception(vm, ExceptionGP, 0, 0);
          }

        parse_selector(dest_selector, &cs_selector);

        /* selector must be within its descriptor table limits,
         *   else #GP(code segment selector) */
        fetch_raw_descriptor(&cs_selector, &dword1, &dword2,
          ExceptionGP);
        parse_descriptor(dword1, dword2, &cs_descriptor);

        /* AR byte of selected descriptor must indicate code segment,
         *   else #GP(code segment selector)
         * DPL of selected descriptor must be <= CPL,
         * else #GP(code segment selector)
         */
        if (cs_descriptor.valid==0 ||
            cs_descriptor.segment==0 ||
            cs_descriptor.u.segment.executable==0 ||
            cs_descriptor.dpl > CPL) {
          monpanic(vm, "call_pro: selected desciptor not code\n");
          exception(vm, ExceptionGP, cs_selector.value & 0xfffc, 0);
          }

        /* CALL GATE TO MORE PRIVILEGE
         * if non-conforming code segment and DPL < CPL then
         * ??? use gate_descriptor.dpl or cs_descriptor.dpl ???
         */
        if ( (cs_descriptor.u.segment.c_ed==0)  &&
             (cs_descriptor.dpl < CPL) ) {

          Bit16u SS_for_cpl_x;
          Bit32u ESP_for_cpl_x;
          bx_selector_t   ss_selector;
          bx_descriptor_t ss_descriptor;
          unsigned room_needed;
          Bit8u    param_count;
          Bit16u   return_SS, return_CS;
          Bit32u   return_ESP, return_EIP;
          Bit32u   return_ss_base;
          unsigned i;
          Bit16u   parameter_word[32];
          Bit32u   parameter_dword[32];
          Bit32u   temp_ESP;

          /* get new SS selector for new privilege level from TSS */
          get_SS_ESP_from_TSS(cs_descriptor.dpl,
                              &SS_for_cpl_x, &ESP_for_cpl_x);

/* ??? use dpl or rpl ??? */

          /* check selector & descriptor for new SS:
           * selector must not be null, else #TS(0)
           */
          if ( (SS_for_cpl_x & 0xfffc) == 0 ) {
            monpanic(vm, "call_pro: new SS null\n");
            exception(vm, ExceptionTS, 0, 0);
            return;
            }

          /* selector index must be within its descriptor table limits,
           *   else #TS(SS selector)
           */
          parse_selector(SS_for_cpl_x, &ss_selector);
          fetch_raw_descriptor(&ss_selector, &dword1, &dword2,
            BX_TS_EXCEPTION);

          parse_descriptor(dword1, dword2, &ss_descriptor);

          /* selector's RPL must equal DPL of code segment,
           *   else #TS(SS selector)
           */
          if (ss_selector.rpl != cs_descriptor.dpl) {
            monpanic(vm, "call_pro: SS selector.rpl != CS descr.dpl\n");
            exception(vm, ExceptionTS, SS_for_cpl_x & 0xfffc, 0);
            return;
            }

          /* stack segment DPL must equal DPL of code segment,
           *   else #TS(SS selector)
           */
          if (ss_descriptor.dpl != cs_descriptor.dpl) {
            monpanic(vm, "call_pro: SS descr.rpl != CS descr.dpl\n");
            exception(vm, ExceptionTS, SS_for_cpl_x & 0xfffc, 0);
            return;
            }

          /* descriptor must indicate writable data segment,
           *   else #TS(SS selector)
           */
          if (ss_descriptor.valid==0 ||
              ss_descriptor.segment==0  ||
              ss_descriptor.u.segment.executable ||
              ss_descriptor.u.segment.r_w==0) {
            bx_printf("call_pro: ss descriptor not writable data seg\n");
            exception(vm, ExceptionTS, SS_for_cpl_x & 0xfffc, 0);
            return;
            }

          /* segment must be present, else #SS(SS selector) */
          if (ss_descriptor.p==0) {
            monpanic(vm, "call_pro: ss descriptor not present.\n");
            exception(vm, ExceptionSS, SS_for_cpl_x & 0xfffc, 0);
            return;
            }

          if ( cs_descriptor.u.segment.d_b )
            /* new stack must have room for parameters plus 16 bytes */
            room_needed = 16;
          else
            /* new stack must have room for parameters plus 8 bytes */
            room_needed =  8;

          if (gate_descriptor.type==4) {
            /* get word count from call gate, mask to 5 bits */
            param_count = gate_descriptor.u.gate286.word_count & 0x1f;
            room_needed += param_count*2;
            }
          else {
            /* get word count from call gate, mask to 5 bits */
            param_count = gate_descriptor.u.gate386.dword_count & 0x1f;
            room_needed += param_count*4;
            }

          /* new stack must have room for parameters plus return info
           *   else #SS(SS selector)
           */

          if ( !can_push(&ss_descriptor, ESP_for_cpl_x, room_needed) ) {
            bx_printf("call_pro: stack doesn't have room\n");
            exception(vm, ExceptionSS, SS_for_cpl_x & 0xfffc, 0);
            return;
            }

          /* new eIP must be in code segment limit else #GP(0) */
          if ( new_EIP > cs_descriptor.u.segment.limit_scaled ) {
            monpanic(vm, "call_pro: IP not within CS limits\n");
            exception(vm, ExceptionGP, 0, 0);
            return;
            }


          /* save return SS:eSP to be pushed on new stack */
          return_SS = BX_CPU_THIS_PTR sregs[BX_SEG_REG_SS].selector.value;
          if (BX_CPU_THIS_PTR sregs[BX_SEG_REG_SS].cache.u.segment.d_b)
            return_ESP = ESP;
          else
            return_ESP =  SP;
          return_ss_base = BX_CPU_THIS_PTR sregs[BX_SEG_REG_SS].cache.u.segment.base;

          /* save return CS:eIP to be pushed on new stack */
          return_CS = BX_CPU_THIS_PTR sregs[BX_SEG_REG_CS].selector.value;
          if ( cs_descriptor.u.segment.d_b )
            return_EIP = EIP;
          else
            return_EIP =  IP;


          if (gate_descriptor.type==4) {
            for (i=0; i<param_count; i++) {
              access_linear(return_ss_base + return_ESP + i*2,
                2, 0, BX_READ, &parameter_word[i]);
              }
            }
          else {
            for (i=0; i<param_count; i++) {
              access_linear(return_ss_base + return_ESP + i*4,
                4, 0, BX_READ, &parameter_dword[i]);
              }
            }

          /* load new SS:SP value from TSS */
          /* load SS descriptor */
          load_ss_pro(&ss_selector, &ss_descriptor, ss_descriptor.dpl);
          if (ss_descriptor.u.segment.d_b)
            ESP = ESP_for_cpl_x;
          else
            SP =  (Bit16u) ESP_for_cpl_x;

          /* load new CS:IP value from gate */
          /* load CS descriptor */
          /* set CPL to stack segment DPL */
          /* set RPL of CS to CPL */
          load_cs_pro(&cs_selector, &cs_descriptor, cs_descriptor.dpl);
          EIP = new_EIP;

          /* push pointer of old stack onto new stack */
          if (gate_descriptor.type==4) {
            push_16(return_SS);
            push_16((Bit16u) return_ESP);
            }
          else {
            push_32(return_SS);
            push_32(return_ESP);
            }

          /* get word count from call gate, mask to 5 bits */
          /* copy parameters from old stack onto new stack */
          if (BX_CPU_THIS_PTR sregs[BX_SEG_REG_SS].cache.u.segment.d_b)
            temp_ESP = ESP;
          else
            temp_ESP =  SP;

          if (gate_descriptor.type==4) {
            for (i=param_count; i>0; i--) {
              push_16(parameter_word[i-1]);
              }
            }
          else {
            for (i=param_count; i>0; i--) {
              push_32(parameter_dword[i-1]);
              }
            }

          /* push return address onto new stack */
          if (gate_descriptor.type==4) {
            push_16(return_CS);
            push_16((Bit16u) return_EIP);
            }
          else {
            push_32(return_CS);
            push_32(return_EIP);
            }

          return;
          }

        /* CALL GATE TO SAME PRIVILEGE */
        else {
          Bit32u temp_ESP;

          if (BX_CPU_THIS_PTR sregs[BX_SEG_REG_SS].cache.u.segment.d_b)
            temp_ESP = ESP;
          else
            temp_ESP = SP;

          if (gate_descriptor.type==12) {
            if ( !can_push(&BX_CPU_THIS_PTR sregs[BX_SEG_REG_SS].cache, temp_ESP, 8) ) {
              monpanic(vm, "call_pro: stack doesn't have room for 8 bytes\n");
              exception(vm, ExceptionSS, 0, 0);
              }
            }
          else {
            /* stack must room for 4-byte return address
             *   else #SS(0) */
            if ( !can_push(&BX_CPU_THIS_PTR sregs[BX_SEG_REG_SS].cache, temp_ESP, 4) ) {
              monpanic(vm, "call_pro: stack doesn't have room for 4 bytes\n");
              exception(vm, ExceptionSS, 0, 0);
              }
            }

          /* EIP must be within code segment limit, else #GP(0) */
          if ( new_EIP > cs_descriptor.u.segment.limit_scaled ) {
            monpanic(vm, "call_pro: IP not within code segment limits\n");
            exception(vm, ExceptionGP, 0, 0);
            }

          if (gate_descriptor.type==12) {
            /* push return address onto stack */
            push_32(BX_CPU_THIS_PTR sregs[BX_SEG_REG_CS].selector.value);
            push_32(EIP);
            }
          else {
            /* push return address onto stack */
            push_16(BX_CPU_THIS_PTR sregs[BX_SEG_REG_CS].selector.value);
            push_16(IP);
            }

          /* load CS:EIP from gate
           * load code segment descriptor into CS register
           * set RPL of CS to CPL
           */
          load_cs_pro(&cs_selector, &cs_descriptor, CPL);
          EIP = new_EIP;

          return;
          }

        monpanic(vm, "call_pro: call gate: should not get here\n");
        return;

#endif
      default:
        monprint(vm, "call_pro: type = %u\n",
          (unsigned) cs_cache.desc.type);
        monpanic(vm, "call_pro: type = %u\n",
          (unsigned) cs_cache.desc.type);
        return;
      }
    monpanic(vm, "call_pro: gate segment unfinished\n");
    }

  monpanic(vm, "call_pro: shouldn't get here!\n");
}

  void
return_protected(vm_t *vm, Bit16u pop_bytes)
{
  selector_t cs_selector;
  descriptor_cache_t cs_cache;
  Bit32u stack_cs_offset, stack_param_offset;
  Bit32u return_EIP, temp_ESP;
  Bit16u return_IP;


  /* + 6+N*2: SS      | +12+N*4:     SS */
  /* + 4+N*2: SP      | + 8+N*4:    ESP */
  /*          parm N  | +        parm N */
  /*          parm 3  | +        parm 3 */
  /*          parm 2  | +        parm 2 */
  /*          parm 1  | + 8:     parm 1 */
  /* + 2:     CS      | + 4:         CS */
  /* + 0:     IP      | + 0:        EIP */

  if ( vm->i.os_32 ) {
    /* operand size=32: third word on stack must be within stack limits,
     *   else #SS(0); */
    if (!can_pop(vm, 6)) {
      monpanic(vm, "return_pro: 3rd word not in stack limits\n");
      /* #SS(0) */
      return;
      }
    stack_cs_offset = 4;
    stack_param_offset = 8;
    }
  else {
    /* operand size=16: second word on stack must be within stack limits,
     *   else #SS(0);
     */
    if ( !can_pop(vm, 4) ) {
      monpanic(vm, "return_pro: 2nd word not in stack limits\n");
      /* #SS(0) */
      return;
      }
    stack_cs_offset = 2;
    stack_param_offset = 4;
    }

  /* can_pop() calls cache_sreg(vm, SRegSS); */
  if (vm->guest_cpu.desc_cache[SRegSS].desc.d_b)
    temp_ESP = G_ESP(vm);
  else
    temp_ESP = G_SP(vm);

  /* return selector RPL must be >= CPL, else #GP(return selector) */
  access_linear(vm, vm->guest_cpu.desc_cache[SRegSS].base + temp_ESP +
                stack_cs_offset, 2, G_CPL(vm)==3, OP_READ, &cs_selector.raw);
  if ( cs_selector.fields.rpl < G_CPL(vm) ) {
    monprint(vm, "return_pro: CS.rpl < CPL\n");
    monprint(vm, "  CS.rpl=%u CPL=%u\n", (unsigned) cs_selector.fields.rpl,
      (unsigned) G_CPL(vm));
    exception(vm, ExceptionGP, cs_selector.raw & 0xfffc);
    }

  /* if return selector RPL == CPL then
   *   RETURN TO SAME LEVEL */
  if ( cs_selector.fields.rpl == G_CPL(vm) ) {
    /* return selector must be non-null, else #GP(0) */
    if ( IsNullSelector(cs_selector.raw) ) {
      monpanic(vm, "return_pro: CS null\n");
      /* #GP(0) */
      }

    /* selector index must be within its descriptor table limits,
     * else #GP(selector) */
    fetch_raw_descriptor(vm, cs_selector, &cs_cache.desc, ExceptionGP);
    descriptor2cache(vm, &cs_cache);

    /* descriptor AR byte must indicate code segment, else #GP(selector) */
    if ( !(cs_cache.valid) ||
         !(cs_cache.desc.type & D_S) ||
         !(cs_cache.desc.type & D_EXECUTE) ) {
      monprint(vm, "return_pro: same: AR byte not code\n");
      exception(vm, ExceptionGP, cs_selector.raw & 0xfffc);
      }

    /* if non-conforming then code segment DPL must = CPL,
     * else #GP(selector) */
    if ( !(cs_cache.desc.type & D_CONFORM) &&
         (cs_cache.desc.dpl!=G_CPL(vm)) ) {
      monpanic(vm, "return_pro: non-conforming, DPL!=CPL\n");
      /* #GP(selector) */
      }

    /* if conforming then code segment DPL must be <= CPL,
     * else #GP(selector) */
    if ( (cs_cache.desc.type & D_CONFORM) &&
         (cs_cache.desc.dpl>G_CPL(vm)) ) {
      monprint(vm, "return_pro: conforming, DPL>CPL\n");
      exception(vm, ExceptionGP, cs_selector.raw & 0xfffc);
      }

    /* code segment must be present, else #NP(selector) */
    if (cs_cache.desc.p==0) {
      monprint(vm, "return_pro: not present\n");
      exception(vm, ExceptionNP, cs_selector.raw & 0xfffc);
      }

    /* top word on stack must be within stack limits, else #SS(0) */
    if ( !can_pop(vm, stack_param_offset + pop_bytes) ) {
      monpanic(vm, "return_pro: top word not in stack limits\n");
      /* #SS(0) */
      }

    /* eIP must be in code segment limit, else #GP(0) */
    if (vm->i.os_32) {
      access_linear(vm, vm->guest_cpu.desc_cache[SRegSS].base + temp_ESP + 0,
        4, G_CPL(vm)==3, OP_READ, &return_EIP);
      }
    else {
      access_linear(vm, vm->guest_cpu.desc_cache[SRegSS].base + temp_ESP + 0,
        2, G_CPL(vm)==3, OP_READ, &return_IP);
      return_EIP = return_IP;
      }

    if ( return_EIP > cs_cache.limit_scaled ) {
      monpanic(vm, "return_pro: return IP > CS.limit\n");
      /* #GP(0) */
      }

    /* load CS:eIP from stack
     * load CS register with descriptor
     * increment eSP */
    load_cs_pro(vm, cs_selector, &cs_cache, G_CPL(vm));
    G_EIP(vm) = return_EIP;
    if (vm->guest_cpu.desc_cache[SRegSS].desc.d_b)
      G_ESP(vm) += stack_param_offset + pop_bytes;
    else
      G_SP(vm) += stack_param_offset + pop_bytes;

    return;
    }

  /* RETURN TO OUTER PRIVILEGE LEVEL */
  else {
    selector_t ss_selector;
    descriptor_cache_t ss_cache;
    Bit32u return_ESP;

    monpanic(vm, "return_pro: outer level\n");
#if 0
    /* + 6+N*2: SS      | +12+N*4:     SS */
    /* + 4+N*2: SP      | + 8+N*4:    ESP */
    /*          parm N  | +        parm N */
    /*          parm 3  | +        parm 3 */
    /*          parm 2  | +        parm 2 */
    /*          parm 1  | + 8:     parm 1 */
    /* + 2:     CS      | + 4:         CS */
    /* + 0:     IP      | + 0:        EIP */

    if (vm->i.os_32) {
      /* top 16+immediate bytes on stack must be within stack limits, else #SS(0) */
      if ( !can_pop(16 + pop_bytes) ) {
        monpanic(vm, "return_pro: 8 bytes not within stack limits\n");
        /* #SS(0) */
        return;
        }
      }
    else {
      /* top 8+immediate bytes on stack must be within stack limits, else #SS(0) */
      if ( !can_pop(8 + pop_bytes) ) {
        monpanic(vm, "return_pro: 8 bytes not within stack limits\n");
        /* #SS(0) */
        return;
        }
      }

    /* examine return CS selector and associated descriptor */

    /* selector must be non-null else #GP(0) */
    if ( (raw_cs_selector & 0xfffc) == 0 ) {
      monpanic(vm, "return_pro: CS selector null\n");
      /* #GP(0) */
      return;
      }

    /* selector index must be within its descriptor table limits,
     * else #GP(selector) */
    fetch_raw_descriptor(&cs_selector, &dword1, &dword2,
      ExceptionGP);
    parse_descriptor(dword1, dword2, &cs_descriptor);

    /* descriptor AR byte must indicate code segment else #GP(selector) */
    if (cs_descriptor.valid==0 ||
        cs_descriptor.segment==0  ||
        cs_descriptor.u.segment.executable==0) {
      monpanic(vm, "return_pro: AR byte not code\n");
      /* #GP(selector) */
      return;
      }

    /* if non-conforming code then code seg DPL must equal return selector RPL
     * else #GP(selector) */
    if (cs_descriptor.u.segment.c_ed==0 &&
        cs_descriptor.dpl!=cs_selector.rpl) {
      monpanic(vm, "return_pro: non-conforming seg DPL != selector.rpl\n");
      /* #GP(selector) */
      return;
      }

    /* if conforming then code segment DPL must be <= return selector RPL
     * else #GP(selector) */
    if (cs_descriptor.u.segment.c_ed &&
        cs_descriptor.dpl>cs_selector.rpl) {
      monpanic(vm, "return_pro: conforming seg DPL > selector.rpl\n");
      /* #GP(selector) */
      return;
      }

    /* segment must be present else #NP(selector) */
    if (cs_descriptor.p==0) {
      monpanic(vm, "return_pro: segment not present\n");
      /* #NP(selector) */
      return;
      }

    /* examine return SS selector and associated descriptor: */
    if (vm->i.os_32) {
      access_linear(BX_CPU_THIS_PTR sregs[BX_SEG_REG_SS].cache.u.segment.base + temp_ESP + 12 + pop_bytes,
        2, 0, BX_READ, &raw_ss_selector);
      access_linear(BX_CPU_THIS_PTR sregs[BX_SEG_REG_SS].cache.u.segment.base + temp_ESP + 8 + pop_bytes,
        4, 0, BX_READ, &return_ESP);
      access_linear(BX_CPU_THIS_PTR sregs[BX_SEG_REG_SS].cache.u.segment.base + temp_ESP + 0,
        4, 0, BX_READ, &return_EIP);
      }
    else {
      Bit16u return_SP;

      access_linear(BX_CPU_THIS_PTR sregs[BX_SEG_REG_SS].cache.u.segment.base + temp_ESP + 6 + pop_bytes,
        2, 0, BX_READ, &raw_ss_selector);
      access_linear(BX_CPU_THIS_PTR sregs[BX_SEG_REG_SS].cache.u.segment.base + temp_ESP + 4 + pop_bytes,
        2, 0, BX_READ, &return_SP);
      return_ESP = return_SP;
      access_linear(BX_CPU_THIS_PTR sregs[BX_SEG_REG_SS].cache.u.segment.base + temp_ESP + 0,
        2, 0, BX_READ, &return_IP);
      return_EIP = return_IP;
      }

    /* selector must be non-null else #GP(0) */
    if ( (raw_ss_selector & 0xfffc) == 0 ) {
      monpanic(vm, "return_pro: SS selector null\n");
      /* #GP(0) */
      return;
      }

    /* selector index must be within its descriptor table limits,
     * else #GP(selector) */
    parse_selector(raw_ss_selector, &ss_selector);
    fetch_raw_descriptor(&ss_selector, &dword1, &dword2,
      ExceptionGP);
    parse_descriptor(dword1, dword2, &ss_descriptor);

    /* selector RPL must = RPL of the return CS selector,
     * else #GP(selector) */
    if (ss_selector.rpl != cs_selector.rpl) {
      bx_printf("return_pro: ss.rpl != cs.rpl\n");
      exception(vm, ExceptionGP, raw_ss_selector & 0xfffc, 0);
      return;
      }

    /* descriptor AR byte must indicate a writable data segment,
     * else #GP(selector) */
    if (ss_descriptor.valid==0 ||
        ss_descriptor.segment==0 ||
        ss_descriptor.u.segment.executable ||
        ss_descriptor.u.segment.r_w==0) {
      monpanic(vm, "return_pro: SS.AR byte not writable data\n");
      /* #GP(selector) */
      return;
      }

    /* descriptor dpl must = RPL of the return CS selector,
     * else #GP(selector) */
    if (ss_descriptor.dpl != cs_selector.rpl) {
      monpanic(vm, "return_pro: SS.dpl != cs.rpl\n");
      /* #GP(selector) */
      return;
      }

    /* segment must be present else #SS(selector) */
    if (ss_descriptor.p==0) {
      monpanic(vm, "ss.p == 0\n");
      /* #NP(selector) */
      return;
      }

    /* eIP must be in code segment limit, else #GP(0) */
    if (return_EIP > cs_descriptor.u.segment.limit_scaled) {
      monpanic(vm, "return_pro: eIP > cs.limit\n");
      /* #GP(0) */
      return;
      }

    /* set CPL to RPL of return CS selector */
    /* load CS:IP from stack */
    /* set CS RPL to CPL */
    /* load the CS-cache with return CS descriptor */
    load_cs(&cs_selector, &cs_descriptor, cs_selector.rpl);
    BX_CPU_THIS_PTR eip = return_EIP;

    /* load SS:SP from stack */
    /* load SS-cache with return SS descriptor */
    load_ss(&ss_selector, &ss_descriptor, cs_selector.rpl);
    if (ss_descriptor.u.segment.d_b)
      ESP = return_ESP;
    else
      SP  = (Bit16u) return_ESP;

    /* check ES, DS, FS, GS for validity */
    validate_seg_regs();

    return;
#endif
    }

  return;
}


  void
validate_seg_regs(vm_t *vm)
{
  selector_t null_sel;

  null_sel.raw = 0;

  cache_sreg(vm, SRegES);
  cache_sreg(vm, SRegDS);
  cache_sreg(vm, SRegFS);
  cache_sreg(vm, SRegGS);

  if ( vm->guest_cpu.desc_cache[SRegES].desc.dpl<G_CPL(vm) ) {
    invalidate_sreg_pro(vm, SRegES, null_sel);
    }
  if ( vm->guest_cpu.desc_cache[SRegDS].desc.dpl<G_CPL(vm) ) {
    invalidate_sreg_pro(vm, SRegDS, null_sel);
    }
  if ( vm->guest_cpu.desc_cache[SRegFS].desc.dpl<G_CPL(vm) ) {
    invalidate_sreg_pro(vm, SRegFS, null_sel);
    }
  if ( vm->guest_cpu.desc_cache[SRegGS].desc.dpl<G_CPL(vm) ) {
    invalidate_sreg_pro(vm, SRegGS, null_sel);
    }
}
