/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2000 Kevin P. Lawton
 *
 *  data_xfer16.c:  emulation of 16-bit data transfer instructions
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "monitor.h"



  void
CMOV_GwEw(vm_t *vm)
{
  if (cpuid_info.featureFlags.fields.cmov) {
    /* Note: CMOV accesses a memory source operand (read), regardless
     *       of whether condition is true or not.  Thus, exceptions may
     *       occur even if the MOV does not take place.
     */
   
    Boolean condition;
    Bit16u op2;
   
    switch (vm->i.b1) {
      /* CMOV opcodes: */
      case 0x140: condition = G_GetOF(vm); break;
      case 0x141: condition = !G_GetOF(vm); break;
      case 0x142: condition = G_GetCF(vm); break;
      case 0x143: condition = !G_GetCF(vm); break;
      case 0x144: condition = G_GetZF(vm); break;
      case 0x145: condition = !G_GetZF(vm); break;
      case 0x146: condition = G_GetCF(vm) || G_GetZF(vm); break;
      case 0x147: condition = !G_GetCF(vm) && !G_GetZF(vm); break;
      case 0x148: condition = G_GetSF(vm); break;
      case 0x149: condition = !G_GetSF(vm); break;
      case 0x14A: condition = G_GetPF(vm); break;
      case 0x14B: condition = !G_GetPF(vm); break;
      case 0x14C: condition = G_GetSF(vm) != G_GetOF(vm); break;
      case 0x14D: condition = G_GetSF(vm) == G_GetOF(vm); break;
      case 0x14E: condition = G_GetZF(vm) ||
                  (G_GetSF(vm) != G_GetOF(vm)); break;
      case 0x14F: condition = !G_GetZF(vm) &&
                  (G_GetSF(vm) == G_GetOF(vm)); break;
      default:
        condition = 0;
        monpanic(vm, "CMOV_GwEw: default case\n");
      }
   
    if (vm->i.mod == 0xc0) {
      op2 = ReadReg16(vm, vm->i.rm);
      }
    else {
      /* pointer, segment address pair */
      read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op2);
      }
   
    if (condition) {
      WriteReg16(vm, vm->i.nnn, op2);
      }
    }
  else {
    monpanic(vm, "CMOV_GwEw:\n");
    UndefinedOpcode(vm);
    }
}


  void
MOVZX_GwEw(vm_t *vm)
{
  Bit16u op2_16;
 
  if (vm->i.mod == 0xc0) {
    op2_16 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op2_16);
    }
 
  /* normal move */
  WriteReg16(vm, vm->i.nnn, op2_16);
}

  void
MOVSX_GwEw(vm_t *vm)
{
  Bit16u op2;
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
 
  /* normal move */
  WriteReg16(vm, vm->i.nnn, op2);
}

  void
XCHG_EwGw(vm_t *vm)
{
  Bit16u op2, op1;
 
  op2 = ReadReg16(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg16(vm, vm->i.rm);
    WriteReg16(vm, vm->i.rm, op2);
    }
  else {
    read_RMW_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1);
    write_RMW_virtual_word(vm, &op2);
    }
 
  WriteReg16(vm, vm->i.nnn, op1);
}

  void
XCHG_RXAX(vm_t *vm)
{
  Bit16u temp16;
  unsigned reg;
 
  reg = vm->i.b1 & 0x07;
 
  temp16 = G_AX(vm);
  G_AX(vm) = ReadReg16(vm, reg);
  WriteReg16(vm, reg, temp16);
}


  void
MOV_GwEw(vm_t *vm)
{
  Bit16u op2;

  if (vm->i.mod == 0xc0) {
    op2 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }

  WriteReg16(vm, vm->i.nnn, op2);
}

  void
MOV_EwSw(vm_t *vm)
{
  Bit16u val16;

  cache_selector(vm, vm->i.nnn);
  val16 = vm->guest_cpu.selector[vm->i.nnn].raw;

  if (vm->i.mod == 0xc0) {
    if ( vm->i.os_32 ) {
      WriteReg32(vm, vm->i.rm, val16);
      }
    else {
      WriteReg16(vm, vm->i.rm, val16);
      }
    }
  else {
    write_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &val16);
    }
}

  void
MOVZX_GwEb(vm_t *vm)
{
  Bit8u  op2_8;
 
  if (vm->i.mod == 0xc0) {
    op2_8 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op2_8);
    }
 
  /* zero extend byte op2 into word op1 */
  WriteReg16(vm, vm->i.nnn, (Bit16u) op2_8);
}

  void
MOV_EwGw(vm_t *vm)
{
  Bit16u op2;

  op2 = ReadReg16(vm, vm->i.nnn);

  if (vm->i.mod == 0xc0) {
    WriteReg16(vm, vm->i.rm, op2);
    }
  else {
    write_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
}

  void
LEA_GwM(vm_t *vm)
{
  if (vm->i.mod == 0xc0) {
    monpanic(vm, "LEA_GvM: op2 is a register");
    UndefinedOpcode(vm);
    }

  WriteReg16(vm, vm->i.nnn, (Bit16u) vm->i.rm_addr);
}

  void
MOV_EwIw(vm_t *vm)
{
  Bit16u op2;

  op2 = vm->i.Iw;

  if (vm->i.mod == 0xc0) {
    WriteReg16(vm, vm->i.rm, op2);
    }
  else {
    write_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
}

  void
MOV_OwAX(vm_t *vm)
{
  Bit16u temp;
  Bit32u addr_32;

  addr_32 = vm->i.Id;

  temp = G_AX(vm);

  /* write to memory address */
  if (!NULL_SEG_REG(vm->i.seg)) {
    write_virtual_word(vm, vm->i.seg, addr_32, &temp);
    }
  else {
    write_virtual_word(vm, SRegDS, addr_32, &temp);
    }
}

  void
MOV_AXOw(vm_t *vm)
{
  Bit16u temp;
  Bit32u addr_32;
 
  addr_32 = vm->i.Id;
 
  if (!NULL_SEG_REG(vm->i.seg)) {
    read_virtual_word(vm, vm->i.seg, addr_32, &temp);
    }
  else {
    read_virtual_word(vm, SRegDS, addr_32, &temp);
    }
 
  G_AX(vm) = temp;
}

  void
MOVSX_GwEb(vm_t *vm)
{
  Bit8u op2_8;
 
  if (vm->i.mod == 0xc0) {
    op2_8 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op2_8);
    }
 
  /* sign extend byte op2 into word op1 */
  WriteReg16(vm, vm->i.nnn, (Bit8s) op2_8);
}
