/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2000 Kevin P. Lawton
 *
 *  data_xfer32.c:  emulation of 32-bit data transfer instructions
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "monitor.h"



  void
CMOV_GdEd(vm_t *vm)
{
  if (cpuid_info.featureFlags.fields.cmov) {
    /* Note: CMOV accesses a memory source operand (read), regardless
     *       of whether condition is true or not.  Thus, exceptions may
     *       occur even if the MOV does not take place.
     */
   
    Boolean condition;
    Bit32u op2;
   
    switch (vm->i.b1) {
      /* CMOV opcodes: */
      case 0x140: condition = G_GetOF(vm); break;
      case 0x141: condition = !G_GetOF(vm); break;
      case 0x142: condition = G_GetCF(vm); break;
      case 0x143: condition = !G_GetCF(vm); break;
      case 0x144: condition = G_GetZF(vm); break;
      case 0x145: condition = !G_GetZF(vm); break;
      case 0x146: condition = G_GetCF(vm) || G_GetZF(vm); break;
      case 0x147: condition = !G_GetCF(vm) && !G_GetZF(vm); break;
      case 0x148: condition = G_GetSF(vm); break;
      case 0x149: condition = !G_GetSF(vm); break;
      case 0x14A: condition = G_GetPF(vm); break;
      case 0x14B: condition = !G_GetPF(vm); break;
      case 0x14C: condition = G_GetSF(vm) != G_GetOF(vm); break;
      case 0x14D: condition = G_GetSF(vm) == G_GetOF(vm); break;
      case 0x14E: condition = G_GetZF(vm) ||
                  (G_GetSF(vm) != G_GetOF(vm)); break;
      case 0x14F: condition = !G_GetZF(vm) &&
                  (G_GetSF(vm) == G_GetOF(vm)); break;
      default:
        condition = 0;
        monpanic(vm, "CMOV_GwEw: default case\n");
      }
   
    if (vm->i.mod == 0xc0) {
      op2 = ReadReg32(vm, vm->i.rm);
      }
    else {
      /* pointer, segment address pair */
      read_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op2);
      }
   
    if (condition) {
      WriteReg32(vm, vm->i.nnn, op2);
      }
    }
  else {
    monpanic(vm, "CMOV_GdEd:\n");
    UndefinedOpcode(vm);
    }
}

  void
MOVZX_GdEw(vm_t *vm)
{
  Bit16u op2_16;
 
  if (vm->i.mod == 0xc0) {
    op2_16 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op2_16);
    }
 
  /* zero extend word op2 into dword op1 */
  WriteReg32(vm, vm->i.nnn, (Bit32u) op2_16);
}

  void
MOVSX_GdEw(vm_t *vm)
{
  Bit16u op2_16;
 
  if (vm->i.mod == 0xc0) {
    op2_16 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op2_16);
    }
 
  /* sign extend word op2 into dword op1 */
  WriteReg32(vm, vm->i.nnn, (Bit16s) op2_16);
}

  void
XCHG_EdGd(vm_t *vm)
{
  Bit32u op2, op1;
 
  op2 = ReadReg32(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    WriteReg32(vm, vm->i.rm, op2);
    }
  else {
    read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    write_RMW_virtual_dword(vm, &op2);
    }
 
  WriteReg32(vm, vm->i.nnn, op1);
}

  void
XCHG_ERXEAX(vm_t *vm)
{
  Bit32u temp32;
  unsigned reg;
 
  reg = vm->i.b1 & 0x07;

  temp32 = G_EAX(vm);
  G_EAX(vm) = ReadReg32(vm, reg);
  WriteReg32(vm, reg, temp32);
}

  void
MOV_GdEd(vm_t *vm)
{
  Bit32u op2_32;

  if (vm->i.mod == 0xc0) {
    op2_32 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op2_32);
    }

  WriteReg32(vm, vm->i.nnn, op2_32);
}

  void
MOVZX_GdEb(vm_t *vm)
{
  Bit8u  op2_8;

  if (vm->i.mod == 0xc0) {
    op2_8 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op2_8);
    }

  /* zero extend byte op2 into dword op1 */
  WriteReg32(vm, vm->i.nnn, (Bit32u) op2_8);
}

  void
MOV_EdGd(vm_t *vm)
{
  Bit32u op2_32;

  op2_32 = ReadReg32(vm, vm->i.nnn);

  if (vm->i.mod == 0xc0) {
    WriteReg32(vm, vm->i.rm, op2_32);
    }
  else {
    write_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op2_32);
    }
}

  void
LEA_GdM(vm_t *vm)
{
  if (vm->i.mod == 0xc0) {
    monpanic(vm, "LEA_GvM: op2 is a register");
    UndefinedOpcode(vm);
    }

  WriteReg32(vm, vm->i.nnn, vm->i.rm_addr);
}

  void
MOV_EdId(vm_t *vm)
{
  Bit32u op2_32;

  op2_32 = vm->i.Id;

  if (vm->i.mod == 0xc0) {
    WriteReg32(vm, vm->i.rm, op2_32);
    }
  else {
    write_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op2_32);
    }
}

  void
MOV_OdEAX(vm_t *vm)
{
  Bit32u temp_32;
  Bit32u addr_32;

  addr_32 = vm->i.Id;

  temp_32 = G_EAX(vm);

  if (!NULL_SEG_REG(vm->i.seg)) {
    write_virtual_dword(vm, vm->i.seg, addr_32, &temp_32);
    }
  else {
    write_virtual_dword(vm, SRegDS, addr_32, &temp_32);
    }
}

  void
MOV_EAXOd(vm_t *vm)
{
  Bit32u temp_32;
  Bit32u addr_32;
 
  addr_32 = vm->i.Id;
 
  if (!NULL_SEG_REG(vm->i.seg)) {
    read_virtual_dword(vm, vm->i.seg, addr_32, &temp_32);
    }
  else {
    read_virtual_dword(vm, SRegDS, addr_32, &temp_32);
    }
 
  G_EAX(vm) = temp_32;
}

  void
MOVSX_GdEb(vm_t *vm)
{
  Bit8u op2_8;
 
  if (vm->i.mod == 0xc0) {
    op2_8 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op2_8);
    }
 
  /* sign extend byte op2 into dword op1 */
  WriteReg32(vm, vm->i.nnn, (Bit8s) op2_8);
}
