/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2000 Kevin P. Lawton
 *
 *  emulation.c:  This file contains functions to emulate IA32 instructions
 *                
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "monitor.h"




  void
emulate_instr(vm_t *vm, guest_context_t *context, unsigned reason)
{
  unsigned ret;
  Bit32u port;
  Boolean seg32;
  unsigned max_len, room_in_page;
  Bit8u *ptr;
  unsigned group;
  unsigned len;
  Bit32u data, laddr0, guest_ppi, perror;
  unsigned us, rw;


  if (SetJmp(vm->jmp_buf_env)) {
    /* nonzero return, means returning from longjmp() */
    return;
    }

  /* Store EIP,ESP in case we need to rewind them due to an */
  /* exception caused by the instruction. */
  vm->guest_cpu.prev_eip = vm->guest.addr.guest_context->eip;
  vm->guest_cpu.prev_esp = vm->guest.addr.guest_context->esp;

  vm->guest_cpu.EXT = 0;
  vm->guest_cpu.errorno = 0;

  async_checks(vm);

  if (G_GetTF(vm)) {
    /* TF is set before execution of this instruction.  Schedule */
    /* a debug trap (#DB) after execution.  After completion of */
    /* the instruction, handle_async_event will invoke the trap. */
    vm->guest_cpu.debug_trap |= 0x00004000; /* BS flag in DR6 */
    vm->guest_cpu.async_event = 1;
    }

  cache_sreg(vm, SRegCS);
  seg32 = vm->guest_cpu.desc_cache[SRegCS].desc.d_b;
  laddr0 = vm->guest_cpu.desc_cache[SRegCS].base + context->eip;
  max_len = MaxIA32OpcodeLen;
  room_in_page = 4096 - (laddr0 & 0xfff);
  if ( room_in_page < MaxIA32OpcodeLen)
    max_len = room_in_page;

/* +++ this code ignores guest segment check, possibly */
/* +++ paging checks */

  /* See if we can read(fetch) using the current monitor mapping. */
  /* This would be the case if we are virtualizing an instruction, */
  /* where the page is mapped in to the monitor and accessible from */
  /* the currently executing guest code. */
  us = G_GetCPL(vm)==3;
  rw = 0;
  switch ( (ret = map_guest_laddr(vm, laddr0, &guest_ppi,
            us, rw, PageUsageVCode, &perror)) ) {
    case MapLinOK:
    case MapLinAlreadyMapped:
      break;
    case MapLinException:
      vm->guest_cpu.cr2 = laddr0;
      exception(vm, ExceptionPF, perror);
      break;
    case MapLinMonConflict:
    case MapLinPPageOOB:
    default:
      monpanic(vm, "emulate:1 default case of %u\n", ret);
    }
  /* Page is OK for reading (fetching) using natural CPU reads */
  ptr = (Bit8u*) Guest2Monitor(vm, laddr0);
  ret = fetchDecode(ptr, &vm->i, max_len, seg32);
  if ( ret==0 ) {
    Bit32u laddr1;
    unsigned map_ret;
    /* Instruction runs over into next page.  See if it is OK to */
    /* read natively in the next page.  If so, then no problem. */
    laddr1 = (laddr0 | 0x00000fff) + 1;
    map_ret = map_guest_laddr(vm, laddr1, &guest_ppi,
                              us, rw, PageUsageVCode, &perror);
    switch (map_ret) {
      case MapLinOK:
      case MapLinAlreadyMapped:
        break;
      case MapLinException:
        vm->guest_cpu.cr2 = laddr1;
        exception(vm, ExceptionPF, perror);
        break;
      case MapLinMonConflict:
      case MapLinPPageOOB:
      default:
        monpanic(vm, "emulate:2 default case of %u\n", map_ret);
      }
    ret = fetchDecode(ptr, &vm->i, MaxIA32OpcodeLen, seg32);
    if ( ret==0 )
      monpanic(vm, "fetchDecode returned 0 (multipage)\n");
    }
  if ( vm->i.attr & InvalidOpcode ) {
    monprint(vm, "fetchDecode InvalidOpcode b1=0x%x nnn=%u\n",
      vm->i.b1, vm->i.nnn);
    UndefinedOpcode(vm);
    }

  /* If instruction requires resolution of mod-rm address, */
  /* then call resolve function with current CPU context. */
  if (vm->i.ResolveModrm)
    vm->i.ResolveModrm(&vm->i, context);

  if (vm->i.rep_used && (vm->i.attr & Repeatable)) {
    if (vm->i.attr & RepeatableZF) {
      if (vm->i.as_32) {
        if (G_ECX(vm) != 0) {
          goto execute;
          }
        goto advance;
        }
      else {
        if (G_CX(vm) != 0) {
          goto execute;
          }
        goto advance;
        }
      }
    else { /* normal repeat, no concern for ZF */
      if (vm->i.as_32) {
        if (G_ECX(vm) != 0) {
          goto execute;
          }
        goto advance;
        }
      else { /* 16bit addrsize */
        if (G_CX(vm) != 0) {
          goto execute;
          }
        goto advance;
        }
      }
    }
  else {
    /* Advance EIP by length of instruction.  An exception will */
    /* rewind it back to prev_eip. */
    context->eip += vm->i.ilen;
    }

execute:

  /* Execute instruction. */
  if (vm->i.attr & GroupN)
    goto group_n;

/* */
/* Standard 1 or 2-byte opcodes: */
/* */

  switch (vm->i.b1) {
    case 0x00:
#if 0
      if (vm->i.modrm == 0) {
        monpanic(vm, "Potentially uunning in the weeds\n");
        }
#endif
      ADD_EbGb(vm);
      goto advance;

    case 0x01:
      if (vm->i.os_32)
        ADD_EdGd(vm);
      else
        ADD_EwGw(vm);
      goto advance;

    case 0x02:
      ADD_GbEb(vm);
      goto advance;

    case 0x03:
      if (vm->i.os_32)
        ADD_GdEd(vm);
      else
        ADD_GwEw(vm);
      goto advance;

    case 0x04:
      ADD_ALIb(vm);
      goto advance;

    case 0x05:
      if (vm->i.os_32)
        ADD_EAXId(vm);
      else
        ADD_AXIw(vm);
      goto advance;

    case 0x06:
      PUSH_SREG(vm, SRegES);
      goto advance;

    case 0x07:
      POP_SREG(vm, SRegES);
      goto advance;

    case 0x08:
      OR_EbGb(vm);
      goto advance;

    case 0x09:
      if (vm->i.os_32)
        OR_EdGd(vm);
      else
        OR_EwGw(vm);
      goto advance;

    case 0x0a:
      OR_GbEb(vm);
      goto advance;

    case 0x0b:
      if (vm->i.os_32)
        OR_GdEd(vm);
      else
        OR_GwEw(vm);
      goto advance;

    case 0x0c:
      OR_ALIb(vm);
      goto advance;

    case 0x0d:
      if (vm->i.os_32)
        OR_EAXId(vm);
      else
        OR_AXIw(vm);
      goto advance;

    case 0x0e:
      PUSH_SREG(vm, SRegCS);
      goto advance;

    case 0x10:
      ADC_EbGb(vm);
      goto advance;

    case 0x11:
      if (vm->i.os_32)
        ADC_EdGd(vm);
      else
        ADC_EwGw(vm);
      goto advance;

    case 0x12:
      ADC_GbEb(vm);
      goto advance;

    case 0x13:
      if (vm->i.os_32)
        ADC_GdEd(vm);
      else
        ADC_GwEw(vm);
      goto advance;

    case 0x14:
      ADC_ALIb(vm);
      goto advance;

    case 0x15:
      if (vm->i.os_32)
        ADC_EAXId(vm);
      else
        ADC_AXIw(vm);
      goto advance;

    case 0x16:
      PUSH_SREG(vm, SRegSS);
      goto advance;

    case 0x17:
      POP_SREG(vm, SRegSS);
      goto advance;

    case 0x18:
      SBB_EbGb(vm);
      goto advance;

    case 0x19:
      if (vm->i.os_32)
        SBB_EdGd(vm);
      else
        SBB_EwGw(vm);
      goto advance;

    case 0x1a:
      SBB_GbEb(vm);
      goto advance;

    case 0x1b:
      if (vm->i.os_32)
        SBB_GdEd(vm);
      else
        SBB_GwEw(vm);
      goto advance;

    case 0x1c:
      SBB_ALIb(vm);
      goto advance;

    case 0x1d:
      if (vm->i.os_32)
        SBB_EAXId(vm);
      else
        SBB_AXIw(vm);
      goto advance;

    case 0x1e:
      PUSH_SREG(vm, SRegDS);
      goto advance;

    case 0x1f:
      POP_SREG(vm, SRegDS);
      goto advance;

    case 0x20:
      AND_EbGb(vm);
      goto advance;

    case 0x21:
      if (vm->i.os_32)
        AND_EdGd(vm);
      else
        AND_EwGw(vm);
      goto advance;

    case 0x22:
      AND_GbEb(vm);
      goto advance;

    case 0x23:
      if (vm->i.os_32)
        AND_GdEd(vm);
      else
        AND_GwEw(vm);
      goto advance;

    case 0x24:
      AND_ALIb(vm);
      goto advance;

    case 0x25:
      if (vm->i.os_32)
        AND_EAXId(vm);
      else
        AND_AXIw(vm);
      goto advance;

    case 0x27:
      DAA(vm);
      goto advance;

    case 0x28:
      SUB_EbGb(vm);
      goto advance;

    case 0x29:
      if (vm->i.os_32)
        SUB_EdGd(vm);
      else
        SUB_EwGw(vm);
      goto advance;

    case 0x2a:
      SUB_GbEb(vm);
      goto advance;

    case 0x2b:
      if (vm->i.os_32)
        SUB_GdEd(vm);
      else
        SUB_GwEw(vm);
      goto advance;

    case 0x2c:
      SUB_ALIb(vm);
      goto advance;

    case 0x2d:
      if (vm->i.os_32)
        SUB_EAXId(vm);
      else
        SUB_AXIw(vm);
      goto advance;

    case 0x2f:
      DAS(vm);
      goto advance;

    case 0x30:
      XOR_EbGb(vm);
      goto advance;

    case 0x31:
      if (vm->i.os_32)
        XOR_EdGd(vm);
      else
        XOR_EwGw(vm);
      goto advance;

    case 0x32:
      XOR_GbEb(vm);
      goto advance;

    case 0x33:
      if (vm->i.os_32)
        XOR_GdEd(vm);
      else
        XOR_GwEw(vm);
      goto advance;

    case 0x34:
      XOR_ALIb(vm);
      goto advance;

    case 0x35:
      if (vm->i.os_32)
        XOR_EAXId(vm);
      else
        XOR_AXIw(vm);
      goto advance;

    case 0x37:
      AAA(vm);
      goto advance;

    case 0x38:
      CMP_EbGb(vm);
      goto advance;

    case 0x39:
      if (vm->i.os_32)
        CMP_EdGd(vm);
      else
        CMP_EwGw(vm);
      goto advance;

    case 0x3a:
      CMP_GbEb(vm);
      goto advance;

    case 0x3b:
      if (vm->i.os_32)
        CMP_GdEd(vm);
      else
        CMP_GwEw(vm);
      goto advance;

    case 0x3c:
      CMP_ALIb(vm);
      goto advance;

    case 0x3d:
      if (vm->i.os_32)
        CMP_EAXId(vm);
      else
        CMP_AXIw(vm);
      goto advance;

    case 0x3f:
      AAS(vm);
      goto advance;

    case 0x40:
    case 0x41:
    case 0x42:
    case 0x43:
    case 0x44:
    case 0x45:
    case 0x46:
    case 0x47:
      if (vm->i.os_32)
        INC_ERX(vm);
      else
        INC_RX(vm);
      goto advance;

    case 0x48:
    case 0x49:
    case 0x4a:
    case 0x4b:
    case 0x4c:
    case 0x4d:
    case 0x4e:
    case 0x4f:
      if (vm->i.os_32)
        DEC_ERX(vm);
      else
        DEC_RX(vm);
      goto advance;

    case 0x50: /* PUSH_eAX */
    case 0x51: /* PUSH_eCX */
    case 0x52: /* PUSH_eDX */
    case 0x53: /* PUSH_eBX */
    case 0x54: /* PUSH_eSP */
    case 0x55: /* PUSH_eBP */
    case 0x56: /* PUSH_eSI */
    case 0x57: /* PUSH_eDI */
      if (vm->i.os_32)
        push32(vm, ReadReg32(vm, vm->i.b1 & 0x07));
      else
        push16(vm, ReadReg16(vm, vm->i.b1 & 0x07));
      goto advance;

    case 0x58: /* POP_eAX */
    case 0x59: /* POP_eCX */
    case 0x5a: /* POP_eDX */
    case 0x5b: /* POP_eBX */
    case 0x5c: /* POP_eSP */
    case 0x5d: /* POP_eBP */
    case 0x5e: /* POP_eSI */
    case 0x5f: /* POP_eDI */
      if (vm->i.os_32) {
        Bit32u erx;
        pop32(vm, &erx);
        WriteReg32(vm, vm->i.b1 & 0x07, erx);
        }
      else {
        Bit16u rx;
        pop16(vm, &rx);
        WriteReg16(vm, vm->i.b1 & 0x07, rx);
        }
      goto advance;

    case 0x60:
      if (vm->i.os_32)
        PUSHAD32(vm);
      else
        PUSHAD16(vm);
      goto advance;

    case 0x61:
      if (vm->i.os_32)
        POPAD32(vm);
      else
        POPAD16(vm);
      goto advance;

    case 0x63:
      ARPL_EwGw(vm);
      goto advance;

    case 0x68: /* PUSH_Iv */
      if (vm->i.os_32)
        push32(vm, vm->i.Id);
      else
        push16(vm, vm->i.Iw);
      goto advance;

    case 0x69:
    case 0x6b:
      if (vm->i.os_32)
        IMUL_GdEdId(vm);
      else
        IMUL_GwEwIw(vm);
      goto advance;

    case 0x6a: /* PUSH_Ib (byte extended to word or dword) */
      PUSH_Iv(vm);
      goto advance;

    case 0x6c:
      INSB_YbDX(vm);
      goto advance;

    case 0x6d:
      INSW_YvDX(vm);
      goto advance;

    case 0x6f:
      OUTSW_DXXv(vm);
      goto advance;

    case 0x70:
    case 0x71:
    case 0x72:
    case 0x73:
    case 0x74:
    case 0x75:
    case 0x76:
    case 0x77:
    case 0x78:
    case 0x79:
    case 0x7a:
    case 0x7b:
    case 0x7c:
    case 0x7d:
    case 0x7e:
    case 0x7f:
      if (vm->i.os_32)
        JCC_Jd(vm);
      else
        JCC_Jw(vm);
      goto advance;

    case 0x84: TEST_EbGb(vm); goto advance;

    case 0x85:
      if (vm->i.os_32)
        TEST_EdGd(vm);
      else
        TEST_EwGw(vm);
      goto advance;

    case 0x86: XCHG_EbGb(vm); goto advance;

    case 0x87:
      if (vm->i.os_32)
        XCHG_EdGd(vm);
      else
        XCHG_EwGw(vm);
      goto advance;

    case 0x88: MOV_EbGb(vm); goto advance;

    case 0x89:
      if (vm->i.os_32)
        MOV_EdGd(vm);
      else
        MOV_EwGw(vm);
      goto advance;

    case 0x8a:
      MOV_GbEb(vm);
      goto advance;

    case 0x8b:
      if (vm->i.os_32)
        MOV_GdEd(vm);
      else
        MOV_GwEw(vm);
      goto advance;

    case 0x8c:
      MOV_EwSw(vm);
      goto advance;

    case 0x8d:
      if (vm->i.os_32)
        LEA_GdM(vm);
      else
        LEA_GwM(vm);
      goto advance;

    case 0x8e:
      MOV_SwEw(vm);
      goto advance;

    case 0x8f:
      if (vm->i.os_32)
        POP_Ed(vm);
      else
        POP_Ew(vm);
      goto advance;

    case 0x90: /* NOP */
      /* Nothing to do. */
      goto advance;

    case 0x91:
    case 0x92:
    case 0x93:
    case 0x94:
    case 0x95:
    case 0x96:
    case 0x97:
      if (vm->i.os_32)
        XCHG_ERXEAX(vm);
      else
        XCHG_RXAX(vm);
      goto advance;

    case 0x98:
      if (vm->i.os_32)
        CWDE(vm);
      else
        CBW(vm);
      goto advance;

    case 0x99:
      if (vm->i.os_32)
        CDQ(vm);
      else
        CWD(vm);
      goto advance;

    case 0x9a:
      if (vm->i.os_32)
        CALL32_Ap(vm);
      else
        CALL16_Ap(vm);
      goto advance;

    case 0x9b:
      FWAIT(vm);
      goto advance;

    case 0x9c:
      PUSHF_Fv(vm);
      goto advance;

    case 0x9d:
      POPF_Fv(vm);
      goto advance;

    case 0x9e:
      SAHF(vm);
      goto advance;

    case 0x9f:
      LAHF(vm);
      goto advance;

    case 0xa0:
      MOV_ALOb(vm);
      goto advance;

    case 0xa1:
      if (vm->i.os_32)
        MOV_EAXOd(vm);
      else
        MOV_AXOw(vm);
      goto advance;

    case 0xa2:
      MOV_ObAL(vm);
      goto advance;

    case 0xa3:
      if (vm->i.os_32)
        MOV_OdEAX(vm);
      else
        MOV_OwAX(vm);
      goto advance;

    case 0xa4:
      MOVSB_XbYb(vm);
      goto advance;

    case 0xa5:
      MOVSW_XvYv(vm);
      goto advance;

    case 0xa6:
      CMPSB_XbYb(vm);
      goto advance;

    case 0xa7:
      CMPSW_XvYv(vm);
      goto advance;

    case 0xa8:
      TEST_ALIb(vm);
      goto advance;

    case 0xa9:
      if (vm->i.os_32)
        TEST_EAXId(vm);
      else
        TEST_AXIw(vm);
      goto advance;

    case 0xaa:
      STOSB_YbAL(vm);
      goto advance;

    case 0xab:
      STOSW_YveAX(vm);
      goto advance;

    case 0xac:
      LODSB_ALXb(vm);
      goto advance;

    case 0xad:
      LODSW_eAXXv(vm);
      goto advance;

    case 0xae:
      SCASB_ALXb(vm);
      goto advance;

    case 0xaf:
      SCASW_eAXXv(vm);
      goto advance;

    case 0xb0: /* MOV_ALIb */
    case 0xb1:
    case 0xb2:
    case 0xb3:
    case 0xb4:
    case 0xb5:
    case 0xb6:
    case 0xb7:
      WriteReg8(vm, vm->i.b1 & 0x07, vm->i.Ib);
      goto advance;

    case 0xb8:
    case 0xb9:
    case 0xba:
    case 0xbb:
    case 0xbc:
    case 0xbd:
    case 0xbe:
    case 0xbf:
      if (vm->i.os_32)
        WriteReg32(vm, vm->i.b1 & 0x07, vm->i.Id);
      else
        WriteReg16(vm, vm->i.b1 & 0x07, vm->i.Iw);
      goto advance;

    case 0xc2:
      if (vm->i.os_32)
        RETnear32_Iw(vm);
      else
        RETnear16_Iw(vm);
      goto advance;

    case 0xc3:
      if (vm->i.os_32)
        RETnear32(vm);
      else
        RETnear16(vm);
      goto advance;

    case 0xc4:
      LxS_GvMp(vm, SRegES);
      goto advance;

    case 0xc5:
      LxS_GvMp(vm, SRegDS);
      goto advance;

    case 0xc6:
      MOV_EbIb(vm);
      goto advance;

    case 0xc7:
      if (vm->i.os_32)
        MOV_EdId(vm);
      else
        MOV_EwIw(vm);
      goto advance;

    case 0xc8:
      ENTER_IwIb(vm);
      goto advance;

    case 0xc9:
      LEAVE(vm);
      goto advance;

    case 0xca:
      if (vm->i.os_32)
        RETfar32_Iw(vm);
      else
        RETfar16_Iw(vm);
      goto advance;

    case 0xcb:
      if (vm->i.os_32)
        RETfar32(vm);
      else
        RETfar16(vm);
      goto advance;

    case 0xcc:
      INT3(vm);
      goto advance;

    case 0xcd:
      INT_Ib(vm);
      goto advance;

    case 0xcf:
      if (vm->i.os_32)
        IRET32(vm);
      else
        IRET16(vm);
      goto advance;

    case 0xd4:
      AAM(vm);
      goto advance;

    case 0xd5:
      AAD(vm);
      goto advance;

    case 0xd6:
      SALC(vm);
      goto advance;

    case 0xd7:
      XLAT(vm);
      goto advance;

    case 0xd8: ESC0(vm); goto advance;
    case 0xd9: ESC1(vm); goto advance;
    case 0xda: ESC2(vm); goto advance;
    case 0xdb: ESC3(vm); goto advance;
    case 0xdc: ESC4(vm); goto advance;
    case 0xdd: ESC5(vm); goto advance;
    case 0xde: ESC6(vm); goto advance;
    case 0xdf: ESC7(vm); goto advance;

    case 0xe0:
      LOOPNE_Jb(vm);
      goto advance;

    case 0xe1:
      LOOPE_Jb(vm);
      goto advance;

    case 0xe2:
      LOOP_Jb(vm);
      goto advance;

    case 0xe3:
      JCXZ_Jb(vm);
      goto advance;

    case 0xe4:
      IN_ALIb(vm);
      goto advance;

    case 0xe5:
      IN_eAXIb(vm);
      goto advance;

    case 0xe6:
      OUT_IbAL(vm);
      goto advance;

    case 0xe7:
      OUT_IbeAX(vm);
      goto advance;

    case 0xe8:
      if (vm->i.os_32)
        CALL_Ad(vm);
      else
        CALL_Aw(vm);
      goto advance;

    case 0xe9: /* JMP_Jv */
    case 0xeb: /* JMP_Jb */
      if (vm->i.os_32)
        JMP_Jd(vm);
      else
        JMP_Jw(vm);
      goto advance;

    case 0xea:
      JMP_Ap(vm);
      goto advance;

    case 0xec:
      IN_ALDX(vm);
      goto advance;

    case 0xed:
      IN_eAXDX(vm);
      goto advance;

    case 0xee: /* out dx, al */
      port = context->edx & 0xffff;
      IOPermCheck(vm, port, 1);
      sysIOOut(vm, port, 1, context->eax & 0xff);
      goto advance;

    case 0xef: /* out dx, eAX */
      port = context->edx & 0xffff;
      if (vm->i.os_32) {
        data = context->eax;
        len = 4;
        }
      else {
        data = context->eax & 0xffff;
        len = 2;
        }
      IOPermCheck(vm, port, len);
      sysIOOut(vm, port, len, data);
      goto advance;

    case 0xf4:
      HLT(vm);
      goto advance;

    case 0xf5:
      CMC(vm);
      goto advance;

    case 0xf8: /* CLC */
      G_SetCF(vm, 0);
      goto advance;

    case 0xf9: /* STC */
      G_SetCF(vm, 1);
      goto advance;

    case 0xfa:
      CLI_guest(vm);
      goto advance;

    case 0xfb:
      STI_guest(vm);
      goto advance;

    case 0xfc: /* CLD */
      G_SetDF(vm, 0);
      goto advance;

    case 0xfd: /* STD */
      G_SetDF(vm, 1);
      goto advance;

    case 0x102:
      LAR_GvEw(vm);
      goto advance;

    case 0x103:
      LSL_GvEw(vm);
      goto advance;

    case 0x106:
      CLTS(vm);
      goto advance;

    case 0x109:
      WBINVD(vm);
      goto advance;

    case 0x120:
      MOV_RdCd(vm);
      goto advance;

    case 0x121:
      MOV_RdDd(vm);
      goto advance;

    case 0x122:
      MOV_CdRd(vm);
      goto advance;

    case 0x123:
      MOV_DdRd(vm);
      goto advance;

    case 0x131:
      RDTSC(vm);
      goto advance;

    case 0x140:
    case 0x141:
    case 0x142:
    case 0x143:
    case 0x144:
    case 0x145:
    case 0x146:
    case 0x147:
    case 0x148:
    case 0x149:
    case 0x14a:
    case 0x14b:
    case 0x14c:
    case 0x14d:
    case 0x14e:
    case 0x14f:
      if (vm->i.os_32)
        CMOV_GdEd(vm);
      else
        CMOV_GwEw(vm);
      goto advance;

    case 0x180:
    case 0x181:
    case 0x182:
    case 0x183:
    case 0x184:
    case 0x185:
    case 0x186:
    case 0x187:
    case 0x188:
    case 0x189:
    case 0x18a:
    case 0x18b:
    case 0x18c:
    case 0x18d:
    case 0x18e:
    case 0x18f:
      if (vm->i.os_32)
        JCC_Jd(vm);
      else
        JCC_Jw(vm);
      goto advance;

    case 0x190: SETCC_Eb(vm); goto advance;
    case 0x191: SETCC_Eb(vm); goto advance;
    case 0x192: SETCC_Eb(vm); goto advance;
    case 0x193: SETCC_Eb(vm); goto advance;
    case 0x194: SETCC_Eb(vm); goto advance;
    case 0x195: SETCC_Eb(vm); goto advance;
    case 0x196: SETCC_Eb(vm); goto advance;
    case 0x197: SETCC_Eb(vm); goto advance;
    case 0x198: SETCC_Eb(vm); goto advance;
    case 0x199: SETCC_Eb(vm); goto advance;
    case 0x19A: SETCC_Eb(vm); goto advance;
    case 0x19B: SETCC_Eb(vm); goto advance;
    case 0x19C: SETCC_Eb(vm); goto advance;
    case 0x19D: SETCC_Eb(vm); goto advance;
    case 0x19E: SETCC_Eb(vm); goto advance;
    case 0x19F: SETCC_Eb(vm); goto advance;

    case 0x1a0:
      PUSH_SREG(vm, SRegFS);
      goto advance;

    case 0x1a1:
      POP_SREG(vm, SRegFS);
      goto advance;

    case 0x1a2: CPUID(vm); goto advance;

    case 0x1a3: BT_EvGv(vm); goto advance;

    case 0x1a4:
    case 0x1a5:
      if (vm->i.os_32)
        SHLD_EdGd(vm);
      else
        SHLD_EwGw(vm);
      goto advance;

    case 0x1a8:
      PUSH_SREG(vm, SRegGS);
      goto advance;

    case 0x1a9:
      POP_SREG(vm, SRegGS);
      goto advance;

    case 0x1ab: BTS_EvGv(vm); goto advance;

    case 0x1ac:
    case 0x1ad:
      if (vm->i.os_32)
        SHRD_EdGd(vm);
      else
        SHRD_EwGw(vm);
      goto advance;

    case 0x1af:
      if (vm->i.os_32)
        IMUL_GdEd(vm);
      else
        IMUL_GwEw(vm);
      goto advance;

    case 0x1b0:
      CMPXCHG_EbGb(vm);
      goto advance;

    case 0x1b1:
      if (vm->i.os_32)
        CMPXCHG_EdGd(vm);
      else
        CMPXCHG_EwGw(vm);
      goto advance;

    case 0x1b2: LxS_GvMp(vm, SRegSS); goto advance;

    case 0x1b3: BTR_EvGv(vm); goto advance;

    case 0x1b4: LxS_GvMp(vm, SRegFS); goto advance;
    case 0x1b5: LxS_GvMp(vm, SRegGS); goto advance;

    case 0x1b6:
      if (vm->i.os_32)
        MOVZX_GdEb(vm);
      else
        MOVZX_GwEb(vm);
      goto advance;

    case 0x1b7:
      if (vm->i.os_32)
        MOVZX_GdEw(vm);
      else
        MOVZX_GwEw(vm);
      goto advance;

    case 0x1bb: BTC_EvGv(vm); goto advance;
    case 0x1bc: BSF_GvEv(vm); goto advance;
    case 0x1bd: BSR_GvEv(vm); goto advance;

    case 0x1be:
      if (vm->i.os_32)
        MOVSX_GdEb(vm);
      else
        MOVSX_GwEb(vm);
      goto advance;

    case 0x1bf:
      if (vm->i.os_32)
        MOVSX_GdEw(vm);
      else
        MOVSX_GwEw(vm);
      goto advance;

    case 0x1c0:
      XADD_EbGb(vm);
      goto advance;

    case 0x1c1:
      if (vm->i.os_32)
        XADD_EdGd(vm);
      else
        XADD_EwGw(vm);
      goto advance;

    case 0x1C8:
    case 0x1C9:
    case 0x1CA:
    case 0x1CB:
    case 0x1CC:
    case 0x1CD:
    case 0x1CE:
    case 0x1CF:
      BSWAP_ERX(vm);
      goto advance;

    default:
      monpanic(vm, "default case on i.b1 of 0x%x, reason=0x%x\n",
        vm->i.b1, reason);
    }

/* Special Group opcodes: */
group_n:
  group = GetGroupIndex(vm->i.attr);

  /*ucode_map.groups[group][vm->i.nnn]; */
  switch ((group<<3) | vm->i.nnn) {
    case (G1EbIb<<3) | 0: ADD_EbIb(vm); break;
    case (G1EbIb<<3) | 1: OR_EbIb(vm);  break;
    case (G1EbIb<<3) | 2: ADC_EbIb(vm); break;
    case (G1EbIb<<3) | 3: SBB_EbIb(vm); break;
    case (G1EbIb<<3) | 4: AND_EbIb(vm); break;
    case (G1EbIb<<3) | 5: SUB_EbIb(vm); break;
    case (G1EbIb<<3) | 6: XOR_EbIb(vm); break;
    case (G1EbIb<<3) | 7: CMP_EbIb(vm); break;

    case (G1Ew<<3) | 0: ADD_EwIw(vm); break;
    case (G1Ew<<3) | 1: OR_EwIw(vm); break;
    case (G1Ew<<3) | 2: ADC_EwIw(vm); break;
    case (G1Ew<<3) | 3: SBB_EwIw(vm); break;
    case (G1Ew<<3) | 4: AND_EwIw(vm); break;
    case (G1Ew<<3) | 5: SUB_EwIw(vm); break;
    case (G1Ew<<3) | 6: XOR_EwIw(vm); break;
    case (G1Ew<<3) | 7: CMP_EwIw(vm); break;

    case (G1Ed<<3) | 0: ADD_EdId(vm); break;
    case (G1Ed<<3) | 1:  OR_EdId(vm); break;
    case (G1Ed<<3) | 2: ADC_EdId(vm); break;
    case (G1Ed<<3) | 3: SBB_EdId(vm); break;
    case (G1Ed<<3) | 4: AND_EdId(vm); break;
    case (G1Ed<<3) | 5: SUB_EdId(vm); break;
    case (G1Ed<<3) | 6: XOR_EdId(vm); break;
    case (G1Ed<<3) | 7: CMP_EdId(vm); break;

    case (G2Eb<<3) | 0: ROL_Eb(vm); break;
    case (G2Eb<<3) | 1: ROR_Eb(vm); break;
    case (G2Eb<<3) | 2: RCL_Eb(vm); break;
    case (G2Eb<<3) | 3: RCR_Eb(vm); break;
    case (G2Eb<<3) | 4: SHL_Eb(vm); break;
    case (G2Eb<<3) | 5: SHR_Eb(vm); break;
    case (G2Eb<<3) | 6: SHL_Eb(vm); break;
    case (G2Eb<<3) | 7: SAR_Eb(vm); break;

    case (G2Ew<<3) | 0: ROL_Ew(vm); break;
    case (G2Ew<<3) | 1: ROR_Ew(vm); break;
    case (G2Ew<<3) | 2: RCL_Ew(vm); break;
    case (G2Ew<<3) | 3: RCR_Ew(vm); break;
    case (G2Ew<<3) | 4: SHL_Ew(vm); break;
    case (G2Ew<<3) | 5: SHR_Ew(vm); break;
    case (G2Ew<<3) | 6: SHL_Ew(vm); break;
    case (G2Ew<<3) | 7: SAR_Ew(vm); break;

    case (G2Ed<<3) | 0: ROL_Ed(vm); break;
    case (G2Ed<<3) | 1: ROR_Ed(vm); break;
    case (G2Ed<<3) | 2: RCL_Ed(vm); break;
    case (G2Ed<<3) | 3: RCR_Ed(vm); break;
    case (G2Ed<<3) | 4: SHL_Ed(vm); break;
    case (G2Ed<<3) | 5: SHR_Ed(vm); break;
    case (G2Ed<<3) | 6: SHL_Ed(vm); break;
    case (G2Ed<<3) | 7: SAR_Ed(vm); break;

    case (G3Eb<<3) | 0:
    case (G3Eb<<3) | 1: TEST_EbIb(vm); break;
    case (G3Eb<<3) | 2: NOT_Eb(vm); break;
    case (G3Eb<<3) | 3: NEG_Eb(vm); break;
    case (G3Eb<<3) | 4: MUL_ALEb(vm); break;
    case (G3Eb<<3) | 5: IMUL_ALEb(vm); break;
    case (G3Eb<<3) | 6: DIV_ALEb(vm); break;
    case (G3Eb<<3) | 7: IDIV_ALEb(vm); break;

    case (G3Ew<<3) | 0:
    case (G3Ew<<3) | 1: TEST_EwIw(vm); break;
    case (G3Ew<<3) | 2: NOT_Ew(vm); break;
    case (G3Ew<<3) | 3: NEG_Ew(vm); break;
    case (G3Ew<<3) | 4: MUL_AXEw(vm); break;
    case (G3Ew<<3) | 5: IMUL_AXEw(vm); break;
    case (G3Ew<<3) | 6: DIV_AXEw(vm); break;
    case (G3Ew<<3) | 7: IDIV_AXEw(vm); break;

    case (G3Ed<<3) | 0: TEST_EdId(vm); break;
    case (G3Ed<<3) | 1: TEST_EdId(vm); break;
    case (G3Ed<<3) | 2: NOT_Ed(vm); break;
    case (G3Ed<<3) | 3: NEG_Ed(vm); break;
    case (G3Ed<<3) | 4: MUL_EAXEd(vm); break;
    case (G3Ed<<3) | 5: IMUL_EAXEd(vm); break;
    case (G3Ed<<3) | 6: DIV_EAXEd(vm); break;
    case (G3Ed<<3) | 7: IDIV_EAXEd(vm); break;

    case (G4<<3) | 0: INC_Eb(vm); break;
    case (G4<<3) | 1: DEC_Eb(vm); break;
    case (G4<<3) | 2: monprint(vm, "G4.2 unhandled\n"); goto unhandled;
    case (G4<<3) | 3: monprint(vm, "G4.3 unhandled\n"); goto unhandled;
    case (G4<<3) | 4: monprint(vm, "G4.4 unhandled\n"); goto unhandled;
    case (G4<<3) | 5: monprint(vm, "G4.5 unhandled\n"); goto unhandled;
    case (G4<<3) | 6: monprint(vm, "G4.6 unhandled\n"); goto unhandled;
    case (G4<<3) | 7: monprint(vm, "G4.7 unhandled\n"); goto unhandled;

    case (G5w<<3) | 0: INC_Ew(vm); break;
    case (G5w<<3) | 1: DEC_Ew(vm); break;
    case (G5w<<3) | 2: CALL_Ew(vm); break;
    case (G5w<<3) | 3: CALL16_Ep(vm); break;
    case (G5w<<3) | 4: JMP_Ew(vm); break;
    case (G5w<<3) | 5: JMP16_Ep(vm); break;
    case (G5w<<3) | 6: PUSH_Ew(vm); break;
    case (G5w<<3) | 7: monprint(vm, "G5w7 unhandled\n"); goto unhandled;

    case (G5d<<3) | 0: INC_Ed(vm); break;
    case (G5d<<3) | 1: DEC_Ed(vm); break;
    case (G5d<<3) | 2: CALL_Ed(vm); break;
    case (G5d<<3) | 3: CALL32_Ep(vm); break;
    case (G5d<<3) | 4: JMP_Ed(vm); break;
    case (G5d<<3) | 5: JMP32_Ep(vm); break;
    case (G5d<<3) | 6: PUSH_Ed(vm); break;
    case (G5d<<3) | 7: monprint(vm, "G5d7 unhandled\n"); goto unhandled;

    case (G6<<3) | 0: SLDT_Ew(vm); break;
    case (G6<<3) | 1: STR_Ew(vm); break;
    case (G6<<3) | 2: LLDT_Ew(vm); break;
    case (G6<<3) | 3: LTR_Ew(vm); break;
    case (G6<<3) | 4: VERR_Ew(vm); break;
    case (G6<<3) | 5: VERW_Ew(vm); break;
    case (G6<<3) | 6: monprint(vm, "G6.6 unhandled\n"); goto unhandled;
    case (G6<<3) | 7: monprint(vm, "G6.7 unhandled\n"); goto unhandled;

    case (G7<<3) | 0: SGDT_Ms(vm); break;
    case (G7<<3) | 1: SIDT_Ms(vm); break;
    case (G7<<3) | 2: LGDT_Ms(vm); break;
    case (G7<<3) | 3: LIDT_Ms(vm); break;
    case (G7<<3) | 4: SMSW_Ew(vm); break;
    case (G7<<3) | 5: monprint(vm, "G7.5 unhandled\n"); goto unhandled;
    case (G7<<3) | 6: LMSW_Ew(vm); break;
    case (G7<<3) | 7: INVLPG(vm); break;

    case (G8EvIb<<3) | 0:
    case (G8EvIb<<3) | 1:
    case (G8EvIb<<3) | 2:
    case (G8EvIb<<3) | 3:
      monprint(vm, "G8EvIb unhandled\n");
      goto unhandled;
    case (G8EvIb<<3) | 4: BT_EvIb(vm);  break;
    case (G8EvIb<<3) | 5: BTS_EvIb(vm); break;
    case (G8EvIb<<3) | 6: BTR_EvIb(vm); break;
    case (G8EvIb<<3) | 7: BTC_EvIb(vm); break;

    case (G9<<3) | 0:
    default:
      monprint(vm, "GroupN unhandled\n");
      goto unhandled;
    }

advance:

    if (vm->i.rep_used && (vm->i.attr & Repeatable)) {
      if (vm->i.attr & RepeatableZF) {
        if (vm->i.as_32) {
          if (G_ECX(vm) != 0) {
            G_ECX(vm) -= 1;
            }
          if ((vm->i.rep_used==0xf3) && (G_GetZF(vm)==0)) goto repeat_done;
          if ((vm->i.rep_used==0xf2) && (G_GetZF(vm)!=0)) goto repeat_done;
          if (G_ECX(vm) == 0) goto repeat_done;
          goto repeat_not_done;
          }
        else {
          if (G_CX(vm) != 0) {
            G_CX(vm) -= 1;
            }
          if ((vm->i.rep_used==0xf3) && (G_GetZF(vm)==0)) goto repeat_done;
          if ((vm->i.rep_used==0xf2) && (G_GetZF(vm)!=0)) goto repeat_done;
          if (G_CX(vm) == 0) goto repeat_done;
          goto repeat_not_done;
          }
        }
      else { /* normal repeat, no concern for ZF */
        if (vm->i.as_32) {
          if (G_ECX(vm) != 0) {
            G_ECX(vm) -= 1;
            }
          if (G_ECX(vm) == 0) goto repeat_done;
          goto repeat_not_done;
          }
        else { /* 16bit addrsize */
          if (G_CX(vm) != 0) {
            G_CX(vm) -= 1;
            }
          if (G_CX(vm) == 0) goto repeat_done;
          goto repeat_not_done;
          }
        }
      /* shouldn't get here from above */

repeat_done:
      context->eip += vm->i.ilen;
repeat_not_done:
      }

  /* Execution complete, Commit EIP & ESP.  Actually the following
   * code signals to the debugger, that the values in eip/esp are
   * good and that the instruction completed.
   */
  vm->guest_cpu.prev_eip = -1;
  vm->guest_cpu.prev_esp = -1;

  if (vm->executeN) {
    vm->executeN--;
    if (vm->executeN==0) {
      sysEOICount(vm);
      }
    }
  /* Pretend instruction took 1 cycle to execute. */
  vm->system.elapsed += 400;

  return;

unhandled:
  monpanic(vm, "i.b1=0x%x, i.nnn=%u, group=%u, reason=0x%x\n",
    vm->i.b1, vm->i.nnn, group, reason);
  return;
}

  void
emulate_interrupt(vm_t *vm, unsigned vector)
{
/*monprint(vm, "EmInt:\n");*/
  /* Store EIP,ESP in case we need to rewind them due to an */
  /* exception caused by the instruction. */
  vm->guest_cpu.prev_eip = vm->guest.addr.guest_context->eip;
  vm->guest_cpu.prev_esp = vm->guest.addr.guest_context->esp;
 
  vm->guest_cpu.errorno = 0;
  vm->guest_cpu.EXT     = 1; /* external event */
  if (SetJmp(vm->jmp_buf_env)) {
    /* nonzero return, means returning from longjmp() */
    return;
    }
  interrupt(vm, vector, 0, 0, 0);
}

  void
emulate_exception(vm_t *vm, unsigned vector, Bit32u gerror)
{
/*monprint(vm, "EmExc:\n");*/
  /* Store EIP,ESP in case we need to rewind them due to an */
  /* exception caused by the instruction. */
  vm->guest_cpu.prev_eip = vm->guest.addr.guest_context->eip;
  vm->guest_cpu.prev_esp = vm->guest.addr.guest_context->esp;
 
  vm->guest_cpu.EXT = 0;
  vm->guest_cpu.errorno = 0;

  if (SetJmp(vm->jmp_buf_env)) {
    /* nonzero return, means returning from longjmp() */
    return;
    }
  exception(vm, vector, gerror);
}


  void
emulate_async_checks(vm_t *vm)
{
  if (SetJmp(vm->jmp_buf_env)) {
    return;
    }
  async_checks(vm);
}


  void
async_checks(vm_t *vm)
{
  if (vm->guest_cpu.async_event) {
    /* I made up this bitmask to mean HALT state. */
    if (vm->guest_cpu.debug_trap & 0x80000000) {
      vm->guest_cpu.debug_trap = 0; /* clear traps for after resume */
      vm->guest_cpu.inhibit_mask = 0; /* clear inhibits for after resume */
      /* Need to fix this for cosimulation. */
  
      while (1) {
        if (vm->system.intr && G_GetIF(vm)) {
          break;
          }
        vm->system.elapsed += 400;
        guest_cycles_elapsed(vm);
        }
      }
  
    /* Priority 1: Hardware Reset and Machine Checks */
    /*   RESET */
    /*   Machine Check */
    /* (plex86 doesn't support these) */
  
    /* Priority 2: Trap on Task Switch */
    /*   T flag in TSS is set */
    if (vm->guest_cpu.debug_trap & 0x00008000) {
      monpanic(vm, "Trap on Task Switch\n");
      vm->guest_cpu.dr6 |= vm->guest_cpu.debug_trap;
      exception(vm, ExceptionDB, 0);
      }
  
    /* Priority 3: External Hardware Interventions */
    /*   FLUSH */
    /*   STOPCLK */
    /*   SMI */
    /*   INIT  */
    /* (plex86 doesn't support these) */
  
    /* Priority 4: Traps on Previous Instruction */
    /*   Breakpoints */
    /*   Debug Trap Exceptions (TF flag set or data/IO breakpoint) */
    if ( vm->guest_cpu.debug_trap &&
         !(vm->guest_cpu.inhibit_mask & InhibitDebug) ) {
      /* A trap may be inhibited on this boundary due to an instruction */
      /* which loaded SS.  If so we clear the inhibit_mask below */
      /* and don't execute this code until the next boundary. */
      /* Commit debug events to DR6 */
      monpanic(vm, "Trap on Prev Instr\n");
      vm->guest_cpu.dr6 |= vm->guest_cpu.debug_trap; 
      exception(vm, ExceptionDB, 0);
      }
  
    /* Priority 5: External Interrupts */
    /*   NMI Interrupts */
    /*   Maskable Hardware Interrupts */
    if (vm->guest_cpu.inhibit_mask & InhibitInterrupts) {
      /* Processing external interrupts is inhibited on this */
      /* boundary because of certain instructions like STI. */
      /* inhibit_mask is cleared below, in which case we will have */
      /* an opportunity to check interrupts on the next instruction */
      /* boundary. */
      }
  #define DbgAsyncIntr 1 /* don't ask! */
    else if (vm->system.intr && G_GetIF(vm) && DbgAsyncIntr) {
      Bit8u vector;
  
  /* +++ Need to look into getting rid of this */
  /* Store EIP,ESP in case we need to rewind them due to an */
  /* exception caused by the instruction. */
  vm->guest_cpu.prev_eip = vm->guest.addr.guest_context->eip;
  vm->guest_cpu.prev_esp = vm->guest.addr.guest_context->esp;
   
      /* NOTE: similar code in ::take_irq() */
      vector = sysIAC(vm); /* may set INTR with next interrupt */
      vm->guest_cpu.errorno = 0;
      vm->guest_cpu.EXT     = 1; /* external event */
      interrupt(vm, vector, 0, 0, 0);
      }
  #define DbgAsyncDma 1 /* don't ask! */
    else if (vm->system.hrq && DbgAsyncDma) {
      /* NOTE: similar code in ::take_dma() */
      /* assert Hold Acknowledge (HLDA) and go into a bus hold state */
      monpanic(vm, "DMA event\n");
      sysRaiseHLDA(vm);
      }
  
    /* Priority 6: Faults from fetching next instruction */
    /*   Code breakpoint fault */
    /*   Code segment limit violation (priority 7 on 486/Pentium) */
    /*   Code page fault (priority 7 on 486/Pentium) */
    /* (handled in main decode loop) */
  
    /* Priority 7: Faults from decoding next instruction */
    /*   Instruction length > 15 bytes */
    /*   Illegal opcode */
    /*   Coprocessor not available */
    /* (handled in main decode loop etc) */
  
    /* Priority 8: Faults on executing an instruction */
    /*   Floating point execution */
    /*   Overflow */
    /*   Bound error */
    /*   Invalid TSS */
    /*   Segment not present */
    /*   Stack fault */
    /*   General protection */
    /*   Data page fault */
    /*   Alignment check */
    /* (handled by rest of the code) */
  
  
    if ( !(vm->system.intr ||
           vm->guest_cpu.debug_trap ||
           vm->system.hrq) )
      vm->guest_cpu.async_event = 0;
    }

  /* Now we can handle things which are synchronous to instruction */
  /* execution. */
  if (G_GetRF(vm)) {
    G_SetRF(vm, 0);
    }
#if 0
#if BX_X86_DEBUGGER /* +++ */
  else {
    /* only bother comparing if any breakpoints enabled */
    if ( BX_CPU_THIS_PTR dr7 & 0x000000ff ) {
      Bit32u iaddr =
        BX_CPU_THIS_PTR sregs[BX_SEG_REG_CS].cache.u.segment.base +
        BX_CPU_THIS_PTR prev_eip;
      Bit32u dr6_bits;
      if ( (dr6_bits = hwdebug_compare(iaddr, 1, BX_HWDebugInstruction,
                                       BX_HWDebugInstruction)) ) {
        /* Add to the list of debug events thus far. */
        BX_CPU_THIS_PTR debug_trap |= dr6_bits;
        BX_CPU_THIS_PTR async_event = 1;
        /* If debug events are not inhibited on this boundary, */
        /* fire off a debug fault.  Otherwise handle it on the next */
        /* boundary. (becomes a trap) */
        if ( !(BX_CPU_THIS_PTR inhibit_mask & BX_INHIBIT_DEBUG) ) {
          /* Commit debug events to DR6 */
          BX_CPU_THIS_PTR dr6 = BX_CPU_THIS_PTR debug_trap;
          exception(BX_DB_EXCEPTION, 0, 0); /* no error, not interrupt */
          }
        }
      }
    }
#endif  /* BX_X86_DEBUGGER */
#endif  /* 0 */

  /* Events in the mask may have been inhibited during processing */
  /* of the events above.  Now that they have been properly ignored, */
  /* we can clear the inhibit mask. */
  vm->guest_cpu.inhibit_mask = 0; /* clear inhibits */
}
