/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2000 Kevin P. Lawton
 *
 *  mult16.c:  emulation of 16-bit multiplication instructions
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "monitor.h"


  void
IMUL_GwEwIw(vm_t *vm)
{
  Bit16s op2, op3, product;
  Bit32u eflags, arithmetic_eflags;
 
  op3 = vm->i.Iw;
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }

  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %5, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "imulw %4, %3          \n\t"
    "pushfl                \n\t"
    "popl   %0"
    : "=r" (eflags), "=r" (product)
    : "0" (eflags), "1" (op2), "g" (op3), "g" (arithmetic_eflags)
    : "memory", "cc"
    );

  /* now write product back to destination */
  WriteReg16(vm, vm->i.nnn, product);

  SetOSZAPC(vm, eflags);
}

  void
IMUL_GwEw(vm_t *vm)
{
  Bit16s op1, op2, product;
  Bit32u eflags, arithmetic_eflags;
 
  op1 = ReadReg16(vm, vm->i.nnn);

  if (vm->i.mod == 0xc0) {
    op2 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }

  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %5, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "imulw %4, %3          \n\t"
    "pushfl                \n\t"
    "popl   %0"
    : "=r" (eflags), "=r" (product)
    : "0" (eflags), "1" (op1), "g" (op2), "g" (arithmetic_eflags)
    : "memory", "cc"
    );

  /* now write product back to destination */
  WriteReg16(vm, vm->i.nnn, product);

  SetOSZAPC(vm, eflags);
}

  void
MUL_AXEw(vm_t *vm)
{
  Bit16s op2, ax, dx;
  Bit32u eflags, arithmetic_eflags;
 
  ax = G_AX(vm);
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
 
  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %6, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "mulw %5               \n\t"
    "pushfl                \n\t"
    "popl   %0"
    : "=r" (eflags), "=a" (ax), "=d" (dx)
    : "0" (eflags), "1" (ax), "g" (op2), "g" (arithmetic_eflags)
    : "memory", "cc"
    );
 
  G_AX(vm) = ax;
  G_DX(vm) = dx;
  SetOSZAPC(vm, eflags);
}

  void
IMUL_AXEw(vm_t *vm)
{
  Bit16s op2, ax, dx;
  Bit32u eflags, arithmetic_eflags;
 
  ax = G_AX(vm);
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
 
  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %6, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "imulw %5              \n\t"
    "pushfl                \n\t"
    "popl   %0"
    : "=r" (eflags), "=a" (ax), "=d" (dx)
    : "0" (eflags), "1" (ax), "g" (op2), "g" (arithmetic_eflags)
    : "memory", "cc"
    );
 
  G_AX(vm) = ax;
  G_DX(vm) = dx;
  SetOSZAPC(vm, eflags);
}

  void
DIV_AXEw(vm_t *vm)
{
  Bit16u op2, ax, dx;
  Bit32u eflags, arithmetic_eflags;
 
  dx = G_DX(vm);
  ax = G_AX(vm);
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
 
  if (op2 == 0) {
    exception(vm, ExceptionDE, 0);
    }

  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %7, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "divw  %6              \n\t"
    "pushfl                \n\t"
    "popl   %0"
    : "=r" (eflags), "=a" (ax), "=d" (dx)
    : "0" (eflags), "1" (ax), "2" (dx), "g" (op2), "g" (arithmetic_eflags)
    : "memory", "cc"
    );

  G_AX(vm) = ax;
  G_DX(vm) = dx;
  SetOSZAPC(vm, eflags);
}

  void
IDIV_AXEw(vm_t *vm)
{
  Bit16u op2, ax, dx;
  Bit32u eflags, arithmetic_eflags;
 
  dx = G_DX(vm);
  ax = G_AX(vm);
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
 
  if (op2 == 0) {
    exception(vm, ExceptionDE, 0);
    }

  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %7, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "idivw  %6             \n\t"
    "pushfl                \n\t"
    "popl   %0"
    : "=r" (eflags), "=a" (ax), "=d" (dx)
    : "0" (eflags), "1" (ax), "2" (dx), "g" (op2), "g" (arithmetic_eflags)
    : "memory", "cc"
    );

  G_AX(vm) = ax;
  G_DX(vm) = dx;
  SetOSZAPC(vm, eflags);
}
