/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2000 Kevin P. Lawton
 *
 *  mult32.c:  emulation of 32-bit multiplication instructions
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "monitor.h"



  void
IMUL_GdEdId(vm_t *vm)
{
  Bit32s op2, op3, product;
  Bit32u eflags, arithmetic_eflags;
 
  op3 = vm->i.Id;
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }

  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %5, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "imull %4, %3          \n\t"
    "pushfl                \n\t"
    "popl   %0"
    : "=r" (eflags), "=r" (product)
    : "0" (eflags), "1" (op2), "g" (op3), "g" (arithmetic_eflags)
    : "memory", "cc"
    );

  /* now write product back to destination */
  WriteReg32(vm, vm->i.nnn, product);

  SetOSZAPC(vm, eflags);
}

  void
IMUL_GdEd(vm_t *vm)
{
  Bit32s op1, op2, product;
  Bit32u eflags, arithmetic_eflags;
 
  op1 = ReadReg32(vm, vm->i.nnn);

  if (vm->i.mod == 0xc0) {
    op2 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }

  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %5, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "imull %4, %3          \n\t"
    "pushfl                \n\t"
    "popl   %0"
    : "=r" (eflags), "=r" (product)
    : "0" (eflags), "1" (op1), "g" (op2), "g" (arithmetic_eflags)
    : "memory", "cc"
    );

  /* now write product back to destination */
  WriteReg32(vm, vm->i.nnn, product);

  SetOSZAPC(vm, eflags);
}

  void
MUL_EAXEd(vm_t *vm)
{
  Bit32s op2, eax, edx;
  Bit32u eflags, arithmetic_eflags;
 
  eax = G_EAX(vm);
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
 
  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %6, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "mull %5               \n\t"
    "pushfl                \n\t"
    "popl   %0"
    : "=r" (eflags), "=a" (eax), "=d" (edx)
    : "0" (eflags), "1" (eax), "g" (op2), "g" (arithmetic_eflags)
    : "memory", "cc"
    );
 
  G_EAX(vm) = eax;
  G_EDX(vm) = edx;
  SetOSZAPC(vm, eflags);
}

  void
IMUL_EAXEd(vm_t *vm)
{
  Bit32s op2, eax, edx;
  Bit32u eflags, arithmetic_eflags;
 
  eax = G_EAX(vm);
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
 
  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %6, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "imull %5              \n\t"
    "pushfl                \n\t"
    "popl   %0"
    : "=r" (eflags), "=a" (eax), "=d" (edx)
    : "0" (eflags), "1" (eax), "g" (op2), "g" (arithmetic_eflags)
    : "memory", "cc"
    );
 
  G_EAX(vm) = eax;
  G_EDX(vm) = edx;
  SetOSZAPC(vm, eflags);
}

  void
DIV_EAXEd(vm_t *vm)
{
  Bit32u op2, eax, edx;
  Bit32u eflags, arithmetic_eflags;
 
  edx = G_EDX(vm);
  eax = G_EAX(vm);
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
 
  if (op2 == 0) {
    exception(vm, ExceptionDE, 0);
    }

  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %7, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "divl  %6              \n\t"
    "pushfl                \n\t"
    "popl   %0"
    : "=r" (eflags), "=a" (eax), "=d" (edx)
    : "0" (eflags), "1" (eax), "2" (edx), "g" (op2), "g" (arithmetic_eflags)
    : "memory", "cc"
    );

  G_EAX(vm) = eax;
  G_EDX(vm) = edx;
  SetOSZAPC(vm, eflags);
}

  void
IDIV_EAXEd(vm_t *vm)
{
  Bit32u op2, eax, edx;
  Bit32u eflags, arithmetic_eflags;
 
  edx = G_EDX(vm);
  eax = G_EAX(vm);
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
 
  if (op2 == 0) {
    exception(vm, ExceptionDE, 0);
    }

  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %7, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "idivl  %6             \n\t"
    "pushfl                \n\t"
    "popl   %0"
    : "=r" (eflags), "=a" (eax), "=d" (edx)
    : "0" (eflags), "1" (eax), "2" (edx), "g" (op2), "g" (arithmetic_eflags)
    : "memory", "cc"
    );

  G_EAX(vm) = eax;
  G_EDX(vm) = edx;
  SetOSZAPC(vm, eflags);
}
