/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2000 Kevin P. Lawton
 *
 *  paging.c:  linear->physical memory access functionality
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "monitor.h"







  Bit32u
dtranslate_linear(vm_t *vm, Bit32u laddr, unsigned req_us,
                  unsigned req_rw)
{
  Bit32u       pdi, pti, offset;
  Bit32u       guest_lpage_index;
  pageEntry_t *guestPDir, guestPDE, *guestPTbl, guestPTE;
  Bit32u       guest_pdir_page_index, ptbl_ppage_index;
  Bit32u       error;
  unsigned     RMW_rw;

  VM_ASSERT(vm, vm->guest_cpu.cr0.fields.pg);
  offset = laddr & 0x00000fff;
  guest_lpage_index = laddr >> 12;
  pdi = guest_lpage_index >> 10;
  pti = guest_lpage_index & 0x3ff;

  RMW_rw = req_rw;
  if (req_rw==OP_RW)
    req_rw = OP_WRITE;


  /* First, get the guest PDE */
  guest_pdir_page_index = A20Addr(vm, vm->guest_cpu.cr3) >> 12;
  if (guest_pdir_page_index >= vm->pages.guest_n_pages)
    monpanic(vm, "dtranslate_linear:PG=1 guest PDE OOB\n");
  /* Open a window into guest physical memory */
  guestPDir = open_guest_phy_page(vm, guest_pdir_page_index,
                                  vm->guest.addr.tmp_phy_page0);
  guestPDE = guestPDir[pdi];

  /* See if present, before fetching PTE */
  if (guestPDE.fields.P==0) {
    error = 0x00000000; /* RSVD=0, P=0 */
    goto np_exception;
    }

  if (cpuid_info.procSignature.fields.family < 6) {
    /* Update A bit of PDE memory image if not already */
    if ( guestPDE.fields.A == 0 ) {
      guestPDE.fields.A = 1;
      guestPDir[pdi] = guestPDE;
      }
    }

  /* Second, get the guest PDE */
  ptbl_ppage_index = A20PageIndex(vm, guestPDE.fields.base);
  if (ptbl_ppage_index >= vm->pages.guest_n_pages)
    monpanic(vm, "dtranslate_linear: guest PTE OOB\n");
  guestPTbl = open_guest_phy_page(vm, ptbl_ppage_index,
                                  vm->guest.addr.tmp_phy_page1);
  guestPTE = guestPTbl[pti];

  if (guestPTE.fields.P==0) {
    error = 0x00000000; /* RSVD=0, P=0 */
    goto np_exception;
    }

#if 0
/* +++ */
  if (guestPDE.raw & PDEUnhandled)
    monpanic(vm, "dtranslate_linear: guestPDE 0x%08x\n", guestPDE.raw);
#endif
  /* See if requested guest priv is weaker than guest PDE priv */
  if (req_us > guestPDE.fields.US) {
    error = 0x00000001; /* RSVD=0, P=1 */
    goto access_exception;
    }
  if ( (req_rw > guestPDE.fields.RW) &&
       (vm->guest_cpu.cr0.fields.wp || req_us) ) {
    error = 0x00000001; /* RSVD=0, P=1 */
    goto access_exception;
    }

  if (guestPTE.raw & PTEUnhandled)
    monpanic(vm, "dtranslate_linear: guestPTE 0x%08x\n", guestPTE.raw);
  if (req_us > guestPTE.fields.US) {
    error = 0x00000001; /* RSVD=0, P=1 */
    goto access_exception;
    }
  if ( (req_rw > guestPTE.fields.RW) &&
       (vm->guest_cpu.cr0.fields.wp || req_us) ) {
    error = 0x00000001; /* RSVD=0, P=1 */
    goto access_exception;
    }

  if (cpuid_info.procSignature.fields.family >= 6) {
    /* Update A bit of PDE memory image if not already */
    if ( guestPDE.fields.A == 0 ) {
      guestPDE.fields.A = 1;
      guestPDir[pdi] = guestPDE;
      }
    }
  
  /* Update A bit in PTE memory image if not already */
  if ( (guestPTE.fields.A == 0) ||
       ((req_rw==OP_WRITE) && !guestPTE.fields.D) ) {
    guestPTE.fields.A = 1;
    if (req_rw==OP_WRITE)
      guestPTE.fields.D = 1;
    guestPTbl[pti] = guestPTE;
    }

  return( (A20Addr(vm, guestPTE.raw) & 0xfffff000) | offset );

np_exception:
  /* For exceptions due to P==0, R-M-W operations should
   * report a Read.
   */
#if 0
  if (RMW_rw==OP_RW)
    req_rw = OP_READ;
#endif

access_exception:
  error |= (req_us<<2) | (req_rw<<1);
  vm->guest_cpu.cr2 = laddr;
  exception(vm, ExceptionPF, error);
}

  void
access_linear(vm_t *vm, Bit32u laddr, unsigned length, unsigned pl,
              unsigned rw, void *data)
{
  Bit32u mod4096;
  unsigned xlate_rw;

#if SUPPORT_X86_BREAKPOINTS
  /* Only compare debug registers if any breakpoints are enabled */
  if ( vm->guest_cpu.dr7 & 0x000000ff ) {
    monpanic(vm, "access_linear: hardware breakpoints enabled\n");
    }
#endif

  xlate_rw = rw;
  if (rw==OP_RW)
    rw = OP_READ;

  if (vm->guest_cpu.cr0.fields.pg) { /* paging on */
    /* check for access across multiple pages */
    mod4096 = laddr & 0x00000fff;
    if ( (mod4096 + length) <= 4096 ) {
      /* Bit32u paddress1; */

      /* access within single page */
      vm->guest_cpu.address_xlation.paddress1 =
        dtranslate_linear(vm, laddr, pl, xlate_rw);
      vm->guest_cpu.address_xlation.pages     = 1;

      if (rw == OP_READ) {
        read_physical(vm, vm->guest_cpu.address_xlation.paddress1, length, data);
        }
      else {
        write_physical(vm, vm->guest_cpu.address_xlation.paddress1, length, data);
        }
      return;
      }
    else {
      /* access across 2 pages */
      vm->guest_cpu.address_xlation.paddress1 =
        dtranslate_linear(vm, laddr, pl, xlate_rw);
      vm->guest_cpu.address_xlation.len1  = 4096 - mod4096;
      vm->guest_cpu.address_xlation.len2  =
        length - vm->guest_cpu.address_xlation.len1;
      vm->guest_cpu.address_xlation.pages = 2;

      vm->guest_cpu.address_xlation.paddress2 =
        dtranslate_linear(vm, laddr + vm->guest_cpu.address_xlation.len1, pl, xlate_rw);

      if (rw == OP_READ) {
        read_physical(vm, vm->guest_cpu.address_xlation.paddress1,
                             vm->guest_cpu.address_xlation.len1, data);
        read_physical(vm, vm->guest_cpu.address_xlation.paddress2,
                             vm->guest_cpu.address_xlation.len2,
                             ((Bit8u*)data) + vm->guest_cpu.address_xlation.len1);
        }
      else {
        write_physical(vm, vm->guest_cpu.address_xlation.paddress1,
                              vm->guest_cpu.address_xlation.len1, data);
        write_physical(vm, vm->guest_cpu.address_xlation.paddress2,
                              vm->guest_cpu.address_xlation.len2,
                              ((Bit8u*)data) + vm->guest_cpu.address_xlation.len1);
        }
      return;
      }
    }
  else {
    /* paging off */
    if (rw == OP_READ) {
      read_physical(vm, A20Addr(vm, laddr), length, data);
      }
    else {
      write_physical(vm, A20Addr(vm, laddr), length, data);
      }
    vm->guest_cpu.address_xlation.paddress1 = A20Addr(vm, laddr);
    }
}

  void
CR3_change(vm_t *vm, Bit32u val32)
{
  /* Reserved bits take on value of MOV instruction */
  vm->guest_cpu.cr3 = val32;

  /* If paging is on, we have to rehash the virtualized
   * paging environment.
   */
  if (vm->guest_cpu.cr0.fields.pg)
    vm->modeChange |= ModeChangePaging;
}
