/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2000 Kevin P. Lawton
 *
 *  shift8.c:  emulation of 8-bit bitshift instructions
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "monitor.h"


  void
ROL_Eb(vm_t *vm)
{
  Bit8u    op1;
  unsigned count;
  Bit32u   eflags, arithmetic_eflags;
 
  if ( vm->i.b1 == 0xc0 )
    count = vm->i.Ib & 0x07;
  else if ( vm->i.b1 == 0xd0 )
    count = 1;
  else /* 0xd2 */
    count = G_CL(vm) & 0x07;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  if (!count) return;
  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %5, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "rolb %%cl, %1         \n\t" /* Do operation */
    "pushfl                \n\t" /* Get resultant flags */
    "popl   %0"
    : "=r" (eflags), "=mq" (op1)
    : "1" (op1), "c" (count), "0" (eflags), "g" (arithmetic_eflags)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg8(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_byte(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
ROR_Eb(vm_t *vm)
{
  Bit8u op1;
  unsigned count;
  Bit32u   eflags, arithmetic_eflags;
 
  if (vm->i.b1 == 0xc0)
    count = vm->i.Ib & 0x07;
  else if (vm->i.b1 == 0xd0)
    count = 1;
  else /* 0xd2 */
    count = G_CL(vm) & 0x07;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  if (!count) return;
  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %5, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "rorb %%cl, %1         \n\t" /* Do operation */
    "pushfl                \n\t" /* Get resultant flags */
    "popl   %0"
    : "=r" (eflags), "=mq" (op1)
    : "1" (op1), "c" (count), "0" (eflags), "g" (arithmetic_eflags)
    : "memory", "cc"
    );

  if (vm->i.mod == 0xc0) {
    WriteReg8(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_byte(vm, &op1);
    }

  SetOSZAPC(vm, eflags);
}

  void
RCL_Eb(vm_t *vm)
{
  Bit8u    op1;
  unsigned count;
  Bit32u   eflags, arithmetic_eflags;
 
  if ( vm->i.b1 == 0xc0 )
    count = vm->i.Ib;
  else if ( vm->i.b1 == 0xd0 )
    count = 1;
  else /* 0xd2 */
    count = G_CL(vm);
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %5, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "rclb %%cl, %1         \n\t" /* Do operation */
    "pushfl                \n\t" /* Get resultant flags */
    "popl   %0"
    : "=r" (eflags), "=mq" (op1)
    : "1" (op1), "c" (count), "0" (eflags), "g" (arithmetic_eflags)
    : "memory", "cc"
    );

  if (vm->i.mod == 0xc0) {
    WriteReg8(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_byte(vm, &op1);
    }

  SetOSZAPC(vm, eflags);
}

  void
RCR_Eb(vm_t *vm)
{
  Bit8u op1;
  unsigned count;
  Bit32u eflags, arithmetic_eflags;
 
  if ( vm->i.b1 == 0xc0 )
    count = vm->i.Ib;
  else if ( vm->i.b1 == 0xd0 )
    count = 1;
  else /* 0xd2 */
    count = G_CL(vm);
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %5, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "rcrb %%cl, %1         \n\t" /* Do operation */
    "pushfl                \n\t" /* Get resultant flags */
    "popl   %0"
    : "=r" (eflags), "=mq" (op1)
    : "1" (op1), "c" (count), "0" (eflags), "g" (arithmetic_eflags)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg8(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_byte(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
SHL_Eb(vm_t *vm)
{
  Bit8u op1;
  unsigned count;
  Bit32u eflags, arithmetic_eflags;
 
  if (vm->i.b1 == 0xc0)
    count = vm->i.Ib & 0x1f;
  else if (vm->i.b1 == 0xd0)
    count = 1;
  else /* 0xd2 */
    count = G_CL(vm) & 0x1f;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  if (!count) return;
 
  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %5, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "shlb %%cl, %1         \n\t"
    "pushfl                \n\t"
    "popl   %0"
    : "=r" (eflags), "=mq" (op1)
    : "1" (op1), "c" (count), "0" (eflags), "g" (arithmetic_eflags)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg8(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_byte(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
SHR_Eb(vm_t *vm)
{
  Bit8u op1;
  unsigned count;
  Bit32u eflags, arithmetic_eflags;
 
  if (vm->i.b1 == 0xc0)
    count = vm->i.Ib & 0x1f;
  else if (vm->i.b1 == 0xd0)
    count = 1;
  else /* 0xd2 */
    count = G_CL(vm) & 0x1f;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  if (!count) return;
 
  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %5, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "shrb %%cl, %1         \n\t"
    "pushfl                \n\t"
    "popl   %0"
    : "=r" (eflags), "=mq" (op1)
    : "1" (op1), "c" (count), "0" (eflags), "g" (arithmetic_eflags)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg8(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_byte(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
SAR_Eb(vm_t *vm)
{
  Bit8u op1;
  unsigned count;
  Bit32u eflags, arithmetic_eflags;
 
  if (vm->i.b1 == 0xc0)
    count = vm->i.Ib & 0x1f;
  else if (vm->i.b1 == 0xd0)
    count = 1;
  else /* 0xd2 */
    count = G_CL(vm) & 0x1f;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  if (!count) return;
 
  arithmetic_eflags = read_eflags(vm) & 0x000008d5;
  eflags = 0; /* keep compiler quiet */
 
  asm volatile (
    /* First copy guest arithmetic flags to monitor flags */
    "pushfl                \n\t" /* Get mon flags */
    "popl   %0             \n\t"
    "andl  $0xfffff700, %0 \n\t" /* Erase mon arithmetic flags */
    "orl   %5, %0          \n\t" /* Or in guest arithmetic flags */
    "pushl %0              \n\t" /* Set flags in mon */
    "popfl                 \n\t"
    "sarb %%cl, %1         \n\t"
    "pushfl                \n\t"
    "popl   %0"
    : "=r" (eflags), "=mq" (op1)
    : "1" (op1), "c" (count), "0" (eflags), "g" (arithmetic_eflags)
    : "memory", "cc"
    );

  if (vm->i.mod == 0xc0) {
    WriteReg8(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_byte(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}
