/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2000 Kevin P. Lawton
 *
 *  string.c:  string oriented instructions
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "monitor.h"



  void
SCASB_ALXb(vm_t *vm)
{
  Bit8u op1, op2;
  Bit32u eflags;
 
  if (vm->i.as_32) {
    Bit32u edi;
 
    edi = G_EDI(vm);
 
    op1 = G_AL(vm);
    read_virtual_byte(vm, SRegES, edi, &op2);
 
    asm volatile (
      "cmpb %2, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags)
      : "r" (op1), "g" (op2)
      : "memory", "cc"
      );
 
    SetOSZAPC(vm, eflags);
 
    if (G_GetDF(vm)) {
      edi--;
      }
    else {
      edi++;
      }
 
    G_EDI(vm) = edi;
    }
  else {
    Bit16u di;
 
    di = G_DI(vm);
 
    op1 = G_AL(vm);
    read_virtual_byte(vm, SRegES, di, &op2);
 
    asm volatile (
      "cmpb %2, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags)
      : "r" (op1), "g" (op2)
      : "memory", "cc"
      );
 
    SetOSZAPC(vm, eflags);
 
    if (G_GetDF(vm)) {
      di--;
      }
    else {
      di++;
      }
 
    G_DI(vm) = di;
    }
}


  void
SCASW_eAXXv(vm_t *vm)
{
  Bit32u eflags;

  if (vm->i.as_32) {
    Bit32u edi;

    edi = G_EDI(vm);

    if (vm->i.os_32) {
      Bit32u op1, op2;

      op1 = G_EAX(vm);
      read_virtual_dword(vm, SRegES, edi, &op2);

      asm volatile (
        "cmpl %2, %1\n\t"
        "pushfl     \n\t"
        "popl %0"
        : "=g" (eflags)
        : "r" (op1), "g" (op2)
        : "memory", "cc"
        );
 
      SetOSZAPC(vm, eflags);

      if (G_GetDF(vm)) {
        edi -= 4;
        }
      else {
        edi += 4;
        }
      }
    else { /* 16 bit opsize */
      Bit16u op1, op2;

      op1 = G_AX(vm);
      read_virtual_word(vm, SRegES, edi, &op2);

      asm volatile (
        "cmpw %2, %1\n\t"
        "pushfl     \n\t"
        "popl %0"
        : "=g" (eflags)
        : "r" (op1), "g" (op2)
        : "memory", "cc"
        );

      SetOSZAPC(vm, eflags);

      if (G_GetDF(vm)) {
        edi -= 2;
        }
      else {
        edi += 2;
        }
      }

    G_EDI(vm) = edi;
    }
  else {
    Bit16u di;

    di = G_DI(vm);

    if (vm->i.os_32) {
      Bit32u op1, op2;

      op1 = G_EAX(vm);
      read_virtual_dword(vm, SRegES, di, &op2);

      asm volatile (
        "cmpl %2, %1\n\t"
        "pushfl     \n\t"
        "popl %0"
        : "=g" (eflags)
        : "r" (op1), "g" (op2)
        : "memory", "cc"
        );
 
      SetOSZAPC(vm, eflags);

      if (G_GetDF(vm)) {
        di -= 4;
        }
      else {
        di += 4;
        }
      }
    else {
      Bit16u op1, op2;

      op1 = G_AX(vm);
      read_virtual_word(vm, SRegES, di, &op2);

      asm volatile (
        "cmpw %2, %1\n\t"
        "pushfl     \n\t"
        "popl %0"
        : "=g" (eflags)
        : "r" (op1), "g" (op2)
        : "memory", "cc"
        );

      SetOSZAPC(vm, eflags);

      if (G_GetDF(vm)) {
        di -= 2;
        }
      else {
        di += 2;
        }
      }

    G_DI(vm) = di;
    }
}


  void
CMPSB_XbYb(vm_t *vm)
{
  unsigned seg;
  Bit8u op1, op2;
  Bit32u eflags;

  if (!NULL_SEG_REG(vm->i.seg)) {
    seg = vm->i.seg;
    }
  else {
    seg = SRegDS;
    }

  if (vm->i.as_32) {
    Bit32u esi, edi;

    esi = G_ESI(vm);
    edi = G_EDI(vm);

    read_virtual_byte(vm, seg, esi, &op1);
    read_virtual_byte(vm, SRegES, edi, &op2);

    asm volatile (
      "cmpb %2, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags)
      : "r" (op1), "g" (op2)
      : "memory", "cc"
      );

    SetOSZAPC(vm, eflags);

    if (G_GetDF(vm)) {
      /* decrement ESI */
      esi--;
      edi--;
      }
    else {
      /* increment ESI */
      esi++;
      edi++;
      }

    G_EDI(vm) = edi;
    G_ESI(vm) = esi;
    }
  else {
    Bit16u si, di;

    si = G_SI(vm);
    di = G_DI(vm);

    read_virtual_byte(vm, seg, si, &op1);
    read_virtual_byte(vm, SRegES, di, &op2);

    asm volatile (
      "cmpb %2, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags)
      : "r" (op1), "g" (op2)
      : "memory", "cc"
      );

    SetOSZAPC(vm, eflags);

    if (G_GetDF(vm)) {
      /* decrement ESI */
      si--;
      di--;
      }
    else {
      /* increment ESI */
      si++;
      di++;
      }

    G_DI(vm) = di;
    G_SI(vm) = si;
    }
}

  void
CMPSW_XvYv(vm_t *vm)
{
  unsigned seg;
  Bit32u eflags;

  if (!NULL_SEG_REG(vm->i.seg)) {
    seg = vm->i.seg;
    }
  else {
    seg = SRegDS;
    }

  if (vm->i.as_32) {
    Bit32u esi, edi;

    esi = G_ESI(vm);
    edi = G_EDI(vm);

    if (vm->i.os_32) {
      Bit32u op1, op2;

      read_virtual_dword(vm, seg, esi, &op1);
      read_virtual_dword(vm, SRegES, edi, &op2);

      asm volatile (
        "cmp  %2, %1\n\t"
        "pushfl     \n\t"
        "popl %0"
        : "=g" (eflags)
        : "r" (op1), "g" (op2)
        : "memory", "cc"
        );

      SetOSZAPC(vm, eflags);

      if (G_GetDF(vm)) {
        esi -= 4;
        edi -= 4;
        }
      else {
        esi += 4;
        edi += 4;
        }
      }
    else { /* 16 bit opsize */
      Bit16u op1, op2;

      read_virtual_word(vm, seg, esi, &op1);
      read_virtual_word(vm, SRegES, edi, &op2);

      asm volatile (
        "cmpw %2, %1\n\t"
        "pushfl     \n\t"
        "popl %0"
        : "=g" (eflags)
        : "r" (op1), "g" (op2)
        : "memory", "cc"
        );

      SetOSZAPC(vm, eflags);

      if (G_GetDF(vm)) {
        esi -= 2;
        edi -= 2;
        }
      else {
        esi += 2;
        edi += 2;
        }
      }


    G_EDI(vm) = edi;
    G_ESI(vm) = esi;
    }
  else { /* 16 bit address mode */
    Bit16u si, di;

    si = G_SI(vm);
    di = G_DI(vm);

    if (vm->i.os_32) {
      Bit32u op1, op2;

      read_virtual_dword(vm, seg, si, &op1);
      read_virtual_dword(vm, SRegES, di, &op2);

      asm volatile (
        "cmp  %2, %1\n\t"
        "pushfl     \n\t"
        "popl %0"
        : "=g" (eflags)
        : "r" (op1), "g" (op2)
        : "memory", "cc"
        );

      SetOSZAPC(vm, eflags);

      if (G_GetDF(vm)) {
        si -= 4;
        di -= 4;
        }
      else {
        si += 4;
        di += 4;
        }
      }
    else { /* 16 bit opsize */
      Bit16u op1, op2;

      read_virtual_word(vm, seg, si, &op1);
      read_virtual_word(vm, SRegES, di, &op2);

      asm volatile (
        "cmpw %2, %1\n\t"
        "pushfl     \n\t"
        "popl %0"
        : "=g" (eflags)
        : "r" (op1), "g" (op2)
        : "memory", "cc"
        );

      SetOSZAPC(vm, eflags);

      if (G_GetDF(vm)) {
        si -= 2;
        di -= 2;
        }
      else {
        si += 2;
        di += 2;
        }
      }

    G_DI(vm) = di;
    G_SI(vm) = si;
    }
}

  void
MOVSB_XbYb(vm_t *vm)
{
  unsigned seg;
  Bit8u temp8;

  if (!NULL_SEG_REG(vm->i.seg)) {
    seg = vm->i.seg;
    }
  else {
    seg = SRegDS;
    }

  if (vm->i.as_32) {
    Bit32u esi, edi;

    esi = G_ESI(vm);
    edi = G_EDI(vm);

    read_virtual_byte(vm, seg, esi, &temp8);

    write_virtual_byte(vm, SRegES, edi, &temp8);

    if (G_GetDF(vm)) {
      /* decrement ESI, EDI */
      esi--;
      edi--;
      }
    else {
      /* increment ESI, EDI */
      esi++;
      edi++;
      }

    G_ESI(vm) = esi;
    G_EDI(vm) = edi;
    }

  else {
    Bit16u si, di;

    si = G_SI(vm);
    di = G_DI(vm);

    read_virtual_byte(vm, seg, si, &temp8);

    write_virtual_byte(vm, SRegES, di, &temp8);

    if (G_GetDF(vm)) {
      /* decrement SI, DI */
      si--;
      di--;
      }
    else {
      /* increment SI, DI */
      si++;
      di++;
      }

    G_SI(vm) = si;
    G_DI(vm) = di;
    }
}

  void
MOVSW_XvYv(vm_t *vm)
{
  unsigned seg;

  if (!NULL_SEG_REG(vm->i.seg)) {
    seg = vm->i.seg;
    }
  else {
    seg = SRegDS;
    }

  if (vm->i.as_32) {
    Bit32u  temp32;

    Bit32u esi, edi;

    esi = G_ESI(vm);
    edi = G_EDI(vm);

    if (vm->i.os_32) {
      read_virtual_dword(vm, seg, esi, &temp32);

      write_virtual_dword(vm, SRegES, edi, &temp32);

      if (G_GetDF(vm)) {
        esi -= 4;
        edi -= 4;
        }
      else {
        esi += 4;
        edi += 4;
        }
      }
    else {
      Bit16u  temp16;

      read_virtual_word(vm, seg, esi, &temp16);

      write_virtual_word(vm, SRegES, edi, &temp16);

      if (G_GetDF(vm)) {
        esi -= 2;
        edi -= 2;
        }
      else {
        esi += 2;
        edi += 2;
        }
      }

    G_ESI(vm) = esi;
    G_EDI(vm) = edi;
    }
  else {
    Bit16u si, di;

    si = G_SI(vm);
    di = G_DI(vm);

    if (vm->i.os_32) {
      Bit32u temp32;

      read_virtual_dword(vm, seg, si, &temp32);

      write_virtual_dword(vm, SRegES, di, &temp32);

      if (G_GetDF(vm)) {
        si -= 4;
        di -= 4;
        }
      else {
        si += 4;
        di += 4;
        }
      }
    else {
      Bit16u  temp16;

      read_virtual_word(vm, seg, si, &temp16);

      write_virtual_word(vm, SRegES, di, &temp16);

      if (G_GetDF(vm)) {
        si -= 2;
        di -= 2;
        }
      else {
        si += 2;
        di += 2;
        }
      }

    G_SI(vm) = si;
    G_DI(vm) = di;
    }
}

  void
LODSB_ALXb(vm_t *vm)
{
  unsigned seg;
  Bit8u al;
 
  if (!NULL_SEG_REG(vm->i.seg)) {
    seg = vm->i.seg;
    }
  else {
    seg = SRegDS;
    }
 
  if (vm->i.as_32) {
    Bit32u esi;
 
    esi = G_ESI(vm);
 
    read_virtual_byte(vm, seg, esi, &al);
 
    G_AL(vm) = al;
    if (G_GetDF(vm)) {
      /* decrement ESI */
      esi--;
      }
    else {
      /* increment ESI */
      esi++;
      }
 
    G_ESI(vm) = esi;
    }
  else { /* 16bit address mode */
    Bit16u si;
 
    si = G_SI(vm);
 
    read_virtual_byte(vm, seg, si, &al);
 
    G_AL(vm) = al;
    if (G_GetDF(vm)) {
      /* decrement ESI */
      si--;
      }
    else {
      /* increment ESI */
      si++;
      }
 
    G_SI(vm) = si;
    }
}

  void
LODSW_eAXXv(vm_t *vm)
{
  unsigned seg;

  if (!NULL_SEG_REG(vm->i.seg)) {
    seg = vm->i.seg;
    }
  else {
    seg = SRegDS;
    }

  if (vm->i.as_32) {
    Bit32u esi;

    esi = G_ESI(vm);

    if (vm->i.os_32) {
      Bit32u eax;

      read_virtual_dword(vm, seg, esi, &eax);

      G_EAX(vm) = eax;
      if (G_GetDF(vm)) {
        esi -= 4;
        }
      else {
        esi += 4;
        }
      }
    else {
      Bit16u ax;
      read_virtual_word(vm, seg, esi, &ax);

      G_AX(vm) = ax;
      if (G_GetDF(vm)) {
        esi -= 2;
        }
      else {
        esi += 2;
        }
      }

    G_ESI(vm) = esi;
    }
  else {
    Bit16u si;

    si = G_SI(vm);

    if (vm->i.os_32) {
      Bit32u eax;

      read_virtual_dword(vm, seg, si, &eax);

      G_EAX(vm) = eax;
      if (G_GetDF(vm)) {
        si -= 4;
        }
      else {
        si += 4;
        }
      }
    else {
      Bit16u ax;

      read_virtual_word(vm, seg, si, &ax);

      G_AX(vm) = ax;
      if (G_GetDF(vm)) {
        si -= 2;
        }
      else {
        si += 2;
        }
      }

    G_SI(vm) = si;
    }
}

  void
STOSW_YveAX(vm_t *vm)
{
  if (vm->i.as_32) {
    Bit32u edi;

    edi = G_EDI(vm);

    if (vm->i.os_32) {
        Bit32u eax;

        eax = G_EAX(vm);
        write_virtual_dword(vm, SRegES, edi, &eax);

        if (G_GetDF(vm)) {
          edi -= 4;
          }
        else {
          edi += 4;
          }
      }
    else {
        Bit16u ax;

        ax = G_AX(vm);
        write_virtual_word(vm, SRegES, edi, &ax);

        if (G_GetDF(vm)) {
          edi -= 2;
          }
        else {
          edi += 2;
          }
      }

    G_EDI(vm) = edi;
    }
  else {
    Bit16u di;

    di = G_DI(vm);

    if (vm->i.os_32) {
        Bit32u eax;

        eax = G_EAX(vm);
        write_virtual_dword(vm, SRegES, di, &eax);

        if (G_GetDF(vm)) {
          di -= 4;
          }
        else {
          di += 4;
          }
      }
    else {
        Bit16u ax;

        ax = G_AX(vm);
        write_virtual_word(vm, SRegES, di, &ax);

        if (G_GetDF(vm)) {
          di -= 2;
          }
        else {
          di += 2;
          }
      }

    G_DI(vm) = di;
    }
}

  void
STOSB_YbAL(vm_t *vm)
{
  Bit8u al;
 
  if (vm->i.as_32) {
    Bit32u edi;
 
    edi = G_EDI(vm);
 
    al = G_AL(vm);
    write_virtual_byte(vm, SRegES, edi, &al);
 
    if (G_GetDF(vm)) {
      edi--;
      }
    else {
      edi++;
      }
 
    G_EDI(vm) = edi;
    }
  else {
    Bit16u di;
 
    di = G_DI(vm);
 
    al = G_AL(vm);
    write_virtual_byte(vm, SRegES, di, &al);
 
    if (G_GetDF(vm)) {
      di--;
      }
    else {
      di++;
      }
 
    G_DI(vm) = di;
    }
}
