/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2000 Kevin P. Lawton
 *
 *  host-all.c:  Non-host-specific components of kernel module code.
 *               Placed here to abstract as much code as possible
 *               from the host-specific kernel module.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#define IN_HOST_SPACE
#include "monitor.h"

cpuid_info_t cpuid_info;


  void
ioctlSetIntr(vm_t *vm, unsigned long intr)
{
    vm->system.intr = intr;
    vm->guest_cpu.async_event = 1;
}

  int
ioctlSetA20E(vm_t *vm, unsigned long val)
{
  unsigned prev_a20;

  if ( !val && vm->guest_cpu.cr0.fields.pg ) {
    /* Disabling the A20 line with paging on, is not
     * currently handled
     */
    monprint(vm, "SetA20E: val=0, CR0.PG=1 unhandled\n");
    return 0; /* fail */
    }
  val = (val > 0); /* make 0 or 1 */
  prev_a20 = vm->system.a20;

  vm->system.a20 = val;
  vm->system.a20AddrMask  = 0xffefffff | (val << 20);
  vm->system.a20IndexMask = 0x000ffeff | (val << 8);

  if ( prev_a20 != vm->system.a20 ) {
    /* change in A20 line enable status */
    vm->modeChange |= ModeChangePaging;
    }

  return 1; /* OK */
}


  int
ioctlMessageQ(vm_t *vm, vm_messages_t *user_msgs)
{
    if ( vm->mon_state != MON_STATE_RUNNABLE )
    {
        monprint(vm, "ioctlMessageQ failed: not runnable state\n");
        goto panic;
    }

    switch (user_msgs->header.msg_type) 
    {
    case VMMessageIOInResponse:
    {
        IO_msg_t *user_io_msg = (IO_msg_t *) user_msgs->msg;
        IO_msg_t *mon_io_msg  = (IO_msg_t *) vm->mon_msgs.msg;

        /* Make sure user response is valid for monitor request */
        if ( (vm->mon_request != MON_REQ_RESPONSE_PENDING) ||
             (vm->mon_msgs.header.msg_type != VMMessageIOInRequest) ||
             (user_io_msg->port != mon_io_msg->port) ||
             (user_io_msg->len != mon_io_msg->len) ||
             (user_io_msg->op != mon_io_msg->op) ) 
        {
            monprint(vm, "plex86: IOResponse mismatch.\n");
            goto panic;
        }

        mon_io_msg->data = user_io_msg->data;
        vm->mon_request = MON_REQ_NONE; /* request is satisfied */
        if (run_guest_loop(vm)) 
        {
            monprint(vm, "run_guest failed after IOInResponse\n");
            goto panic;
        }
        break;
    }

    case VMMessageIOInBatchResponse:
    {
        IOBatch_msg_t *user_io_msg = (IOBatch_msg_t *) user_msgs->msg;
        IOBatch_msg_t *mon_io_msg  = (IOBatch_msg_t *) vm->mon_msgs.msg;

        /* Make sure user response is valid for monitor request */
        if ( (vm->mon_request != MON_REQ_RESPONSE_PENDING) ||
             (vm->mon_msgs.header.msg_type != VMMessageIOInBatchRequest) ||
             (user_io_msg->port != mon_io_msg->port) ||
             (user_io_msg->len != mon_io_msg->len) ||
             (user_io_msg->op != mon_io_msg->op) ) 
        {
            monprint(vm, "plex86: IOBatchResponse mismatch.\n");
            goto panic;
        }

        vm->mon_request = MON_REQ_NONE; /* request is satisfied */
        if (run_guest_loop(vm)) 
        {
            monprint(vm, "run_guest failed after IOInResponse\n");
            goto panic;
        }
        break;
    }

    case VMMessageMemMapIOReadResponse:
    {
        memMapIO_msg_t *user_msg = (memMapIO_msg_t *) user_msgs->msg;
        memMapIO_msg_t *mon_msg  = (memMapIO_msg_t *) vm->mon_msgs.msg;
 
        /* Make sure user response is valid for monitor request */
        if ( (vm->mon_request != MON_REQ_RESPONSE_PENDING) ||
             (vm->mon_msgs.header.msg_type != VMMessageMemMapIOReadRequest) ||
             (user_msg->addr != mon_msg->addr) ||
             (user_msg->len != mon_msg->len) ||
             (user_msg->op != mon_msg->op) )
        {
            monprint(vm, "plex86: MemMapIOReadResponse mismatch.\n");
            goto panic;
        }
 
        mon_msg->data = user_msg->data;
        vm->mon_request = MON_REQ_NONE; /* request is satisfied */
        if (run_guest_loop(vm))
        {
            monprint(vm, "run_guest failed after MemMapIOReadResponse\n");
            goto panic;
        }
        break;
    }

    case VMMessageIACResponse:
    {
        IAC_msg_t *user_iac_msg = (IAC_msg_t *) user_msgs->msg;
        IAC_msg_t *mon_iac_msg  = (IAC_msg_t *) vm->mon_msgs.msg;

        /* Make sure user response is valid for monitor request */
        if ( (vm->mon_request != MON_REQ_RESPONSE_PENDING) ||
             (vm->mon_msgs.header.msg_type != VMMessageIACRequest) ) 
        {
            monprint(vm, "plex86: IACResponse mismatch.\n");
            goto panic;
        }

        mon_iac_msg->vector = user_iac_msg->vector;
        vm->mon_request = MON_REQ_NONE; /* request is satisfied */
        if (run_guest_loop(vm)) 
        {
            monprint(vm, "run_guest failed after IACResponse\n");
            goto panic;
        }
        break;
    }

    case VMMessageIntResponse:
    {
        INT_msg_t *user_int_msg = (INT_msg_t *) user_msgs->msg;
        INT_msg_t *mon_int_msg  = (INT_msg_t *) vm->mon_msgs.msg;

        /* Make sure user response is valid for monitor request */
        if ( (vm->mon_request != MON_REQ_RESPONSE_PENDING) ||
             (vm->mon_msgs.header.msg_type != VMMessageIntRequest) ||
             (user_int_msg->vector != mon_int_msg->vector) )
        {
            monprint(vm, "plex86: IntResponse mismatch.\n");
            goto panic;
        }

        mon_int_msg->reflect = user_int_msg->reflect;
        vm->mon_request = MON_REQ_NONE; /* request is satisfied */
        if (run_guest_loop(vm)) 
        {
            monprint(vm, "run_guest failed after IntResponse\n");
            goto panic;
        }
        break;
    }

    case VMMessageRunGuestN:
        {
        run_guest_n_t *msg_p;
        if ( (vm->mon_request != MON_REQ_NONE) ||
             (vm->mon_state != MON_STATE_RUNNABLE) )
        {
            monprint(vm, "plex86: RunGuest: not runnable state (%u)\n",
                         vm->mon_state);
            goto panic;
        }

        ClearMonMessageQ(vm);

        msg_p = (run_guest_n_t *) user_msgs->msg;

        switch (msg_p->method) {
          case RunGuestNMethodExecute:
            /* Normal execution; run indefinitely.  As we will run guest
             * code natively (but under SBE control), we don't have
             * enough control to stop execution exactly after a certain
             * number of instructions.  Thus, a count of
             * ICOUNT_INDEFINITE must be passed.
             */
            if (msg_p->icount != ICOUNT_INDEFINITE) {
              monprint(vm, "RunGuestN: MethodExecute w/ icount!=indef\n");
              goto panic;
              }
            vm->executeN = 0;
            vm->executeMethod = msg_p->method;
            break;

          case RunGuestNMethodEmulate:
          case RunGuestNMethodBreakpoint:
            /* Execute guest code for N instructions. */
            if (msg_p->icount == ICOUNT_INDEFINITE) {
              /* We can use these methods to run code indefinitely,
               * but execution will be very slow comparatively.
               * Usually, this is for debugging when you don't know
               * how long before some interesting event occurs.
               */
              vm->executeN = 0;
              }
            else if (msg_p->icount == ICOUNT_CONTINUE) {
              /* Continue executing, using the current icount
               * value.  This is for the case when execution was
               * interrupted to service something (like IO) and
               * control was transferred back to the host temporarily.
               */
              if (msg_p->icount == 0) {
                monprint(vm, "ICOUNT_CONTINUE w/ icount=0\n");
                goto panic;
                }
              if (msg_p->method != vm->executeMethod) {
                monprint(vm, "RunGuestN: continue: different method.\n");
                goto panic;
                }
              /* Use existing vm->executeN value */
              }
            else {
              vm->executeN = msg_p->icount;
              }
            vm->executeMethod = msg_p->method;
            break;

          default:
            monprint(vm, "RunGuestN: unknown method %u\n", msg_p->method);
            goto panic;
          }

        if (run_guest_loop(vm)) 
        {
            monprint(vm, "run_guest failed after RunGuest\n");
            goto panic;
        }
        break;
        }

    default:
        monprint(vm, "run_guest_loop: unknown message type (%u)\n",
                 user_msgs->header.msg_type);
        return 1;
    }


    /* Copy monitor message to user space */
    copy_memory(user_msgs, &vm->mon_msgs, 
                vm->mon_msgs.header.msg_len + sizeof(vm->mon_msgs.header));

    /*if (vm->log_buffer_info.event && !vm->log_buffer_info.locked)
     *    host_print_buf(vm); */
    return 0;


 panic:
    vm->mon_request = MON_REQ_PANIC;
    vm->mon_state   = MON_STATE_PANIC;
    vm->mon_msgs.header.msg_type = VMMessagePanic;
    vm->mon_msgs.header.msg_len = 0;

    copy_memory(user_msgs, &vm->mon_msgs, 
                vm->mon_msgs.header.msg_len + sizeof(vm->mon_msgs.header));

    /*if (vm->log_buffer_info.event && !vm->log_buffer_info.locked)
     *    host_print_buf(vm); */
    return 0;
}


  void
get_guest_cpu_state(vm_t *vm, guest_cpu_t *cpu)
{
  /* Get complete guest CPU state */
  guest_context_t *context;

  cache_sreg(vm, SRegCS);
  cache_sreg(vm, SRegSS);
  cache_sreg(vm, SRegDS);
  cache_sreg(vm, SRegES);
  cache_sreg(vm, SRegFS);
  cache_sreg(vm, SRegGS);

  context = vm->host.addr.guest_context;

  cpu->eax = context->eax;
  cpu->ebx = context->ebx;
  cpu->ecx = context->ecx;
  cpu->edx = context->edx;
  cpu->ebp = context->ebp;
  cpu->esi = context->esi;
  cpu->edi = context->edi;
  cpu->esp = context->esp;
  cpu->eflags = read_eflags(vm);
  cpu->eip = context->eip;
  if (vm->guest_cpu.prev_eip != -1) {
    /* Rewind EIP,ESP back to beginning of instruction so that */
    /* debug disassembly will show correct instruction. */
    cpu->eip = vm->guest_cpu.prev_eip;
    cpu->esp = vm->guest_cpu.prev_esp;
    }

  cpu->cs.sel.raw = vm->guest_cpu.selector[SRegCS].raw;
  cpu->cs.des = vm->guest_cpu.desc_cache[SRegCS].desc;
  cpu->cs.valid = vm->guest_cpu.desc_cache[SRegCS].valid;

  cpu->ss.sel.raw = vm->guest_cpu.selector[SRegSS].raw;
  cpu->ss.des = vm->guest_cpu.desc_cache[SRegSS].desc;
  cpu->ss.valid = vm->guest_cpu.desc_cache[SRegSS].valid;

  cpu->ds.sel.raw = vm->guest_cpu.selector[SRegDS].raw;
  cpu->ds.des = vm->guest_cpu.desc_cache[SRegDS].desc;
  cpu->ds.valid = vm->guest_cpu.desc_cache[SRegDS].valid;

  cpu->es.sel.raw = vm->guest_cpu.selector[SRegES].raw;
  cpu->es.des = vm->guest_cpu.desc_cache[SRegES].desc;
  cpu->es.valid = vm->guest_cpu.desc_cache[SRegES].valid;

  cpu->fs.sel.raw = vm->guest_cpu.selector[SRegFS].raw;
  cpu->fs.des = vm->guest_cpu.desc_cache[SRegFS].desc;
  cpu->fs.valid = vm->guest_cpu.desc_cache[SRegFS].valid;

  cpu->gs.sel.raw = vm->guest_cpu.selector[SRegGS].raw;
  cpu->gs.des = vm->guest_cpu.desc_cache[SRegGS].desc;
  cpu->gs.valid = vm->guest_cpu.desc_cache[SRegGS].valid;

  cpu->ldtr.sel = vm->guest_cpu.ldtr_selector;
  cpu->ldtr.des = vm->guest_cpu.ldtr_cache.desc;
  cpu->ldtr.valid = vm->guest_cpu.ldtr_cache.valid;

  cpu->tr.sel = vm->guest_cpu.tr_selector;
  cpu->tr.des = vm->guest_cpu.tr_cache.desc;
  cpu->tr.valid = vm->guest_cpu.tr_cache.valid;

  cpu->gdtr.base = vm->guest_cpu.gdtr.base;
  cpu->gdtr.limit = vm->guest_cpu.gdtr.limit;

  cpu->idtr.base = vm->guest_cpu.idtr.base;
  cpu->idtr.limit = vm->guest_cpu.idtr.limit;

  cpu->dr0 = vm->guest_cpu.dr0;
  cpu->dr1 = vm->guest_cpu.dr1;
  cpu->dr2 = vm->guest_cpu.dr2;
  cpu->dr3 = vm->guest_cpu.dr3;
  cpu->dr6 = vm->guest_cpu.dr6;
  cpu->dr7 = vm->guest_cpu.dr7;

  cpu->tr3 = vm->guest_cpu.tr3;
  cpu->tr4 = vm->guest_cpu.tr4;
  cpu->tr5 = vm->guest_cpu.tr5;
  cpu->tr6 = vm->guest_cpu.tr6;
  cpu->tr7 = vm->guest_cpu.tr7;

  cpu->cr0 = vm->guest_cpu.cr0.raw;
  cpu->cr1 = vm->guest_cpu.cr1;
  cpu->cr2 = vm->guest_cpu.cr2;
  cpu->cr3 = vm->guest_cpu.cr3;
  cpu->cr4 = vm->guest_cpu.cr4.raw;

  cpu->inhibit_mask = vm->guest_cpu.inhibit_mask;
}


/* */
/* Allocate various pages/memory needed by monitor */
/* */

  int
alloc_vm_pages( vm_t *vm, unsigned nmegs )
{
    vm_pages_t *pg = &vm->pages;
    vm_addr_t  *ad = &vm->host.addr;
    unsigned where = 0xffffffff;

    /* clear out allocated pages lists */
    zero_memory(pg, sizeof(*pg));
    zero_memory(ad, sizeof(*ad));


    /* Guest physical memory pages */
    pg->guest_n_megs  = nmegs;
    pg->guest_n_pages = nmegs * 256;
    pg->guest_n_bytes = nmegs * 1024 * 1024;

    if ( !(ad->guest = host_alloc(pg->guest_n_megs * 1024 * 1024)) ) {
      where = 1;
      goto error;
      }
    if (!host_map(pg->guest, MON_GUEST_PAGES, 
             ad->guest, pg->guest_n_megs * 1024 * 1024) ) {
      where = 2;
      goto error;
      }

    /* Monitor page directory */
    if ( !(ad->page_dir = (pageEntry_t *)host_alloc_page()) ) {
      where = 3;
      goto error;
      }
    if (!(pg->page_dir = host_map_page(ad->page_dir))) {
      where = 4;
      goto error;
      }

    /* Monitor page tables */
    if ( !(ad->page_tbl = host_alloc(4096 * MON_PAGE_TABLES)) ) {
      where = 5;
      goto error;
      }
    if (!host_map(pg->page_tbl, MON_PAGE_TABLES, 
             ad->page_tbl, 4096 * MON_PAGE_TABLES)) {
      where = 6;
      goto error;
      }

    /* Map of the linear addresses of page tables currently */
    /* mapped into the monitor space. */
    if ( !(ad->page_tbl_laddr_map = (unsigned *)host_alloc_page()) ) {
      where = 7;
      goto error;
      }
    if ( !(pg->page_tbl_laddr_map = host_map_page(ad->page_tbl_laddr_map)) ) {
      where = 8;
      goto error;
      }

    /* Nexus page table */
    if ( !(ad->nexus_page_tbl = (page_t *)host_alloc_page()) ) {
      where = 9;
      goto error;
      }
    if ( !(pg->nexus_page_tbl = host_map_page(ad->nexus_page_tbl)) ) {
      where = 10;
      goto error;
      }

    /* Transition page table */
    if ( !(ad->transition_PT = (page_t *)host_alloc_page()) ) {
      where = 11;
      goto error;
      }
    if ( !(pg->transition_PT = host_map_page(ad->transition_PT)) ) {
      where = 12;
      goto error;
      }

    if ( !(ad->log_buffer = host_alloc(4096 * LOG_BUFF_PAGES)) ) {
      where = 13;
      goto error;
      }
    if (!host_map(pg->log_buffer, LOG_BUFF_PAGES, 
             ad->log_buffer, 4096 * LOG_BUFF_PAGES)) {
      where = 14;
      goto error;
      }

    /* Nexus page */
    if ( !(ad->nexus = (nexus_t *)host_alloc_page()) ) {
      where = 15;
      goto error;
      }
    if ( !(pg->nexus = host_map_page(ad->nexus)) ) {
      where = 16;
      goto error;
      }

zero_memory(&vm->code_cache, sizeof(vm->code_cache));
/* Pages used for the code cache for scan-before-execute technique. */
/* No need to zero out these pages.  When a new meta page is used, it */
/* has to be zeroed on demand.  The virtualized page has to be copied */
/* from the unvirtualized page. +++ */

    /* Virtualized code pages for prescan technique */
    if ( !(ad->icache = host_alloc(ICACHE_SIZE)) ) {
      where = 17;
      goto error;
      }
    if (!host_map(pg->icache, ICACHE_PAGES, ad->icache, ICACHE_SIZE)) {
      where = 18;
      goto error;
      }

    /* Code page meta info for prescan technique */
    if ( !(ad->meta = host_alloc(ICACHE_SIZE)) ) {
      where = 19;
      goto error;
      }
    if (!host_map(pg->meta, ICACHE_PAGES, ad->meta, ICACHE_SIZE)) {
      where = 20;
      goto error;
      }

    /* Monitor IDT */
    if ( !(ad->idt = host_alloc(MON_IDT_SIZE)) ) {
      where = 21;
      goto error;
      }
    if (!host_map(pg->idt, MON_IDT_PAGES, ad->idt, MON_IDT_SIZE)) {
      where = 22;
      goto error;
      }

    /* Monitor GDT */
    if ( !(ad->gdt = host_alloc(MON_GDT_SIZE)) ) {
      where = 23;
      goto error;
      }
    if (!host_map(pg->gdt, MON_GDT_PAGES, ad->gdt, MON_GDT_SIZE)) {
      where = 24;
      goto error;
      }

    /* Monitor LDT */
    if ( !(ad->ldt = host_alloc(MON_LDT_SIZE)) ) {
      where = 25;
      goto error;
      }
    if (!host_map(pg->ldt, MON_LDT_PAGES, ad->ldt, MON_LDT_SIZE)) {
      where = 26;
      goto error;
      }

    /* Monitor TSS */
    if ( !(ad->tss = host_alloc(MON_TSS_SIZE)) ) {
      where = 27;
      goto error;
      }
    if (!host_map(pg->tss, MON_TSS_PAGES, ad->tss, MON_TSS_SIZE)) {
      where = 28;
      goto error;
      }

    /* Monitor IDT stubs */
    if ( !(ad->idt_stubs = host_alloc(MON_IDT_STUBS_SIZE)) ) {
      where = 29;
      goto error;
      }
    if (!host_map(pg->idt_stubs, MON_IDT_STUBS_PAGES, 
             ad->idt_stubs, MON_IDT_STUBS_SIZE)) {
      where = 30;
      goto error;
      }

    /* Get the physical pages associated with the vm_t structure. */
    if (!host_map(pg->vm, MAX_VM_STRUCT_PAGES, vm, sizeof(*vm))) {
      where = 31;
      goto error;
      }

    return 0;

 error:
    unalloc_vm_pages( vm );
    return where;
}


/* */
/* Unallocate pages/memory used by monitor */
/* */

  void
unalloc_vm_pages( vm_t *vm )
{
    vm_pages_t *pg = &vm->pages;
    vm_addr_t  *ad = &vm->host.addr;

    /* Guest physical memory pages */
    if (ad->guest) host_free(ad->guest);

    /* Monitor page directory */
    if (ad->page_dir) host_free_page(ad->page_dir);

    /* Monitor page tables */
    if (ad->page_tbl) host_free(ad->page_tbl);

    /* Map of linear addresses of page tables mapped into monitor. */
    if (ad->page_tbl_laddr_map) host_free_page(ad->page_tbl_laddr_map);

    /* Nexus page table */
    if (ad->nexus_page_tbl) host_free_page(ad->nexus_page_tbl);

    /* Transition page table */
    if (ad->transition_PT) host_free_page(ad->transition_PT);

    if (ad->log_buffer) host_free(ad->log_buffer);

    /* Nexus page */
    if (ad->nexus) host_free_page(ad->nexus);

    /* Code cache pages */
    if (ad->icache) host_free(ad->icache);
    if (ad->meta) host_free(ad->meta);

    /* Monitor IDT */
    if (ad->idt) host_free(ad->idt);

    /* Monitor GDT */
    if (ad->gdt) host_free(ad->gdt);

    /* Monitor LDT */
    if (ad->ldt) host_free(ad->ldt);

    /* Monitor TSS */
    if (ad->tss) host_free(ad->tss);

    /* Monitor IDT stubs */
    if (ad->idt_stubs) host_free(ad->idt_stubs);


    /* clear out allocated pages lists */
    zero_memory(pg, sizeof(*pg));
    zero_memory(ad, sizeof(*ad));
}

  unsigned
get_cpu_capabilities(void)
{
  Bit32u eax, ebx, ecx, edx;

  /* Get the highest allowed cpuid level */
  asm volatile (
    "xorl %%eax,%%eax\n\t"
    "cpuid"
    : "=a" (eax), "=b" (ebx), "=c" (ecx), "=d" (edx)
    :
    : "cc"
    );
  cpuid_info.maxval = eax;
  if (eax < 1)
    return(0); /* not enough capabilities */

  /* Copy vendor string */
  * (Bit32u*) &cpuid_info.vendorID[0] = ebx;
  * (Bit32u*) &cpuid_info.vendorID[4] = edx;
  * (Bit32u*) &cpuid_info.vendorID[8] = ecx;
  cpuid_info.vendorID[12] = 0;

  /* CPUID w/ EAX==1: Processor Signature & Feature Flags */
  asm volatile (
    "movl $1,%%eax\n\t"
    "cpuid"
    : "=a" (eax), "=b" (ebx), "=c" (ecx), "=d" (edx)
    :
    : "cc"
    );
  cpuid_info.procSignature.raw = eax;
  cpuid_info.featureFlags.raw = edx;
  /* Plex86 needs TSC */
  if (cpuid_info.featureFlags.fields.tsc==0)
    return(0);

#if 1
#warning "get_cpu_capabilites hacked to always return stripped Pentium"
  cpuid_info.procSignature.raw = 0;
  cpuid_info.featureFlags.raw = 0;
  cpuid_info.procSignature.fields.family = 5;
  cpuid_info.procSignature.fields.model = 1;
  cpuid_info.procSignature.fields.stepping = 3;
#else
#warning "get_cpu_capabilites returning host CPU values"
#endif

  return(1);
}
