/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2000  Kevin P. Lawton
 *
 *  system.c: The 'motherboard' logic which connects the entire
 *    PC system.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include "plex86.h"
#define IN_MONITOR_SPACE
#include "monitor.h"

#include "ioport.h"

static void call_timer_callbacks(vm_t *vm);


  Bit32u
sysIOIn(vm_t *vm, Bit32u port, unsigned len)
{
    IO_msg_t *io_msg = (IO_msg_t *) vm->mon_msgs.msg;
    Bit32u data;

    CLI();
    vm->mon_msgs.header.msg_type = VMMessageIOInRequest;
    vm->mon_msgs.header.msg_len  = sizeof(IO_msg_t);
    io_msg->port = port;
    io_msg->len  = len;
    io_msg->op   = IO_IN;

    vm->mon_request = MON_REQ_RESPONSE_PENDING;
    vm->guest.__mon2host();

    data = io_msg->data;
    ClearMonMessageQ(vm);
    STI();

    return data;
}

  unsigned
sysIOInBatch(vm_t *vm, Bit32u port, unsigned len, unsigned n, Bit32u paddr)
{
    IOBatch_msg_t *msg = (IOBatch_msg_t *) vm->mon_msgs.msg;
 
    CLI();
    vm->mon_msgs.header.msg_type = VMMessageIOInBatchRequest;
    vm->mon_msgs.header.msg_len  = sizeof(IOBatch_msg_t);
    msg->port  = port;
    msg->len   = len;
    msg->op    = IO_IN;
    msg->n     = n;
    msg->paddr = paddr;
 
    vm->mon_request = MON_REQ_RESPONSE_PENDING;
    vm->guest.__mon2host();
    n = msg->n;
 
    ClearMonMessageQ(vm);
    STI();

    return n;
}

  void
sysIOOut(vm_t *vm, Bit32u port, unsigned len, Bit32u data)
{
    IO_msg_t *io_msg = (IO_msg_t *) vm->mon_msgs.msg;

    CLI();
    vm->mon_msgs.header.msg_type = VMMessageIOOutRequest;
    vm->mon_msgs.header.msg_len  = sizeof(IO_msg_t);
    io_msg->port = port;
    io_msg->len  = len;
    io_msg->op   = IO_OUT;
    io_msg->data = data;

    vm->mon_request = MON_REQ_NO_RESPONSE;
    vm->guest.__mon2host();

    ClearMonMessageQ(vm);
    STI();
}

  void
sysMemMapIOWrite(vm_t *vm, Bit32u addr, unsigned len, Bit32u data)
{
    memMapIO_msg_t *memMapIO_msg = (memMapIO_msg_t *) vm->mon_msgs.msg;

    CLI();
    vm->mon_msgs.header.msg_type = VMMessageMemMapIOWriteRequest;
    vm->mon_msgs.header.msg_len  = sizeof(memMapIO_msg_t);
    memMapIO_msg->addr = addr;
    memMapIO_msg->len  = len;
    memMapIO_msg->op   = IO_OUT;
    memMapIO_msg->data = data;

    vm->mon_request = MON_REQ_NO_RESPONSE;
    vm->guest.__mon2host();

    ClearMonMessageQ(vm);
    STI();
}

  Bit32u
sysMemMapIORead(vm_t *vm, Bit32u addr, unsigned len)
{
    memMapIO_msg_t *memMapIO_msg = (memMapIO_msg_t *) vm->mon_msgs.msg;
    Bit32u data;

    CLI();
    vm->mon_msgs.header.msg_type = VMMessageMemMapIOReadRequest;
    vm->mon_msgs.header.msg_len  = sizeof(memMapIO_msg_t);

    memMapIO_msg->addr = addr;
    memMapIO_msg->len  = len;
    memMapIO_msg->op   = IO_IN;

    vm->mon_request = MON_REQ_RESPONSE_PENDING;
    vm->guest.__mon2host();

    data = memMapIO_msg->data;
    ClearMonMessageQ(vm);
    STI();

    return data;
}


  unsigned
sysIAC(vm_t *vm)
{
    IAC_msg_t *iac_msg = (IAC_msg_t *) vm->mon_msgs.msg;
    unsigned iac;

    CLI();
    vm->mon_msgs.header.msg_type = VMMessageIACRequest;
    vm->mon_msgs.header.msg_len  = 0;
  
    vm->mon_request = MON_REQ_RESPONSE_PENDING;
    vm->guest.__mon2host();

    iac = iac_msg->vector;
    ClearMonMessageQ(vm);
    STI();

    return iac;
}

  void
sysRaiseHLDA(vm_t *vm)
{
    /* IF handling ??? */
    monpanic(vm, "sysRaiseHLDA:\n");
}

  unsigned
sysReflectInt(vm_t *vm, unsigned vector)
{
    INT_msg_t *int_msg = (INT_msg_t *) vm->mon_msgs.msg;
    unsigned reflect;

monpanic(vm, "sysReflectInt:\n");
    CLI();
    vm->mon_msgs.header.msg_type = VMMessageIntRequest;
    vm->mon_msgs.header.msg_len  = sizeof(INT_msg_t);
    int_msg->vector = vector;

    vm->mon_request = MON_REQ_RESPONSE_PENDING;
    vm->guest.__mon2host();

    reflect = int_msg->reflect;
    ClearMonMessageQ(vm);
    STI();

    return reflect;
}

  unsigned
register_timer(vm_t *vm, unsigned space, void (*callback)(void *, unsigned),
  Bit64u cpu_cycles, unsigned continuous, unsigned active)
{
  unsigned i;

/* +++ */
return 0;

  /* Sanity checks. */
  if (vm->system.num_timers >= MAX_TIMERS) {
    monpanic(vm, "register_timer: too many registered timers.");
    }

  if (callback == (void (*)(void *, unsigned)) 0 )
    monpanic(vm, "register_timer: callback is NULL\n");

  i = vm->system.num_timers;
  vm->system.num_timers++;
  vm->system.timer[i].period     = cpu_cycles;
  vm->system.timer[i].remaining  = cpu_cycles;
  vm->system.timer[i].active     = active;
  vm->system.timer[i].continuous = continuous;
  vm->system.timer[i].triggered  = 0;
  vm->system.timer[i].callback   = callback;
  vm->system.timer[i].space      = space;

  if (active) {
    if (vm->system.cpu_cycles_in_period == 0) {
      /* no active timers */
      vm->system.cpu_cycles_in_period = cpu_cycles;
      vm->system.cpu_cycles_left      = cpu_cycles;
      }
    else {
      if (cpu_cycles < vm->system.cpu_cycles_left) {
        vm->system.cpu_cycles_in_period = cpu_cycles;
        vm->system.cpu_cycles_left      = cpu_cycles;
        }
      }
    }

  /* return timer id */
  return(i);
}

  void
guest_cycles_elapsed(vm_t *vm)
{
  /* If no active timers, then we don't care how much guest execution */
  /* time has elapsed. */
  /*if (vm->system.cpu_cycles_in_period == 0) */
  /*return; */

while ( vm->system.elapsed >= 400 ) {
  vm->system.elapsed -= 400;
  mycallback(vm, 400);
  }
return;



  while ( (vm->system.cpu_cycles_in_period != 0) &&
          (vm->system.elapsed >= vm->system.cpu_cycles_left) ) {
    vm->system.elapsed -= vm->system.cpu_cycles_left;
    call_timer_callbacks(vm);
    /* The above function will compute the cpu_cycles_left until */
    /* the next timing event is to be delivered. */
    }

  /* If there are still any active timers, we still have */
  /* to subtract off the remaining elapsed time. */
  if (vm->system.cpu_cycles_in_period != 0)
    vm->system.cpu_cycles_left -= vm->system.elapsed;
}

  void
call_timer_callbacks(vm_t *vm)
{
  /* A timing event is to be delivered on this timing boundary. */

  Bit64u min, elapsed;
  unsigned i;
  timer_t *timer = vm->system.timer;
 
  elapsed = vm->system.cpu_cycles_in_period;
 
  for (i=0; i < vm->system.num_timers; i++) {
    timer[i].triggered = 0; /* Mark not triggered intially */
    if (timer[i].active) {
      if (timer[i].remaining < elapsed) {
        monpanic(vm, "call_timer_callback: remain < period\n");
        }
      timer[i].remaining -= elapsed;
      if (timer[i].remaining == 0) {
        timer[i].triggered = 1;
        /* reset remaining period for triggered timer */
        timer[i].remaining = timer[i].period;
 
        /* if triggered timer is one-shot, deactive */
        if (timer[i].continuous==0)
          timer[i].active = 0;
        }
      }
    }
 
  /* Now compute the minimum time of all active timers, until */
  /* the requested timing event needs to be delivered. */
  min = (Bit64u) -1; /* max number in Bit64u range */
  for (i=0; i < vm->system.num_timers; i++) {
    if (timer[i].active && (timer[i].remaining < min))
      min = timer[i].remaining;
    }
  vm->system.cpu_cycles_in_period = min;
  vm->system.cpu_cycles_left = min;
 
  /* Now invoke all callbacks that were triggered at this */
  /* time boundary. */
  for (i=0; i < vm->system.num_timers; i++) {
    /* call requested timer function.  It may request a different */
    /* timer period or deactivate, all cases handled below */
    if (timer[i].triggered) {
      timer[i].callback(vm, i);
      }
    }
}

/* +++ if all disabled, then */
/*  vm->system.cpu_cycles_in_period = 0; */

  void
mycallback(void *voidptr, unsigned t)
{
  vm_t *vm = (vm_t *) voidptr;

  time_elapsed_t *time_elapsed = (time_elapsed_t *) vm->mon_msgs.msg;

  CLI();
  vm->mon_msgs.header.msg_type = VMMessageTimeElapsed;
  vm->mon_msgs.header.msg_len  = sizeof(time_elapsed_t);
  time_elapsed->elapsed = t;

  vm->mon_request = MON_REQ_NO_RESPONSE;
  vm->guest.__mon2host();

  ClearMonMessageQ(vm);
  STI();
}

  void
sysDisasm(vm_t *vm)
{
    CLI();
    vm->mon_msgs.header.msg_type = VMMessageDisasm;
    vm->mon_msgs.header.msg_len  = 0;
    vm->mon_request = MON_REQ_NO_RESPONSE;
    vm->guest.__mon2host();
 
    ClearMonMessageQ(vm);
    STI();
}

  void
sysFlushPrintBuf(vm_t *vm)
{
#if 0
    CLI();
    vm->mon_request = MON_REQ_FLUSH_PRINT_BUF;
    vm->guest.__mon2host();
    STI();
#endif

    CLI();
    vm->mon_msgs.header.msg_type = VMMessagePrintBuf;
    vm->mon_msgs.header.msg_len  = 0;
    vm->mon_request = MON_REQ_NO_RESPONSE;
    vm->guest.__mon2host();
 
    ClearMonMessageQ(vm);
    STI();
}

  void
sysEOICount(vm_t *vm)
{
    EOICount_t *msg = (EOICount_t *) vm->mon_msgs.msg;

    CLI();
    vm->mon_msgs.header.msg_type = VMMessageEOICount;
    vm->mon_msgs.header.msg_len  = sizeof(EOICount_t);
    cache_sreg(vm, SRegCS);
    msg->cs = vm->guest_cpu.selector[SRegCS].raw;
    msg->eip = G_EIP(vm);
    msg->laddr = vm->guest_cpu.desc_cache[SRegCS].base + G_EIP(vm);
    msg->seg32 = vm->guest_cpu.desc_cache[SRegCS].desc.d_b;
    vm->mon_request = MON_REQ_NO_RESPONSE;
    vm->guest.__mon2host();
 
    ClearMonMessageQ(vm);
    STI();
}

  void
sysReqComplete(vm_t *vm)
{
    CLI();
    vm->mon_msgs.header.msg_type = VMMessageReqComplete;
    vm->mon_msgs.header.msg_len  = 0;
    vm->mon_request = MON_REQ_NO_RESPONSE;
    vm->guest.__mon2host();
 
    ClearMonMessageQ(vm);
    STI();
}

  void
sysRemapMonitor(vm_t *vm)
{
    CLI();
    vm->mon_request = MON_REQ_REMAP_MONITOR;
    vm->guest.__mon2host();
    STI();
}
