/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2000  Kevin P. Lawton
 *
 *  nexus-mode.c: deals with which mode to run the guest CPU in,
 *    and mapping the monitor segments according to those modes.
 *    (features available to either space)
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#define IN_NEXUS_SPACE
#include "monitor.h"


const selector_t nullSelector = { raw: 0 };
const descriptor_t nullDescriptor = {
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  };



/* Given the values already in vm->guest_cpu, and the passed 'eflags',
 * map the monitor and guest into the VM.
 */

  unsigned
mapMonitor(vm_t *vm, Bit32u eflags)
{
    Bit16u mon_cs = 0, mon_ss = 0, mon_tss = 0;
    Bit32u laddr, base;
    int i;
    unsigned virtualizeSegRegs = 0;

    /* We first need to determine the guest mode, and the mode the
     * monitor will execute it in.
     */

/* +++ should take eflags into account */
    if (RealMode(vm)) {
      if (isV86MCompatible(vm)) {
        SetMonGuestMode(vm, MonModeVM, GuestModeRM);
        vm->vOpcodeMap = &vOpcodeMapV86;
        }
      else {
        SetMonGuestMode(vm, MonModePMR3, GuestModeRM);
        vm->vOpcodeMap = &vOpcodeMapStrongest;
        }
      }
    else if (ProtectedMode(vm)) {
      SetMonGuestMode(vm, MonModePMR3, GuestModePMR0 | G_GetCPL(vm));
      vm->vOpcodeMap = &vOpcodeMapStrongest;
      virtualizeSegRegs = 1;
      }
    else { /* Must be in V86M */
      SetMonGuestMode(vm, MonModeVM, GuestModeVM);
      vm->vOpcodeMap = &vOpcodeMapV86;
      }

    /* EFLAGS: virtualized guest flags are stored in veflags, */
    /*   the rest are stored in guest_context->eflags. */
    /* Start out by setting the virtualized flags (the ones */
    /* needed for the monitor). */
    vm->guest_cpu.veflags.raw = 0;
    vm->addr->guest_context->eflags = FLG_IF | 0x02;
    write_eflags(vm, eflags, ~0);

    /* Set the actual EFLAGS.VM flag according to the mode the
     * guest is monitor in.
     */
    if (GetMonMode(vm) == MonModeVM)
      vm->addr->guest_context->eflags |= FLG_VM;
    else
      vm->addr->guest_context->eflags &= ~FLG_VM;

    /* Assume that SBE is on.  With SBE on, we maintain 6 descriptors */
    /* which correspond to the shadow descriptor cache values which */
    /* normally would have been loaded into ES,CS,SS,DS,FS,GS by */
    /* instructions in the guest. */

    /* We also need to maintain two descriptors for the monitor's */
    /* code (CS) and data (ES,SS,DS,FS,GS) segments, and one for */
    /* the monitor's TSS segment. */

    if (!vm->vOpcodeMap) {
      monprint(vm, "mapMonitor: sbe is off\n");
      return(0);
      }

/* +++ should zero out GDT, so prev entries do not remain */
/* +++ do that in unmap_mon */

    /* monitor selectors */
    mon_cs  = Selector(1, 0, RPL0);
    mon_ss  = Selector(2, 0, RPL0);
    mon_tss = Selector(3, 0, RPL0);

    /*
     * Virtualized guest data selectors and shadow cache descriptors
     */
    if ( GetMonMode(vm) == MonModeVM ) {
      /* Guest VM/RM code running in monitor VM mode.  Just use guest
       * selectors as-is. */

      vm->addr->guest_context->es = vm->guest_cpu.selector[SRegES].raw;
      vm->addr->guest_context->cs = vm->guest_cpu.selector[SRegCS].raw;
      vm->addr->guest_context->ss = vm->guest_cpu.selector[SRegSS].raw;
      vm->addr->guest_context->ds = vm->guest_cpu.selector[SRegDS].raw;
      vm->addr->guest_context->fs = vm->guest_cpu.selector[SRegFS].raw;
      vm->addr->guest_context->gs = vm->guest_cpu.selector[SRegGS].raw;
      }
    else if (GetGuestMode(vm) == GuestModeRM) {
      for (i=0; i<6; i++) {
        if (vm->guest_cpu.desc_cache[i].valid) {
          vm->addr->gdt[4+i] = vm->guest_cpu.desc_cache[i].desc;
          vm->addr->gdt[4+i].dpl = 3;
          }
        else {
          vm->addr->gdt[4+i] = nullDescriptor;
          }
        }
      }
    else {
      /* ModModePMR3 */
      unsigned i;
  
  /* +++ unify with similar code in mon-fault.c */
      /* If the descriptor caches for each segment register is
       * invalid, then set the virtualized selector to NULL so
       * an exception is not generated upon return to the guest.
       * Otherwise, point them at virtualized descriptors representing
       * the shadow cache values of each segment register.
       */
      if ( !vm->guest_cpu.desc_cache[SRegES].valid )
        vm->addr->guest_context->es = nullSelector.raw;
      else
        vm->addr->guest_context->es = Selector(4, 0, RPL3);
   
      if ( !vm->guest_cpu.desc_cache[SRegCS].valid )
        vm->addr->guest_context->cs = nullSelector.raw;
      else
        vm->addr->guest_context->cs = Selector(5, 0, RPL3);
   
      if ( !vm->guest_cpu.desc_cache[SRegSS].valid )
        vm->addr->guest_context->ss = nullSelector.raw;
      else
        vm->addr->guest_context->ss = Selector(6, 0, RPL3);
   
      if ( !vm->guest_cpu.desc_cache[SRegDS].valid )
        vm->addr->guest_context->ds = nullSelector.raw;
      else
        vm->addr->guest_context->ds = Selector(7, 0, RPL3);
   
      if ( !vm->guest_cpu.desc_cache[SRegFS].valid )
        vm->addr->guest_context->fs = nullSelector.raw;
      else
        vm->addr->guest_context->fs = Selector(8, 0, RPL3);
   
      if ( !vm->guest_cpu.desc_cache[SRegGS].valid )
        vm->addr->guest_context->gs = nullSelector.raw;
      else
        vm->addr->guest_context->gs = Selector(9, 0, RPL3);
  
      /* Set the virtualized descriptors.  These represent the
       * contents of the shadow cache descriptors of each of the guest
       * segment registers.  Thus a separate one is needed, one
       * for each guest descriptor shadow cache.  For now, these
       * are store at fixed locations in the GDT.  To virtualize,
       * just copy guest descriptor shadow caches to the GDT entries,
       * and downgrade the DPL to 3.
       */
      for (i=0; i<6; i++) {
        if (vm->guest_cpu.desc_cache[i].valid) {
          vm->addr->gdt[4+i] = vm->guest_cpu.desc_cache[i].desc;
          vm->addr->gdt[4+i].dpl = 3;
          }
        else {
          vm->addr->gdt[4+i] = nullDescriptor;
          }
        }
      }

    /* All guest segments are now synchronized with their virtualized
     * counterparts */
    vm->segmentUpdated = 0;

    /* Search for unused PDE for nexus PT  (fixed for now) */
    laddr = 0x70000000;
    vm->mon_pde_mask = laddr & 0xffc00000;
    vm->mon_pdi = vm->mon_pde_mask >> 22;
    base = MON_BASE_FROM_LADDR(laddr);

    /* Map nexus into monitor/guest address space */
    vm->addr->page_dir[laddr >> 22] = vm->host.nexus_pde;

    /* Monitor segments (code/data/TSS).  Put at fixed GDT location for now */
    SET_DESCRIPTOR(vm->addr->gdt[mon_cs >> 3], base, 0xfffff, 
                   D_PG, D_D32, D_AVL0, D_PRESENT, D_DPL0, D_CODE | D_READ)
    SET_DESCRIPTOR(vm->addr->gdt[mon_ss >> 3], base, 0xfffff, 
                   D_PG, D_D32, D_AVL0, D_PRESENT, D_DPL0, D_DATA | D_WRITE)
    SET_DESCRIPTOR(vm->addr->gdt[mon_tss >> 3],
                   base + (Bit32u) vm->guest.addr.tss,
                   sizeof(tss_t)-1,
                   D_BG, 0, D_AVL0, D_PRESENT, D_DPL0, D_TSS)

#if 0
    /* Search for unused GDT selectors */
    for ( i = 1; i < MON_GDT_PAGES*PAGESIZE/8; i++ )
        if (    ((Bit32u *)&vm->addr->gdt[i])[0] == 0
             && ((Bit32u *)&vm->addr->gdt[i])[1] == 0 )
        {
            cs = i << 3;
            break;
        }

    for ( i++; i < MON_GDT_PAGES*PAGESIZE/8; i++ )
        if (    ((Bit32u *)&vm->addr->gdt[i])[0] == 0
             && ((Bit32u *)&vm->addr->gdt[i])[1] == 0 )
        {
            ss = i << 3;
            break;
        }

    for ( i++; i < MON_GDT_PAGES*PAGESIZE/8; i++ )
        if (    ((Bit32u *)&vm->addr->gdt[i])[0] == 0
             && ((Bit32u *)&vm->addr->.gdt[i])[1] == 0 )
        {
            tss = i << 3;
            break;
        }

    if ( !cs || !ss || !tss )
    {
        vm->state = STATE_UNINITIALIZED;
        return;
    }
#endif


    /* Fix up the selectors of all IDT entries */
    for ( i = 0; i < 256; i++ )
        vm->addr->idt[i].selector.raw = mon_cs;

    /* The monitor GDT/IDT loading info */
    vm->addr->nexus->mon_gdt_info.base  = base + (Bit32u) vm->guest.addr.gdt;
    vm->addr->nexus->mon_gdt_info.limit = 0xffff; /* MON_GDT_SIZE; +++ */
    vm->addr->nexus->mon_idt_info.base  = base + (Bit32u) vm->guest.addr.idt;
    vm->addr->nexus->mon_idt_info.limit = 0xffff; /* MON_IDT_SIZE; +++ */

    /* We don't have a monitor LDT for now */
    vm->addr->nexus->mon_ldt_sel = 0;

    /* The monitor TSS */
    vm->addr->nexus->mon_tss_sel = mon_tss;
#warning "deal with esp0 according to monitor mode"
    if ( GetMonMode(vm) == MonModeVM ) {
      vm->addr->tss->esp0 =
        ((Bit32u)vm->guest.addr.nexus) + PAGESIZE;
      }
    else {
      vm->addr->tss->esp0 =
        ((Bit32u)vm->guest.addr.nexus) + PAGESIZE - sizeof(v86_sregs_t);
      }
    vm->addr->tss->ss0  = mon_ss;

    /* Monitor code and stack segments */
    vm->addr->nexus->mon_jmp_info.selector   = mon_cs;
    vm->addr->nexus->mon_stack_info.selector = mon_ss;

    /* Monitor PDBR */
#warning "Monitor CRx hacks"
    vm->addr->nexus->mon_cr0 = 0x80000033;
    vm->addr->nexus->mon_cr3 = vm->pages.page_dir << 12;
    vm->addr->nexus->mon_cr4 = 0x00000004; /* TSD=1 */

    /* Monitor code/data segment base */
    vm->addr->nexus->mon_base = base;
    return(1);
}

  unsigned
isV86MCompatible(vm_t *vm)
{
  Bit32u desc_raw_dword1, bits, compatValue;
  unsigned sreg;
  descriptor_cache_t *cache;
 
  for (sreg=0; sreg<6; sreg++) {
    cache = &vm->guest_cpu.desc_cache[sreg];
    if ( !(vm->descriptorInEmu & (1<<sreg)) )
      monprint(vm, "isV86MCompatible: desc not synced\n");
    if ( !(vm->selectorInEmu & (1<<sreg)) )
      monprint(vm, "isV86MCompatible: sel not synced\n");
    if (cache->valid==0)
      return 0;
    if (cache->limit_scaled != 0xffff)
      return 0;
    /* Base must = selector << 4, or else a reload of the guest
     * segment will not yield the proper base */
    if ( cache->base != (vm->guest_cpu.selector[sreg].raw << 4) )
      return 0;
 
    /* Get the high dword of the descriptor */
    desc_raw_dword1 = ((Bit32u *) &cache->desc)[1];
 
    /* Make sure that important fields are compatible with RM */
#define RMCompatMask  0x00c0fe00 /* G/D_B/P/DPL/S/Type */
#define RMCompatValueData 0x00009200
#define RMCompatValueCode 0x00009A00
    bits = desc_raw_dword1 & RMCompatMask;
    if (sreg == SRegCS)
      compatValue = RMCompatValueCode;
    else
      compatValue = RMCompatValueData;
    if ( bits != compatValue ) {
      monprint(vm, "isV86MCompatible: not compat, sreg=%u, bits=0x%x\n",
               sreg, bits);
      }
    }
 
  return 1;
}
