/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2000  Kevin P. Lawton
 *
 *  monprint.c:  Monitor debug print facility
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#define IN_NEXUS_SPACE
#include "monitor.h"

int mon_vsnprintf(char *str, unsigned size, const char *fmt,
                         va_list args);
int mon_vprint(vm_t *vm, char *fmt, va_list args);

static unsigned int power_of_ten[] = {
  1,
  10,
  100,
  1000,
  10000,
  100000,
  1000000,
  10000000,
  100000000,
  1000000000,
  };

  int
monprint(vm_t *vm, char *fmt, ...)
{
  va_list args;
  int ret;

  va_start(args, fmt);
  ret = mon_vprint(vm, fmt, args);
  va_end(args);
  return(ret);
}

  int
mon_vprint(vm_t *vm, char *fmt, va_list args)
{
  unsigned offset, size;
  unsigned char *log_buff_p;
  int ret;

  if (vm->log_buffer_info.locked)
    return 0;
wrap:
  vm->log_buffer_info.locked = 1;
  vm->log_buffer_info.event = 1;
  offset = vm->log_buffer_info.offset;
  /* Sanity check */
  if (offset >= LOG_BUFF_SIZE) {
    vm->log_buffer_info.offset = 0; /* Reset */
    vm->log_buffer_info.error++;  /* Flag error */
    ret = 0;
    goto done;
    }
  size = LOG_BUFF_SIZE - offset;
  log_buff_p = &vm->addr->log_buffer[offset];

  ret = mon_vsnprintf(log_buff_p, size, fmt, args);

#if 1
#warning "clean up this hack"
  if (ret == -1) {
    void monpanic_nomess(vm_t *);
    if (vm->addr == &vm->host.addr)
      return 0;
    else {
      unsigned i;
      for (i=0; i<LOG_BUFF_SIZE; i++) {
        if (fmt[i])
          vm->addr->log_buffer[i] = fmt[i];
        else
          break;
        }
      vm->addr->log_buffer[LOG_BUFF_SIZE-1] = 0;
      vm->log_buffer_info.locked = 0;
      flush_print_buffer(vm);
      vm->abort_code = 3;
      monpanic_nomess(vm);
      }
    }
  else {
    vm->log_buffer_info.locked = 0;
    flush_print_buffer(vm);
    return(ret);
    }
#endif


  if (ret == -1) {
    /* Message could not fit in the log buffer, probably due to
     * previous messages filling up the buffer.  First flush the buffer
     * to the host print service, then try again.
     */
    *log_buff_p = 0; /* Put NULL back in place; was overwritten */
   
    if (offset == 0) {
      unsigned i;
      unsigned char *ptr;
      /* This string can't possibly fit.  Flag a log error and return. */
      vm->log_buffer_info.offset = 0;
      vm->log_buffer_info.error++;
      ptr = fmt;
      for (i=0; i<LOG_BUFF_SIZE; i++) {
        if (*ptr) {
          vm->addr->log_buffer[i] = *ptr++;
          }
        else {
          vm->addr->log_buffer[i] = 0;
          break;
          }
        }
      vm->log_buffer_info.locked = 0;
      flush_print_buffer(vm);
      return(0);
      }

    vm->log_buffer_info.locked = 0;
    flush_print_buffer(vm);
    goto wrap;
    }
  vm->log_buffer_info.offset += ret;

done:
  vm->log_buffer_info.locked = 0;
  return(ret);
}

  void
flush_print_buffer(vm_t *vm)
{
  if (vm->addr == &vm->host.addr) {
    /* In host space */
    }
  else {
    /* In monitor space, request a switch to host space to
     * flush the buffer */
    sysFlushPrintBuf(vm);
    }
}

  void
resetPrintBuf(vm_t *vm)
{
  vm->log_buffer_info.event = 0;
  vm->log_buffer_info.locked = 0;
  vm->log_buffer_info.offset = 0;
  vm->log_buffer_info.error = 0;
}


/* For now, this is a simple vsnprintf() type of function.  We need
 * to fill a little
 */

  int
mon_vsnprintf(char *str, unsigned size, const char *fmt, va_list args)
{
  int count = 0;
  unsigned format_width;
  unsigned char c;

  while (*fmt) {
    switch (*fmt) {

      case '%':
        format_width = 0;
        fmt++;
        c = *fmt++;
        /* Get optional field width */
        if ( (c>='0') && (c<='9') ) {
          do {
            format_width = (format_width * 10) + (c - '0');
            c = *fmt++;
            } while ( (c>='0') && (c<='9') );
          }
        /* %x: hexadecimal */
        if ( c == 'x' ) {
          unsigned int val, leadin;
          int j;
          unsigned nibble;

          val = va_arg(args, unsigned int);
          leadin = 1;

          for (j=7; j>=0; j--) {
            nibble = (val >> (4 * j)) & 0x0f;
            if (leadin && j && !format_width && !nibble)
              continue;
            if (leadin && j && format_width && ((j+1)>format_width) &&
                !nibble)
              continue;
            leadin = 0;
            if ( (count+2) >= size ) goto error;
            if (nibble <= 9)
              *str++ = nibble + '0';
            else
              *str++ = (nibble-10) + 'A';
            count++;
            }
          break;
          }

        /* %c: character */
        if ( c == 'c' ) {
          unsigned char val;
          val = va_arg(args, unsigned);
          if ( (count+2) >= size ) goto error;
          *str++ = val;
          count++;
          break;
          }
        /* %u: unsigned int */
        if ( c == 'u' ) {
          unsigned int val, leadin;
          int j;
          unsigned digit;

          val = va_arg(args, unsigned int);
          leadin = 1;

          for (j=9; j>=0; j--) {
            if (leadin && j && !format_width && (val < power_of_ten[j]))
              continue;
            if (leadin && j && format_width && ((j+1)>format_width) &&
                (val < power_of_ten[j]))
              continue;
            leadin = 0;
            digit = (val / power_of_ten[j]);
            if ( (count+2) >= size ) goto error;
            *str++ = digit + '0';
            count++;
            val -= (digit * power_of_ten[j]);
            }
          break;
          }
        /* %b : binary (non-standard but useful) */
        if ( c == 'b' ) {
          unsigned int val, bit, leadin;
          int j;
          val = va_arg(args, unsigned int);
          leadin = 1;
          for (j=31; j>=0; j--) {
            bit = (val >> j) & 1;
            if (leadin && j && !format_width && !bit)
              continue;
            if (leadin && j && format_width && ((j+1)>format_width) && !bit)
              continue;
            leadin = 0;
            if ( (count+2) >= size ) goto error;
            *str++ = bit + '0';
            count++;
            }
          break;
          }

        /* Error, unrecognized format char */
        goto error;
        break;

      default:
        /* pass char through */
        if ( (count+2) >= size ) goto error;
        *str++ = *fmt++;
        count++;
        break;
      }
    }

  *str = 0; /* Complete string with null char */
  return(count);

error:
  return(-1);
}
