/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2000  Kevin P. Lawton
 *
 *  vsegment_nexus.c:  Virtualized (monitor) segmentation functionality.
 *    Function exported to either host or monitor spaces.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#define IN_NEXUS_SPACE
#include "monitor.h"



  void
cache_selector(vm_t *vm, unsigned sreg)
{
  guest_context_t *context;
  selector_t       selector;

  if ( vm->selectorInEmu & (1<<sreg) ) {
    /* Selector value in guest_cpu is already valid.  This is
     * the case, if the segment is being virtualized */
    return;
    }

  if ( GetMonMode(vm) == MonModeVM ) {
    /* Selector is not being virtualized.  Guest code is executing
     * using the real selector values, stored in the guest_context.
     */
    context = vm->addr->guest_context;
    switch (sreg) {
      case SRegES:
        selector.raw = context->es & 0xffff;
        break;
      case SRegCS:
        selector.raw = context->cs & 0xffff;
        break;
      case SRegSS:
        selector.raw = context->ss & 0xffff;
        break;
      case SRegDS:
        selector.raw = context->ds & 0xffff;
        break;
      case SRegFS:
        selector.raw = context->fs & 0xffff;
        break;
      case SRegGS:
        selector.raw = context->gs & 0xffff;
        break;
      default:
        monprint(vm, "cache_selector: default case\n");
        break;
      }
  
    /* Move selector into guest_cpu structure, and mark as such */
    vm->guest_cpu.selector[sreg] = selector;
    vm->selectorInEmu |= (1<<sreg);
    }
  else {
    monprint(vm, "cache_selector: MonMode!=VM\n");
    }
}



  void
cache_sreg(vm_t *vm, unsigned sreg)
{
  guest_context_t *context;
  descriptor_cache_t    *cache;

  /* Note code expects this function to cache selector too. */

  cache_selector(vm, sreg);

  if ( vm->descriptorInEmu & (1<<sreg) ) {
    /* Descriptor value in guest_cpu is already valid */
    return;
    }

  context = vm->addr->guest_context;
  cache = &vm->guest_cpu.desc_cache[sreg];

  switch ( GetMonMode(vm) ) {
    case MonModeVM:
      {
      Bit32u base;
      /* If the guest code is being executing within v86 mode,
       * then it is in either v86 mode or RM with RM-compatible
       * descriptor caches.  In this case, descriptor cache values
       * can be derived simply from the selector value.
       */

      base = vm->guest_cpu.selector[sreg].raw << 4;
      cache->desc.limit_low = 0xffff;
      cache->desc.base_low = (base & 0xffff);
      cache->desc.base_med = (base >> 16) & 0xff;
      if (sreg==SRegCS)
        cache->desc.type = 0x1B; /* E/R/A */
      else /* DS,ES,FS,GS,SS */
        cache->desc.type = 0x13; /* data seg, W, A */
      if (vm->guest_cpu.veflags.fields.vm)
        cache->desc.dpl = 3; /* guest actually in VM */
      else
        cache->desc.dpl = 0; /* guest actually in RM */
      cache->desc.p = 1;
      cache->desc.limit_high = 0;
      cache->desc.avl = 0;
      cache->desc.reserved = 0;
      cache->desc.d_b = 0;
      cache->desc.g = 0;
      cache->desc.base_high = (base >> 24) & 0xff;
      cache->base = base;
      cache->limit_scaled = 0xffff;
      cache->valid = 1;
      vm->descriptorInEmu |= (1<<sreg);
      return;
      }

    default:
      monprint(vm, "cache_sreg: default mode\n");
    }
}
