#!/usr/local/bin/perl -w
use strict;
require 5.002;
# ************************************************************ #
# NAME: file-util
# PURPOSE: Utilities for reading from files.
# FUNCTIONS:
#   &stdin_to_tmp: Create a Unique Tmp file for stdin.
#   &cat_region: Copy the specified region of a file from INPUT to
#          OUTPUT, optional flag enables DSC filtering.
#   &find_closing_rexp: Skip forward in file through pairs of rexp's.
#   &next_line: Read next line, plus of current offset.
#   &previous_line: read previous line from a file, plus current offset.
#   &strip_nl: Strip optional newline from end of line.
#         
# NOTES: 
# SCCS: @(#)file-util.pl	2.3 5/26/92
# HISTORY:
#       murray - Mar 25, 1992: Created.
# ************************************************************ #

#
# port to perl5 by Denis N. Antonioli <antonio@ifi.unizh.ch>
#  1.0   27 4 1997
#  1.0.1 14 7 1997 cleaned up for 5.004
#

# ##################################################################### #
# (C) 1992 D Murray Laing, D.M.Laing@uk.ac.edinburgh
#          c/o Department of Chemical Engineering,
#              University of Edinburgh,
#              Edinburgh,
#              Scotland
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License version 2 as
#    published by the Free Software Foundation.
# 
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
# 
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
# ###################################################################### #

# -------------------------------------------------- #
#   Routine to Copy Stdin to a temporary file
# -------------------------------------------------- #

sub main::stdin_to_tmp {
    my ($prefix)=@_;
    my ($num, $tmpstat, $tmpfile);
	local $_;

    # Find a Free Filename in /tmp #
    if ( defined  $tmpfile ) {
		if ( ! open(TMP,">$tmpfile") ) {
			$@ = 'Unable to open temporary file for Stdin';
            return '';
		}
	} else {
		for ( $num=0; $num<999; $num++) {
			$tmpfile="$prefix.$num";
            last if (! -f $tmpfile && open(TMP,">$tmpfile"));
            undef $tmpfile;
		}
        if ( ! defined $tmpfile ) {
			$@='Unable to temporary open file for stdin';
            return '';
		}
	}

    # Cat the Stdin to the tmpfile #
    while ( <STDIN> ) {
		print TMP $_;
	}
    close(TMP);
    
    # Check for Empty Stdin #
    if ( ! (stat($tmpfile))[7] > 0 ) {
		$@ = 'Standard input empty';
        unlink $tmpfile;
        return '';
	}

    # Return the Name of the Temporary File #
    return $tmpfile;
}

# --------------------------------------------------- #
# routine to print a selected area of text
# If flag is true filter lines matching /^%%/ to /^
# --------------------------------------------------- #

sub main::cat_region {
    my($INFILE, $OUTFILE, $start_offset, $end_offset, $flag) = @_;
    my($offset);
	local $_;

    if ( ! defined $start_offset ) {
		while (<$INFILE>) {
			print $OUTFILE $_;
		}
        return 1; 
	}

    $flag = 0 unless defined $flag;

    if ( $start_offset < 0 ) {
		$offset = 0;
	} elsif ( seek($INFILE, $start_offset, 0) ) {
        $offset = $start_offset;
	} else {
		$@ = "Seek Failed - $@"; return 0;
	}

    if ( defined $end_offset ) {
        while (<$INFILE> ) {
            $offset += length($_);

            s/^%%/%*%%/o if $flag;

            if ( $offset == $end_offset ) {
				print  $OUTFILE $_;
                return 1;
			} elsif ( $end_offset > 0 && $offset > $end_offset ) {
                print $OUTFILE substr($_, 0, length($_) + $end_offset - $offset);
                return 1;
			} else {
				print $OUTFILE $_;
			}
		}
	} else {
		while ( <$INFILE> ) {
			s/^%%/%*%%/o if $flag;
            print $OUTFILE $_;
		}
        return 1;
	}

    $@="Unable to reach offset $end_offset";
    return 0;
}

# ------------------------------------------------------- #
# Routine to skip round blocks such as Begin/EndDocument
# ------------------------------------------------------- #
sub main::find_closing_rexp {
    my($FILE, $offset, $open_rexp, $close_rexp, $depth) = @_;
    my($line);
	local $_;

        # Assume search starts after first match of open_rexp #
    $depth=1 unless defined $depth;

    while( $depth > 0 && ($line=<$FILE>) ) {
		if ( $line =~ /$open_rexp/ )     { $depth++; }
        elsif ( $line =~ /$close_rexp/ ) { $depth--; }
        $$offset += length($line);
	}

	return $depth > 0 ? 0 : 1;
}

# -------------------------------------------------- #
# Routines to keep track of position while reading
# line from a $FILE.
# -------------------------------------------------- #
sub main::next_line {
    my($FHANDLE,$offset,$flag)= @_;
    my $line = '';
	local $_;

    if ( $$offset < 0 ) {
		$$offset = 0; $$flag=1;
	}

    if ( defined $flag && $$flag ) {
		if ( ! seek($FHANDLE, $$offset, 0) ) {
			$@ = "Seek Failed - $@";
            return '';
		}
        $flag=0;
	}

    if ( $line = <$FHANDLE> ) {
		$$offset += length($line);
	} else {
		$@ = "Read Failed - $@";
        $line = '';
	}
    return $line;
}

sub main::previous_line {
    my ($FHANDLE, $offset, $flag) = @_;
    my($char, $line, $newoffset) = ('', '', 0);
	local $_;

    # Return null string when zero offset reached #
    return '' if $$offset == 0;

    # Negative Offset is flag to reset to end of $FILE #
    $$offset = (stat($FHANDLE))[7] if $$offset < 0;

    $$flag = 0;
    $newoffset = $$offset;
    do {
        $newoffset--;

        if ( ! seek($FHANDLE, $newoffset, 0) ) {
			$@ = "Seek Failed - $@";
            return '';
		}

        if ( read($FHANDLE, $char, 1, 0) != 1 ) {
			$@ = "Read Failed - $@";
            return '';
		}

        $line = $char . $line;
	} until ( ($char eq "\n" && $newoffset+1 < $$offset ) || $newoffset == 0 );

    # If there is a leading newlines remove it and set offset #
    if( $newoffset > 0 ) {
		$line = substr($line,1); $$offset = $newoffset+1;
	} else {
		$$offset = $newoffset;
	}

    return $line;
}

# ============================================================ #
# END of Package $FILE-util.pl
# ============================================================ #
1;
