#!/usr/local/bin/perl -w
use strict;
require 5.002;

use FileHandle;

# ************************************************************ #
# NAME: pspp-util.pl
# PURPOSE: Utilities to Process PS Documents, used by psmulti.
# NOTES: 
# SCCS: @(#)pspp-util.pl	2.6 5/26/92
# HISTORY:
#       murray - Mar 24, 1992: Created.
# ************************************************************ #

#
# port to perl5 by Denis N. Antonioli <antonio@ifi.unizh.ch>
#  1.0  27 4 1997
#

# ##################################################################### #
# (C) 1992 D Murray Laing, D.M.Laing@uk.ac.edinburgh
#          c/o Department of Chemical Engineering,
#              University of Edinburgh,
#              Edinburgh,
#              Scotland
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License version 2 as
#    published by the Free Software Foundation.
# 
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
# 
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
# ###################################################################### #

# ************************************************************ #
# PSPP Syntax:
#
#   /^%%IncludeDocument: <DocName>$/
#       Include document <DocName> at current point in text
#    surrounded by %%Begin/EndDocument comments.
#
#   /^%%RequireDocument: <DocName>$/
#       If <DocName> has not already been included then
#    include as per %%IncludeDocument.
#
#   /^%%PSMacro(Macro): <MacroExpansion>$/
#       Define a macro <Macro> having expansion <MacroExpansion>
#
#   /^%%+ <MacroContinuation>$/
#       If found directly following a psmacro declaration
#       taken as a continuation of that macro.
#
#   /@Macro\s+/ || /@{Macro}/
#       If macro <Macro> is know substitute its expansion
#     otherwise leave text as is.
# ************************************************************ #

package PSPP_Util;

@main::PSPPIncludePath = ( '.' ) unless defined @main::PSPPIncludePath;


# ============================================================ #
#        Routines Local to Package
# ============================================================ #

sub find_document {
	my ($docname) = @_;
	my $doc;

	return $docname if -f $docname;

	foreach(@main::PSPPIncludePath) {  
		if ( -f "$_/$docname" ) {
			$doc = "$_/$docname"; last;
		}
	}

	if ( ! defined $doc ) {
		$@="Can't locate document $docname on include path";
	}
	return $doc;
}

sub expand_line {
    my ($line) = @_;

    if ( $line =~ /\@{([^\s}]+)}/o ) {
     	my ($leading, $macro, $trailing) = ($`, $1, $');

		$leading = &expand_line($leading);
		$trailing = &expand_line($trailing);
        
		if ( defined $PSPP_Util::PSMacro{$macro} ) {
			$macro = &expand_line($PSPP_Util::PSMacro{$macro});
		} else {
			$macro = '@{' . $macro . '}';
		}
		$line = $leading . $macro . $trailing;
	} elsif ( $line =~ /\@([^\s]+)(\s+|$)/o ) {
		my ($leading, $macro, $trailing) = ($`, $1, $2 . $');

		$leading = &expand_line($leading);
		$trailing = &expand_line($trailing);

		if ( defined $PSPP_Util::PSMacro{$macro} ) {
			$macro = &expand_line($PSPP_Util::PSMacro{$macro});
		} else {
			$macro = '@' . $macro;
		}
		$line = $leading . $macro . $trailing;
	}
    return $line;
}

# ============================================================ #
#         Routines Exported to Main
# ============================================================ #

sub main::reset_pspp {
	package PSPP_Util;
    @PSPP_Util::PSPPIncluded = ();
    %PSPP_Util::PSMacros = ();
}

sub main::require_document {
	package PSPP_Util;

    my ($docname) = @_;
    local $_;

    foreach( @PSPP_Util::PSPPIncluded ) {
		return 1 if /$docname$/;
	}

    &main::include_document($docname) || return 0;

    return 1;
}


sub main::include_document {
	package PSPP_Util;
    
    my($docname) = @_;
    my(@DocStack, %NextChar, $cmacro, $doc);
    my($cmacro_count,$count) = ( -1, 0);
    local $_;

    ($doc = &find_document($docname)) || return 0;

	my $input = FileHandle->new();
    unless ( open($input, $doc) ) {
		$@ = "Can't open $doc: $!\n"; return 0;
	}

    push(@PSPP_Util::PSPPIncluded, $doc);

    print "%%BeginDocument: $doc\n";

    while( <$input> ) {
        $_ = &expand_line($_);
        $count++;

        if  ( /^%%IncludeDocument:\s*([^\s]+)\s*$/o ) {
			&main::include_document($1) || return 0;
		} elsif ( /^%%RequireDocument:\s*([^\s]+)\s*$/o ) {
			&main::require_document($1) || return 0;
		} elsif ( /^%%PSMacro\(\s*([^\s]+)\s*\):(.+)$/o ) {
			$PSPP_Util::PSMacro{$1} = $2;
            $cmacro = $1; $cmacro_count = $count+1;
		} elsif ( /^%%\+(.*)$/o && $cmacro_count == $count ) {
			$PSPP_Util::PSMacro{$cmacro} .= "\n$1";
            $cmacro_count = $count+1;
		} else {
			print  $_;
		}
	}

    print "%%EndDocument\n";

    return 1;
}

sub main::pspp_cat {
    package PSPP_Util;
    my($file, $InitMacros) = @_;
	local $_;
    &main::reset_pspp unless (defined $InitMacros->{'NoReset'} && $InitMacros->{'NoReset'} eq 'true' );

    foreach(keys(%$InitMacros)) {
		$PSPP_Util::PSMacro{$_} = $InitMacros->{$_};
	}

    &main::include_document($file) || return 0;
}

# -------------------------------------------------- #
# End of Package PSMacroz_Util
# -------------------------------------------------- #
1;
