/***************************************************************************
                          ractzoom.cpp  -  description
                             -------------------
    begin                : Mon Sep 27 1999
    copyright            : (C) 1999 by Andreas Mustun
    email                : andrew@ribbonsoft.com
 ***************************************************************************/

#include <qbitmap.h>
#include <qcursor.h>

#include "rappwin.h"
#include "ractzoom.h"
#include "rbehaviordef.h"
#include "rcolor.h"
#include "rgraphic.h"
#include "rlog.h"
#include "rpainter.h"
#include "rstatuspanel.h"


/*! \class RActZoom
    \brief Action handling for zoom functions

    \author Andrew Mustun
*/


/*! Constructor
  */
RActZoom::RActZoom(RGraphic* _graphic)
{
  graphic = _graphic;
  otherGraphic = 0;
  prevVisible=false;
  mouseInside=false;
  px=py=32000;
  sx=sy=32000;
}



/*! Destructor
*/
RActZoom::~RActZoom()
{
}



/*! Serves an event to the corresponding function
    \param _action The Action (e.g. ACT_ZOOM_WINDOW)
    \param _rEvent The event (e.g. REVENT_BEGIN)
*/
void
RActZoom::serveEvent(int _action,
                     int _rEvent )
{
  if(!graphic) return;
  
  if(_action>=ACT_ZOOM_FIRST &&
     _action<=ACT_ZOOM_LAST     ) {

    switch(_action) {
    
      case ACT_ZOOM_WINDOW:
        zoomWindow(_rEvent);
        break;

      case ACT_ZOOM_PAN:
        zoomPan(_rEvent);
        break;

      default:
        break;
    }
  }
}



/*! Lets the user zoom in a window
    \param _rEvent The event that happened (e.g. REVENT_LBUTTONDOWN, ...)

    States:
    <ul>
      <li>1: Set 1st window corner
      <li>2: Set 2nd window corner
    </ul>
*/
void
RActZoom::zoomWindow(int _rEvent)
{
  if(!graphic) return;
  
  if(graphic==currentDoc()) {
    otherGraphic=currentOut();
  }
  else {
    otherGraphic=currentDoc();
  }
  
  switch(_rEvent) {

    case REVENT_BEGIN:
      //graphic->setBehavior(BVR_FREEPOINT);
      graphic->setCurrentState(1);
      graphic->setCursor(*RAppWin::getRAppWin()->curGlass);
      statusPanel()->setActionStatus(tr("L: First Corner of zoom range"),
                                     tr("R: Back"),
                                     tr("Zoom Window"));
      prevVisible=false;
      mouseInside=true;
      break;
      
    case REVENT_MOUSELEAVE:
      zoomWindow(REVENT_DELPREVIEW);
      mouseInside=false;
      break;
      
    case REVENT_MOUSEENTER:
      mouseInside=true;
      break;
      
    case REVENT_DRAWPREVIEW:
    case REVENT_DELPREVIEW:
      if(graphic->getCurrentState()==2) {
        RPainter paint;        // painter
        
        paint.begin(graphic);
        paint.setPen(rCyan);
        paint.setRasterOp(XorROP);
        paint.setClipRect(2, 2, graphic->width()-4, graphic->height()-4);

        // Delete Preview:
        //
        if(prevVisible) {
          paint.drawRect(px, py, 
                         sx-px, sy-py);
          prevVisible=false;
        }
        
        // Draw Preview:
        //
        if(!prevVisible && _rEvent==REVENT_DRAWPREVIEW) {
          paint.drawRect(px, py,
                         graphic->getMousePosX()-px, 
                         graphic->getMousePosY()-py);
          prevVisible=true;
        }
        paint.end();
      }
      break;
      
    case REVENT_MOUSEMOVE:
      if(mouseInside && graphic->getCurrentState()==2) {
        zoomWindow(REVENT_DRAWPREVIEW);
        graphic->getMousePos(&sx, &sy);
      }
      break;

    case REVENT_LBUTTONDOWN:
      if(graphic->getCurrentState()==1) {
        graphic->getMousePos(&px, &py);
        graphic->getMousePos(&sx, &sy);
        graphic->setCurrentState(2);
        statusPanel()->setLeftButtonStatus(tr("L: Second corner of zoom range"));        
      }
      break;
  
    case REVENT_LBUTTONUP:
      if(graphic->getCurrentState()==2) {
        graphic->getMousePos(&sx, &sy);
        // Avoid too small zoom ranges:
        //   Zoom range gets selected with two klicks OR one drag
        //
        if(sx>px+2 || sx<px-2 ||
           sy>py+2 || sy<py-2    ) {
          graphic->setZoom(px, py, sx, sy, 0);
          //graphic->setBehavior(BVR_FREEPOINT);
          graphic->repaint();
          RAppWin::getRAppWin()->synchronizeZoom(graphic==currentOut());
          if(otherGraphic) otherGraphic->repaint();
          graphic->setCurrentState(1);
          statusPanel()->setLeftButtonStatus(tr("L: First Corner of zoom range"));
        }
      }
      break;

    case REVENT_RBUTTONUP:
      if(graphic->getCurrentState()==1) {
        graphic->setBehavior(BVR_NO);
        graphic->resetCurrentState();
        graphic->resetCurrentAction();
        graphic->setCursor(arrowCursor);
        statusPanel()->clear();
        
        graphic->restoreLastActionSettings();
        
        if(otherGraphic) {
          otherGraphic->setBehavior(BVR_NO);
          otherGraphic->resetCurrentState();
          otherGraphic->resetCurrentAction();
          otherGraphic->setCursor(arrowCursor);
          otherGraphic->restoreLastActionSettings();
        }
      }
      break;

    default:
      break;
  }
}



/*! Lets the user move the current view (pan zoom)
    \param _rEvent The event that happened (e.g. REVENT_LBUTTONDOWN, ...)

    States:
    <ul>
      <li>1: Set new center for panning
    </ul>
*/
void
RActZoom::zoomPan(int _rEvent)
{
  if(!graphic) return;
  
  if(graphic==currentDoc()) {
    otherGraphic=currentOut();
  }
  else {
    otherGraphic=currentDoc();
  }
  
  sx=graphic->width();
  sy=graphic->height();

  switch(_rEvent) {

    case REVENT_BEGIN:
      graphic->setCurrentState(1);
      graphic->setCursor(*RAppWin::getRAppWin()->curGlass);
      statusPanel()->setActionStatus(tr("L: New center of zoom range"),
                                     tr("R: Back"),
                                     tr("Panzoom"));
      
      graphic->getMousePos(&px, &py);
      prevVisible=false;
      break;
    
    case REVENT_MOUSELEAVE:
      zoomPan(REVENT_DELPREVIEW);
      mouseInside=false;
      break;
      
    case REVENT_MOUSEENTER:
      mouseInside=true;
      break;
    
    case REVENT_DRAWPREVIEW:
    case REVENT_DELPREVIEW:
      if(true) {
        RPainter paint;        // painter

        paint.begin(graphic);
        paint.setPen(rCyan);
        paint.setRasterOp(XorROP);
        paint.setClipRect(2, 2, graphic->width()-4, graphic->height()-4);

        // Delete Preview:
        //
        if(prevVisible) {
          paint.drawRect(px-sx/2, py-sy/2, 
                         sx, sy);
          prevVisible=false;
        }
        
        // Draw Preview:
        //
        if(!prevVisible && _rEvent==REVENT_DRAWPREVIEW) {
          paint.drawRect(graphic->getMousePosX()-sx/2, 
                         graphic->getMousePosY()-sy/2,
                         sx, sy);
          prevVisible=true;
        }
        paint.end();
      }
      break;
    
    case REVENT_MOUSEMOVE:
      if(mouseInside) {
        zoomPan(REVENT_DRAWPREVIEW);
        graphic->getMousePos(&px, &py);
      }
      break;
    
    case REVENT_LBUTTONUP:
      zoomPan(REVENT_DELPREVIEW);
    
      graphic->getMousePos(&px, &py);
      graphic->setZoom(px-sx/2, py-sy/2, 
                       px+sx/2, py+sy/2, 0);
      RAppWin::getRAppWin()->synchronizeZoom(graphic==currentOut());
      
      graphic->repaint();
      if(otherGraphic) otherGraphic->repaint();
      
      break;

    case REVENT_RBUTTONUP:
      if(graphic->getCurrentState()==1) {
        graphic->resetCurrentState();
        graphic->resetCurrentAction();
        graphic->setCursor(arrowCursor);
        
        zoomPan(REVENT_DELPREVIEW);
        
        statusPanel()->clear();
        
        graphic->restoreLastActionSettings();
        
        if(otherGraphic) {
          otherGraphic->setBehavior(BVR_NO);
          otherGraphic->resetCurrentState();
          otherGraphic->resetCurrentAction();
          otherGraphic->setCursor(arrowCursor);
          otherGraphic->restoreLastActionSettings();
        }
      }
      break;

    default:
      break;
  }
  
}


// EOF





