#ifndef DOWNLOAD_H
#define DOWNLOAD_H

#include <fstream>
#include <ctime>

#include <qobject.h>
#include <qdatetime.h>
#include <qtimer.h>

#include "QueryHitEntry.h"

class QSocket;
class QListViewItem;
class QFile;

class DownloadManager;

using namespace std;

class Download : public QObject
{
  Q_OBJECT

 public:
  Download(QueryHitEntry qhe, QListViewItem *item, DownloadManager *parent);
  ~Download();

  void                run();
  const int           readState();
  const double        readBandwidth();
  const int           readRetryTimeout();
  const unsigned long bytesRead();
  const unsigned long bytesToRead();
  void                startPushedDownload(QSocket *socket);
  void                abortDownload();
  void                resumeDownload();
  void                finish();
  bool                isActive();
  void                moveFile(std::string directory);

  enum { dtIdle,             /*!< Initial state                                     */
	 dtError,            /*!< Error occured.                                    */
	 dtAbort,            /*!< Download aborted by user.                         */
	 dtClosed,           /*!< Connection was closed without any bytes read.     */
	 dtClosedCompleted,  /*!< Connection was closed after finishing download.   */
	 dtClosedUnfinished, /*!< Connection was closed but download not completed. */
	 dtConnecting,       /*!< Try to establish a connection.                    */
	 dtNotFound,         /*!< */
	 dtTimeout,          /*!< */
	 dtBusy,             /*!< */
	 dtConnected,        /*!< Connected.                                        */
	 dtExists,           /*!< File already exists.                              */
	 dtQueued,
	 dtPushed
  };

  //! Pointer to the download item in the download list
  /*!
    This item is deleted automatically when ~DownloadThread() is called.
    \sa ~DownloadThread()
  */
  QListViewItem* _item;

  //! Holds information about the file to be downloaded.
  QueryHitEntry  _qhe;

  //! File to which incoming data is written to.
  QFile*         _f;

 private:
  std::string   createFilename(std::string file, unsigned long size, std::string host);
  std::string   findFile(std::string file, unsigned long size);
  std::string   findFile(std::string file, std::string host);
  std::string   findFile(std::string file, unsigned long size, std::string host);

  void          setState(int);

  //! Timer which manages the update of the status of this download.
  //QTimer*        _status_update_timer;

  //! Name of the server.
  std::string         _client;

  //! Current state of the this download.
  int            _state;

  //! Size of the file.
  unsigned long  _content_length;

  //! Time when download started.
  time_t         _start_time;

  //! Time when download finished.
  time_t         _end_time;

  //! Buffer to store incoming data.
  std::string    _buffer;

  //! True if run() method was not invoked yet.
  bool           _first_start;

  //! Number of retries.
  int            _retries;

  //! Stores the HTML-Header
  std::string    _header;

  DownloadManager* _parent;

  std::string    complete_path;
  std::string    incomplete_path;

  bool           _active;

  QSocket        *sock;
  QTimer         *timer_bandwidth;
  QTimer         *timer_retry;
  int            retry_timeout;
  unsigned long  tmp_read;
  double         bandwidth;

  QTimer         *timeout;

  std::string    _filename;  // current complete filename for moveFile

 public slots:
  void           slotReadyRead();
  void           slotConnectionClosed();
  void           slotError(int);
  void           slotConnected();
  void           slotTimerBandwidth();
  void           slotTimeout();
  void           slotRetry();
  void           slotUpdateStatus();
};

#endif
