#ifndef SERVENT_H
#define SERVENT_H

#include <qobject.h>
#include <qtimer.h>

#include <string>
#include <list>
#include <set>
#include <ctime>

#include "Gnutella.h"
#include "Address.h"

class Ping;
class Host;
class Address;
class Message;
class Connections;
class PongCache;

class QListViewItem;
class QSocket;

using namespace std;

//! This class is used to communicate to other Gnutella host.
/*! 
    It connects to only one host at once. When data arrives it
    is automatically appended it to the ping, pong, query,
    queryhit and push vectors. These vectors are being precessed by
    the Connection class every 200ms. Since Qtella is mutlithreaded
    we have to call this->mutex.lock() every time we access the
    vectors to prevent conflicts.
    \sa Connection vPing vPong vQuery vQueryHit vPush
*/

class Servent : public QObject
{
  Q_OBJECT

 public:
  //! Creates an object with the given Gnutella ID.
  /*!
      \param id The Gnutella ID used for this host.
  */
  Servent(Connections* parent, std::string id, QListViewItem* item, QSocket* socket = NULL);

  //! Destructor.
  ~Servent();

  //! Establish a connection to another host at the given address.
  /*!
      This method returns immediately. 
      \param address Address of the other host.
      \sa Address
   */
  void    connect(Address address);

  void    close();

  //! Returns the client's state.
  /*!
      \return serventIdle, serventConnected, serventError serventClosed
  */
  int     state();
  
  void    writeConnectString();

  void    limitVectors();

  int     getDirection();

  void    setDirection(int direction);

  void    sendMessage(std::string& buffer, bool cache = true);
  void    sendMessageDirect(std::string& buffer);

  std::string       _connect_str;
  std::string       _response_str;

  enum { Outgoing, Incoming };

  //! States for connection to other host.
  enum { Idle,       /*!< Not connected.                    */
	 Connected,  /*!< Connected to other Gnutella host. */
	 Error,      /*!< An error occurred.                */
	 Connecting, /*!< Connecting...                     */
	 Waiting,    /*!< Socket is open.                   */
	 Closed      /*!< Connection closed                 */
  };

  QSocket*          _socket;
  QListViewItem*    _item;
  int               _timeout;

  //! Unique Gnutella ID of this host.
  std::string       _id;

  QTimer*           _ping_timer;

  time_t            _start;

 public slots:
    void    flushTimeout();
    void    pingTimeout();
    void    slotDelayedClosed();

    //! Called if an error occured.
    /*!
        State is set to serventError.
    */
    void    slotError(int);

    //! Called if connection has been established.
    /*! 
        State is set to serventConnected.
    */
    void    slotConnect();

    //! Called if new data has been arrived.
    /*!
     */
    void    slotReadyRead();

    //! Called if other client has closed the connection.
    /*!
        State is set to serventIdle.
    */
    void    slotClosed();

    void    bandwidthTimeout();

    void    timeout();

 private:
    void            setState(int state);
    static int      getNextServentNumber();
    
    //! Holds the current state of the connection.
    int             _state;

    //! Buffer which takes all incoming data.
    std::string     _buffer;

    //! Used for connectToServent(...) to have a valid object.
    std::string     _strip;

    Connections*    _parent;


    int             _direction;

    QTimer*         _bandwidth_timer;
    int             _bytes_in;
    int             _bytes_out;

    QTimer*         _timeout_timer;

    time_t          _last_input;

 public:
    double          _bandwidth_in;
    double          _bandwidth_out;

    std::string     _send_buffer;
    QTimer*         _flush_timer;

    std::list<std::string>    _message_buffer;
    
    unsigned int              _servant_number;
    time_t                    _last_ping_send;
    std::string               _last_accepted_ping_id;

    std::set<std::string>     _id_set;

    Address                   _address;

    //! Number of input messages
    long                      _input;
    
    //! Number of output messages
    long                      _output;
};

#endif
