#include "Connections.h"

#include "QtellaSub.h"
#include "Servent.h"
#include "Pong.h"
#include "Host.h"
#include "QueryHit.h"
#include "Ping.h"
#include "Query.h"
#include "QueryHitEntry.h"
#include "Push.h"
#include "SearchWidget.h"
#include "Search.h"
#include "SharedFile.h"
#include "QueryResult.h"
#include "Message.h"
#include "UploadManager.h"
#include "StringManipulation.h"
#include "SearchViewItem.h"
#include "PongCache.h"

#include "Gnutella.h"

#include <iostream>
#include <map>
#include <list>
#include <algorithm>
#include <memory>
#include <strstream>
#include <iomanip>

#include <cstdlib>
#include <ctime>
#include <cctype>   // tolower

#include <qevent.h>
#include <qlistbox.h>
#include <qlistview.h>
#include <qspinbox.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qtimer.h>
#include <qdatetime.h>
#include <qpixmap.h>
#include <qimage.h>
#include <qcolor.h>
#include <qsocket.h>
#include <qtabwidget.h>

Connections::Connections(QtellaSub *parent = 0)
  : _parent(parent), _address(), _statistic(), _max_hosts(300), _timer(NULL),
    _last_query_time(0)
{
  _max_servants = _parent->ui_spinbox_hosts->value();

  _id = createID();

  _statistic_timer = new QTimer();
  connect(_statistic_timer, SIGNAL(timeout()), this, SLOT(slotUpdateStatistic()));
  _statistic_timer->start(1000);

  _uptime = new QDateTime( QDate::currentDate(), QTime::currentTime() );

  _timer = new QTimer(this);
  connect(_timer, SIGNAL(timeout()), SLOT(check()));
  _timer->start(500);
}


std::string Connections::createID()
{
  #ifdef QTELLA_DEBUG
    std::clog << "Connections: create ID" << std::endl;
  #endif

  std::string id;

  srand(time(NULL));

  for(int i = 0; i < 16; i++)
    {
      unsigned char c = rand()%255;
      id += c;
    }

  id[8] = 0xff;
  id[15] = 1;

  return id;
}


Connections::~Connections()
{
  #ifdef QTELLA_DEBUG
    std::clog << "Connections: destructor" << std::endl;
  #endif

  delete _statistic_timer;
  delete _uptime;
  if(_timer) delete _timer;
}


void Connections::disconnect()
{
  #ifdef QTELLA_DEBUG
    std::clog << "Connections: disconnect" << std::endl;
  #endif

  for(int i = 0; i < _servants.size(); ++i) delete _servants[i];
  _servants.clear();  
}


void Connections::start()
{
  #ifdef QTELLA_DEBUG
    std::clog << "Connections: start" << std::endl;
  #endif

  std::string  s;
  int          i = 0;
  bool         ignore = false;

  while( (_servants.size() < _max_servants) && (i < _parent->ui_listbox_hostlist->count()) )
    {
      s = static_cast<std::string>(_parent->ui_listbox_hostlist->text(i));
      _parent->ui_listbox_hostlist->removeItem(i++);
      for(int j = 0; j < _servants.size(); ++j) if( s == std::string(_servants[j]->_item->text(0))) ignore = true;
      if(!ignore) addServant(Address(s));
      ignore = false;
    }

  i = 0;
  while( (_servants.size() < _max_servants) && (i < _parent->ui_listbox_autoconnect->count()) )
    {
      s = static_cast<std::string>(_parent->ui_listbox_autoconnect->text(i++));
      for(int j = 0; j < _servants.size(); ++j) if( s == std::string(_servants[j]->_item->text(0))) ignore = true;
      if(!ignore) addServant(Address(s));
      ignore = false;
    }
}


void Connections::addServant(Address address, QSocket* socket)
{
  QListViewItem *item;
  
  // for incoming connections
  if( (_servants.size() >= _max_servants) && (socket) )
    {
      std::vector<Servent*>::iterator pos;

      for(pos = _servants.begin(); pos != _servants.end(); )
	if((*pos)->state() == Servent::Connecting)
	  {
	    delete *pos;
	    pos = _servants.erase(pos);
	    break;
	  }
	else ++pos;
      
      if(_servants.size() >= _max_servants)
	{
	  delete socket;
	  return;
	}
    }

  if(_servants.empty())
    item = new QListViewItem(_parent->ui_listview_connections);
  else
    item = new QListViewItem(_parent->ui_listview_connections, _servants.back()->_item);

  _servants.push_back(new Servent(this, _id, item, socket));

  item->setText(0, address.toString().c_str());
  if(socket) 
    {
      item->setText(1, "Incoming"); 
      _servants.back()->setDirection(Servent::Incoming);
    }
  else 
    {
      item->setText(1, "Connecting...");
      _servants.back()->setDirection(Servent::Outgoing);
      _servants.back()->connect(address);
    }  

  if(!_parent->timer_connection->isActive()) _parent->timer_connection->start(500);
}


void Connections::checkServants()
{
  #ifdef QTELLA_DEBUG
    std::clog << "Connections: check " << _servants.size() << std::endl;
  #endif

  unsigned long hosts = 0;
  unsigned long files = PongCache::getInstance()->getNumberOfFiles();
  unsigned long size  = PongCache::getInstance()->getSharedSize();

  double bandwidth_in = 0;
  double bandwidth_out = 0;

  try{

  for( int i = 0; i < _servants.size(); ++i)
    switch( _servants[i]->state() )
      {
      case Servent::Connected:
	{
          #ifdef QTELLA_DEBUG
	    std::clog << "Connections: switch " << i << std::endl;
          #endif

	  if(_servants[i]->getDirection() == Servent::Outgoing)
	    _servants[i]->_item->setText(1, "Outgoing");
	  else
	    _servants[i]->_item->setText(1, "Incoming");

	  _servants[i]->limitVectors();

	  std::strstream str;
	  str << _servants[i]->_input << " / " << _servants[i]->_output << std::ends;
	  _servants[i]->_item->setText(2, str.str());
	  str.freeze(false);

	  unsigned sh = PongCache::getInstance()->getHosts(_servants[i]->_servant_number);
	  hosts += sh;
	  str.seekp(0, std::ios::beg);
	  str << sh << std::ends;
	  _servants[i]->_item->setText(3, str.str());
	  str.freeze(false);

	  str.seekp(0, std::ios::beg);
	  str.setf(std::ios::fixed);
	  str << setprecision(2) << _servants[i]->_bandwidth_in << " / " << _servants[i]->_bandwidth_out << std::ends;
	  _servants[i]->_item->setText(4, str.str());
	  str.freeze(false);

	  bandwidth_in += _servants[i]->_bandwidth_in;
	  bandwidth_out += _servants[i]->_bandwidth_out;

	  // uptime
	  unsigned u = time(NULL) - _servants[i]->_start;
	  unsigned h = u / 3600;
	  unsigned m = (u - h * 3600) / 60;
	  unsigned s = u - h * 3600 - m * 60;
	  std::strstream uptime;
	  uptime.fill('0');
	  uptime << setw(2) << h << ":" << setw(2) << m << ":" << setw(2) << s << std::ends;
	  _servants[i]->_item->setText(5, uptime.str());
	  uptime.freeze(false);
	}
	break;

      case Servent::Closed:
	_servants[i]->_item->setText(1, "Closed");
	break;

      case Servent::Connecting:
	_servants[i]->_item->setText(1, "Connecting...");
	break;

      case Servent::Error:
	_servants[i]->_item->setText(1, "Error");
	break;

      case Servent::Waiting:
	_servants[i]->_item->setText(1, "Waiting");
	break;
      }

  } catch(...) { std::cerr << "EXC Connections::checkServant A" << std::endl; }
  

  try{

  // delete servent which are in close or error state and increase timeout counter of all servents
  std::vector<class Servent*>::iterator    pos;

  #ifdef QTELLA_DEBUG
    std::clog << "Connections: check close and error" << std::endl;
  #endif

  // timeout = 7.5 sec because DNS lookup signal caused SEGV
  for(pos = _servants.begin(); pos != _servants.end(); )
    if( ((*pos)->state() == Servent::Closed || (*pos)->state() == Servent::Error) || 
	((*pos)->_timeout++ > 15 && (*pos)->state() != Servent::Connected) )
      {
	delete *pos;
	pos = _servants.erase(pos);
      }
    else ++pos;
  
  } catch(...) { std::cerr << "EXC Connections::checkServant B" << std::endl; }

  // connect to other hosts
  start();

  // number of hosts

  std::strstream str;
  str << hosts << std::ends;
  _parent->ui_textlabel_hosts->setText(str.str());
  str.freeze(false);

  std::string u;
  str.seekp(0, ios::beg);
  if(files > 1024) { files = files >> 10; u = "K"; };
  if(files > 1024) { files = files >> 10; u = "M"; };
  str << files << u << std::ends;
  _parent->ui_textlabel_files->setText(str.str());
  str.freeze(false);
  
  u = "K";
  str.seekp(0, ios::beg);
  if(size > 1024) { size = size >> 10; u = "MB"; }
  if(size > 1024) { size = size >> 10; u = "GB"; }
  if(size > 1024) { size = size >> 10; u = "TB"; }
  str << size << u << std::ends;
  _parent->ui_textlabel_size->setText(str.str());
  str.freeze(false);

  str.seekp(0, ios::beg);
  str.setf(std::ios::fixed);
  str << setprecision(1) << bandwidth_in << "KB(I) + " << bandwidth_out << "KB(O) = " 
      << (bandwidth_in + bandwidth_out) << "KB" << std::ends;
  _parent->ui_label_connect_bandwidth->setText(str.str());
  str.freeze(false);

  #ifdef QTELLA_DEBUG
    std::clog << "Connections: ok checkServants" << std::endl;
  #endif
}


void Connections::check()
{
  for(int i = 0; i < _servants.size(); ++i)
    if(_servants[i]->state() == Servent::Connected ||
       _servants[i]->state() == Servent::Waiting) checkMessages(_servants[i]);
}


// void Connections::checkMessages(Message *msg, Servent* s)
void Connections::checkMessages(Servent* s)
{
  #ifdef QTELLA_DEBUG
    std::clog << "Connections: check messages" << std::endl;
  #endif

  try{

  _address.setIP( static_cast<std::string>(_parent->ui_lineedit_ip->text()) );
  _address.setPort( static_cast<std::string>(_parent->ui_lineedit_listenport->text()) );

  while(!s->_message_buffer.empty())
    {
      Message msg(s->_message_buffer.front(), s->_message_buffer.front().size());
      s->_message_buffer.pop_front();
  
  switch(msg.cmd())
    {
    case CMD_PING:
      {
	try{

	auto_ptr<Ping>  p(msg.getPing());

	_statistic.in_ping_n++;
	_statistic.in_ping_b += p->_data.size();
	
	if( (p->_id == _id) || (p->_ttl < 1) ) break;

	s->_last_accepted_ping_id = p->_id;

	if( p->_ttl != 1) // host needs pongs
	  {
	    std::list<Pong>            pong_list;
	    std::list<Pong>::iterator  pos;
	    
	    PongCache::getInstance()->getPong(pong_list, 10);
	    
	    for(pos = pong_list.begin(); pos != pong_list.end(); ++pos)
	      {
		pos->_id = p->_id;
		s->sendMessage(pos->_data);
		_statistic.out_pong_n++;
		_statistic.out_pong_b += pos->_data.size();
	      }
	  }

	unsigned long size = static_cast<unsigned long>(_parent->shared_size/1024);
	Pong          pong(p->_hops, 0, p->_id, _address, _parent->number_shared_files, size);

	s->sendMessage(pong._data, false);
	_statistic.out_pong_n++;
	_statistic.out_pong_b += pong._data.size();

	} catch(...) { std::cerr << "EXC Connections::checkMessages Ping" << std::endl; }
      }
      break;
      
    case CMD_PONG:
      {
	try{
	auto_ptr<Pong> p(msg.getPong());

	_statistic.in_pong_n++;
	_statistic.in_pong_b += p->_data.size();

	PongCache::getInstance()->addPong(*p, s->_servant_number);
	
	if( !p->_address.isPrivate() || (p->_address.isPrivate() && !_parent->ui_checkbox_filterip->isChecked()) )
	  if(_parent->ui_listbox_hostlist->count() < _max_hosts)
	    _parent->ui_listbox_hostlist->insertItem(p->_address.toString().c_str());

	} catch(...) { std::cerr << "EXC Connections::checkMessages Pong" << std::endl; }
      }
      break;

    case CMD_QUERY:
      {
 	try
 	  {
 	    auto_ptr<Query> q(msg.getQuery());
	    
	    PongCache::getInstance()->addHost(msg.id(), msg.hops(), s->_servant_number);

 	    _statistic.in_query_n++;
 	    _statistic.in_query_b += q->_data.size();
	    
 	    if(_parent->ui_checkbox_incomingsearch->isChecked())
 	      _parent->ui_listbox_incomingsearch->insertItem(q->_search.c_str(), 0);
	    
 	    while(_parent->ui_listbox_incomingsearch->count() > 40)
 	      _parent->ui_listbox_incomingsearch->removeItem(40);
	    
 	    int p = -1;
	    
 	    for(int i = 0; i < _search.size(); i++) if(q->_id == _search[i]->id) p = 0;
 	    if( (p == 0) || (q->_ttl < 1) ) break;
	    
 	    q->_ttl--;
 	    q->_hops++;
	    
	    for(int i = 0; i < _servants.size(); ++i)
	      if(_servants[i] != s) 
		{
		  _servants[i]->sendMessage(q->_data);
		  _statistic.out_query_n++;
		  _statistic.out_query_b += q->_data.size();
		}
	    
 	    s->_id_set.insert( q->_id );

	    if( _last_query_time >= (time(NULL) - 1) ) break; // 1 Query /s
	    _last_query_time = time(NULL);

	    // search query in shared file list
	    std::vector<std::string>   vstr;
	    std::string                search = q->_search;
	    std::string::size_type     idx;
	    
	    // transform(search.begin(), search.end(), search.begin(), tolower);
	    for(int i = 0; i < search.size(); ++i) search[i] = tolower(search[i]);

	    while( (idx = search.find_first_of(" ")) != std::string::npos )
	      {
		vstr.push_back(search.substr(0, idx));
		search = search.substr(idx+1);
	      }

	  
	    vstr.push_back(search);
	    
	    std::vector<QueryResult> qr;

	    // cpu usage problem 
	    for(int k = 0; k < _parent->vSharedFiles.size(); k++)  // for each shared file...
	      {
		bool         found_all = true;
		std::string  file = _parent->vSharedFiles[k]->file;
		
		//transform(file.begin(), file.end(), file.begin(), tolower);
		for(int i = 0; i < file.size(); ++i) file[i] = tolower(file[i]);
		
		for(int l = 0; l < vstr.size(); l++) // for each word in query...
		  if( file.find(vstr[l].c_str()) == std::string::npos ) found_all = false;
		
		if(found_all) qr.push_back(QueryResult(k, _parent->vSharedFiles[k]->size, _parent->vSharedFiles[k]->file));
	      }
	    //

	    int speed[] = {56, 64, 128, 350, 1000, 3000};
	    
	    if(qr.size() > 0)
	      {
		QueryHit qh(q->_id, q->_hops, 0, _address, speed[_parent->ui_combobox_speed->currentItem()], _id, qr);
		s->sendMessageDirect( qh._data );
		_statistic.out_hit_n++;
		_statistic.out_hit_b += qh._data.size();
	      }
	    
	  }
	catch(...) 
	  { 
	    #ifdef QTELLA_WARNINGS
 	      std::cerr << "EXC Connections::checkMessages Query" << std::endl; 
	    #endif
	  }
      }
      break;

    case CMD_QUERYHIT:
      {
	try{

	auto_ptr<QueryHit> q(msg.getQueryHit());

	PongCache::getInstance()->addHost(msg.id(), msg.hops(), s->_servant_number);

	_statistic.in_hit_n++;
	_statistic.in_hit_b += q->_data.size();

	int pos = -1;
	for(int i = 0; i < _search.size(); i++) if(q->_id == _search[i]->id) pos = i;

	if( pos > -1 )
	  {
	    if( (_search[pos]->active) && (q->_speed>=_search[pos]->_min_speed) )
	      for(int n = 0;  n < q->_results.size(); n++)
		{
		  // start content type check
		  std::string type;
		  std::string filename(q->_results[n].filename);
		  bool        found(false);
		  switch(_search[pos]->_type)
		    {
		    case 1: type = _parent->_types_images; break;
		    case 2: type = _parent->_types_videos; break;
		    case 3: type = _parent->_types_music; break;
		    }
		  if(!type.empty())
		    {
		      std::vector<std::string> type_vect;
		      StringManipulation       sm(type);
		      sm.split_to_vector(type_vect, ';');
		      
		      for(int j = 0; j < type_vect.size(); ++j)
			if(filename.find( ("."+type_vect[j]) ) != std::string::npos) found = true;
		    }
		  else found = true;
		  // end content type check

		  // check status of host
		  QColor      color(0x60, 0x60, 0x60);
		  std::string vendor;
		  int         status = QueryHitEntry::Unknown;

		  if(q->_trailer.size() > 6)
		    {
		      vendor = q->_trailer.substr(0, 4);
		      
		      char flag  = q->_trailer[5];
		      char flag2 = q->_trailer[6];
		      
		      if( flag & 4 )
			{
			  if( flag2 & 4 )
			    {
			      status = QueryHitEntry::Busy;
			      color = QColor(0xB0, 0x30, 0x30);  // busy, red
			      if(_parent->ui_checkbox_ignorebusy->isChecked()) found = false;
			    }
			  else
			    {
			      status = QueryHitEntry::Ready;
			      color = QColor(0x30, 0xB0, 0x30);  // not busy, green
			    }
			}
		      
		      if( ((flag2 & 1)&&(flag & 1)) || q->_address.isPrivate() )  // firewalled
			{
			  status |= QueryHitEntry::Firewalled;
			  if(color==QColor(0x30,0xB0,0x30)) color = QColor(0xB0,0xB0,0x30);
			  if(_parent->ui_checkbox_ignorefirewalled->isChecked()) found = false;
			}
		    }
		  else
		    {
		      if(q->_address.isPrivate()) 
			{
			  status |= QueryHitEntry::Firewalled;  
			  color = QColor(0xB0,0xB0,0x30);
			}
		      if(_parent->ui_checkbox_ignorefirewalled->isChecked()) found = false;
		    }

		  if(vendor == "QTEL") _statistic.hits_qtel_n++;
		  else if(vendor == "LIME") _statistic.hits_lime_n++;
		  else if(vendor == "BEAR") _statistic.hits_bear_n++;
		  else _statistic.hits_othe_n++;

		  //end check status


		  if(found)
		    {
		      std::strstream str_size;
		      str_size << q->_results[n].filesize << std::ends;
		      std::string ssize;
		      ssize = str_size.str();
		      str_size.freeze(false);
		      ssize = StringManipulation::insertDelimiter(ssize, ',', 3);

		      // create pixmap
		      uint32_t speed = q->_speed;
		      if(speed > 1100) speed = 1100;
		      speed = 62 * speed / 1100;
		      
		      QImage img(64, 11, 32, 16);
		      img.fill(qRgb(0xff, 0xff, 0xff));
		      for(int x = 0; x < 64; x++)
			{
			  img.setPixel(x, 0, qRgb(0xaa, 0xaa, 0xaa));
			  img.setPixel(x, 10, qRgb(0xaa, 0xaa, 0xaa));
			}
		      for(int y = 0; y < 11; y++)
			{
			  img.setPixel(0, y, qRgb(0xaa, 0xaa, 0xaa));
			  img.setPixel(63, y, qRgb(0xaa, 0xaa, 0xaa));
			}
		      QPixmap pict;
		      
		      for(int y = 1; y <= 9; ++y)
			for(int x = 1; x <= speed; ++x) img.setPixel(x, y, color.rgb());
		      
		      pict.convertFromImage(img);
		      // end pixmap

		      _search[pos]->vSearchResults.push_back(new QueryHitEntry(q->_results[n].filename,
									       q->_results[n].filesize,
									       q->_results[n].index,
									       q->_speed,
									       status,
									       q->_address,
									       q->_id_result, NULL) );
		      // create item
		      QListViewItem *item;
		      
		      if(_search[pos]->vSearchResults.size() == 0)
			item = new SearchViewItem(_search[pos]->sw->ui_listview_search, this);
		      else
			item = new SearchViewItem(_search[pos]->sw->ui_listview_search, this, _search[pos]->vSearchResults.back()->item);
		      
		      std::strstream str_hit;
		      str_hit << _search[pos]->vSearchResults.size() << std::ends;
		      item->setText(0, str_hit.str());
		      str_hit.freeze(false);

		      item->setPixmap(1, pict);
		      item->setText(2, q->_results[n].filename.c_str());
		      
		      std::string content;
		      if(filename.find_last_of(".") != std::string::npos) content = filename.substr(filename.find_last_of(".")+1);
		      item->setText(3, content.c_str());
		      item->setText(4, ssize.c_str());
		      item->setText(5, q->_address.strIP().c_str());		      
		      item->setText(6, vendor.c_str());

		      std::strstream str;
		      str << _search[pos]->query << " (" << _search[pos]->vSearchResults.size() << ")" << std::ends;

		      _parent->ui_tabwidget_search->changeTab((QWidget*)_search[pos]->sw, str.str());

		      if(_parent->ui_tabwidget_search->currentPage() != _search[pos]->sw)
			{
			  // TODO bold
			}

		      _search[pos]->vSearchResults.back()->item = item;
		      str.freeze(false);
		    }
		}
	  }
	else  // pos = -1
	  if(q->_ttl > 1)
	    {
	      q->_ttl--;
	      q->_hops++;
	      
	      for(int k = 0; k < _servants.size(); k++)
		if( (_servants[k]!=s) && (_servants[k]->_id_set.find(q->_id) != _servants[k]->_id_set.end()) )
		  {
		    _servants[k]->sendMessage(q->_data);
		    _statistic.out_hit_n++;
		    _statistic.out_hit_b += q->_data.size();
		  }
	    }
	} catch(...) { std::cerr << "EXC Connections::checkMessages QueryHit" << std::endl; }
      }
      break;

    case CMD_PUSH:
      {
	try{

	auto_ptr<Push> p(msg.getPush());

	PongCache::getInstance()->addHost(msg.id(), msg.hops(), s->_servant_number);

	_statistic.in_push_n++;
	_statistic.in_push_b += msg.size();

	if(p->_id_payload == _id)
	  _parent->upload_manager->addPushUpload(*p);
	else
	  if(p->_ttl >= 1)
	    {
	      p->_ttl--;
	      p->_hops++;
	      for(int i = 0; i < _servants.size(); ++i)
		if(_servants[i] != s)
		  {
		    _servants[i]->sendMessage(p->_data);
		    _statistic.out_push_n++;
		    _statistic.out_push_b += p->_data.size();
		  }
	    }
	
	} catch(...) { std::cerr << "EXC Connections::checkMessages Push" << std::endl; }
	break;
      }
    }
    }//while
  }catch(...) { std::cerr << "EXC Connections::checkMessages" << std::endl; }

  #ifdef QTELLA_DEBUG
    std::clog << "Connections: ok check messages" << std::endl;
  #endif
}


void Connections::sendAll(std::string data)
{
  try{

  for(int i = 0; i < _servants.size(); ++i)
    {
      _servants[i]->sendMessage(data, false);
      _statistic.out_push_n++;
      _statistic.out_push_b += data.size();
    }

  } catch(...) { std::cerr << "EXC Connections::sendAll" << std::endl; }
}


std::string Connections::toHex(std::string data)
{
  std::string s;

  for(int i = 0; i < data.size(); ++i)
    {
      std::strstream str;
      str.width(2);
      str.fill('0');
      str.unsetf(std::ios::dec);
      str.setf(std::ios::hex);
      int a = (unsigned char)data[i];
      str << a << std::ends;
      s += str.str();
      str.freeze(false);
    }

  return s;
}


void Connections::search(std::string query, SearchWidget *sw)
{
  #ifdef QTELLA_DEBUG
    std::clog << "Connections: search" << std::endl;
  #endif

  try{

  std::vector<Servent*>::iterator  pos;
  uint16_t                         speed[] = {0, 64, 128, 350, 1000, 3000};

  std::string search_id = createID();
  _search.push_back(new Search(search_id, sw, speed[_parent->ui_combobox_searchspeed->currentItem()],
			       _parent->ui_combobox_search_type->currentItem() ));
  
  _search.back()->query = query;

  for(pos = _servants.begin(); pos != _servants.end(); ++pos) 
    {
      #ifdef QTELLA_DEBUG
        std::clog << "Connections: search send query" << std::endl;
      #endif
      Query q(7, 0, search_id, speed[_parent->ui_combobox_searchspeed->currentItem()], query);
      (*pos)->sendMessage(q._data, false);

      _statistic.out_query_n++;
      _statistic.out_query_b += q._data.size();
    }

  } catch(...) { std::cerr << "EXC Connections::search" << std::endl; }

  #ifdef QTELLA_DEBUG
    std::clog << "Connections: ok search" << std::endl;
  #endif
}


void Connections::slotRemoveHost()
{
  #ifdef QTELLA_DEBUG
    std::clog << "Connections: remove host" << std::endl;
  #endif

  try{

  QListViewItem *item = _parent->ui_listview_connections->currentItem();

  if(item)
    {
      std::vector<class Servent*>::iterator pos = _servants.begin();

      for(pos; pos != _servants.end(); )
	if( (*pos)->_item == item )
	  {
	    delete *pos;
	    pos = _servants.erase(pos);
	    break;
	  }
	else
	  ++pos;
    }

  } catch(...) { std::cerr << "EXC Connections::slotRemoveHost" << std::endl; }

  #ifdef QTELLA_DEBUG
    std::clog << "Connections: ok check host" << std::endl;
  #endif
}



void Connections::slotAddHost()
{
  try{

  std::string hostname = static_cast<string>(_parent->ui_lineedit_host->text());
  std::string port = static_cast<string>(_parent->ui_lineedit_port->text());

  Host host = Host(Address(hostname+":"+port));

  QListViewItem *item = 
    new QListViewItem(_parent->ui_listview_connections, host.address.toString().c_str(), "Connecting...");

  _servants.push_back(new Servent(this, _id, item));
  _servants.back()->connect(host.address);
  _parent->ui_listview_connections->insertItem(item);

  } catch(...) { std::cerr << "EXC Connections::slotAddHost" << std::endl; }
}



void Connections::slotMaxServents(int value)
{
  _max_servants = value;
}


void Connections::slotListClear()
{
  _parent->ui_listbox_hostlist->clear();
}


void Connections::slotListConnect()
{
  int i = _parent->ui_listbox_hostlist->currentItem();

  if(i != -1)
    {
      std::string addr = static_cast<string>(_parent->ui_listbox_hostlist->text(i));

      Host host = Host(Address(addr));
      
      QListViewItem *item = 
	new QListViewItem(_parent->ui_listview_connections, host.address.toString().c_str(), "Connecting...");

      if(!_parent->timer_connection->isActive()) _parent->timer_connection->start(500);

      _servants.push_back(new Servent(this, _id, item));
      _servants.back()->connect(host.address);
      _parent->ui_listview_connections->insertItem(item);
    }
}


void Connections::slotListRemove()
{
  int i = _parent->ui_listbox_hostlist->currentItem();

  if(i != -1)
    _parent->ui_listbox_hostlist->removeItem(i);
}


void Connections::slotUpdateStatistic()
{
  // incoming messages
  _parent->ui_in_ping_n->setText( _statistic.str(_statistic.in_ping_n, 1).c_str() );
  _parent->ui_in_pong_n->setText( _statistic.str(_statistic.in_pong_n, 1).c_str() );
  _parent->ui_in_query_n->setText( _statistic.str(_statistic.in_query_n, 1).c_str() );
  _parent->ui_in_hit_n->setText( _statistic.str(_statistic.in_hit_n, 1).c_str() );
  _parent->ui_in_push_n->setText( _statistic.str(_statistic.in_push_n, 1).c_str() );

  _parent->ui_in_all_n->setText( _statistic.str(_statistic.in_all_n(), 1).c_str() );

  _parent->ui_in_ping_b->setText( _statistic.str(_statistic.in_ping_b, 1).c_str() );
  _parent->ui_in_pong_b->setText( _statistic.str(_statistic.in_pong_b, 1).c_str() );
  _parent->ui_in_query_b->setText( _statistic.str(_statistic.in_query_b, 1).c_str() );
  _parent->ui_in_hit_b->setText( _statistic.str(_statistic.in_hit_b, 1).c_str() );
  _parent->ui_in_push_b->setText( _statistic.str(_statistic.in_push_b, 1).c_str() );

  _parent->ui_in_all_b->setText( _statistic.str(_statistic.in_all_b(), 1).c_str() );

  // outgoing messages
  _parent->ui_out_ping_n->setText( _statistic.str(_statistic.out_ping_n, 1).c_str() );
  _parent->ui_out_pong_n->setText( _statistic.str(_statistic.out_pong_n, 1).c_str() );
  _parent->ui_out_query_n->setText( _statistic.str(_statistic.out_query_n, 1).c_str() );
  _parent->ui_out_hit_n->setText( _statistic.str(_statistic.out_hit_n, 1).c_str() );
  _parent->ui_out_push_n->setText( _statistic.str(_statistic.out_push_n, 1).c_str() );

  _parent->ui_out_all_n->setText( _statistic.str(_statistic.out_all_n(), 1).c_str() );

  _parent->ui_out_ping_b->setText( _statistic.str(_statistic.out_ping_b, 1).c_str() );
  _parent->ui_out_pong_b->setText( _statistic.str(_statistic.out_pong_b, 1).c_str() );
  _parent->ui_out_query_b->setText( _statistic.str(_statistic.out_query_b, 1).c_str() );
  _parent->ui_out_hit_b->setText( _statistic.str(_statistic.out_hit_b, 1).c_str() );
  _parent->ui_out_push_b->setText( _statistic.str(_statistic.out_push_b, 1).c_str() );

  _parent->ui_out_all_b->setText( _statistic.str(_statistic.out_all_b(), 1).c_str() );

  // hits
  _parent->ui_hits_qtel_n->setText( _statistic.str(_statistic.hits_qtel_n, 1).c_str() );
  _parent->ui_hits_lime_n->setText( _statistic.str(_statistic.hits_lime_n, 1).c_str() );
  _parent->ui_hits_bear_n->setText( _statistic.str(_statistic.hits_bear_n, 1).c_str() );
  _parent->ui_hits_othe_n->setText( _statistic.str(_statistic.hits_othe_n, 1).c_str() );

  unsigned long a = _statistic.hits_all_n();
  if( a > 0 )
    {
      _parent->ui_hits_qtel_p->setText( _statistic.str(static_cast<double>(100*_statistic.hits_qtel_n)/a, 1).c_str() );
      _parent->ui_hits_lime_p->setText( _statistic.str(static_cast<double>(100*_statistic.hits_lime_n)/a, 1).c_str() );
      _parent->ui_hits_bear_p->setText( _statistic.str(static_cast<double>(100*_statistic.hits_bear_n)/a, 1).c_str() );
      _parent->ui_hits_othe_p->setText( _statistic.str(static_cast<double>(100*_statistic.hits_othe_n)/a, 1).c_str() );
    }

  // down / uploads
  _parent->ui_loads_usucc->setText( _statistic.str(_statistic.loads_usucc, 0).c_str() );
  _parent->ui_loads_uabor->setText( _statistic.str(_statistic.loads_uabor, 0).c_str() );
  _parent->ui_loads_usize->setText( _statistic.str(_statistic.loads_usize, 1).c_str() );
  _parent->ui_loads_dsucc->setText( _statistic.str(_statistic.loads_dsucc, 0).c_str() );
  _parent->ui_loads_dabor->setText( _statistic.str(_statistic.loads_dabor, 0).c_str() );
  _parent->ui_loads_dsize->setText( _statistic.str(_statistic.loads_dsize, 1).c_str() );

  // shared files
  _parent->ui_files_n->setText( _statistic.str(_statistic.shared_files, 0).c_str() );
  _parent->ui_files_b->setText( _statistic.str(_statistic.shared_size, 1).c_str() );

  // uptime
  QDateTime* dt = new QDateTime( QDate::currentDate(), QTime::currentTime() );

  int sto = _uptime->secsTo(*dt);
  
  int days = sto / 86400;
  sto -= days * 86400;
  int h = sto / 3600;
  sto -= h * 3600;
  int m = sto / 60;
  sto -= m * 60;
  int s = sto;

  std::strstream str;

  if(days > 0) str << days << "d ";
  if(h > 0) str << h << "h ";
  if(h > 0 || m > 0) str << m << "m ";
  if(days==0 && h==0) str << s << "s";
  str << std::ends;

  _parent->ui_uptime->setText( str.str() );
  str.freeze(false);
  delete dt;
}
