#include "Download.h"
#include "DownloadManager.h"
#include "StringManipulation.h"
#include "QtellaSub.h"
#include "Connections.h"
#include "Push.h"

#include <qsocket.h>
#include <qlistview.h>
#include <qfile.h>
#include <qdir.h>
#include <qradiobutton.h>
#include <qspinbox.h>
#include <qlineedit.h>

#include <iostream>
#include <iomanip>
#include <strstream>
#include <ctime>

Download::Download(QueryHitEntry qhe, QListViewItem *item, DownloadManager *parent) : 
  _parent(parent), _item(item), _qhe(qhe), _state(dtQueued), _content_length(qhe.size), _retries(0),
  sock(NULL), _active(false)
{
  timeout = new QTimer();
  timer_bandwidth = new QTimer();
  timer_retry = new QTimer();
  
  if(!_qhe.firewalled())
    {
      sock = new QSocket();

      connect(sock, SIGNAL(readyRead()), this, SLOT(slotReadyRead()));
      connect(sock, SIGNAL(connectionClosed()), this, SLOT(slotConnectionClosed()));
      connect(sock, SIGNAL(error(int)), this, SLOT(slotError(int)));
      connect(sock, SIGNAL(connected()), this, SLOT(slotConnected()));
    }
      
  connect(timeout, SIGNAL(timeout()), this, SLOT(slotTimeout()));
  connect(timer_retry, SIGNAL(timeout()), this, SLOT(slotRetry()));
  connect(timer_bandwidth, SIGNAL(timeout()), this, SLOT(slotTimerBandwidth()));

  _client.erase();
  _first_start = true;
  _f = NULL;
}


void Download::run()
{
  setState(dtConnecting);
  _active = true;

  if(timer_retry->isActive()) timer_retry->stop();
  if(timeout->isActive()) timeout->stop();
  if(timer_bandwidth->isActive()) timer_bandwidth->stop();

  bandwidth     = 0;
  tmp_read      = 0;
  _buffer.erase();
  _header.erase();
  retry_timeout = 30;

  std::string host = _qhe.addr.strIP();


  complete_path   = _parent->_parent->ui_lineedit_downloaddirectory->text();
  if(complete_path[complete_path.size()-1] != '/') complete_path += "/";
  complete_path += _qhe.filename;

  incomplete_path = _parent->_parent->ui_lineedit_incompleted->text();
  if(incomplete_path[incomplete_path.size()-1] != '/') incomplete_path += "/";
  incomplete_path += _qhe.filename;


  QDir d(incomplete_path.substr(0, incomplete_path.find_last_of("/")).c_str());

  
  if(!d.exists())
    {
      std::cerr << "Directory for incomplete downloads does not exist." << std::endl;
      setState(dtError);
      return;
    }

  d.setPath(complete_path.substr(0, complete_path.find_last_of("/")).c_str());

  if(!d.exists())
    {
      std::cerr << "Directory for completed downloads does not exist." << std::endl;
      setState(dtError);
      return;
    }


  if(_first_start)
    {
      _first_start = false;

      QFile file(complete_path.c_str());

      if( file.exists() && _parent->_parent->ui_radiobutton_e_abort->isChecked() )
	{
	  _f = new QFile(complete_path.c_str());
	  setState(dtExists);
	  return;
	}

      file.setName(incomplete_path.c_str());


      if( _parent->_parent->ui_radiobutton_a_startnew->isChecked() )
	{
	  std::string s = createFilename(incomplete_path, _qhe.size, _qhe.addr.strIP());
	  _f = new QFile(s.c_str());
	}
      else
	{
	  std::string s;

	  if( _parent->_parent->ui_radiobutton_a_contsize->isChecked() )
	    s = findFile(incomplete_path, _qhe.size);

	  if( _parent->_parent->ui_radiobutton_a_conthost->isChecked() )
	    s = findFile(incomplete_path, _qhe.addr.strIP());

	  if( _parent->_parent->ui_radiobutton_a_contsizehost->isChecked() )
	    s = findFile(incomplete_path, _qhe.size, _qhe.addr.strIP());
	      
	  if(!s.empty())
	    _f = new QFile(s.c_str());
	  else
	    {
	      std::string s = createFilename(incomplete_path, _qhe.size, _qhe.addr.strIP());
	      _f = new QFile(s.c_str());
	    }
	}

    }
  else
    if(_f->isOpen()) _f->close();

  
  if(!_f->isOpen()) _f->open(IO_Append|IO_ReadWrite);
  if(!_f->isOpen())
    {
      std::cerr << "Cannot open file for writing." << std::endl;
      slotError(dtError);
      return;
    }

  if(_f->size() > 0) retry_timeout = 15;

  timeout->start(20000, true);

  if(!_qhe.firewalled())
    sock->connectToHost(_qhe.addr.strIP().c_str(), _qhe.addr.port());
  else
    {
      std::string id = _parent->_parent->connections->_id;
      Address addr( std::string(_parent->_parent->ui_lineedit_ip->text()) );
      addr.setPort( std::string(_parent->_parent->ui_lineedit_listenport->text()) );
      Push p(7, 0, id, _qhe.id, _qhe.index, addr);
      _parent->_parent->connections->sendAll( p._data );
      setState(dtPushed);
    }
}


void Download::startPushedDownload(QSocket *socket)
{
  setState(dtConnecting);
  sock = socket;
  
  connect(sock, SIGNAL(readyRead()), this, SLOT(slotReadyRead()));
  connect(sock, SIGNAL(connectionClosed()), this, SLOT(slotConnectionClosed()));
  connect(sock, SIGNAL(error(int)), this, SLOT(slotError(int)));
  
  slotConnected();
}


const int Download::readState()
{
  return _state;
}


void Download::setState(int i)
{
  _state = i;
}


const unsigned long Download::bytesRead()
{
  if(_f)
    if(_f->exists())
      return _f->size();

  return 0;
}


const unsigned long Download::bytesToRead()
{
  return _content_length - bytesRead();
}


void Download::slotReadyRead()
{
  char cbuff[32768+10];  // do we really need these 10 bytes?? 
                         // without them I often get an segfault

  int w = sock->readBlock(cbuff, 32768);

  _buffer.append(cbuff, w);
  tmp_read += w;

  if(_header.empty())
    {
      std::string::size_type  pos = _buffer.find("\r\n\r\n");
      
      if( pos != std::string::npos)
	{
	  _header = _buffer.substr(0, pos+4);
	  _buffer = _buffer.substr(pos+4);

	  std::string::size_type  idx1 = _header.find(" ");
	  std::string::size_type  idx2 = _header.find(" ", idx1+1);

	  if( _header.substr(idx1+1, idx2-idx1-1) == "404" ) 
	    {
	      setState(dtNotFound);
	      return;
	    }
	  
	  if( _header.substr(idx1+1, idx2-idx1-1) == "503" ) 
	    {
	      setState(dtBusy);
	      timer_retry->start(1000);
	      return;
	    }
	  
	  if( _header.substr(idx1+1, idx2-idx1-1) == "200" ) 
	    {
	      setState(dtConnected);
	      timer_bandwidth->start(4000);
	      bandwidth = 0;
	    }

	  // get server name
	  
	  StringManipulation sm(_header);
	  if( ((idx1 = sm.to_lower().find("server:")) != std::string::npos) && (_client.empty()) )
	    {
	      idx1 += 7;  // point to character after ':'
	      idx1 = _header.find_first_not_of(" ", idx1);
	      idx2 = _header.find("\r\n", idx1);
	      _client =  _header.substr(idx1, idx2-idx1);
	    }
	  
	  // save time
	  _start_time = time(NULL);
	}
    }

  if( (!_buffer.empty()) && (readState()==dtConnected) )
    {
      int w = _f->writeBlock(_buffer.data(), _buffer.size());
      _f->flush();
      _parent->_parent->connections->_statistic.loads_dsize += w;
      _buffer.erase(0, w);
    }
  
  if(bytesRead() == _content_length) finish();
}


void Download::finish()
{
  sock->close();
  setState(dtClosed);
  _active   = false;
  _end_time = time(NULL);

  if(readState() != dtBusy)
    if(bytesRead() == _content_length)
      {
	_parent->_parent->connections->_statistic.loads_dsucc++;
	setState(dtClosedCompleted);
	
	_f->close();

	std::string destination = complete_path;
	int    i = 0;

	if(_parent->_parent->ui_radiobutton_e_rename->isChecked() || _parent->_parent->ui_radiobutton_e_abort->isChecked())
	  {
	    QFile d(destination.c_str());

	    std::string di = complete_path.substr(0, complete_path.find_last_of("/"));
	    std::string fi = complete_path.substr(complete_path.find_last_of("/")+1);

	    while(d.exists())
	      {
		destination = (di + "/");
		
		std::strstream str;
		str << i++ << std::ends;
		destination += str.str();
		str.freeze(false);
		
		destination += "_" + fi;

		d.setName(destination.c_str());
	      }
	  }

	QFile dest(destination.c_str());

	QDir d;
	if( !d.rename(_f->name(), destination.c_str()) )
	  std::cerr << "Error while moving file from incomleted path to completed path." << std::endl;
	else
	  {
	    _f->setName(destination.c_str());
	    _filename = destination;
	  }
      }
    else
      {
	_parent->_parent->connections->_statistic.loads_dabor++;
	setState(dtClosed);
	timer_retry->start(1000);
      }
}


void Download::slotConnectionClosed()
{
  while(sock->bytesAvailable() > 0) slotReadyRead();

  if(readState() == dtConnected) _f->writeBlock(_buffer.data(), _buffer.size());
  _buffer.erase();
  _f->flush();

  finish();

  _end_time = time(NULL);
}


void Download::slotError(int error)
{
  setState(dtError);
  timer_retry->start(1000);
}


void Download::slotConnected()
{
  std::string     s = _qhe.filename;
  unsigned long   begin = _f->size();
  std::strstream  str;

  str << "GET /get/" << _qhe.index << "/" << s << " HTTP/1.0\r\n"
      << "Range: bytes=" << begin << "-\r\n"
      << "User-Agent: Qtella " << VERSION << "\r\n\r\n" << std::ends;

  s = str.str();
  str.freeze(false);

  sock->writeBlock(s.c_str(), s.size());
}



void Download::slotTimerBandwidth()
{
  bandwidth = static_cast<double>(tmp_read) / 4096;  // KB/s
  tmp_read = 0;
}


const double Download::readBandwidth()
{
  return bandwidth;
}


void Download::slotTimeout()
{
  timeout->stop();

  if(readState() == dtConnecting)
    {
      setState(dtTimeout);
      sock->close();
      timer_retry->start(1000);
    }
}


void Download::abortDownload()
{
  timeout->stop();
  timer_bandwidth->stop();
  timer_retry->stop();
  if(sock) sock->close();
  
  if(_f) _f->close();
  
  _parent->_parent->connections->_statistic.loads_dabor++;
  setState(dtAbort);
  _active = false;
}


void Download::resumeDownload()
{
  _retries = 0;
  setState(dtQueued);
}


Download::~Download()
{
  delete timeout;
  delete timer_bandwidth;
  delete timer_retry;
  if(sock) delete sock;
  delete _item;

  if(_f)
    {
      if(_f->isOpen()) _f->close();
      if(_f->size() == 0) _f->remove();
      delete _f;
    }
}

void Download::slotRetry()
{
  retry_timeout--;

  if( retry_timeout == 0 )  
    {
      if(timer_retry->isActive()) timer_retry->stop();
      if(timeout->isActive()) timeout->stop();
      if(timer_bandwidth->isActive()) timer_bandwidth->stop();

      setState(dtQueued);
      _retries++;

      int n = _parent->_parent->ui_spinbox_conf_retries->value();
      if( (n != -1) && (_retries > n) ) 
	{
	  setState(dtError);
	  _active = false;
	}
    }
}


const int Download::readRetryTimeout()
{
  return retry_timeout;
}


std::string Download::createFilename(std::string file, unsigned long size, std::string host)
{
  time_t t  = time(NULL);

  for(int id = rand()%1000; ; ++id)
    {
      std::strstream str;
      std::string    s;

      std::string::size_type  idx = file.find_last_of(".");
      std::string             ext;
      std::string             filename = file;
      if(idx != std::string::npos)
	{
	  filename = file.substr(0, idx);
	  ext = file.substr(idx);  // with "." !!
	}

      str << ".part_" << t << "_" << size << "_" << host << "_" << id << std::ends;
      s = filename + std::string(str.str()) + ext;
      str.freeze(false);

      QFile f(s.c_str());

      if(!f.exists()) return s;
    }
}


std::string Download::findFile(std::string file, unsigned long size)
{
  std::string::size_type  idx = file.find_last_of("/");

  std::string    directory = file.substr(0, idx);
  std::string    filename  = file.substr(idx+1);
  unsigned long  current_size = 0;

  QDir d(directory.c_str(), (filename+"*").c_str(), QDir::DefaultSort, QDir::Files);

  std::string s;

  for(int i = 0; i < d.count(); ++i)
    {
      filename = d[i];
      idx = filename.find(".part_");

      if( idx != std::string::npos )
	{
	  idx = filename.find("_", idx+6);  // find first "_"
	  std::string::size_type  idx2 = filename.find("_", idx+1); // find second "_" 

	  std::string s_size = filename.substr(idx+1, idx2-idx-1);  // extract filesize
	  
	  std::strstream str;
	  str << s_size << std::ends;
	  unsigned long str_size;
	  str >> str_size;
	  
	  std::string s_new = directory + "/" + filename;
	  QFile f(s_new.c_str());

	  if( (str_size == size) && (f.size() > current_size) )
	    {
	      s = s_new;
	      current_size = f.size();
	    }
	}
    }

  return s;
}


std::string Download::findFile(std::string file, std::string host)
{
  std::string::size_type  idx = file.find_last_of("/");

  string         directory = file.substr(0, idx);
  string         filename  = file.substr(idx+1);
  unsigned long  current_size = 0;

  QDir d(directory.c_str(), (filename+"*").c_str(), QDir::DefaultSort, QDir::Files);

  std::string s;

  for(int i = 0; i < d.count(); ++i)
    {
      filename = d[i];
      idx = filename.find(".part_");

      if( idx != std::string::npos )
	{
	  idx = filename.find("_", idx+6);
	  idx = filename.find("_", idx+1);
	  std::string::size_type  idx2 = filename.find("_", idx+1);

	  std::string s_host = filename.substr(idx+1, idx2-idx-1);

	  if( (s_host == host) )
	    {
	      std::string s_new = directory + "/" + filename;
	      QFile f(s_new.c_str());

	      if( f.size() > current_size )
		{
		  s = s_new;
		  current_size = f.size();
		}
	    }
	}
    }

  return s;
}


std::string Download::findFile(std::string file, unsigned long size, std::string host)
{
  std::string::size_type  idx = file.find_last_of("/");

  std::string         directory = file.substr(0, idx);
  std::string         filename  = file.substr(idx+1);
  unsigned long  current_size = 0;

  QDir d(directory.c_str(), (filename+"*").c_str(), QDir::DefaultSort, QDir::Files);

  std::string s;

  for(int i = 0; i < d.count(); ++i)
    {
      filename = d[i];
      idx = filename.find(".part_");

      if( idx != std::string::npos )
	{
	  idx  = filename.find("_", idx+6);

	  std::string::size_type  idx2;
	  std::string::size_type  idx3;

	  idx2 = filename.find("_", idx+1);
	  idx3 = filename.find("_", idx2+1);

	  std::string s_size = filename.substr(idx+1, idx2-idx-1);
	  std::string s_host = filename.substr(idx2+1, idx3-idx2-1);

	  std::strstream str;
	  str << s_size << std::ends;
	  unsigned long str_size;
	  str >> str_size;

	  if( (s_host == host) && (str_size == size) )
	    {
	      std::string s_new = directory + "/" + filename;
	      QFile f(s_new.c_str());

	      if( f.size() > current_size )
		{
		  s = s_new;
		  current_size = f.size();
		}
	    }
	}
    }

  return s;
}


void Download::slotUpdateStatus()
{
  static const std::string status[] = 
  {
    "Idle", "Error", "Abort", "Closed", "Completed", "Closed",
    "Connecting", "NotFound", "Timeout", "Busy", "Connected", "Exist",
    "Queued", "Push" 
  };

  std::string s[8];
  for(int i = 0; i < 8; ++i) s[i] = "";

  s[1] = status[readState()];

  if( (readState()==dtBusy) || (readState()==dtClosed) || (readState()==dtTimeout) || 
      ( (readState()==dtError) && _active ) )
    {
      s[1] += " - ";
      std::strstream str;
      str << readRetryTimeout() << "s" << std::ends;
      s[1] += str.str();
      str.freeze(false);
    }

  if( readState() == dtClosedCompleted )
    {
      time_t         diff = _end_time - _start_time;
      std::strstream str;
      int            h   = diff / 3600;
      int            m   = (diff - h * 3600) / 60;
      int            sec = diff - h * 3600 - m * 60;
      str.fill('0');
      str << h << ":" << std::setw(2) << m << ":" << std::setw(2) << sec << std::ends;
      s[6].assign(std::string(str.str()));
      str.freeze(false);

      if(diff>0)
	{
	  double b = bytesRead() / diff / 1024;
	  std::strstream str;
	  str.setf(ios::fixed);
	  str << std::setprecision(2) << b << " KB/s" << std::ends;
	  s[5] = str.str();
	  str.freeze(false);
	}
    }

  if( readState() == dtConnected )
    {
      // bandwidth
      std::strstream str;
      str.setf(ios::fixed);
      str << std::setprecision(2) << readBandwidth() << " KB/s" << std::ends;
      s[5] = str.str();
      str.freeze(false);

      // calculate time to go
      uint32_t seconds   = 0;
      uint32_t toread    = bytesToRead();
      uint32_t read      = bytesRead();
      double bandwidth   = readBandwidth();
      if(read > 0) seconds = toread / read * (time(NULL) - _start_time);
      //if(bandwidth > 0.01) seconds = static_cast<double>(toread) / 1024 / bandwidth;
      int      h   = seconds / 3600;
      int      m   = (seconds - h * 3600) / 60;
      int      sec = seconds - h * 3600 - m * 60;
      std::strstream time_str;
      time_str.fill('0');
      time_str << h << ":" << std::setw(2) << m << ":" << std::setw(2) << sec << std::ends;
      s[6] = time_str.str();
      time_str.freeze(false);
    }

  // progress
  double p = 100;
  if(_content_length > 0) p = static_cast<double>(100) * static_cast<double>(bytesRead()) / static_cast<double>(_content_length);
  std::strstream str;
  str.setf(std::ios::fixed);
  str << setprecision(0) << p << "%" << std::ends;
  s[3] = str.str();
  str.freeze(false);

  s[0] = _qhe.filename;
  s[4] = _qhe.addr.strIP();
  s[7] = _client;

  str.seekp(0);
  str << _qhe.size << std::ends;
  s[2] = StringManipulation::insertDelimiter(str.str(), ',', 3);
  str.freeze(false);

  for(int i = 0; i < 8; ++i) _item->setText(i, s[i].c_str());
}


bool Download::isActive()
{
  return _active;
}


void Download::moveFile(std::string directory)
{
  std::string current(_filename);
  if(current.find_last_of("/") != std::string::npos) current.erase(0, current.find_last_of("/")+1);

  QDir d;
  d.rename(_filename.c_str(), (directory + current).c_str());
  _f->setName( (directory+current).c_str() );

  _filename = directory + current;
}
