#include "QtellaSub.h"
#include "Gnutella.h"

#include "Connections.h"
#include "StringManipulation.h"
#include "QueryHitEntry.h"
#include "Info.h"
#include "SearchWidget.h"
#include "Search.h"
#include "SharedFile.h"
#include "UploadManager.h"
#include "DownloadManager.h"
#include "IncomingConnect.h"
#include "SharedViewItem.h"
#include "AddApplication.h"
#include "Download.h"
#include "MP3property.h"

#include <fstream>
#include <algorithm>
#include <iomanip>
#include <strstream>
#include <iostream>
#include <cctype>
#include <unistd.h>

#include <qspinbox.h>
#include <qlistbox.h>
#include <qmenubar.h>
#include <qpopupmenu.h>
#include <qlayout.h>
#include <qtimer.h>
#include <qlineedit.h>
#include <qlistview.h>
#include <qtabwidget.h>
#include <qtabbar.h>
#include <qlabel.h>
#include <qfile.h>
#include <qdir.h>
#include <qstringlist.h>
#include <qstring.h>
#include <qcombobox.h>
#include <qradiobutton.h>
#include <qfiledialog.h>
#include <qcheckbox.h>
#include <qthread.h>
#include <qapplication.h>
#include <qpushbutton.h>
#include <qpopupmenu.h>
#include <qfiledialog.h>
#include <qiconset.h>

#include <signal.h>

#ifdef KDE
#include "ktmainwindow.h"
#endif

/* 
 *  Constructs a QtellaSub which is a child of 'parent', with the 
 *  name 'name' and widget flags set to 'f' 
 *
 *  The dialog will by default be modeless, unless you set 'modal' to
 *  TRUE to construct a modal dialog.
 */
QtellaSub::QtellaSub( QApplication* parent,  const char* name, WFlags fl )
  : Qtella( 0, name, fl ), _binary(name)
{
  #ifdef KDE
    KPopupMenu *help = new KPopupMenu((KTMainWindow*)this);
  #endif

  #ifdef QT
    QPopupMenu *help = new QPopupMenu(this);
  #endif

  help->insertItem("&Info", this, SLOT(slotInfo()));


  #ifdef KDE
    KPopupMenu *file = new KPopupMenu((KTMainWindow*)this);
  #endif

  #ifdef QT
    QPopupMenu *file = new QPopupMenu(this);
  #endif

  file->insertItem("&Connect", this, SLOT(slotConnect()));
  file->insertItem("&Disconnect", this, SLOT(slotDisconnect()));
  file->insertSeparator();
  file->insertItem("E&xit", parent, SLOT(quit()));


  #ifdef KDE
    menu = new KMenuBar((KTMainWindow*)this);
  #endif

  #ifdef QT
    menu = new QMenuBar(this);
  #endif

  menu->insertItem("&File", file);
  menu->insertSeparator();
  menu->insertItem("&Help", help);



  
  std::strstream str;
  str << "Qtella - " << VERSION << " - http://www.qtella.net" << std::ends;
  std::string caption = str.str();
  str.freeze(false);
  setCaption( caption.c_str() );

  connections = new Connections(this);

  timer_connection = new QTimer(this);
  connect(timer_connection, SIGNAL(timeout()), this, SLOT(check_connections()));

  connect((QObject*)ui_pushbutton_remove, SIGNAL(clicked()), (QObject*)connections, SLOT(slotRemoveHost()));
  connect((QObject*)ui_spinbox_hosts, SIGNAL(valueChanged(int)), (QObject*)connections, SLOT(slotMaxServents(int)));
  connect((QObject*)ui_pushbutton_listclear, SIGNAL(clicked()), (QObject*)connections, SLOT(slotListClear()));
  connect((QObject*)ui_pushbutton_listconnect, SIGNAL(clicked()), (QObject*)connections, SLOT(slotListConnect()));
  connect((QObject*)ui_pushbutton_listremove, SIGNAL(clicked()), (QObject*)connections, SLOT(slotListRemove()));


  ui_listview_download->setColumnWidthMode(0, QListView::Manual);
  ui_listview_download->setColumnWidthMode(1, QListView::Manual);
  ui_listview_download->setColumnWidthMode(2, QListView::Manual);
  ui_listview_download->setColumnWidthMode(3, QListView::Manual);
  ui_listview_download->setColumnWidthMode(4, QListView::Manual);
  ui_listview_download->setColumnWidthMode(5, QListView::Manual);
  ui_listview_download->setColumnWidthMode(6, QListView::Manual);
  ui_listview_download->setColumnWidthMode(7, QListView::Manual);
  ui_listview_download->setColumnWidth(0, 16*10);
  ui_listview_download->setColumnWidth(1, 7*10);
  ui_listview_download->setColumnWidth(2, 7*10);
  ui_listview_download->setColumnWidth(3, 5*10);
  ui_listview_download->setColumnWidth(4, 9*10);
  ui_listview_download->setColumnWidth(5, 7*10);
  ui_listview_download->setColumnWidth(6, 5*10);
  ui_listview_download->setColumnWidth(7, 7*10);
  ui_listview_download->setColumnAlignment(0, AlignLeft);
  ui_listview_download->setColumnAlignment(1, AlignLeft);
  ui_listview_download->setColumnAlignment(2, AlignRight);
  ui_listview_download->setColumnAlignment(3, AlignRight);
  ui_listview_download->setColumnAlignment(4, AlignRight);
  ui_listview_download->setColumnAlignment(5, AlignRight);
  ui_listview_download->setColumnAlignment(6, AlignRight);
  ui_listview_download->setColumnAlignment(7, AlignLeft);
  ui_listview_download->setSorting(-1);
  ui_listview_download->setColumnWidth(0, 30*10);
  
  ui_listview_connections->setColumnWidthMode(0, QListView::Manual);
  ui_listview_connections->setColumnWidthMode(1, QListView::Manual);
  ui_listview_connections->setColumnWidthMode(2, QListView::Manual);
  ui_listview_connections->setColumnWidthMode(3, QListView::Manual);
  ui_listview_connections->setColumnWidth(0, 18*10);
  ui_listview_connections->setColumnWidth(1, 9*10);
  ui_listview_connections->setColumnWidth(2, 11*10);
  ui_listview_connections->setColumnWidth(3, 7*10);
  ui_listview_connections->setColumnWidth(5, 7*10);
  ui_listview_connections->setColumnAlignment(0, AlignLeft);
  ui_listview_connections->setColumnAlignment(1, AlignLeft);
  ui_listview_connections->setColumnAlignment(2, AlignHCenter);
  ui_listview_connections->setColumnAlignment(3, AlignHCenter);
  ui_listview_connections->setColumnAlignment(4, AlignHCenter);
  ui_listview_connections->setColumnAlignment(5, AlignRight);
  ui_listview_connections->setSorting(-1);

  ui_listview_shared->setColumnWidthMode(0, QListView::Manual);
  ui_listview_shared->setColumnAlignment(0, AlignLeft);
  ui_listview_shared->setColumnAlignment(1, AlignRight);
  ui_listview_shared->setColumnAlignment(2, AlignRight);
  ui_listview_shared->setColumnAlignment(3, AlignRight);
  ui_listview_shared->setColumnWidth(0, 46*10);
  ui_listview_shared->setColumnWidth(1, 8*10);
  ui_listview_shared->setColumnWidth(2, 8*10);
  ui_listview_shared->setColumnWidth(3, 10*10);
  ui_listview_shared->setSorting(0);

  ui_listview_uploads->setColumnWidthMode(0, QListView::Manual);
  ui_listview_uploads->setColumnWidthMode(1, QListView::Manual);
  ui_listview_uploads->setColumnWidthMode(2, QListView::Manual);
  ui_listview_uploads->setColumnWidthMode(5, QListView::Manual);
  ui_listview_uploads->setColumnAlignment(0, AlignLeft);
  ui_listview_uploads->setColumnAlignment(1, AlignLeft);
  ui_listview_uploads->setColumnAlignment(2, AlignRight);
  ui_listview_uploads->setColumnAlignment(3, AlignRight);
  ui_listview_uploads->setColumnAlignment(4, AlignRight);
  ui_listview_uploads->setColumnAlignment(5, AlignRight);
  ui_listview_uploads->setColumnWidth(0, 29*10);
  ui_listview_uploads->setColumnWidth(1, 10*10);
  ui_listview_uploads->setColumnWidth(2, 10*10);
  ui_listview_uploads->setColumnWidth(5, 7*10);

  // initialise extensions for images, videos and music
  _types_images = IMAGES;
  _types_videos = VIDEOS;
  _types_music  = MUSIC;

  // read configuration
  readConfiguration();

  // catch signal 13, PIPE BROKEN
  struct sigaction sa;
  sa.sa_handler = sighandler;
  sigaction(SIGPIPE, &sa, NULL);
  //sigaction(SIGABRT, &sa, NULL);

  // initialize list of shared files
  slotRefreshSharedList();


  search_tabs = 0;
  ui_tabwidget_search->removePage(tab);
  delete tab;
  tab = new SearchWidget(this);
  connect(tab, SIGNAL(doubleClicked(QListViewItem*)), this, SLOT(slotSearchDoubleClicked(QListViewItem*)));
  connect(tab, SIGNAL(rightButton(QListViewItem*, const QPoint&, int)), this, SLOT(slotShowSearchContext(QListViewItem*, const QPoint&, int)));
  ui_tabwidget_search->addTab(tab, "");
  ui_tabwidget_search->showPage(tab);

  // init IncomingConnect
  Address *addr = new Address();
  addr->setPort( std::string(ui_lineedit_listenport->text()) );
  ui_lineedit_ip->setText(addr->strIP().c_str());
  ui_lineedit_listenport->setText(addr->strPort().c_str());

  _incoming_connect = new IncomingConnect(addr->port(), 0, this);
  if( !_incoming_connect->ok() ) std::cerr << "Error: IncomingConnect." << std::endl;
  delete addr;

  // init upload manager
  upload_manager = new UploadManager(this);

  // init download manager
  download_manager = new DownloadManager(this);

  QObject *o = (QObject*)download_manager;

  connect((QObject*)ui_pushbutton_abortdownload, SIGNAL(clicked()), o, SLOT(slotAbortDownload()));
  connect((QObject*)ui_pushbutton_removedownload, SIGNAL(clicked()), o, SLOT(slotRemoveDownload()));
  connect((QObject*)ui_pushbutton_resumedownload, SIGNAL(clicked()), o, SLOT(slotResumeDownload()));
  connect((QObject*)ui_pushbutton_clearinactive, SIGNAL(clicked()), o, SLOT(slotClearInactive()));

  // read host list
  readHostList();

  // connect if "on start auto-connect" is checked
  if(ui_checkbox_conf_autoconnect->isChecked()) slotConnect();

  // download context menu
  _context_download = new QPopupMenu(this, "Download");
  _context_download->insertItem( "Launch", this, SLOT(slotLaunchSelectedDownloadItem()), CTRL+Key_L);
  _context_download->insertItem( "Move into Directory", this, SLOT(slotMoveDownloads()), CTRL+Key_M);
  _context_download->insertSeparator();
  _context_download->insertItem( "Abort", o, SLOT(slotAbortDownload()), CTRL+Key_A);
  _context_download->insertItem( "Remove", o, SLOT(slotRemoveDownload()), CTRL+Key_Delete);
  _context_download->insertItem( "Resume", o, SLOT(slotResumeDownload()), CTRL+Key_R);
  _context_download->insertSeparator();
  _context_download->insertItem( "Clear Inactive", o, SLOT(slotClearInactive()), CTRL+Key_I);

  // search context menu
  _context_search = new QPopupMenu(this, "Search");
  _context_search->insertItem( "Download", this, SLOT(slotDownloadAll()), CTRL+Key_D);

  // upload context menu
  _context_upload = new QPopupMenu(this, "Upload");
  _context_upload->insertItem( "Abort", this, SLOT(slotAbortUpload()), CTRL+Key_A);
  _context_upload->insertItem( "Remove", this, SLOT(slotRemoveUpload()), CTRL+Key_Delete);
  _context_upload->insertSeparator();
  _context_upload->insertItem( "Clear Inactive", this, SLOT(slotClearInactiveUploads()), CTRL+Key_I);

  // shared files context menu
  _context_shared = new QPopupMenu(this, "Shared Files");
  _context_shared->insertItem( "Launch", this, SLOT(slotSharedFilesLaunch()), CTRL+Key_L);
//   _context_shared->insertItem( "Properties", this, SLOT(slotSharedProperties()), CTRL+Key_P);
  _context_shared->insertItem( "Update List", this, SLOT(slotRefreshSharedList()), CTRL+Key_U);
  _context_shared->insertItem( "Delete", this, SLOT(slotDeleteSharedFile()), CTRL+Key_Delete);

  ui_pushbutton_properties->hide();
}


void QtellaSub::readConfiguration()
{
  QDir d;
  std::string homedir = static_cast<std::string>(d.homeDirPath());
      
  ui_lineedit_downloaddirectory->setText(homedir.c_str());
  ui_lineedit_incompleted->setText(homedir.c_str());

  conffile  = homedir + "/.qtella";
  std::fstream   f;

  f.open(conffile.c_str(), std::ios::in );
  if(f)
    {
      while( !f.eof() )
	{
	  char   buffer[1024];
	  f.getline(buffer, 1023);
	  std::string input = buffer;
	  
	  StringManipulation s(input);
	  
	  s.remove_comments();
	  
	  std::string s1, s2;
	  s.split(s1, s2, '=');
	  
	  s.assign(s1);
	  s.remove_chars("\n ");
	  s1 = s.str();
	  
	  s.assign(s2);
	  s.remove_chars("\n ");
	  s2 = s.str();
	  
	  //transform(s1.begin(), s1.end(), s1.begin(), toupper);
	  for(int i = 0; i < s1.size(); ++i) s1[i] = toupper(s1[i]);

	  if(!s1.empty())
	    {
	      if(s1=="DOWNLOAD")
		ui_lineedit_downloaddirectory->setText(s2.c_str());

	      if(s1=="INCOMPLETE")
		ui_lineedit_incompleted->setText(s2.c_str());
	
	      if(s1=="PORT")
		ui_lineedit_listenport->setText(s2.c_str());

	      if(s1=="SHAREDDIR")
		{
		  std::vector<std::string> vString;
		  s.assign(s2);
		  s.split_to_vector(vString, ';');
		  for(int i=0; i<vString.size(); i++)
		    ui_listbox_shareddirectories->insertItem(vString[i].c_str());
		}

	      if(s1=="APPS")
		{
		  std::vector<std::string> v;
		  s.assign(s2);
		  s.split_to_vector(v, ':');
		  _applications.push_back( make_pair(v[0], v[1]) );
		  new QListViewItem(ui_listview_apps, v[0].c_str(), v[1].c_str());
		}

	      if(s1=="SPEED")
		{
		  if(s2=="Modem") ui_combobox_speed->setCurrentItem(0);
		  if(s2=="ISDN") ui_combobox_speed->setCurrentItem(1);
		  if(s2=="ISDN2x") ui_combobox_speed->setCurrentItem(2);
		  if(s2=="Cable/DSL") ui_combobox_speed->setCurrentItem(3);
		  if(s2=="T1") ui_combobox_speed->setCurrentItem(4);
		  if(s2=="T3") ui_combobox_speed->setCurrentItem(5);
		}

	      if(s1=="MAXDOWNLOADS")
		{
		  std::strstream str;
		  int       max_downloads;
		  str << s2 << std::ends;
		  str >> max_downloads;
		  ui_spinbox_maxdownloads->setValue(max_downloads);
		}

 	      if(s1=="MAXUPLOADS")
		{
		  std::strstream str;
		  int       max_uploads;
		  str << s2 << std::ends;
		  str >> max_uploads;
		  ui_spinbox_maxuploads->setValue(max_uploads);
		}

 	      if(s1=="UPLOADBANDWIDTH")
		{
		  std::strstream str;
		  int       upload_bandwidth;
		  str << s2 << std::ends;
		  str >> upload_bandwidth;
		  ui_spinbox_upbandwidth->setValue(upload_bandwidth);
		}

	      if(s1=="AUTOCONNECT")
		{
		  std::vector<string> vString;
		  s.assign(s2);
		  s.split_to_vector(vString, ';');
		  for(int i=0; i<vString.size(); i++)
		    ui_listbox_autoconnect->insertItem(vString[i].c_str());
		}

	      if(s1=="EXISTING_FILES")
		{
		  if(s2=="0") ui_radiobutton_e_overwrite->setChecked(true);
		  else
		    if(s2=="1") ui_radiobutton_e_rename->setChecked(true);
		    else
		      ui_radiobutton_e_abort->setChecked(true);
		}

	      if(s1=="SAVE_HOST_LIST")
		{
		  if(s2=="0") ui_checkbox_savehostlist->setChecked(false);
		  else
		    ui_checkbox_savehostlist->setChecked(true);
		}

	      if(s1=="HOST_LIST_FILE")
		{
		  ui_lineedit_hostlistfile->setText(s2.c_str());
		}

	      if(s1=="ONSTART_CONNECT")
		{
		  if(s2=="1") ui_checkbox_conf_autoconnect->setChecked(true);
		}

	      if(s1=="ONEXIT_REMOVE")
		{
		  if(s2=="1") ui_checkbox_conf_removefiles->setChecked(true);
		}

	      if(s1=="ONEXIT_REMOVEALL")
		{
		  if(s2=="1") ui_checkbox_conf_removeall->setChecked(true);
		}

	      if(s1=="ONEXIT_REMOVE_SIZE")
		{
		  std::strstream str;
		  int            size;
		  str << s2 << std::ends;
		  str >> size;
		  ui_spinbox_conf_filesize->setValue(size);
		}

	      if(s1=="IGNORE_FIREWALLED")
		{
		  if(s2=="1") ui_checkbox_ignorefirewalled->setChecked(true);
		}

	      if(s1=="IGNORE_BUSY")
		{
		  if(s2=="1") ui_checkbox_ignorebusy->setChecked(true);
		}

	      if(s1=="FILTER_PRIVATE_IP")
		{
		  if(s2=="0") ui_checkbox_filterip->setChecked(false);
		}

	      if(s1=="RETRIES")
		{
		  std::strstream str;
		  int            n;
		  str << s2 << std::ends;
		  str >> n;
		  ui_spinbox_conf_retries->setValue(n);
		}

	      if(s1=="IMAGES") _types_images = s2;
	      if(s1=="VIDEOS") _types_videos = s2;
	      if(s1=="MUSIC") _types_music = s2;

	      if(s1=="ABORT_DOWNLOADS")
		{
		  if(s2=="0") ui_radiobutton_a_contsize->setChecked(true);
		  else
		    if(s2=="1") ui_radiobutton_a_conthost->setChecked(true);
		    else
		      if(s2=="2") ui_radiobutton_a_contsizehost->setChecked(true);
		      else
			ui_radiobutton_a_startnew->setChecked(true);
		}
	    }
	}
      f.close();
    }
  else  // default configuration
    {
      ui_listbox_autoconnect->insertItem("connect1.gnutellanet.com:6346");
      ui_listbox_autoconnect->insertItem("connect2.gnutellanet.com:6346");
      ui_listbox_autoconnect->insertItem("connect3.gnutellanet.com:6346");
      ui_listbox_autoconnect->insertItem("connect4.gnutellanet.com:6346");
    }
}


void QtellaSub::readHostList()
{
  if(!ui_checkbox_savehostlist->isChecked()) return;

  QFile f(ui_lineedit_hostlistfile->text());

  if(f.open(IO_ReadOnly))
    {
      char buffer[64];
      int  r;
      while( (r = f.readLine(buffer, 64)) > 0)
	{
	  std::strstream str;
	  str << buffer << std::ends;
	  std::string s;
	  str >> s;
	  ui_listbox_hostlist->insertItem(s.c_str());
	}
      f.close();
    }
  else
    std::cerr << "Cannot open host list file" << std::endl;
}


QtellaSub::~QtellaSub()
{
  delete _incoming_connect;
  delete upload_manager;
  delete download_manager;

  // remove all files on exit ?
  if(ui_checkbox_conf_removeall->isChecked()) removeFiles(-1);

  // remove files with size less/equal than x ?
  if(ui_checkbox_conf_removefiles->isChecked()) removeFiles(ui_spinbox_conf_filesize->value());

  // save configuration
  slotApplyConfig();

  // save host list
  slotDisconnect();

  if(ui_checkbox_savehostlist->isChecked())
    {
      QFile f(ui_lineedit_hostlistfile->text());
      
      if(f.open(IO_WriteOnly))
	{
	  for(int i = 0; i < ui_listbox_hostlist->count(); ++i)
	    {
	      std::string s = static_cast<std::string>(ui_listbox_hostlist->text(i));
	      s += "\n";
	      f.writeBlock(s.data(), s.size());
	    }
	  f.close();
	}
      else
	std::cerr << "Error opening host list file." << std::endl;
    }
}


void QtellaSub::slotConnect()
{
  if(!timer_connection->isActive())
    {
      connections->start();
      timer_connection->start(1000);
    }
}


void QtellaSub::slotDisconnect()
{
  if(timer_connection->isActive())
    {
      timer_connection->stop();
      connections->disconnect();
    }
}


void QtellaSub::slotAddHost()
{
  if(!timer_connection->isActive())
    timer_connection->start(500);
  connections->slotAddHost();
}


void QtellaSub::check_connections()
{
  connections->checkServants();
}

void QtellaSub::slotAddDirectory()
{
  QString dirName =
    QFileDialog::getExistingDirectory(0, this, 0, "Select shared directory", true);

  if ( !dirName.isEmpty() ) ui_listbox_shareddirectories->insertItem( dirName );

  slotRefreshSharedList();
}

void QtellaSub::slotRemoveDirectory()
{
  int index = ui_listbox_shareddirectories->currentItem();

  if(index>-1)
    ui_listbox_shareddirectories->removeItem(index);

  slotRefreshSharedList();
}

void QtellaSub::slotApplyConfig()
{
  std::fstream         f;

  f.open(conffile.c_str(), ios::out|ios::trunc);
  if(!f)
    {
      std::cerr << "Cannot open configuration file. " << conffile << std::endl;
      return;
    }

  f << "DOWNLOAD=" << (const char*)ui_lineedit_downloaddirectory->text() << std::endl
    << "INCOMPLETE=" << (const char*)ui_lineedit_incompleted->text() << std::endl
    << "PORT=" << (const char*)ui_lineedit_listenport->text() << std::endl
    << "SPEED=" << (const char*)ui_combobox_speed->currentText() << std::endl
    << "MAXDOWNLOADS=" << ui_spinbox_maxdownloads->value() << std::endl
    << "MAXUPLOADS=" << ui_spinbox_maxuploads->value() << std::endl
    << "UPLOADBANDWIDTH=" << ui_spinbox_upbandwidth->value() << std::endl
    << "SAVE_HOST_LIST=" << ui_checkbox_savehostlist->isChecked() << std::endl
    << "ONSTART_CONNECT=" << ui_checkbox_conf_autoconnect->isChecked() << std::endl
    << "ONEXIT_REMOVE=" << ui_checkbox_conf_removefiles->isChecked() << std::endl
    << "ONEXIT_REMOVEALL=" << ui_checkbox_conf_removeall->isChecked() << std::endl
    << "ONEXIT_REMOVE_SIZE=" << ui_spinbox_conf_filesize->value() << std::endl
    << "RETRIES=" << ui_spinbox_conf_retries->value() << std::endl
    << "IGNORE_FIREWALLED=" << ui_checkbox_ignorefirewalled->isChecked() << std::endl
    << "IGNORE_BUSY=" << ui_checkbox_ignorebusy->isChecked() << std::endl
    << "FILTER_PRIVATE_IP=" << ui_checkbox_filterip->isChecked() << std::endl
    << "HOST_LIST_FILE=" << ui_lineedit_hostlistfile->text() << std::endl
    << "IMAGES=" << _types_images << std::endl
    << "VIDEOS=" << _types_videos << std::endl
    << "MUSIC=" << _types_music << std::endl
    << "SHAREDDIR=";

  for(int i = 0; i < ui_listbox_shareddirectories->count(); i++)
    f << (const char*)ui_listbox_shareddirectories->text(i) << ";";

  f << std::endl;

  for(int i = 0; i < _applications.size(); ++i)
    f << "APPS=" << (_applications[i].first + ":" + _applications[i].second) << std::endl;

  f << "AUTOCONNECT=";
  for(int i = 0; i < ui_listbox_autoconnect->count(); i++)
    f << (const char*)ui_listbox_autoconnect->text(i) << ";";
  f << std::endl;

  f << "EXISTING_FILES=";
  if(ui_radiobutton_e_overwrite->isChecked()) f << "0";
  if(ui_radiobutton_e_rename->isChecked()) f << "1";
  if(ui_radiobutton_e_abort->isChecked()) f << "2";
  f << std::endl;

  f << "ABORT_DOWNLOADS=";
  if(ui_radiobutton_a_contsize->isChecked()) f << "0";
  if(ui_radiobutton_a_conthost->isChecked()) f << "1";
  if(ui_radiobutton_a_contsizehost->isChecked()) f << "2";
  if(ui_radiobutton_a_startnew->isChecked()) f << "3";
  f << std::endl;

  f.close();
}


void QtellaSub::checkDirectories(std::string directory, std::vector<SharedFile*> *old)
{
  QDir   dir(directory.c_str());

  if(dir.exists())
    {
      QStringList dirlist = dir.entryList(QDir::Dirs);

      for( QStringList::Iterator it = dirlist.begin(); it != dirlist.end(); ++it )
	{
	  std::string filename = (*it).latin1();
	  if(filename!="." && filename!="..")
	    checkDirectories( (directory+"/"+filename), old );
	}

      QStringList list = dir.entryList(QDir::Files);
      
      for( QStringList::Iterator it = list.begin(); it!=list.end(); ++it )
	{
	  std::string filename = (*it).latin1();
	  
	  std::string complete = directory;
	  if( complete[complete.size()-1] != '/' ) complete += "/";
	  complete += filename;
	  
	  QFile f(complete.c_str());
	  
	  if(f.exists())
	    {
	      SharedViewItem *item = new SharedViewItem(ui_listview_shared, filename.c_str());
		  
	      item->setText(1, "0");  // uploads
	      item->setText(2, "0");  // requests
	      
	      std::strstream stream_fsize;
	      stream_fsize << f.size() << std::ends;
	      std::string    str(stream_fsize.str());
	      item->setText(3, StringManipulation::insertDelimiter(str, ',', 3).c_str() );
	      stream_fsize.freeze(false);
		  
	      vSharedFiles.push_back(new SharedFile(item, filename, directory, f.size()));
		  
	      for(int i = 0; i < (*old).size(); i++)
		if( *((*old)[i]) == *vSharedFiles.back() )
		  {
		    int up = (*old)[i]->uploads;
		    int rq = (*old)[i]->requests;
		    
		    vSharedFiles.back()->uploads = up;
		    vSharedFiles.back()->requests = rq;
		    
		    std::strstream str;
		    str << up << std::ends;
		    item->setText(1, str.str());
		    str.freeze(false);
		    str.seekp(0, std::ios::beg);
		    str << rq << std::ends;
		    item->setText(2, str.str());
		    str.freeze(false);
		  }
	    }
	}
    }
}


void QtellaSub::slotRefreshSharedList()
{
  std::vector<SharedFile*>  old = vSharedFiles;

  number_shared_files = 0;
  shared_size = 0;

  mutex_shared_files.lock();

  vSharedFiles.clear();

  for(int i = 0; i < ui_listbox_shareddirectories->count(); i++)
    {
      std::string directory = (const char*)ui_listbox_shareddirectories->text(i);
      checkDirectories(directory, &old);
    }

  for(int i = 0; i < old.size(); i++) delete old[i];

  for(int i = 0; i < vSharedFiles.size(); ++i)
    shared_size += vSharedFiles[i]->size;

  number_shared_files = vSharedFiles.size();

  mutex_shared_files.unlock();

  // status line

  connections->_statistic.shared_files = number_shared_files;
  connections->_statistic.shared_size = shared_size;

  unsigned long files = number_shared_files;
  double        size  = shared_size;
  std::strstream str;
  std::string u;

  str.seekp(0, ios::beg);
  if(files > 1024) { files = files >> 10; u = "K"; };
  if(files > 1024) { files = files >> 10; u = "M"; };
  str << files << u << std::ends;
  ui_textlabel_sharedfiles->setText(str.str());
  str.freeze(false);
  
  u = "";
  str.seekp(0, ios::beg);
  str.setf(std::ios::fixed);
  if(size > 1024) { size /= 1024; u = "KB"; }
  if(size > 1024) { size /= 1024; u = "MB"; }
  str << setprecision(1) << size << u << std::ends;
  ui_textlabel_sharedsize->setText(str.str());
  str.freeze(false);
}



void QtellaSub::downloadItem(QListViewItem *item)
{
  int i = 0;
  int m = 0;

  for(int mp = 0; mp < connections->_search.size(); mp++)
    for(int ip = 0; ip < connections->_search[mp]->vSearchResults.size(); ip++)
      if( connections->_search[mp]->vSearchResults[ip]->item == item) 
	{
	  m = mp;
	  i = ip;
	}

  std::string path = (const char*)ui_lineedit_downloaddirectory->text();
  std::string incomplete = (const char*)ui_lineedit_incompleted->text();

  QListViewItem *item_download = new QListViewItem(ui_listview_download);
  item_download->setText(0, connections->_search[m]->vSearchResults[i]->filename.c_str() );
  item_download->setText(3, connections->_search[m]->vSearchResults[i]->addr.strIP().c_str() );
  ui_listview_download->insertItem(item_download);

  download_manager->addDownload(*(connections->_search[m]->vSearchResults[i]), item_download);
}


void QtellaSub::slotSearchDoubleClicked(QListViewItem *item)
{
  downloadItem(item);
  Tabular->setCurrentPage(2);
}

void QtellaSub::slotSearch()
{
  std::string query = static_cast<std::string>(ui_combobox_search->currentText());
  
  bool found = false;
  for(int i = 0; i < ui_combobox_search->count(); ++i)
    if( ui_combobox_search->text(i) == ui_combobox_search->currentText() ) found = true;
  if(!found) ui_combobox_search->insertItem(ui_combobox_search->currentText(), 0);

  if(search_tabs == 0)
    {
      ui_tabwidget_search->changeTab(ui_tabwidget_search->currentPage(), query.c_str());
      connections->search(query, (SearchWidget*)(ui_tabwidget_search->currentPage()) );
    }
  else
    {
      #ifdef QTELLA_DEBUG
        std::clog << "Sub: tabs>0" << std::endl;
      #endif
      SearchWidget *s = new SearchWidget(this);

      connect(s, SIGNAL(doubleClicked(QListViewItem*)), this, SLOT(slotSearchDoubleClicked(QListViewItem*)));
      connect(s, SIGNAL(rightButton(QListViewItem*, const QPoint&, int)), this, SLOT(slotShowSearchContext(QListViewItem*, const QPoint&, int)));
      ui_tabwidget_search->addTab(s, query.c_str());
      ui_tabwidget_search->showPage(s);

      #ifdef QTELLA_DEBUG
        std::clog << "Sub: tabs>0, sending query" << std::endl;
      #endif
      connections->search(query, s);
    }

  search_tabs++;
  ui_pushbutton_stopsearch->setEnabled(true);
  ui_pushbutton_stopsearch->setText("Stop");
}


void QtellaSub::slotDownloadAll()
{
  int          m = -1;
  SearchWidget *s = (SearchWidget*) ui_tabwidget_search->currentPage();

  for(int i = 0; i < connections->_search.size(); i++) if(connections->_search[i]->sw == s) m = i;

  if(m > -1)
    {
      int counts = 0;

      for(int i = 0; i < connections->_search[m]->vSearchResults.size(); i++)
	if( connections->_search[m]->vSearchResults[i]->item->isSelected() )
	  {
	    downloadItem(connections->_search[m]->vSearchResults[i]->item);
	    counts++;
	  }

      if(counts) Tabular->setCurrentPage(2);
    }
}


void QtellaSub::slotAddAutoConnect()
{
  string s = (const char*)ui_lineedit_autoconnect->text();
  
  if(!s.empty())
    ui_listbox_autoconnect->insertItem(s.c_str());

  slotApplyConfig();
  ui_lineedit_autoconnect->setText("");
}


void QtellaSub::slotRemoveAutoConnect()
{
  int index = ui_listbox_autoconnect->currentItem();

  if(index>-1)
    ui_listbox_autoconnect->removeItem(index);

  slotApplyConfig();
}


void QtellaSub::slotInfo()
{
  Info *i = new Info(this, "Info", true);
  i->ui_textlabel_version->setText(VERSION);
  i->exec();
}


void QtellaSub::sighandler(int i)
{
  #ifdef QTELLA_WARNINGS
    std::cout << "signal: " << i << std::endl;
  #endif
}


void QtellaSub::slotAbortUpload()
{
  upload_manager->abortUpload();
}


void QtellaSub::slotRemoveUpload()
{
  upload_manager->removeUpload();
}


void QtellaSub::slotDeleteSharedFile()
{
  std::vector<SharedFile*>::iterator  pos = vSharedFiles.begin();
  
  for(pos; pos != vSharedFiles.end(); ++pos)
    if( (*pos)->item->isSelected() )
      {
	QFile f( (*pos)->completeFilename().c_str() );
	f.remove();
      }

  slotRefreshSharedList();
}


void QtellaSub::slotClearInactiveUploads()
{
  upload_manager->removeInactiveUploads();
}


void QtellaSub::slotReadFile()
{
  QFileDialog d(this, 0, true);

  if(d.exec())
    {
      std::string s = static_cast<const char*>(d.selectedFile());

      std::fstream f;

      f.open(s.c_str(), ios::in );
      if(f)
	{
	  while( !f.eof() )
	    {
	      char   buffer[1024];
	      f.getline(buffer, 1023);
	      std::string input = buffer;
	      
	      if(input.find(":") != std::string::npos)
		ui_listbox_hostlist->insertItem( input.c_str(), 0);
	    }
	}
    }
}



void QtellaSub::slotStopSearch()
{
  SearchWidget* sw  = (SearchWidget*) ui_tabwidget_search->currentPage();

  std::vector<Search*>::iterator pos = connections->_search.begin();
  while((*pos)->sw != sw) ++pos;

  if(std::string(ui_pushbutton_stopsearch->text()) == "Stop")
    {
      (*pos)->active = false;
      ui_pushbutton_stopsearch->setText("Delete");
    }
  else
    {
      for(int j = 0; j < (*pos)->vSearchResults.size(); j++)
	delete (*pos)->vSearchResults[j];

      connections->_search.erase(pos);

      if(search_tabs==1)
	{
	  sw->ui_listview_search->clear();
	  ui_tabwidget_search->changeTab(sw, "");
	  ui_pushbutton_stopsearch->setText("Stop");
	  ui_pushbutton_stopsearch->setEnabled(false);
	}
      else
	{
	  ui_tabwidget_search->removePage(sw);
	  delete sw;

	  sw = (SearchWidget*) ui_tabwidget_search->currentPage();
	  pos = connections->_search.begin();
	  while((*pos)->sw != sw) ++pos;
	  if((*pos)->active)
	    ui_pushbutton_stopsearch->setText("Stop");
	  else
	    ui_pushbutton_stopsearch->setText("Delete");
	}

      search_tabs--;
    }
}


void QtellaSub::slotCurrentChanged(QWidget* w)
{
  // remove icon
  QIconSet icon;
  ui_tabwidget_search->changeTab(w, icon, ui_tabwidget_search->tabLabel(w));

  std::vector<Search*>::iterator pos = connections->_search.begin();
  while( pos != connections->_search.end() ) 
    {
      if((*pos)->sw == ((SearchWidget*)w)) break;
      ++pos;
    }

  if(pos != connections->_search.end())
    if((*pos)->active)
      ui_pushbutton_stopsearch->setText("Stop");
    else
      ui_pushbutton_stopsearch->setText("Delete");
}


void QtellaSub::removeFiles(const int size)
{
  long              lsize = size;
  std::string       directory(ui_lineedit_incompleted->text());

  if(size>-1) lsize = size << 10;
  QDir   dir(directory.c_str());

  if(directory.size()>0)
    if(directory[directory.size()-1]!='/') directory += "/";

  if(dir.exists())
    {
      QStringList list = dir.entryList(QDir::Files);
      
      for( QStringList::Iterator it = list.begin(); it!=list.end(); ++it )
	{
	  std::string filename = (*it).latin1();
	  std::string complete(directory+filename);
	  
	  QFile f(complete.c_str());

	  if(filename.find(".part_") != std::string::npos)
	    if(f.exists())
	      if( (f.size() <= lsize) || (lsize==-1) ) f.remove();
	}
    }
}


void QtellaSub::launchFile(std::string filename)
{
  std::string::size_type idx = filename.find_last_of(".");
  std::string            ext(filename.substr(idx+1));
  std::string            application;
  
  for(int i = 0; i < _applications.size(); ++i)
    if(_applications[i].first.find(ext) != std::string::npos) application = _applications[i].second;

  if(application.empty()) return;

  if(!fork())
    {
      execlp( application.c_str(), _binary.c_str(), filename.c_str(), NULL );
      exit(0);
    }
}


void QtellaSub::slotSharedFilesLaunch(QListViewItem* item)
{
  std::vector<SharedFile*>::iterator    pos = vSharedFiles.begin();

  while(pos != vSharedFiles.end()) if(item == (*pos)->item) break; else ++pos;
  launchFile( (*pos)->directory + "/" + (*pos)->file );
}


void QtellaSub::slotSharedFilesLaunch()
{
  std::vector<std::string> files;

  std::vector<SharedFile*>::iterator  pos = vSharedFiles.begin();
  while(pos != vSharedFiles.end())
    {
      if((*pos)->item->isSelected()) files.push_back( (*pos)->directory + "/" + (*pos)->file );
      ++pos;
    }

  if(files.empty()) return;

  std::string::size_type idx = files[0].find_last_of(".");
  std::string ext(files[0].substr(idx+1));
  std::string application;

  for(int i = 0; i < _applications.size(); ++i)
    if(_applications[i].first.find(ext) != std::string::npos) application = _applications[i].second;

  if(application.empty()) return;

  if(!fork())
    {
      const int N = 1000;
      char*     args[N];

      for(int i = 1; i <= files.size(); ++i) 
	if(i < N-1) args[i] = const_cast<char*>(files[i-1].c_str());

      args[0] = const_cast<char*>(_binary.c_str());
      if(files.size()+1 < N) args[files.size()+1] = NULL; else args[N-1] = NULL;

      execvp( application.c_str(), args );
      exit(0);
    }

}


void QtellaSub::slotAddApplication()
{
  AddApplication aa(this, "", true);

  if(aa.exec())
    {
      new QListViewItem(ui_listview_apps, aa.ui_lineedit_type->text(), aa.ui_lineedit_application->text());
      _applications.push_back( make_pair(aa.ui_lineedit_type->text(), aa.ui_lineedit_application->text()) );
    }
}


void QtellaSub::slotRemoveApplication()
{
  QListViewItem*    fc = ui_listview_apps->firstChild();
  QListViewItem*    tmp;

  while(fc)
    {
      if(fc->isSelected())
	{
	  tmp = fc;
	  std::string type(fc->text(0));
	  std::string app(fc->text(1));
	  std::vector< std::pair<std::string, std::string> >::iterator     pos = _applications.begin();
	  while( ((*pos).first != type) && ((*pos).second != app) ) ++pos;
	  _applications.erase(pos);
	  fc = tmp->itemBelow();
	  delete tmp;
	}
      else
	fc = fc->itemBelow();
    }
}


void QtellaSub::slotDownloadDoubleClicked(QListViewItem* item)
{
  if(!item) return;

  int i = 0;

  for(i = 0; i < download_manager->_downloads.size(); ++i)
    if(item == download_manager->_downloads[i]->_item) break;

  if(item == download_manager->_downloads[i]->_item)
    if(download_manager->_downloads[i]->_f)
      launchFile( std::string(download_manager->_downloads[i]->_f->name()) );
}


void QtellaSub::slotLaunchSelectedDownloadItem()
{
  for(int i = 0; i < download_manager->_downloads.size(); ++i)
    if(download_manager->_downloads[i]->_item->isSelected())
      {
	launchFile( std::string(download_manager->_downloads[i]->_f->name()) );
	break;
      }
}


void QtellaSub::slotDownloadShowContext(QListViewItem* item, const QPoint& p, int column)
{
  _context_download->popup(p);
}


void QtellaSub::slotShowSearchContext(QListViewItem* item, const QPoint& p, int column)
{
  _context_search->popup(p);
}


void QtellaSub::slotShowUploadContext(QListViewItem* item, const QPoint& p, int column)
{
  _context_upload->popup(p);
}


void QtellaSub::slotShowSharedContext(QListViewItem* item, const QPoint& p, int column)
{
  _context_shared->popup(p);
}


void QtellaSub::slotMoveDownloads()
{
  QFileDialog* d = new QFileDialog(this, "Select Directory", true);

  d->setMode(QFileDialog::DirectoryOnly);
  d->setFilter("All (*)");

  if(d->exec() == QDialog::Accepted)
    {
      std::string dirname(d->selectedFile());

      for(int i = 0; i < download_manager->_downloads.size(); ++i)
	if(download_manager->_downloads[i]->readState() == Download::dtClosedCompleted)
	  download_manager->_downloads[i]->moveFile(dirname);
    }
}


void QtellaSub::slotSharedProperties()
{
  std::vector<SharedFile*>::iterator  pos = vSharedFiles.begin();
  while(pos != vSharedFiles.end())
    {
      if((*pos)->item->isSelected()) 
	{
	  MP3property p(this, "Properties", true);
	  p.setFile((*pos)->directory, (*pos)->file);
	  p.exec();
	  break;
	}
      else
	++pos;
    }
}
