#include "Gnutella.h"
#include "Servent.h"
#include "Connections.h"

#include "Ping.h"
#include "Host.h"
#include "Address.h"
#include "Message.h"
#include "PongCache.h"

#include <string>
#include <vector>
#include <stdint.h>

#include <qsocket.h>
#include <qlistview.h>

Servent::Servent(Connections* parent, std::string id, QListViewItem *item, QSocket* socket)
  : _id(id), _item(item), _timeout(0), _input(0), _output(0), _state(Idle),
    _parent(parent), _direction(0), _last_ping_send(0), _bytes_in(0), _bytes_out(0), 
    _bandwidth_in(0), _bandwidth_out(0)
{
  _last_input = time(NULL);
  _start = time(NULL);

  _connect_str = CONNECT_STR;
  _response_str = RESPONSE_STR;

  if(socket) _socket = socket; else _socket = new QSocket();

  QObject::connect(_socket, SIGNAL(error(int)), SLOT(slotError(int)));
  QObject::connect(_socket, SIGNAL(connected()), SLOT(slotConnect()));
  QObject::connect(_socket, SIGNAL(readyRead()), SLOT(slotReadyRead()));
  QObject::connect(_socket, SIGNAL(connectionClosed()), SLOT(slotClosed()));
  QObject::connect(_socket, SIGNAL(delayedCloseFinished()), SLOT(slotDelayedClosed()));

  if(socket)
    if(_socket->state() == QSocket::Connection)
      {
	sendMessageDirect(_response_str);
	Ping p(1, 0, _id);
	sendMessageDirect(p._data);  // send initial ping
	_parent->_statistic.out_ping_n++;
	_parent->_statistic.out_ping_b += p._data.size();
	setState(Connected);
      }
    else
      setState(Closed);

  _ping_timer = new QTimer(this);
  QObject::connect(_ping_timer, SIGNAL(timeout()), SLOT(pingTimeout()));
  _ping_timer->start(3000);

  _servant_number = Servent::getNextServentNumber();

  _flush_timer = new QTimer(this);
  QObject::connect(_flush_timer, SIGNAL(timeout()), SLOT(flushTimeout()));
  _flush_timer->start(1000);

  _bandwidth_timer = new QTimer(this);
  QObject::connect(_bandwidth_timer, SIGNAL(timeout()), SLOT(bandwidthTimeout()));
  _bandwidth_timer->start(2000);

  _timeout_timer = new QTimer(this);
  QObject::connect(_timeout_timer, SIGNAL(timeout()), SLOT(timeout()));
  _timeout_timer->start(5000);
}


Servent::~Servent()
{
  PongCache::getInstance()->removePong(_servant_number);
  if(_item) delete _item;
  if(_ping_timer) delete _ping_timer;
  if(_timeout_timer) delete _timeout_timer;

  QObject::disconnect(_socket, 0, 0, 0);
  if(_socket) delete _socket;
}


int Servent::getNextServentNumber()
{
  static unsigned int number = 1;
  ++number;
  if(!number) number = 1;
  return number;
}


void Servent::pingTimeout()
{
  if( !_last_accepted_ping_id.empty() )
    if( ((static_cast<unsigned char>(_last_accepted_ping_id[8]) == 0xff) && (_last_accepted_ping_id[15] == 1)) ||
	((time(NULL) - _last_ping_send >= 30) && (state() == Connected)) )
      {
	Ping p(7, 0, _id);
	sendMessage(p._data, false);

	_parent->_statistic.out_ping_n++;
	_parent->_statistic.out_ping_b += p._data.size();

	_last_ping_send = time(NULL);
      }
}


void Servent::connect(Address address)
{
  _address = address;
  _strip.erase();
  _strip = address.strIP();

  setState(Connecting);
  _timeout = 0;
  _socket->connectToHost(_strip.c_str(), address.port());
}



void Servent::slotDelayedClosed()
{
  setState(Closed);
}


void Servent::close()
{
  _socket->close();
  setState(Closed);
}



int Servent::state()
{
  return _state;
}



void Servent::setState(int state)
{
  _state = state;
}



void Servent::slotError(int err)
{
  if(_socket->state() != QSocket::Idle) _socket->close();
  setState(Error);
}



void Servent::writeConnectString()
{
  _timeout = 0;
  sendMessageDirect(_connect_str);
}



void Servent::slotConnect()
{
  setState(Waiting);
  writeConnectString();
}



void Servent::slotClosed()
{
  setState(Closed);
}



void Servent::sendMessageDirect(std::string& buffer)
{
  if(_socket->state() == QSocket::Connection)
    {
      _output++;
      _bytes_out += buffer.size();
      _socket->writeBlock(buffer.data(), buffer.size());
      _socket->flush();
    }
}



void Servent::flushTimeout()
{
  if( (_socket->state() == QSocket::Connection) && (state() == Connected) )
    {
      _bytes_out += _send_buffer.size();
      _socket->writeBlock(_send_buffer.data(), _send_buffer.size());
      _send_buffer.erase();
    }
}



void Servent::sendMessage(std::string& buffer, bool cache)
{
  if( ((_send_buffer.size() < 512) && (_socket->bytesToWrite() < 1024))|| !cache )
    {
      _output++;
      _send_buffer.append(buffer);
    }
}


void Servent::limitVectors()
{
  while(_id_set.size() > 2000) _id_set.erase(_id_set.begin());
}


int Servent::getDirection()
{
  return _direction;
}


void Servent::setDirection(int i)
{
  _direction = i;
}


void Servent::slotReadyRead()
{
  char   cbuff[4096+10];

  _last_input = time(NULL);

  do
    {
      int rd = _socket->readBlock(cbuff, 4096);
      _buffer.append(cbuff, rd);
    }
  while(_socket->bytesAvailable() > 0);

  if( (state() == Waiting) && (_buffer.size() >= _response_str.size()) )
    {
      if(_response_str == _buffer.substr(0, _response_str.size()))
	{
	  setState(Connected);
	  _buffer.erase(0, _response_str.size());

	  Ping p(7, 0, _id);
	  sendMessageDirect(p._data);  // send initial ping
	  _parent->_statistic.out_ping_n++;
	  _parent->_statistic.out_ping_b += p._data.size();
	  return;
	}
    }

  else

    while( _buffer.size() >= HEADER_SIZE )
      {
	uint32_t payload  = *( (uint32_t*)(_buffer.substr(19, 4).c_str()) );

	if(payload > 20000)
	  {
	    _buffer.erase();
	    slotError(-1);
	    return;
	  }
	
	if( _buffer.size() >= (HEADER_SIZE + payload) )
	  {
	    _input++;
	    
	    if(_message_buffer.size() < 30)
	      _message_buffer.push_back(_buffer.substr(0, HEADER_SIZE + payload));
	    
	    _buffer.erase(0, HEADER_SIZE + payload);
	    _bytes_in += HEADER_SIZE + payload;
	  }
	else
	  return;
      }
  
//   _parent->checkMessages(this);
}


void Servent::bandwidthTimeout()
{
  _bandwidth_in = static_cast<double>(_bytes_in) / static_cast<double>(2048);
  _bandwidth_out = static_cast<double>(_bytes_out) / static_cast<double>(2048);
  _bytes_in = 0;
  _bytes_out = 0;
}


void Servent::timeout()
{
  if( (time(NULL) - _last_input) >= 120 ) close();
}
