#include "Gnutella.h"
#include "Upload.h"
#include "UploadManager.h"
#include "SharedFile.h"
#include "QtellaSub.h"
#include "Connections.h"
#include "Push.h"
#include "StringManipulation.h"
#include "SharedViewItem.h"

#include "qsocket.h"
#include "qlistview.h"
#include "qfile.h"
#include "qdatetime.h"
#include "qspinbox.h"
#include "qhostaddress.h"
#include "qtimer.h"

#include <iostream>
#include <iomanip>
#include <strstream>
#include <unistd.h>

const std::string Upload::statestr[] = {"Busy", "Start", "Connected", "NotFound", 
					"Finished", "Error", "Abort", "Closed", "Connecting"};

Upload::Upload(UploadManager *parent, std::string data, QSocket* socket)
  : _parent(parent), _push(false), _state(Connecting), item(NULL), _socket(socket), _data(data), _timer(NULL)
{
  connect(_socket, SIGNAL(connectionClosed()), SLOT(slotClosed()));
  initUpload();
  _timer = new QTimer(this);
  connect(_timer, SIGNAL(timeout()), SLOT(slotTimer()));
  _timer->start(1000);
 
  _time_start.start();
}


Upload::Upload(UploadManager *parent, Push push)
  : _parent(parent), _push(true), _state(Connecting), _push_data(push), item(NULL), _timer(NULL)
{
  _socket = new QSocket();
  connect(_socket, SIGNAL(connectionClosed()), SLOT(slotClosed()));
  connect(_socket, SIGNAL(connected()), SLOT(slotConnected()));
  connect(_socket, SIGNAL(readyRead()), SLOT(slotReadyRead()));
  _socket->connectToHost(_push_data._address.strIP().c_str(), _push_data._address.port());
}


Upload::~Upload()
{
  if(_timer) delete _timer;
  if(_socket) delete _socket;
  if(item) delete item;
}


int Upload::getState()
{
  return _state;
}


void Upload::slotClosed()
{
  if(_socket) delete _socket;
  if(_timer) delete _timer;

  _socket = NULL;
  _timer = NULL;
  setState(Closed);
}

void Upload::abortUpload()
{
  if(_timer) _timer->stop();
  if(_socket) _socket->close();
  _parent->_parent->connections->_statistic.loads_dabor++;
  setState(Abort);
}

void Upload::setState(int state)
{
  if( _state == Finished ) return;
  _state = state;
  if(item) item->setText(1, statestr[_state].c_str());
}



void Upload::slotConnected()  // connected by push
{
  std::string f = _parent->_parent->vSharedFiles[_push_data._index]->file;
  std::strstream str;
  str << "GIV " << _push_data._index << ":" << std::ends;
  std::string s = str.str();
  str.freeze(false);
  s += Connections::toHex(_parent->_parent->connections->_id) + "/" + f + "\n\n";
      
  _socket->writeBlock(s.data(), s.size());
}



void Upload::slotReadyRead()  // connected by push
{
  while(_socket->canReadLine())
    {
      std::string n( _socket->readLine() );
      _data += n;

      if(_data.find("\r\n\r\n") != std::string::npos)
	{
	  disconnect(_socket, 0, 0, 0);
	  initUpload();
	  _timer = new QTimer(this);
	  connect(_timer, SIGNAL(timeout()), SLOT(slotTimer()));
	  _timer->start(1000);
	  _time_start.start();
	}
    }
}



void Upload::initUpload()
{
  StringManipulation sm(_data);

  if(_data.substr(0,3) != "GET")
    {
      setState(Closed);
      return;
    }

  std::string::size_type  idx = sm.to_lower().find("/get/");
  
  if(idx == std::string::npos)
    {
      _socket->close();
      setState(Closed);
      return;
    }

  idx += 5;
  std::string::size_type  idx2 = _data.find("/", idx);

  if(idx == std::string::npos)
    {
      _socket->close();
      setState(Closed);
      return;
    }

  std::string sindex = _data.substr(idx, idx2-idx);

  std::strstream a;
  a << sindex << ends;
  a >> index;

  
  idx = sm.to_lower().find(" http", idx2);

  if(idx == std::string::npos)
    {
      _socket->close();
      setState(Closed);
      return;
    }
  
  filename = _data.substr(idx2+1, idx-idx2-1);


  if( (idx = sm.to_lower().find("user-agent:")) != std::string::npos )
    {
      idx = _data.find_first_not_of(" ", idx+11);
      idx2 = _data.find("\r\n", idx);
      client = _data.substr(idx, idx2-idx);
    }
  else
    client = "";


  if( (idx = sm.to_lower().find("range:")) != std::string::npos )
    {
      idx = _data.find_first_of("1234567890", idx+6);
      idx2 = _data.find("-", idx);
      std::string srange_start = _data.substr(idx, idx2-idx);
      
      std::strstream c;
      c << srange_start << ends;
      c >> range_start;

      idx = _data.find("\r\n", idx2);
      if( (idx = _data.find_first_of("0123456789", idx2+1)) != std::string::npos )
	{
	  idx2 = _data.find_first_not_of("0123456789", idx);
	  std::string srange_end = _data.substr(idx, idx2-idx);

	  std::strstream b;
	  b << srange_end << std::ends;
	  b >> range_end;
	}
      else range_end = 0;
    }
  else
    {
      range_start = 0;
      range_end = 0;
    }

  if(index < _parent->_parent->vSharedFiles.size())
    if( _parent->_parent->vSharedFiles[index]->file == filename )
      {
	_parent->_parent->vSharedFiles[index]->requests++;
	std::strstream str;
	str << _parent->_parent->vSharedFiles[index]->requests << std::ends;
	_parent->_parent->vSharedFiles[index]->item->setText(2, str.str());
	str.freeze(false);
      }

  if(_parent->numberUploads() > _parent->_parent->ui_spinbox_maxuploads->value())
    {
      std::strstream str;
      std::string    s;
      str << "HTTP/1.0 503 Busy\r\nServer: Qtella " << VERSION << "\r\n\r\n" << std::ends;
      s = str.str();
      str.freeze(false);
      _socket->writeBlock(s.c_str(), s.size());
      _socket->close();
      _state = Busy;
      return;
    }

  // 
  std::string f = _parent->_parent->vSharedFiles[index]->directory;

  if(f.size() > 0) if( f[f.size()-1] != '/' ) f += '/';
  f += _parent->_parent->vSharedFiles[index]->file;

  item = new QListViewItem(_parent->_parent->ui_listview_uploads, _parent->_parent->vSharedFiles[index]->file.c_str());
  item->setText(2, _socket->peerAddress().toString());

  _file.setName(f.c_str());

  if( !_file.exists() )
    {
      std::string s;
      s = "HTTP/1.0 404 NOT FOUND\r\nServer: Qtella "+std::string(VERSION)+"\r\n\r\n";
      _socket->writeBlock(s.data(), s.size());
      _socket->close();
      setState(NotFound);
      return;
    }

  setState(Connected);
  if(range_end == 0) range_end = _file.size() - 1;

  _towrite = range_end - range_start + 1;

  std::string s("HTTP/1.0 200 OK\r\nContent-Length: ");
  std::strstream str;
  str << _file.size() << std::ends;  // must be file.size() !!
  s += std::string(str.str());
  str.freeze(false);
  s += std::string("\r\nContent-Type: application/binary\r\nServer: Qtella "+std::string(VERSION)+"\r\n\r\n");

  _socket->writeBlock(s.c_str(), s.size());

  if( !_file.open(IO_ReadOnly|IO_Raw) )
    {
      setState(Error);
      _socket->close();
      return;
    }

  _file.at(range_start);
  _written = 0;
}



void Upload::slotTimer()
{
  char         cbuff[8192];
  unsigned int bytes = _parent->_parent->ui_spinbox_upbandwidth->value() * 1024;
  int          written = 0;

  bytes /= _parent->numberUploads();
  if(_socket->bytesToWrite() > bytes) bytes = 0;
  else
    bytes -= _socket->bytesToWrite();

  while(bytes > 0)
    {
      int r;
      int w;

      int toread = 8192;
      if(toread > bytes) toread = bytes;
      if(toread > _towrite) toread = _towrite;

      r = _file.readBlock(cbuff, toread);

      if(_socket->state() == QSocket::Connection) w = _socket->writeBlock(cbuff, r);
      bytes -= r;
      _towrite -= w;
      _written += w;
      written += w;
      _parent->_parent->connections->_statistic.loads_usize += w;

      if(w != r)
	{
	  _parent->_parent->connections->_statistic.loads_uabor++;
	  setState(Error);
	  _socket->close();
	  _file.close();
	  _timer->stop();
	  return;
	}

      if(_towrite == 0)
	{
	  _timer->stop();
	  _socket->close();
	  setState(Finished);
	  _parent->_parent->connections->_statistic.loads_usucc++;
	  _file.close();
	  
	  // final bandwidth
	  strstream str;
	  double b;
	  b = static_cast<double>((_written-range_start)*1000 / (_time_start.elapsed()*1024));
	  str.seekp(0, ios::beg);
	  str.setf(ios::fixed);
	  str << std::setprecision(1) << b << " KB/s" << std::ends;
	  item->setText(4, str.str());
	  str.freeze(false);
	  
	  // final time
	  QTime time_end = QTime();
	  time_end = time_end.addMSecs(_time_start.elapsed());
	  item->setText(5, (const char*)time_end.toString());

	  item->setText(3, "100%");
	  return;
	}
    }

  // bandwidth
  std::strstream str;
  str.setf(ios::fixed);
  str << std::setprecision(1) << static_cast<double>(written/1024) << " KB/s" << std::ends;
  item->setText(4, str.str());
  str.freeze(false);
 
  // progress
  str.seekp(0, ios::beg);
  str << 100 * _written / (range_end+1) << "%" << std::ends;
  item->setText(3, str.str());
  str.freeze(false);

  // time left
  double u = static_cast<double>( _written / (_time_start.elapsed()/1000) );
  unsigned long sectogo = static_cast<unsigned long>( _towrite / u );
  QTime time_togo = QTime();
  time_togo = time_togo.addSecs(sectogo);
  item->setText(5, (const char*)time_togo.toString());
}


