#ifndef DOWNLOAD_H
#define DOWNLOAD_H

#include <fstream>
#include <ctime>
#include <string>

#include <qobject.h>
#include <qdatetime.h>
#include <qtimer.h>
#include <qlistview.h>

#include "QueryHitEntry.h"

class QSocket;
class QFile;

class DownloadViewItem;
class DownloadManager;
class QtellaSub;

#if !defined(_WIN32)
using namespace std;
#endif

//! Each object of this class represents a download.
/*!
  All instances of this class are managed by the DownloadManager.
 */
class Download : public QObject
{
  Q_OBJECT

 public:
  //! Constructor
  /*!
    \param qhe The QueryHitResult of file. qhe.item should set to NULL because
    this will prevent the deletion of the item from the search list by the destructor.

    \param parent Pointer to the parent, the DownloadManager.

    \param item Pointer to the item in the search tab.

    \param dest_file If this parameter is empty qhe.filename is used for the
    filename for incompleted and completed downloads, otherwise dest_file is used.

    \param int_item If the download is resumed from the interrupted file list this
    parameter is a valid pointer to the item of the interrupted download, otherwise
    it is set to NULL.

    \param dest_dir The directory into which the completed download
    is saved. If this directory is empty, the download is saved into the
    directory specified in the configuration tab.

    \param start Start offset of the download. By default this is set to 0, the
    beginning of the file.

    \param end End offset of the download. By default this is set to -1 which
    indicates that the file should be downloaded until the end has been reached.

    \param segmented if set to true, the download is a segmented download and
    will stay in incompleted path when finished

    \param parent make this item a sub item of parent
   */
  Download(const QueryHitEntry&    qhe, 
	         DownloadManager*  parent, 
	         QListViewItem*    item, 
	   const std::string&      dest_file, 
	         QListViewItem*    int_item, 
	   const std::string&      dest_dir = "",
	   const unsigned long     start = 0, 
	   const long              end = -1,
	   const bool              segmented = false,
	         QListViewItem*    parent_item = NULL);

  //! Destructor
  /*!
    - removes item from download tab <br>
    - clears the status ball from search tab <br>
    - clears the status ball from interrupted downloads
  */
  ~Download();

  void                run();
  const int           readState();
  const double        readBandwidth();
  const int           readRetryTimeout();
  const unsigned long bytesRead();
  const unsigned long bytesToRead();

  const void          updateEnd( const unsigned long new_end );

  const void          createUniqueFilename( std::string& filename );

  //! returns the absolute position where the whole download started
  const unsigned long readStart();

  void                startPushedDownload(QSocket *socket);

  //! abort download
  /*!
    \param replaced set true if this download was replaced by another one
  */
  void                abortDownload(const bool replaced = false);

  void                resumeDownload();
  void                finish();
  bool                isActive();
  bool                moveFile(std::string directory);
  std::string         getFileName();
  bool                copyFile(std::string src, std::string dst);
  //! returns a unique id for the filename from _time and _id
  const std::string   getID();

  
  enum { dtIdle,             /*!< Initial state                                     */
	 dtError,            /*!< Error occured.                                    */
	 dtAbort,            /*!< Download aborted by user.                         */
	 dtClosed,           /*!< Connection was closed without any bytes read.     */
	 dtClosedCompleted,  /*!< Connection was closed after finishing download.   */
	 dtClosedUnfinished, /*!< Connection was closed but download not completed. */
	 dtConnecting,       /*!< Try to establish a connection.                    */
	 dtNotFound,         /*!< */
	 dtTimeout,          /*!< */
	 dtBusy,             /*!< */
	 dtConnected,        /*!< Connected.                                        */
	 dtExists,           /*!< File already exists.                              */
	 dtQueued,
	 dtPushed
  };

  std::string       _interrupted_filename;

  //! Pointer to the download item in the download list
  /*!
    This item is deleted automatically when ~DownloadThread() is called.
    \sa ~DownloadThread()
  */
  DownloadViewItem* _item;

  QListViewItem*    _int_item;

  //! Holds information about the file to be downloaded.
  QueryHitEntry     _qhe;

  //! File to which incoming data is written to.
  QFile*            _f;

 private:
  friend class DownloadManager;

  std::string   createFilename(std::string file, unsigned long size, std::string host);
  std::string   findFile(std::string file, unsigned long size);
  std::string   findFile(std::string file, std::string host);
  std::string   findFile(std::string file, unsigned long size, std::string host);

  //! Time which is encoded into the filename
  time_t            _time;
  //! Randomly generated number which is encoded into the filename
  unsigned          _id;

  //! the whole download started at this position
  unsigned long     _start;
  //! the download ends at this position
  unsigned long     _end;

  unsigned long     _all_read;

  //! Pointer to QtellaSub
  QtellaSub*        _c;
  QListViewItem*    _search_item;

  void              setState(int);

  //! Destination filename.
  std::string       _dest_file;

  //! Name of the server.
  std::string       _client;

  //! Current state of the this download.
  int               _state;

  //! Size of the file.
  unsigned long     _content_length;

  //! Time when download started.
  time_t            _start_time;

  //! Time when download finished.
  time_t            _end_time;

  //! Buffer to store incoming data.
  std::string       _buffer;

  //! True if run() method was not invoked yet.
  bool              _first_start;

  //! Number of retries.
  int               _retries;

  //! Stores the HTML-Header
  std::string       _header;

  DownloadManager*  _parent;

  std::string       complete_path;
  std::string       incomplete_path;

  bool              _active;

  QSocket*          sock;
  QTimer*           timer_bandwidth;
  QTimer*           timer_retry;
  int               retry_timeout;
  Q_UINT32          _start_read;
  unsigned long     tmp_read;
  double            bandwidth;

  QTimer*           timeout;

  std::string       _filename;  // current complete filename for moveFile

  bool              _segmented;

  //! Is set true when finished() is called.
  bool              _finished;

  //! Is set to parent item if this is a segmented download.
  QListViewItem*    _parent_item;

  //! Is set true if this download was replaced by another one.
  bool              _replaced;

  //! The directory given to the constructor where to store the file.
  std::string       _dst_directory;

 public slots:
  void           slotReadyRead();
  void           slotConnectionClosed();
  void           slotError(int);
  void           slotConnected();
  void           slotTimerBandwidth();
  void           slotTimeout();
  void           slotRetry();
  void           slotUpdateStatus();
};

#endif
