#ifndef SERVENT_H
#define SERVENT_H

#include <qobject.h>
#include <qtimer.h>

#include <string>
#include <list>
#include <set>
#include <ctime>
#include <vector>

#include "Gnutella.h"
#include "Address.h"

#define QTELLA_WIN

class Ping;
class Host;
class Address;
class Message;
class Connections;
class PongCache;

class QListViewItem;
class QSocket;

#if !defined(_WIN32)
using namespace std;
#endif

//! This class is used to communicate to other Gnutella host.

class Servent : public QObject
{
  Q_OBJECT

 public:
  //! Creates an object with the given Gnutella ID.
  /*!
      \param id The Gnutella ID used for this host.
  */
  Servent(Connections* parent, std::string id, QListViewItem* item, QSocket* socket = NULL, const std::string user_agent = "", const std::string handshake = "" );

  //! Destructor.
  ~Servent();

  //! Establish a connection to another host at the given address.
  /*!
      This method returns immediately. 
      \param address Address of the other host.
      \sa Address
   */
  void    connect(Address address);

  void    close();

  long    uptime();

  //! Returns the user id of the remote client.
  const std::string& getUserID();

  //! Returns the group id of the remote client.
  const std::string& getGroupID();

  //! Returns the content of the header field.
  static const std::string getHeaderField( const std::string& header, const std::string& field );

  //! Returns the client's state.
  /*!
      \return serventIdle, serventConnected, serventError serventClosed
  */
  int          state();
  
  void         writeConnectString();

  void         limitVectors();

  int          getDirection();

  void         setDirection(int direction);

  void         setState(int state);

  void         setByeCode( const int code );

  void         sendMessage(std::string& buffer, bool cache = true);
  void         sendMessageDirect( const std::string& buffer );

  const void   updateLastSearch();

  std::string       _connect_str;
  std::string       _response_str;

  enum { Outgoing, Incoming };

  //! States for connection to other host.
  enum { Idle,       /*!< Not connected.                    */
	 Connected,  /*!< Connected to other Gnutella host. */
	 Error,      /*!< An error occurred.                */
	 Connecting, /*!< Connecting...                     */
	 Waiting,    /*!< Socket is open.                   */
	 Closed,     /*!< Connection closed                 */
	 Closing     /*!< Connection closed, message buffer not empty */
  };

  QSocket*          _socket;
  QListViewItem*    _item;
  int               _timeout;

  //! Unique Gnutella ID of this host.
  std::string       _id;

  time_t            _start;
  int               _queries;

 public slots:
    void    slotDelayedClosed();

    //! Called if an error occured.
    /*!
        State is set to serventError.
    */
    void    slotError(int);

    //! Called if connection has been established.
    /*! 
        State is set to serventConnected.
    */
    void    slotConnect();

    //! Called if new data has been arrived.
    /*!
     */
    void    slotReadyRead();

    //! Called if other client has closed the connection.
    /*!
        State is set to serventIdle.
    */
    void    slotClosed();

    void    timeout();

    const int       getBufferSize();

 private:
    static int      getNextServentNumber();
    
    //! Holds the current state of the connection.
    int             _state;

    //! Buffer which takes all incoming data.
    std::vector<char>         _buffer;
    std::string::size_type    vect_find( const std::vector<char>& v, const std::string& pattern );
    std::string               vect_substr( const std::vector<char>& v, const unsigned beg, const unsigned num );

    const void                checkXTry();

    //! Used for connectToServent(...) to have a valid object.
    std::string     _strip;

    Connections*    _parent;

    int             _timeout_counter;
    int             _bandwidth_counter;
    int             _ping_counter;
    int             _direction;

    QTimer*         _bandwidth_timer;
    int             _bytes_in;
    int             _bytes_out;

    QTimer*         _timeout_timer;

    time_t          _last_input;

    //! Time of last search.
    time_t          _last_search;

 public:
    double          _bandwidth_in;
    double          _bandwidth_out;

    std::string     _send_buffer;

    std::vector< char >&          getFirstMessage();
    const void                    removeFirstMessage();
    
    std::list<std::vector<char> > _message_buffer;
    
    unsigned int              _servant_number;
    time_t                    _last_ping_send;
    std::string               _last_accepted_ping_id;

    std::set<std::string>     _id_set;

    Address                   _address;

    //! Number of input messages
    long                      _input;
    
    //! Number of output messages
    long                      _output;

    //! Version of protocol
    std::string               _version;

    //! Client name of other host
    std::string               _user_agent;

    //! Header
    std::string               _header;

    //! Status code for Bye packet
    int                       _bye;

 private:
    //! user id of the remote client
    std::string               _user_id;
    //! group id of the remote client
    std::string               _group_id;
};

#endif
