/*!
    $Author: etzi $
    $Revision: 1.9 $
    $Date: 2003/01/20 12:15:36 $
*/    

#include "AudioMp3.h"

#include <iostream>
#include <strstream>
#include <fstream>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <iostream>

#if defined(HAVE_ID3_LIB)
#include <id3/tag.h>
#include <id3/misc_support.h>
#endif

#define MIN_CONSEC_GOOD_FRAMES 4
#define FRAME_HEADER_SIZE 4
#define MIN_FRAME_SIZE 21
#define NUM_SAMPLES 4

#define TEXT_FIELD_LEN       30
#define INT_FIELD_LEN        4

#ifndef BUFFER_SIZE_LEN
#define BUFFER_SIZE_LEN		32768
#endif

using namespace std;
enum VBR_REPORT { VBR_VARIABLE, VBR_AVERAGE, VBR_MEDIAN };

class mp3header
{
public:
	int		get_header(std::ifstream &file);
	int		frame_length();
	int		header_layer();
        const char      *header_version() { switch( version ) 
	  {
	  case 2: return "2.5"; break;
	  case 1: return "1.0"; break;
	  default:return "2.0";
	  }
	}
	unsigned int	getbitrate() { return bitrate; };
	bool		sameConstant(mp3header *h2);
	int		header_bitrate();
	int		header_frequency();
	char		*header_emphasis();
	char		*header_mode();

	void		setbitrate( unsigned int br )
				{ bitrate = br; };

private:
	unsigned long	sync;
	unsigned int	version;
	unsigned int	layer;
	unsigned int	crc;
	unsigned int	bitrate;
	unsigned int	freq;
	unsigned int	padding;
	unsigned int	extension;
	unsigned int	mode;
	unsigned int	mode_extension;
	unsigned int	copyright;
	unsigned int	original;
	unsigned int	emphasis;
};

#if !defined(HAVE_ID3_LIB)
class id3tag
{
public:
	char		title[31];
	char		artist[31];
	char		album[31];
	char		year[5];
	char		comment[31];
	unsigned char	track[1];
	unsigned char	genre[1];
};
#endif

class mp3info
{
 public:
	int		get_first_header(long startpos);
	int		get_next_header();
	int		write_tag();
	bool		get_mp3_info(int scantype, int fullscan_vbr);
	void		setfilename( const char *fn )
				{ filename = (char *)fn; };
	int		getseconds() { return seconds; };
	bool		valid_header() { return header_isvalid; };
	mp3header	header;
#if !defined(HAVE_ID3_LIB)
	int		get_id3();
	id3tag		id3;
	bool		valid_id3() { return id3_isvalid; };
#endif

private:
	char		*pad(char *string, int length);
	char		*unpad(char *string);
	char		*filename;
    std::ifstream	file;
	unsigned int	datasize;
	bool		header_isvalid;
	int		vbr;
	float		vbr_average;
	int		seconds;
	int		frames;
	int		badframes;
#if !defined(HAVE_ID3_LIB)
	bool		id3_isvalid;
#endif
};

static int layer_tab[4] = { 0, 3, 2, 1 };

static int frequencies[3][4] = {
   { 22050, 24000, 16000, 50000 },  /* MPEG 2.0 */
   { 44100, 48000, 32000, 50000 },  /* MPEG 1.0 */
   { 11025, 12000,  8000, 50000 }   /* MPEG 2.5 */
};

static int bitrate_tab[2][3][14] = { 
 { /* MPEG 2.0 */
  { 32, 48, 56,  64,  80,  96, 112, 128, 144, 160, 176, 192, 224, 256 }, /* layer 1 */
  {  8, 16, 24,  32,  40,  48,  56,  64,  80,  96, 112, 128, 144, 160 }, /* layer 2 */
  {  8, 16, 24,  32,  40,  48,  56,  64,  80,  96, 112, 128, 144, 160 }  /* layer 3 */
 },

 { /* MPEG 1.0 */
  { 32, 64, 96, 128, 160, 192, 224, 256, 288, 320, 352, 384, 416, 448 }, /* layer 1 */
  { 32, 48, 56,  64,  80,  96, 112, 128, 160, 192, 224, 256, 320, 384 }, /* layer 2 */
  { 32, 40, 48,  56,  64,  80,  96, 112, 128, 160, 192, 224, 256, 320 }  /* layer 3 */
 }
};

static int frame_size_index[] = { 24000, 72000, 72000 };


static char *mode_text[] = {
   "stereo", "joint stereo", "dual channel", "mono"
};

static char *emphasis_text[] = {
  "none", "50/15 microsecs", "reserved", "CCITT J 17"
};


bool mp3info::get_mp3_info(int scantype, int fullscan_vbr)
{
    bool	had_error = false;
    int		frame_type[15] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
    float	sec = 0,
		total_rate = 0;
    int		xframes = 0,
		frame_types = 0,
		frames_so_far = 0;
    int		l,
		vbr_median = -1;
    int		bitrate,
		lastrate;
    int		counter = 0;
    mp3header	head;
    std::streampos	sample_pos,
		data_start = 0;

    file.open( filename, (std::ios::in | std::ios::binary) );
    if( file.bad() )
	return false;
    else {
	file.seekg( 0, std::ios::end );
	datasize = file.tellg();
    }

#if !defined( HAVE_ID3_LIB )
    get_id3();
#endif

    if( get_first_header(0L) )
    {
#ifdef FULLHEADERSCAN
	data_start = file.tellg();
	while( (bitrate = get_next_header()) )
	{
	    frame_type[15-bitrate]++;
	    xframes++;
	}
	memcpy( &head, &(header), sizeof(mp3header) );
	for( counter = 0; counter < 15; counter++ )
	{
	if( frame_type[counter] )
	{
	    frame_types++;
	    header.setbitrate(counter);
	    frames_so_far += frame_type[counter];
	    sec += (float)(header.frame_length()*frame_type[counter])/
		           (float)(header.header_bitrate()*125);
	    total_rate += (float)((head.header_bitrate())*frame_type[counter]);
	    if( (vbr_median == -1) && (frames_so_far >= xframes/2) )
		vbr_median = counter;
	    }
	}
	seconds = (int)(sec + 0.5);
	header.setbitrate(vbr_median);
	vbr_average = total_rate/(float)xframes;
	frames = xframes;
	if( frame_types > 1 )
	    vbr = 1;
#else // no FULLHEADERSCAN
	data_start = file.tellg();
	lastrate = 15 - header.header_bitrate();
	while( (counter < NUM_SAMPLES) && lastrate )
	{
	    sample_pos = (counter * (datasize/NUM_SAMPLES+1)) + data_start;
	    if( get_first_header(sample_pos) )
	    {
		bitrate = 15 - header.header_bitrate();
	    } else {
		bitrate = -1;
	    }
	    if( bitrate != lastrate )
	    {
		vbr = 1;
	    }
	    lastrate = bitrate;
	    counter++;
	    frames = (datasize - data_start) / (l = header.frame_length() );
	    seconds = (int)((float)(l * frames) /
				(float)(header.header_bitrate()*125)+0.5);
	    vbr_average = (float)header.header_bitrate();
	}
#endif
    }
    file.close();
    return !had_error;
}


int mp3info::get_first_header(long startpos) 
{
    int		k, l = 0, c;
    mp3header	h, h2;
    long	valid_start = 0;
  
    file.seekg( startpos, std::ios::beg );
    while (1)
    {
	while( (c = file.peek()) != 255 && (c != EOF) )
	    file.get();
	if( c == 255 )
	{
	    valid_start = file.tellg();
	    l = h.get_header( file );
	    if( l )
	    {
		file.seekg( l-FRAME_HEADER_SIZE, std::ios::cur );
		for(k=1; (k < MIN_CONSEC_GOOD_FRAMES)
			&& (datasize-file.tellg() >= FRAME_HEADER_SIZE); k++)
	        {
		    l = h2.get_header(file);
		    if( !l ) break;
		    if( !h.sameConstant(&h2) ) break;
		    file.seekg( l-FRAME_HEADER_SIZE, std::ios::cur );
		}
		if( k == MIN_CONSEC_GOOD_FRAMES )
		{
		    file.seekg( valid_start, std::ios::beg );
		    memcpy( &(header), &h2, sizeof(mp3header) );
		    header_isvalid = 1;
		    return 1;
		} 
	    }
	} else {
	    return 0;
	}
    }
    return 0;  
}

/* get_next_header() - read header at current position or look for 
   the next valid header if there isn't one at the current position
*/
int mp3info::get_next_header() 
{
    int		l = 0,
		c,
		skip_bytes = 0;
    mp3header	h;
  
    while( 1 )
    {
	while( (c = file.peek()) != 255 && (file.tellg() < datasize) )
	{
	    skip_bytes++;
	    file.get();
	}

	if( c == 255 )
	{
	    l = h.get_header(file);
	    if( l )
	    {
		if(skip_bytes) badframes++;
		file.seekg(l-FRAME_HEADER_SIZE, std::ios::cur);
		return 15-h.getbitrate();
	    } else {
		skip_bytes += FRAME_HEADER_SIZE;
	    }
	} else {
	    if( skip_bytes )
		badframes++; 

	    return 0;
	}
    }
}


/* Get next MP3 frame header.
   Return codes:
   positive value = Frame Length of this header
   0 = No, we did not retrieve a valid frame header
*/

int mp3header::get_header(std::ifstream &file)
{
    unsigned char	buffer[FRAME_HEADER_SIZE];
    int			fl;

    if( !file.read(reinterpret_cast<char *>(buffer), FRAME_HEADER_SIZE) )
      {
        sync = 0;
        return 0;
      }
    sync=( ((int)buffer[0] << 4) | ((int)(buffer[1] & 0xE0) >> 4) );
    if( buffer[1] & 0x10 )
      version = (buffer[1] >> 3) & 1;
    else
      version = 2;
    layer = (buffer[1] >> 1) & 3;
    if( (sync != 0xFFE) || (layer != 1) )
      {
        sync = 0;
        return 0;
      }
    crc = buffer[1] & 1;
    bitrate = (buffer[2] >> 4) & 0x0F;
    freq = (buffer[2] >> 2) & 0x3;
    padding = (buffer[2] >>1) & 0x1;
    extension = (buffer[2]) & 0x1;
    mode = (buffer[3] >> 6) & 0x3;
    mode_extension = (buffer[3] >> 4) & 0x3;
    copyright = (buffer[3] >> 3) & 0x1;
    original = (buffer[3] >> 2) & 0x1;
    emphasis = (buffer[3]) & 0x3;
    
    return ( (fl = frame_length()) >= MIN_FRAME_SIZE ? fl : 0 ); 
}

int mp3header::frame_length()
{
    return ((sync == 0xFFE) ? 
	(frame_size_index[3-layer]*((version & 1) + 1) *
	header_bitrate() / header_frequency() ) + padding : 1 );
}

int mp3header::header_layer()
{
  return layer_tab[layer];
}

int mp3header::header_bitrate()
{
  return bitrate_tab[version & 1][3-layer][bitrate-1];
}

int mp3header::header_frequency()
{
  return frequencies[version][freq];
}

char *mp3header::header_emphasis()
{
  return emphasis_text[emphasis];
}

char *mp3header::header_mode()
{
  return mode_text[mode];
}

bool mp3header::sameConstant(mp3header *h2)
{
    if((*(uint*)this) == (*(uint*)h2))
	return 1;

    if((this->version       == h2->version         ) &&
       (this->layer         == h2->layer           ) &&
       (this->crc           == h2->crc             ) &&
       (this->freq          == h2->freq            ) &&
       (this->mode          == h2->mode            ) &&
       (this->copyright     == h2->copyright       ) &&
       (this->original      == h2->original        ) &&
       (this->emphasis      == h2->emphasis        )) 
		return 1;

    return 0;
}


#if !defined(HAVE_ID3_LIB)
int mp3info::get_id3()
{
    int	retcode = 0;
    char fbuf[4];

    id3_isvalid = false;

    if( datasize >= 128 )
    {
	if( ! file.seekg( -128, std::ios::end ) )
	{
	  std::cerr << "ERROR: Couldn't read last 128 bytes of "
			<< filename << "!!" << std::endl;
	   retcode |= 4;
	} else {
	   file.read(fbuf, 3 );
	   fbuf[3] = '\0';
	   id3.genre[0] = 255;
	   id3.track[0] = 0;


	   if( !strcmp((const char *)"TAG", (const char *)fbuf) )
	   {
   	      id3_isvalid = true;
	      datasize -= 128;
	      file.seekg( -125, std::ios::end );
              file.read( id3.title, 30 );	id3.title[30] = '\0';
              file.read( id3.artist, 30 );	id3.artist[30] = '\0';
              file.read( id3.album, 30 );	id3.album[30] = '\0';
              file.read( id3.year, 4 );		id3.year[4] = '\0';
              file.read( id3.comment, 30 );	id3.comment[30] = '\0';
   	      if( id3.comment[28] == '\0' )
	      {
   		 id3.track[0] = id3.comment[29];
   	      }
              file.read( reinterpret_cast<char*>(id3.genre), 1 );
	      unpad(id3.title);
	      unpad(id3.artist);
	      unpad(id3.album);
	      unpad(id3.year);
	      unpad(id3.comment);
           }
   	}
   }
   return retcode;

}
#endif // !HAVE_ID3_LIB

/* Remove trailing whitespace from the end of a string */

char *mp3info::unpad(char *string)
{
    char *pos = string + strlen(string) - 1;
    while( isspace(pos[0]) )
	(pos--)[0] = 0;
    return string;
}

///////////////////////////////////////////////////////////
//          AudioMp3 Class starts here
///////////////////////////////////////////////////////////
AudioMp3::AudioMp3( std::string filename )
{
  _length = 0;

  mp3info mp3inf;
  mp3inf.setfilename(filename.c_str());

  if( mp3inf.get_mp3_info( 0, 0) )
  {
    if( mp3inf.valid_header() )
    {
      std::strstream str;
      str << "MPEG " << mp3inf.header.header_version() <<
		" Layer " << mp3inf.header.header_layer() << std::ends;
      _format = str.str();
      _valid = true;
      _bitrate = mp3inf.header.header_bitrate();
      _samplerate = mp3inf.header.header_frequency();
      _channelmode = mp3inf.header.header_mode();
      _length = mp3inf.getseconds();
    }
#if defined(HAVE_ID3_LIB)
    ID3_Tag tag;
    tag.Link( filename.c_str() );
    char 	*ptr;

    if( NULL != (ptr = ID3_GetTitle( &tag )) )
      _title = std::string( ptr );
    if( NULL != (ptr = ID3_GetArtist( &tag )) )
      _artist = std::string( ptr );
    if( NULL != (ptr = ID3_GetAlbum( &tag )) )
      _album = std::string( ptr );
    if( NULL != (ptr = ID3_GetYear( &tag )) )
      _year = std::string( ptr );
    if( NULL != (ptr = ID3_GetComment( &tag )) )
      _comment = std::string( ptr );
    if( NULL != ID3_GetGenre( &tag ) )
      _genre = ID3_GetGenreNum( &tag );
    else
      _genre = 0;
    if( _genre > 150 ) _genre = 150;
    _track = ID3_GetTrackNum( &tag );
// There's a bug in id3lib-3.8pre2 related with a empty ID3v1.1 track tag
// this will be interpreted as a tracknumber of 32 (dec for space)
    if( tag.HasV1Tag() && _track == 32
	&& ID3LIB_MAJOR_VERSION == 3 && ID3LIB_MINOR_VERSION == 8
    ) _track = 0;

    _id3v1 = (tag.HasV1Tag() ? 1 : 0 );
    _id3v2 = (tag.HasV2Tag() ? 1 : 0 );
#else
    _id3v1 = 0;
    _id3v2 = 2;		// ID3V2 not available without id3lib
    if( mp3inf.valid_id3() )
    {
      _id3v1 = 1;
      _title = std::string( mp3inf.id3.title );
      _artist = std::string( mp3inf.id3.artist );
      _album = std::string( mp3inf.id3.album );
      _year = std::string( mp3inf.id3.year );
      _comment = std::string( mp3inf.id3.comment );
      _genre = (unsigned int) mp3inf.id3.genre[0];
      if( _genre > 150 ) _genre = 150;
      _track = (unsigned int) mp3inf.id3.track[0];
    }
#endif // HAVE_ID3_LIB
  } else {
    _format = std::string("Unknown");
    _valid = false;
  }
}

bool AudioMp3::updateInfo( std::string file )
{
#if defined(HAVE_ID3_LIB)
  flags_t flags = ID3TT_NONE;
  ID3_Tag tag;
  tag.Link( file.c_str() );

  if( _id3v1 == 1 ) flags |= ID3TT_ID3V1;
  if( _id3v2 == 1 ) flags |= ID3TT_ID3V2;

  if( tag.HasV1Tag() && _id3v1 == 0 )
  {
     tag.Strip( ID3TT_ID3V1 );
     flags &= ~ID3TT_ID3V1;
  }
  if( tag.HasV2Tag() && _id3v2 == 0 )
  {
     tag.Strip( ID3TT_ID3V2 );
     flags &= ~ID3TT_ID3V2;
  }
  if( _title.length() > 0 )
    ID3_AddTitle( &tag, _title.c_str(), TRUE );
  else
    ID3_RemoveTitles( &tag );
  if( _artist.length() > 0 )
    ID3_AddArtist( &tag, _artist.c_str(), TRUE );
  else
    ID3_RemoveArtists( &tag );
  if( _album.length() > 0 )
    ID3_AddAlbum( &tag, _album.c_str(), TRUE );
  else
    ID3_RemoveAlbums( &tag );
  if( _year.length() > 0 )
    ID3_AddYear( &tag, _year.c_str(), TRUE );
  else
    ID3_RemoveYears( &tag );
  ID3_RemoveComments( &tag );
  if( _comment.length() > 0 )
    ID3_AddComment( &tag, _comment.c_str(), TRUE );
  if( _track > 0 )
    ID3_AddTrack( &tag,  _track, 0, TRUE );
  else
    ID3_RemoveTracks( &tag );
  if( _genre > 0 && _genre < 255 )
    ID3_AddGenre( &tag, (size_t) _genre, TRUE );
  else
    ID3_RemoveGenres( &tag );
#ifdef DEBUG
  std::cout << "updateInfo(): New Values are:" << std::endl;
  std::cout << "title: " << _title.c_str() << std::endl;
  std::cout << "artist:" << _artist.c_str() << std::endl;
  std::cout << "track: " << _track << std::endl;
  std::cout << "flags: " << flags << std::endl;
#endif

  if( tag.Update( flags ) == ID3TT_NONE )
  {
    _updatemesg = "Updating failed";
    return false;
  }
  _updatemesg = "Ok";
  return true;
#else

  bool res = true;
  _updatemesg = "ok";

  bool id3present = true;

  std::fstream in( file.c_str(), std::ios::binary | std::ios::in | std::ios::out );

  in.seekg( -128, std::ios::end );
  unsigned int noheaderlen = in.tellg();

// Check again if the file has a ID3v1 extension
  char head[4];
  in.read( head, 3 );
  id3present = (strncmp( head, "TAG", 3 ) == 0);
  if( (_id3v1 == 1) && !id3present )
  {
     in.close();
     _updatemesg = "No ID3V1 extension tag found.";
     return false;
  }

  if( _id3v1 == 1 )
  {
    // if id3 tag has to be removed, we have to make a copy without the
    // trailing 128 bytes.

    std::string ofname( file + ".1" );
#ifdef DEBUG
    std::cout << "tempfile=" << ofname << std::endl;
#endif

    if( access( ofname.c_str(), R_OK ) == 0 )
    {
       _updatemesg = "Temporay file " + ofname + " exists!";
       return false;
    }

    std::ofstream ou( ofname.c_str(), std::ios::binary );

    unsigned int bytes = 0;
    in.seekg( 0, std::ios::beg );
    while( !in.eof() && (bytes < noheaderlen) )
    {
       unsigned int cnt;
       char buffer[BUFFER_SIZE_LEN];
       if( (noheaderlen - bytes) > BUFFER_SIZE_LEN )
  	in.read( buffer, BUFFER_SIZE_LEN );
       else
	in.read( buffer, (noheaderlen - bytes) );

       cnt = in.gcount();
       if( !ou.write( buffer, cnt ) )
       {
	in.close();
	ou.close();
	_updatemesg = "Error copy";
	res = false;
       }
       bytes += cnt;
    }
    ou.close();
    in.close();

    if( (unlink( file.c_str() ) != 0)
     || (link( ofname.c_str(), file.c_str() ) != 0)
     || (unlink( ofname.c_str() ) != 0)
    )
    {
	_updatemesg = "Unable to copy " + file;
	res = false;
    }
  } else { // Update id3 information

    if( access( file.c_str(), W_OK ) != 0 )
    {
	in.close();
	_updatemesg = strerror( errno );
	return false;
    }

    char buffer[129];

    memset( buffer, '\0', 128 );
    strcpy( buffer, "TAG" );
    strncat( &buffer[3], _title.c_str(), TEXT_FIELD_LEN);
    strncat( &buffer[33], _artist.c_str(), TEXT_FIELD_LEN);
    strncat( &buffer[63], _album.c_str(), TEXT_FIELD_LEN);
    strncat( &buffer[93], _year.c_str(), INT_FIELD_LEN);
    strncat( &buffer[97], _comment.c_str(), TEXT_FIELD_LEN);
    buffer[125] = '\0';
    buffer[126] = (char)_track;
    buffer[127] = (char)_genre;

// debuging output
#ifdef DEBUG
  std::cout << "0x0: ";
  for( int i = 0; i< 129; i++ )
  {
     if( (i % 8) == 0)
	std::cout << endl << "0x" << hex << i << ": ";
     if( isprint(buffer[i]) )
     {
        std::cout << buffer[i];
     } else {
	std::cout << " ";
     }
     std::cout << " 0x";
     if( (int)(buffer[i]&0xff) < 16 )
	std::cout << "0";
     std::cout << hex << (int)(buffer[i]&0xff) << " | ";
     
  }
  std::cout << std::endl;;
#endif

    // Add the id3 tag at the end of the file, or override the existing one
    in.seekg( -128 * (id3present ? 1 : 0), std::ios::end );
    if( !in.write( buffer, 128 ) )
    {
	_updatemesg = "Error on adding/overriding ID3 tag";
	res = false;
    }
    in.close();
  }

  return res;
#endif // HAVE_ID3_LIB
}

