/*!
    $Author: etzi $
    $Revision: 1.5 $
    $Date: 2002/07/28 11:47:50 $
*/    

#if defined(HAVE_OGG_LIB)

#include "AudioOgg.h"
#include "AudioGenre.h"
#include "StringManipulation.h"

#include <iostream>
#include <strstream>
#include <fstream>

#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <codec.h>
#include <ogg/ogg.h>
#include <vorbisfile.h>

///////////////////////////////////////////////////////////
//         Most functions here are taken from the
//         vorbistools-1.0rc3 package
///////////////////////////////////////////////////////////

typedef size_t (*vcedit_read_func)(void *, size_t, size_t, void *);
typedef size_t (*vcedit_write_func)(const void *, size_t, size_t, void *);

class VCedit
{
private:
    ogg_sync_state	*_oy;
    ogg_stream_state	*_os;

    vorbis_comment	*_vc;
    vorbis_info          _vi;

    vcedit_read_func	_read;
    vcedit_write_func	_write;

    void		*_in;
    long		_serial;
    unsigned char	*_mainbuf;
    unsigned char	*_bookbuf;
    int			_mainlen;
    int			_booklen;
    char		*_lasterror;
    char  		*_vendor;
    int			_prevW;
    int			_extrapage;
    int			_eosin;

public:
    ~VCedit();
    char		*vcError() { return _lasterror; };
    vorbis_comment	*vcComments() { return _vc; };
    vorbis_comment	*vcCommentCopy();
    void		vcWriteString( oggpack_buffer *o, char *s, int len );
    int			vcCommentHeaderOut( ogg_packet *op );
    int			vcBlocksize( ogg_packet *p );
    int			vcFetchNextPacket( ogg_packet *p, ogg_page *page );
    int			vcOpen( FILE *in );
    int			vcOpenCallbacks( void *iin,
				vcedit_read_func read_func,
				vcedit_write_func write_func);
    int			vcWrite( void *out );
    int			vcAddComment( const char *tag, const char *value );
}; // vcedit_state;

#define CHUNKSIZE 4096

VCedit::~VCedit()
{
    if(_vc)
    {
        vorbis_comment_clear(_vc);
        free(_vc);
        _vc = NULL;
    }
    if(_os)
    {
        ogg_stream_clear(_os);
        free(_os);
        _os = NULL;
    }
    if(_oy)
    {
        ogg_sync_clear(_oy);
        free(_oy);
        _oy = NULL;
    }
    if(_vendor)
    {
        free(_vendor);
        _vendor = NULL;
    }
}

/* Next two functions pulled straight from libvorbis, apart from one change
 * - we don't want to overwrite the vendor string.
 */
void VCedit::vcWriteString(oggpack_buffer *o,char *s, int len)
{
    while(len--)
    {
        oggpack_write(o,*s++,8);
    }
}

int VCedit::vcCommentHeaderOut(ogg_packet *op)
{
    oggpack_buffer opb;

    oggpack_writeinit(&opb);

    /* preamble */  
    oggpack_write(&opb,0x03,8);
    vcWriteString(&opb,"vorbis", 6);

    /* vendor */
    oggpack_write(&opb, strlen(_vendor), 32);
    vcWriteString(&opb, _vendor, strlen(_vendor));

    /* comments */
    oggpack_write(&opb,_vc->comments,32);
    if(_vc->comments){
        int i;
        for(i=0;i<_vc->comments;i++){
            if(_vc->user_comments[i]){
                oggpack_write(&opb,_vc->comment_lengths[i],32);
                vcWriteString(&opb,_vc->user_comments[i], 
                        _vc->comment_lengths[i]);
            }else{
                oggpack_write(&opb,0,32);
            }
        }
    }
    oggpack_write(&opb,1,1);

    op->packet = (unsigned char *)_ogg_malloc(oggpack_bytes(&opb));
    memcpy(op->packet, opb.buffer, oggpack_bytes(&opb));

    op->bytes=oggpack_bytes(&opb);
    op->b_o_s=0;
    op->e_o_s=0;
    op->granulepos=0;

    return 0;
}

int VCedit::vcBlocksize(ogg_packet *p)
{
    int thisone = vorbis_packet_blocksize(&_vi, p);
    int ret = (thisone + _prevW)/4;

    if(!_prevW)
    {
        _prevW = thisone;
        return 0;
    }

    _prevW = thisone;
    return ret;
}

int VCedit::vcFetchNextPacket(ogg_packet *p, ogg_page *page)
{
    int result;
    char *buffer;
    int bytes;

    result = ogg_stream_packetout(_os, p);

    if(result > 0)
        return 1;
    else
    {
        if(_eosin)
            return 0;
        while(ogg_sync_pageout(_oy, page) <= 0)
        {
            buffer = ogg_sync_buffer(_oy, CHUNKSIZE);
            bytes = _read(buffer,1, CHUNKSIZE, _in);
            ogg_sync_wrote(_oy, bytes);
            if(bytes == 0) 
                return 0;
        }
        if(ogg_page_eos(page))
            _eosin = 1;
        else if(ogg_page_serialno(page) != _serial)
        {
            _eosin = 1;
            _extrapage = 1;
            return 0;
        }

        ogg_stream_pagein(_os, page);
        return vcFetchNextPacket(p, page);
    }
}

int VCedit::vcOpen(FILE *in)
{
    return vcOpenCallbacks((void *)in, 
            (vcedit_read_func)fread, (vcedit_write_func)fwrite);
}

int VCedit::vcOpenCallbacks(void *iin,
        vcedit_read_func read_func, vcedit_write_func write_func)
{

    char *buffer;
    int bytes,i;
    ogg_packet *header;
    ogg_packet    header_main;
    ogg_packet  header_comments;
    ogg_packet    header_codebooks;
    ogg_page    og;

    _in = iin;
    _read = read_func;
    _write = write_func;

    _oy = (ogg_sync_state *)malloc(sizeof(ogg_sync_state));
    ogg_sync_init(_oy);

    buffer = ogg_sync_buffer(_oy, CHUNKSIZE);
    bytes = _read(buffer, 1, CHUNKSIZE, _in);

    ogg_sync_wrote(_oy, bytes);

    if(ogg_sync_pageout(_oy, &og) != 1)
    {
        if(bytes<CHUNKSIZE)
            _lasterror = "Input truncated or empty.";
        else
            _lasterror = "Input is not an Ogg bitstream.";
        goto err;
    }

    _serial = ogg_page_serialno(&og);

    _os = (ogg_stream_state *)malloc(sizeof(ogg_stream_state));
    ogg_stream_init(_os, _serial);

    vorbis_info_init(&_vi);

    _vc = (vorbis_comment *)malloc(sizeof(vorbis_comment));
    vorbis_comment_init(_vc);

    if(ogg_stream_pagein(_os, &og) < 0)
    {
        _lasterror = "Error reading first page of Ogg bitstream.";
        goto err;
    }

    if(ogg_stream_packetout(_os, &header_main) != 1)
    {
        _lasterror = "Error reading initial header packet.";
        goto err;
    }

    if(vorbis_synthesis_headerin(&_vi, _vc, &header_main) < 0)
    {
        _lasterror = "Ogg bitstream does not contain vorbis data.";
        goto err;
    }

    _mainlen = header_main.bytes;
    _mainbuf = (unsigned char *)malloc(_mainlen);
    memcpy(_mainbuf, header_main.packet, header_main.bytes);

    i = 0;
    header = &header_comments;
    while(i<2) {
        while(i<2) {
            int result = ogg_sync_pageout(_oy, &og);
            if(result == 0) break; /* Too little data so far */
            else if(result == 1)
            {
                ogg_stream_pagein(_os, &og);
                while(i<2)
                {
                    result = ogg_stream_packetout(_os, header);
                    if(result == 0) break;
                    if(result == -1)
                    {
                        _lasterror = "Corrupt secondary header.";
                        goto err;
                    }
                    vorbis_synthesis_headerin(&_vi, _vc, header);
                    if(i==1)
                    {
                        _booklen = header->bytes;
                        _bookbuf = (unsigned char *)malloc(_booklen);
                        memcpy(_bookbuf, header->packet, 
                                header->bytes);
                    }
                    i++;
                    header = &header_codebooks;
                }
            }
        }

        buffer = ogg_sync_buffer(_oy, CHUNKSIZE);
        bytes = _read(buffer, 1, CHUNKSIZE, _in);
        if(bytes == 0 && i < 2)
        {
            _lasterror = "EOF before end of vorbis headers.";
            goto err;
        }
        ogg_sync_wrote(_oy, bytes);
    }

    /* Copy the vendor tag */
    _vendor = (char *)malloc(strlen(_vc->vendor) +1);
    strcpy(_vendor, _vc->vendor);

    /* Headers are done! */
    return 0;

err:
    return -1;
}

int VCedit::vcWrite( void *out )
{
    ogg_stream_state streamout;
    ogg_packet header_main;
    ogg_packet header_comments;
    ogg_packet header_codebooks;

    ogg_page ogout, ogin;
    ogg_packet op;
    ogg_int64_t granpos = 0;
    int result;
    char *buffer;
    int bytes;
    int needflush=0, needout=0;

    _lasterror = "";
    _eosin = 0;
    _extrapage = 0;

    header_main.bytes = _mainlen;
    header_main.packet = _mainbuf;
    header_main.b_o_s = 1;
    header_main.e_o_s = 0;
    header_main.granulepos = 0;

    header_codebooks.bytes = _booklen;
    header_codebooks.packet = _bookbuf;
    header_codebooks.b_o_s = 0;
    header_codebooks.e_o_s = 0;
    header_codebooks.granulepos = 0;

    ogg_stream_init(&streamout, _serial);

    vcCommentHeaderOut(&header_comments);

    ogg_stream_packetin(&streamout, &header_main);
    ogg_stream_packetin(&streamout, &header_comments);
    ogg_stream_packetin(&streamout, &header_codebooks);

    while((result = ogg_stream_flush(&streamout, &ogout)))
    {
        if(_write(ogout.header,1,ogout.header_len, out) !=
                (size_t) ogout.header_len)
            goto cleanup;
        if(_write(ogout.body,1,ogout.body_len, out) != 
                (size_t) ogout.body_len)
            goto cleanup;
    }

    while(vcFetchNextPacket(&op, &ogin))
    {
        int size;
        size = vcBlocksize(&op);
        granpos += size;

        if(needflush)
        {
            if(ogg_stream_flush(&streamout, &ogout))
            {
                if(_write(ogout.header,1,ogout.header_len, 
                            out) != (size_t) ogout.header_len)
                    goto cleanup;
                if(_write(ogout.body,1,ogout.body_len, 
                            out) != (size_t) ogout.body_len)
                    goto cleanup;
            }
        }
        else if(needout)
        {
            if(ogg_stream_pageout(&streamout, &ogout))
            {
                if(_write(ogout.header,1,ogout.header_len, 
                            out) != (size_t) ogout.header_len)
                    goto cleanup;
                if(_write(ogout.body,1,ogout.body_len, 
                            out) != (size_t) ogout.body_len)
                    goto cleanup;
            }
        }

        needflush=needout=0;

        if(op.granulepos == -1)
        {
            op.granulepos = granpos;
            ogg_stream_packetin(&streamout, &op);
        }
        else /* granulepos is set, validly. Use it, and force a flush to 
                account for shortened blocks (vcut) when appropriate */ 
        {
            if(granpos > op.granulepos)
            {
                granpos = op.granulepos;
                ogg_stream_packetin(&streamout, &op);
                needflush=1;
            }
            else 
            {
                ogg_stream_packetin(&streamout, &op);
                needout=1;
            }
        }        
    }

    streamout.e_o_s = 1;
    while(ogg_stream_flush(&streamout, &ogout))
    {
        if(_write(ogout.header,1,ogout.header_len, 
                    out) != (size_t) ogout.header_len)
            goto cleanup;
        if(_write(ogout.body,1,ogout.body_len, 
                    out) != (size_t) ogout.body_len)
            goto cleanup;
    }

    /* FIXME: freeing this here probably leaks memory */
    /* Done with this, now */
    vorbis_info_clear(&_vi);

    if (_extrapage)
    {
        if(_write(ogin.header,1,ogin.header_len,
                        out) != (size_t) ogin.header_len)
            goto cleanup;
        if (_write(ogin.body,1,ogin.body_len, out) !=
                (size_t) ogin.body_len)
            goto cleanup;
    }

    _eosin=0; /* clear it, because not all paths to here do */
    while(!_eosin) /* We reached eos, not eof */
    {
        /* We copy the rest of the stream (other logical streams)
         * through, a page at a time. */
        while(1)
        {
            result = ogg_sync_pageout(_oy, &ogout);
            if(result==0) break;
            if(result<0)
                _lasterror = "Corrupt or missing data, continuing...";
            else
            {
                /* Don't bother going through the rest, we can just 
                 * write the page out now */
                if(_write(ogout.header,1,ogout.header_len, 
                        out) != (size_t) ogout.header_len)
                    goto cleanup;
                if(_write(ogout.body,1,ogout.body_len, out) !=
                        (size_t) ogout.body_len)
                    goto cleanup;
            }
        }
        buffer = ogg_sync_buffer(_oy, CHUNKSIZE);
        bytes = _read(buffer,1, CHUNKSIZE, _in);
        ogg_sync_wrote(_oy, bytes);
        if(bytes == 0) 
        {
            _eosin = 1;
            break;
        }
    }
                            

cleanup:
    ogg_stream_clear(&streamout);
    ogg_packet_clear(&header_comments);

    free(_mainbuf);
    free(_bookbuf);

    if(!_eosin)
    {
        _lasterror =  
            "Error writing stream to output. "
            "Output stream may be corrupted or truncated.";
        return -1;
    }

    return 0;
}

vorbis_comment *VCedit::vcCommentCopy()
{
  vorbis_comment *vc_bak = (vorbis_comment *)malloc( sizeof( vorbis_comment ) );
  memcpy( vc_bak, _vc, sizeof( vorbis_comment ) );

  vc_bak->vendor = strdup( _vc->vendor );

  return vc_bak;
}

int VCedit::vcAddComment( const char *tag, const char *value )
{
  QString for_utf8( value );

#ifdef AUDIO_DEBUG
  std::cout << "vcAddComment: " << tag << "=" << value << std::endl;
#endif

  vorbis_comment_add_tag( _vc, (char *)tag, for_utf8.utf8().data() );
  return 0;
}

///////////////////////////////////////////////////////////
//          AudioOgg Class starts here
///////////////////////////////////////////////////////////
AudioOgg::AudioOgg( std::string filename )
{
  OggVorbis_File vf;
  FILE		*fp;
  AudioGenre	genre;

  if( (fp = fopen( filename.c_str(), "r" )) == NULL )
  {
     std::cerr << "cannot open \"" << filename << "\"" << std::endl;
     return;
  }
  if( ov_open( fp, &vf, NULL, 0 ) < 0 )
  {
     std::cerr << "error on opening \"" << filename << "\"" << std::endl;
     fclose( fp );
     return;
  }

  char **ptr = ov_comment(&vf, -1)->user_comments;
  vorbis_info *vi = ov_info(&vf, -1);
  _track = 0;
  _genre = genre.max();
  while( *ptr )
  {
    std::string key, value;
    StringManipulation tmp(*ptr);
    tmp.split( key, value, '=' );

    if( key == "TITLE" )
      _title = std::string( value );
    else if( key == "ARTIST" )
      _artist = std::string( value );
    else if( key == "ALBUM" )
      _album = std::string( value );
    else if( key == "DATE" )
      _year = std::string( value );
    else if( key == "COMMENT" )
      _comment = std::string( value );
    else if( key == "GENRE" )
    {
	if( isdigit( value.at(0) ) )
	  _genre = atoi( value.data() );
        else {
	  _genre = genre.findGenreNum( value.c_str() );
	}
    } else if( key == "TRACKNUMBER" )
    {
	if( isdigit( value.at(0) ) )
	  _track = atoi( value.data() );
    }
    ptr++;
  }

#ifdef AUDIO_DEBUG
  std::cout << "Bitstream has " << std::endl;
  std::cout << vi->channels << " channels," << std::endl;
  std::cout << vi->rate << " Hz," << std::endl;
  std::cout << "encoded by: " << ov_comment(&vf,-1)->vendor << std::endl;
#endif

  std::strstream str;

  switch( vi->channels ) {
  case 1:	_channelmode = std::string( "mono" );
		break;
  case 2:	_channelmode = std::string( "stereo/dual" );
		break;
  default:	str << vi->channels << " channels" << std::ends;
		_channelmode = str.str();
		break;
  }
  str.seekp( 0, std::ios::beg );

  str << "Ogg-Vorbis Version " << vi->version << 
	" (Vendor: " << ov_comment(&vf,-1)->vendor << ")" << std::ends;

  _format = str.str();
  _samplerate = vi->rate;
  _bitrate = ov_bitrate( &vf, -1 ) / 1024;
  _length = (unsigned int) ov_time_total( &vf, -1 );

  ov_clear( &vf );
}

bool AudioOgg::updateInfo( std::string filename )
{
  FILE		*in, *out;
  char		**ptr;
  std::string	of = filename + "tmp";
  bool		res = true;

  _updatemesg = "ok"; // Think positiv!

  if( (in = fopen( filename.c_str(), "r" )) == NULL )
  {
     _updatemesg = strerror( errno );
     return false;
  }

  if( (out = fopen( of.c_str(), "w" )) == NULL )
  {
    fclose( in );
    _updatemesg = strerror( errno );
    return false;
  }

  VCedit state;

  if( state.vcOpen( in ) < 0 )
  {
     fclose( in );
     fclose( out );
     unlink( of.c_str() );
     _updatemesg = "Is that a vorbis file?";
     return false;
  }

  vorbis_comment *vc_bak = state.vcCommentCopy();

  vorbis_comment_init( state.vcComments() );

  AudioGenre genre;

  ptr = vc_bak->user_comments;
  // loop though all comments and replace only the changable tags,
  // leave unsupported tags untouched!
  while( *ptr )
  {
    std::string		key, value;
    StringManipulation	tmp(*ptr);

    tmp.split( key, value, '=' );

    if( key == "TITLE" ) {
      if( !_title.empty() )
      {
        state.vcAddComment( key.c_str(), _title.c_str() );
        _title.erase();
      }
    } else if( key == "ARTIST" ) {
      if( !_artist.empty() )
      {
        state.vcAddComment( key.c_str(), _artist.c_str() );
        _artist.erase();
      }
    } else if( key == "ALBUM" ) {
      if( !_album.empty() )
      {
        state.vcAddComment( key.c_str(), _album.c_str() );
        _album.erase();
      }
    } else if( key == "DATE" ) {
      if( !_year.empty() )
      {
        state.vcAddComment( key.c_str(), _year.c_str() );
        _year.erase();
      }
    } else if( key == "COMMENT" ) {
      if( !_comment.empty() )
      {
        state.vcAddComment( key.c_str(), _comment.c_str() );
        _comment.erase();
      }
    } else if( key == "GENRE" ) {
      if( _genre < genre.max() )
      {
        state.vcAddComment( key.c_str(),
		genre.getGenreNameIdx(genre.findGenreIdx(_genre)) );
        _genre = genre.max()+1;
      }
    } else if( key == "TRACKNUMBER" ) {
      if( _track > 0 )
      { 
        std::strstream str;
        str << _track << std::ends;

        state.vcAddComment( key.c_str(), str.str() );
        _track = 0;
      }
    } else {
      state.vcAddComment( key.c_str(), value.c_str() );
    }

    ptr++;
  }
  // if any changes are present or no original tags were set
  if( !_title.empty() )
    state.vcAddComment( "TITLE", _title.c_str() );
  if( !_artist.empty() )
    state.vcAddComment( "ARTIST", _artist.c_str() );
  if( !_album.empty() )
    state.vcAddComment( "ALBUM", _album.c_str() );
  if( !_year.empty() )
    state.vcAddComment( "DATE", _year.c_str() );
  if( !_comment.empty() )
    state.vcAddComment( "COMMENT", _comment.c_str() );
  if( !_artist.empty() )
    state.vcAddComment( "ARTIST", _artist.c_str() );
  if( _genre < genre.max() )
  {
    state.vcAddComment( "GENRE",
		genre.getGenreNameIdx(genre.findGenreIdx(_genre)) );
  }
  if( _track > 0 )
  {
    std::strstream str;
    str << _track << std::ends;

    state.vcAddComment( "TRACKNUMBER", str.str() );
  }
  
  if( (state.vcWrite( out ) != 0) 
    || unlink( filename.c_str() )
    || link( of.c_str(), filename.c_str() )
    || unlink( of.c_str() ) )
  {
    unlink( of.c_str() );
    if( state.vcError() != "" )
      _updatemesg = state.vcError();
    else
      _updatemesg = "write failed";
    res = false;
  }
  vorbis_comment_clear( vc_bak );

  fclose( in );
  fclose( out );

  return res;
}
#endif // HAVE_OGG_LIB
