#include "Download.h"
#include "DownloadManager.h"
#include "StringManipulation.h"
#include "QtellaSub.h"
#include "Connections.h"
#include "Push.h"
#include "Logging.h"
#include "DownloadViewItem.h"
#include "Icon.h"
#include "InterruptedDownloads.h"
#include "InterruptedFile.h"
#include "InterruptedViewItem.h"
#include "WidgetDownloads.h"
#include "WidgetConfig.h"

#include <qsocket.h>
#include <qlistview.h>
#include <qfile.h>
#include <qdir.h>
#include <qradiobutton.h>
#include <qspinbox.h>
#include <qlineedit.h>
#include <qpixmap.h>
#include <qlistbox.h>
#include <qmessagebox.h>
#include <qimage.h>
#include <qcheckbox.h>

#include <iostream>
#include <iomanip>
#include <strstream>
#include <ctime>

Download::Download(const QueryHitEntry&      qhe, 
		         DownloadManager*    parent, 
		         QListViewItem*      item, 
		   const std::string&        dest_file, 
		         QListViewItem*      int_item, 
		   const std::string&        dest_dir,
		   const unsigned long       start, 
		   const long                end,
		   const bool                segmented,
		         QListViewItem*      parent_item)
  : 
  _parent(parent), _qhe(qhe), _state(dtQueued), _retries(0),
  sock(NULL), _active(false), _search_item(item), _all_read(0), _int_item(int_item),
  _start(start), _end(end), _time(0), _id(0), _segmented(segmented), _finished(false),
  _parent_item(parent_item), _replaced(false)
{
  _dst_directory = dest_dir;
  complete_path = dest_dir;

  if( end == -1 )
    _end = qhe.size - 1;
  
  _content_length = _end - _start + 1;


  if( dest_file.empty() ) 
    _dest_file = qhe.filename;
  else
    _dest_file = dest_file;


  // replace bad characters by "_"
  while( _dest_file.find("/") != std::string::npos )
    _dest_file.replace( _dest_file.find("/"), 1, "_");

  _c = QtellaSub::getInstance();

  if( item )
    item->setPixmap(2, *(Icon::getInstance()->getYellowBall()));
  _c->_interrupted_downloads->setYellow((InterruptedViewItem*)_int_item);

  // if a download if the same size exits => make this item
  // a subitem of this download to show that these downloads will be
  // merged
  if( ! parent_item )
    _item = new DownloadViewItem( _parent->_parent->_widget_downloads->ui_listview_download );
  else
    {
      _item = new DownloadViewItem( parent_item );
      _parent->_parent->_widget_downloads->ui_listview_download->setOpen( parent_item, true );
    }

  timeout = new QTimer();
  timer_bandwidth = new QTimer();
  timer_retry = new QTimer();
  
  if(!_qhe.firewalled())
    {
      sock = new QSocket();

      connect(sock, SIGNAL(readyRead()), this, SLOT(slotReadyRead()));
      connect(sock, SIGNAL(connectionClosed()), this, SLOT(slotConnectionClosed()));
      connect(sock, SIGNAL(error(int)), this, SLOT(slotError(int)));
      connect(sock, SIGNAL(connected()), this, SLOT(slotConnected()));
    }
      
  connect(timeout, SIGNAL(timeout()), this, SLOT(slotTimeout()));
  connect(timer_retry, SIGNAL(timeout()), this, SLOT(slotRetry()));
  connect(timer_bandwidth, SIGNAL(timeout()), this, SLOT(slotTimerBandwidth()));

  _client.erase();
  _first_start = true;
  _f = NULL;
}


void Download::run()
{
  _all_read = 0;

  setState(dtConnecting);
  _active = true;

  if(timer_retry->isActive()) timer_retry->stop();
  if(timeout->isActive()) timeout->stop();
  if(timer_bandwidth->isActive()) timer_bandwidth->stop();

  bandwidth     = 0;
  tmp_read      = 0;
  _buffer.erase();
  _header.erase();
  retry_timeout = 15;

  std::string host = _qhe.addr.strIP();
  
  complete_path = _dst_directory;

  if( complete_path.empty() )
    {
      QLineEdit* e = _parent->_parent->_widget_config->ui_lineedit_download_anything;
      int i;
      int k;
      QLineEdit* p[3] = { _parent->_parent->_widget_config->ui_lineedit_download_audio,
			  _parent->_parent->_widget_config->ui_lineedit_download_video,
			  _parent->_parent->_widget_config->ui_lineedit_download_images };

      std::string types[3];

      // path for audio, video, images
      types[0] = static_cast<const char*>( _parent->_parent->_widget_config->ui_lineedit_audio->text() );
      types[1] = static_cast<const char*>( _parent->_parent->_widget_config->ui_lineedit_video->text() );
      types[2] = static_cast<const char*>( _parent->_parent->_widget_config->ui_lineedit_images->text() );

      std::string d = _dest_file;
      
      // tolower everthing
      for( i = 0; i < d.size(); ++i ) d[i] = tolower( d[i] );
      for( k = 0; k < 3; ++k )
	for( i = 0; i < types[k].size(); ++i ) types[k][i] = tolower( types[k][i] );

      std::string ext;
      if( d.find_last_of(".") != std::string::npos )
	ext = d.substr( d.find_last_of( "." ) + 1 );

      // find the extension
      for( i = 0; i < 3; ++i )
	{
	  std::vector< std::string > v;
	  StringManipulation         sm( types[i] );

	  sm.split_to_vector( v, ';' );
	  
	  for( k = 0; k < v.size(); ++k )
	    if( v[k] == ext )
	      {
		e = p[i];
		break;
	      }
	}

      complete_path = e->text().latin1();
      if(complete_path[complete_path.size()-1] != '/') complete_path += "/";
    }
  
  complete_path += _dest_file;


  incomplete_path = _parent->_parent->_widget_config->ui_lineedit_incompleted->text().latin1();
  if(incomplete_path[incomplete_path.size()-1] != '/') incomplete_path += "/";
  incomplete_path += _dest_file;


  QDir d(incomplete_path.substr(0, incomplete_path.find_last_of("/")).c_str());

  
  if( ! d.exists() )
    {
      std::string error("Directory for incompleted downloads does not exist! [");
      error += incomplete_path.substr(0, incomplete_path.find_last_of("/")) + "]";
      Logging::getInstance()->log(Logging::Error, error);

      QMessageBox::warning(0, tr("Warning"), 
			   tr("Directory for incompleted downloads does not exist!"), QMessageBox::Ok, 0);
      setState(dtError);
      return;
    }

  d.setPath(complete_path.substr(0, complete_path.find_last_of("/")).c_str());

  if( ! d.exists() )
    {
      std::string error("Directory for completed downloads does not exist!");
      error += complete_path.substr(0, complete_path.find_last_of("/")) + "]";
      Logging::getInstance()->log(Logging::Error, error);

      QMessageBox::warning(0, tr("Warning"), 
			   tr("Directory for completed downloads does not exist!"), QMessageBox::Ok, 0);
      setState(dtError);
      return;
    }

  std::string debug("Start downloading file. [" + std::string(incomplete_path) + "]");
  Logging::getInstance()->log(Logging::Debug, debug);

  if(_first_start)
    {
      _first_start = false;

      QFile file(complete_path.c_str());

      if( file.exists() && _parent->_parent->_widget_config->ui_radiobutton_e_abort->isChecked() )
	{
	  _f = new QFile(complete_path.c_str());
	  setState(dtExists);
	  return;
	}
      else // check if file exists in one of the shared directories
	for(int i = 0; i < _parent->_parent->_widget_config->ui_listbox_shareddirectories->count(); ++i)
	  {
	    std::string sf( _parent->_parent->_widget_config->ui_listbox_shareddirectories->text(i).latin1() );
	    if( ! sf.empty() )
	      if( sf[sf.size()-1] != '/' ) sf += "/";
	    sf += complete_path.substr( complete_path.find_last_of("/") + 1 );
	    
	    QFile f(sf.c_str());
	    if( file.exists() && _parent->_parent->_widget_config->ui_radiobutton_e_abort->isChecked() )
	      {
		_f = new QFile(complete_path.c_str());
		setState(dtExists);
		return;
	      }
	  }

      file.setName(incomplete_path.c_str());


      if( _parent->_parent->_widget_config->ui_radiobutton_a_startnew->isChecked() )
	{
	  std::string s = createFilename(incomplete_path, _qhe.size, _qhe.addr.strIP());
	  _f = new QFile(s.c_str());
	}
      else
	{
	  std::string s;

	  // if this download is a segmented download do not continue an
	  // existing one
	  if( ! _segmented )
	    {
	      // continue if size is the same
	      if( _parent->_parent->_widget_config->ui_radiobutton_a_contsize->isChecked() )
		s = findFile(incomplete_path, _qhe.size);
	      
	      // continue if host is the same
	      if( _parent->_parent->_widget_config->ui_radiobutton_a_conthost->isChecked() )
		s = findFile(incomplete_path, _qhe.addr.strIP());
	      
	      // continue if host and size are the same
	      if( _parent->_parent->_widget_config->ui_radiobutton_a_contsizehost->isChecked() )
		s = findFile(incomplete_path, _qhe.size, _qhe.addr.strIP());
	    }

	  if( s.empty() )
	    { // no interrupted download exists - create new filename
	      s = createFilename(incomplete_path, _qhe.size, _qhe.addr.strIP());
	    }
	  else  // search for downloads with same id
	    {
	      std::vector<Download*>::iterator pos;
	      std::string                      id = InterruptedFile::id(s);

#ifndef SHARP
	      DownloadManager::_mutex.lock();
#endif
	      for(pos = _parent->_downloads.begin(); pos != _parent->_downloads.end(); ++pos)
		if( (*pos)->getID() == id )
		  {
		    s = createFilename(incomplete_path, _qhe.size, _qhe.addr.strIP());
		    break;
		  }
#ifndef SHARP
	      DownloadManager::_mutex.unlock();
#endif
	    }
	  
	  _f = new QFile(s.c_str());
	}

      _interrupted_filename = _f->name().latin1();
      QtellaSub::getInstance()->_interrupted_downloads->addInterruptedDownload( _interrupted_filename, 
										_qhe.index,
										_qhe.addr.port() );
    }
  else // not the first start
    if(_f->isOpen()) _f->close();

  InterruptedFile::split( std::string(_f->name().latin1()), _time, _id );
  
  if(!_f->isOpen()) _f->open(IO_Append|IO_ReadWrite|IO_Raw);
  if(!_f->isOpen())
    {
      std::string error("Cannot open file for writing! [" + std::string(_f->name().latin1()) + "]");
      Logging::getInstance()->log(Logging::Error, error);
      slotError(dtError);
      return;
    }

  if(_f->size() > 0) retry_timeout = 15;

  timeout->start(20000, true);

  if( !_qhe.firewalled() )
    sock->connectToHost( _qhe.addr.strIP().c_str(), _qhe.addr.port() );
  else
    {
      std::string id = _parent->_parent->connections->_id;
#ifndef SHARP
      Address addr( std::string(_parent->_parent->_widget_config->ui_lineedit_ip->text().latin1()) );
      addr.setPort( std::string(_parent->_parent->_widget_config->ui_lineedit_listenport->text().latin1()) );
#else
      Address addr( QtellaSub::getInstance()->getMyAddress() );
      addr.setPort( 6346 );
#endif      
      Push p(_parent->_parent->_widget_config->ui_spinbox_ttl->value(), 0, id, _qhe.id, _qhe.index, addr);
      _parent->_parent->connections->sendAll( p._data );
      setState(dtPushed);
    }
}


void Download::startPushedDownload(QSocket *socket)
{
  setState(dtConnecting);
  sock = socket;
  
  connect(sock, SIGNAL(readyRead()), this, SLOT(slotReadyRead()));
  connect(sock, SIGNAL(connectionClosed()), this, SLOT(slotConnectionClosed()));
  connect(sock, SIGNAL(error(int)), this, SLOT(slotError(int)));
  
  slotConnected();
}


const int Download::readState()
{
  return _state;
}


void Download::setState(int i)
{
  if( _search_item )
    if(_c->_search_item_set.find(_search_item) == _c->_search_item_set.end()) _search_item = NULL;
  
  switch(i)
    {
    case dtExists:
    case dtError:
    case dtAbort:
    case dtClosed:
    case dtClosedUnfinished:
    case dtNotFound:
    case dtTimeout:
    case dtBusy:
      if(_search_item) _search_item->setPixmap(2, *(Icon::getInstance()->getRedBall()));
      _c->_interrupted_downloads->setRed((InterruptedViewItem*)_int_item);
      break;

    case dtConnected:
      if(_search_item) _search_item->setPixmap(2, *(Icon::getInstance()->getGreenBall()));
      _c->_interrupted_downloads->setGreen((InterruptedViewItem*)_int_item);
      break;

    case dtConnecting:
    case dtQueued:
    case dtPushed:
    case dtIdle:
      if(_search_item) _search_item->setPixmap(2, *(Icon::getInstance()->getYellowBall()));
      _c->_interrupted_downloads->setYellow((InterruptedViewItem*)_int_item);
      break;

    case dtClosedCompleted:
      if(_search_item) _search_item->setPixmap(2, *(Icon::getInstance()->getSmile()));
      _c->_interrupted_downloads->setSmile((InterruptedViewItem*)_int_item);
      break;
    }

  // don't override state busy with state close
  if( ! ( _state == dtBusy && i == dtClosed ) )
    {
      std::strstream str;
      str << "Set state " << i << "." << std::ends;
      Logging::getInstance()->log(Logging::Debug, str.str());
      str.freeze(false);
      
      _state = i;
    }
}


const unsigned long Download::bytesRead()
{
  if( _finished )
    return _content_length;
  else
    if(_f)
      if(_f->exists())
	return _f->size();

  return 0;
}


const unsigned long Download::bytesToRead()
{
  return _content_length - bytesRead();
}


void Download::slotReadyRead()
{
  int	v,w;
  char  cbuff[32768];

  w = sock->readBlock(cbuff, 32768);
  _buffer.append(cbuff, w);
  tmp_read += w;
  _all_read += w;

  if(_header.empty())
    {
      std::string::size_type  pos = _buffer.find("\r\n\r\n");
      
      if( pos != std::string::npos)
	{
	  _header = _buffer.substr(0, pos+4);
	  _buffer = _buffer.substr(pos+4);

	  std::string::size_type  idx1 = _header.find(" ");
	  std::string::size_type  idx2 = _header.find(" ", idx1+1);

	  if( _header.substr(idx1+1, idx2-idx1-1) == "404" ) 
	    {
	      setState(dtNotFound);
	      return;
	    }
	  
	  if( _header.substr(idx1+1, idx2-idx1-1) == "503" ) 
	    {
	      setState(dtBusy);
	      timer_retry->start(1000);
	      return;
	    }
	  
	  if( _header.substr(idx1+1, idx2-idx1-1) == "200" ) 
	    {
	      setState(dtConnected);
	      timer_bandwidth->start(4000);
	      bandwidth = 0;
	    }

	  // get server name
	  
	  StringManipulation sm(_header);
	  if( ((idx1 = sm.to_lower().find("server:")) != std::string::npos) && (_client.empty()) )
	    {
	      idx1 += 7;  // point to character after ':'
	      idx1 = _header.find_first_not_of(" ", idx1);
	      idx2 = _header.find("\r\n", idx1);
	      _client =  _header.substr(idx1, idx2-idx1);
	    }
	  
	  // save time
	  _start_time = time(NULL);
	}
    }

  if( (!_buffer.empty()) && (readState()==dtConnected) )
    {
	v = _f->size();
	w = _f->writeBlock(_buffer.data(), _buffer.size());
	_f->flush();

	std::strstream str;
	str << "Write bytes: " << v << ", " << _content_length << ", " << w << ", " << _buffer.size() << std::ends;
	Logging::getInstance()->log(Logging::Debug, str.str());
	str.freeze(false);

	// abort download if the write was not ok (e.g. disk full)
	if( ( w != _buffer.size() ) || ( v+w != _f->size()) )  
	  {
	    std::string error("Cannot write to disk [" + std::string(_f->name().latin1()) + "]");
	    Logging::getInstance()->log(Logging::Error, error);

	    abortDownload();
	    return;
	  }

	_parent->_parent->connections->_statistic.loads_dsize += w;
	_buffer.erase(0, w);
   }
  
  std::strstream str;
  str << "bytes read: " << bytesRead() << "/" << _all_read << " header: " << _header.size() << std::ends;
  Logging::getInstance()->log(Logging::Debug, str.str());
  str.freeze(false);
  
  if(bytesRead() >= _content_length) finish();
}


const void Download::createUniqueFilename( std::string& dest)
{
  std::string destination = complete_path;
  int         i = 0;
  QFile       d(destination.c_str());

  std::string di = complete_path.substr(0, complete_path.find_last_of("/"));
  std::string fi = complete_path.substr(complete_path.find_last_of("/")+1);
		
  while( d.exists() )
    {
      std::string local_fi;
      std::string local_ext;
      
      destination = (di + "/");
      
      std::strstream str;
      str << i++ << std::ends;
      
      std::string::size_type idx = fi.find_last_of(".");
      if( idx != std::string::npos )
	{
	  local_fi = fi.substr(0, idx);
	  local_ext = fi.substr(idx);
	}
      else
	local_fi = fi;
      
      destination += local_fi + "_" + std::string(str.str()) + local_ext;
      str.freeze(false);
      
      d.setName(destination.c_str());
    }

  dest = destination;
}


void Download::finish()
{
  int loglevel = Logging::Info;
  std::strstream log;
  log << "Download finished. ";
  
  sock->close();
  setState(dtClosed);
  _active   = false;
  _end_time = time(NULL);

  if(readState() != dtBusy)
    if(bytesRead() >= _content_length)  // all bytes of the file have been read
      {
	if(_search_item) _search_item->setPixmap(2, *(Icon::getInstance()->getSmile()));
	_c->_interrupted_downloads->setSmile((InterruptedViewItem*)_int_item);

	log << "[all bytes read]";

	_parent->_parent->connections->_statistic.loads_dsucc++;
	setState(dtClosedCompleted);
	
	_f->close();

	// if this download is segmented than keep current filename and location
	if( ! _segmented )
	  {
	    std::string destination = complete_path;

	    if( _parent->_parent->_widget_config->ui_radiobutton_e_rename->isChecked() || 
          _parent->_parent->_widget_config->ui_radiobutton_e_abort->isChecked())
	      createUniqueFilename( destination );

	    QFile dest(destination.c_str());
	    
	    QDir d;
	    if( !d.rename(_f->name(), destination.c_str()) )
	      {
		if( ! copyFile(std::string(_f->name().latin1()), destination) )
		  {
		    if(_search_item) _search_item->setPixmap(2, *(Icon::getInstance()->getGreenSmile()));
		    _c->_interrupted_downloads->setGreenSmile((InterruptedViewItem*)_int_item);
		    
		    log << " [error moving file from incomleted path to completed path ("
			<< _f->name() << " -> " << destination << ")]";
		    loglevel = Logging::Error;
		    _filename = _f->name().latin1();
		  }
		else
		  {
		    QtellaSub::getInstance()->_interrupted_downloads->deleteFromIndex( _interrupted_filename );
		    _f->setName(destination.c_str());
		  }
	      }
	    else
	      {
		QtellaSub::getInstance()->_interrupted_downloads->deleteFromIndex( _interrupted_filename );
		_f->setName(destination.c_str());
		_filename = destination;
	      }
	  }

	if( ! _finished )
	  {
	    // wait one second before continuing with another
	    // segment from same host
	    //sleep(1);
	    _parent->downloadFinished(this);
	  }
	_finished = true;
	
      }
    else
      {
	if(_search_item) _search_item->setPixmap(2, *(Icon::getInstance()->getYellowBall()));
	_c->_interrupted_downloads->setYellow((InterruptedViewItem*)_int_item);
	    
	log << "[bytes read (" << bytesRead() << ")" << " != content length (" << _content_length << ")]";
	loglevel = Logging::Error;
	_parent->_parent->connections->_statistic.loads_dabor++;
	setState(dtClosed);
	timer_retry->start(1000);
      }
  else
    {
      log << "[state is busy]";
      loglevel = Logging::Error;
    }

  log << std::ends;
  Logging::getInstance()->log(loglevel, log.str());
  log.freeze(false);
}


void Download::slotConnectionClosed()
{
  std::strstream log;
  log << "Connection closed. Socket buffer: " << sock->bytesAvailable() 
      << " Buffer: " << _buffer.size() << " State: " << readState() 
      << " Filesize: " << _f->size() << std::endl;

  while(sock->bytesAvailable() > 0) slotReadyRead();

  if(readState() == dtConnected) _f->writeBlock(_buffer.data(), _buffer.size());
  _buffer.erase();
  _f->flush();

  log << "Flushed." << std::endl
      << "Socket buffer: " << sock->bytesAvailable()
      << " Buffer: " << _buffer.size() << " State: " << readState()
      << " Filesize: " << _f->size() << std::ends;

  Logging::getInstance()->log(Logging::Debug, log.str());
  log.freeze(false);

  finish();

  _end_time = time(NULL);
}


void Download::slotError(int error)
{
  std::strstream log;
  log << "Download error. [code=" << error << "]";
  Logging::getInstance()->log(Logging::Error, log.str());
  log.freeze(false);

  setState(dtError);
  timer_retry->start(1000);
}


void Download::slotConnected()
{
  std::string     s = _qhe.filename;
  unsigned long   begin = _f->size();
  std::strstream  str;

  str << "GET /get/" << _qhe.index << "/" << s << " HTTP/1.0\r\n"
      << "Range: bytes=" << begin + _start << "-";
  
  if( _end != -1 ) str << _end;

  str << "\r\n"
      << "User-Agent: Qtella " << VERSION << ADDITIONAL_VERSION_STR << "\r\n\r\n" << std::ends;

  _start_read = begin;

  s = str.str();
  str.freeze(false);

  sock->writeBlock(s.c_str(), s.size());
}



void Download::slotTimerBandwidth()
{
  bandwidth = static_cast<double>(tmp_read) / 4096;  // KB/s
  tmp_read = 0;
}


const double Download::readBandwidth()
{
  return bandwidth;
}


void Download::slotTimeout()
{
  timeout->stop();

  if(readState() == dtConnecting)
    {
      setState(dtTimeout);
      sock->close();
      timer_retry->start(1000);
    }
}


void Download::abortDownload(const bool replaced)
{
  _replaced = replaced;

  Logging::getInstance()->log(Logging::Debug, "Abort download.");

  timeout->stop();
  timer_bandwidth->stop();
  timer_retry->stop();
  if(sock) sock->close();
  
  if(_f) _f->close();
  
  _parent->_parent->connections->_statistic.loads_dabor++;
  setState(dtAbort);
  _active = false;
}


void Download::resumeDownload()
{
  _retries = 0;
  setState(dtQueued);
}


Download::~Download()
{
  if(_c->_search_item_set.find(_search_item) == _c->_search_item_set.end()) _search_item = NULL;

  if(_search_item) _search_item->setPixmap(2, QPixmap());
  _c->_interrupted_downloads->clearColored((InterruptedViewItem*)_int_item);

  delete timeout;
  delete timer_bandwidth;
  delete timer_retry;
  if(sock) delete sock;

  QListViewItem* first = _item->firstChild();
  if( first )
    {
      bool b = first->isSelected();
      first->moveItem( _item );
      first->setSelected(b);

      // take all items from this item and make the first child to their new parent
      QListViewItem* i;
      while( (i = _item->firstChild()) )
	{
	  bool selected = i->isSelected();
	  //_item->takeItem(i);
	  first->insertItem(i);
	  i->setSelected(selected);
	}

      _parent->_parent->_widget_downloads->ui_listview_download->setOpen( first, true );
    }

  delete _item;


  if(_f)
    {
      if(_f->isOpen()) _f->close();
      if(_f->exists()) if(_f->size() == 0) _f->remove();
      delete _f;
    }
}

void Download::slotRetry()
{
  retry_timeout--;

  if( retry_timeout == 0 )  
    {
      if(timer_retry->isActive()) timer_retry->stop();
      if(timeout->isActive()) timeout->stop();
      if(timer_bandwidth->isActive()) timer_bandwidth->stop();

      setState(dtQueued);
      _retries++;

      int n = _parent->_parent->_widget_config->ui_spinbox_conf_retries->value();
      if( (n != -1) && (_retries > n) ) 
	{
	  setState(dtError);
	  _active = false;
	}
    }
}


const int Download::readRetryTimeout()
{
  return retry_timeout;
}


std::string Download::createFilename(std::string file, unsigned long size, std::string host)
{
  _time  = time(NULL);
  
  for(_id = rand()%10000; ; ++_id)
    {
      std::strstream str;
      std::string    s;

      std::string::size_type  idx = file.find_last_of(".");
      std::string             ext;
      std::string             filename = file;

      if(idx != std::string::npos)
	{
	  filename = file.substr(0, idx);
	  ext = file.substr(idx);  // with "." !!
	}

      str << ".part_" << _time << "_" << size << "_" << host << "_" << _id << "_" << _start 
	  << "-" << _end << std::ends;

      s = filename + std::string(str.str()) + ext;
      str.freeze(false);

      QFile f(s.c_str());

      if(!f.exists()) return s;
    }
}


std::string Download::findFile(std::string file, unsigned long size)
{
  std::string::size_type  idx = file.find_last_of("/");

  std::string    directory = file.substr(0, idx);
  std::string    filename  = file.substr(idx+1);
  unsigned long  current_size = 0;

  std::string            ext;
  std::string            name;
  std::string::size_type p;
  if( (p = filename.find_last_of(".")) != std::string::npos)
    {
      ext = filename.substr(p+1);
      name = filename.substr(0, p);
    }

  QDir d(directory.c_str(), "*", QDir::DefaultSort, QDir::Files);

  std::string s;

  for(int i = 0; i < d.count(); ++i)
    {
      filename = d[i].latin1();

      if( (idx = filename.find(".part_")) == std::string::npos ) continue;
      if( (filename.substr(0, idx) != name) || (ext != filename.substr(filename.find_last_of(".")+1)) ) continue;

      idx = filename.find("_", idx+6);  // find first "_"
      std::string::size_type  idx2 = filename.find("_", idx+1); // find second "_" 

      std::string s_size = filename.substr(idx+1, idx2-idx-1);  // extract filesize
      
      std::strstream str;
      str << s_size << std::ends;
      unsigned long str_size;
      str >> str_size;
      
      std::string s_new = directory + "/" + filename;
      QFile f(s_new.c_str());
      
      if( (str_size == size) && (f.size() > current_size) )
	{
	  s = s_new;
	  current_size = f.size();
	}
    }

  return s;
}


std::string Download::findFile(std::string file, std::string host)
{
  std::string::size_type  idx = file.find_last_of("/");

  std::string    directory = file.substr(0, idx);
  std::string    filename  = file.substr(idx+1);
  unsigned long  current_size = 0;

  std::string            ext;
  std::string            name;
  std::string::size_type p;
  if( (p = filename.find_last_of(".")) != std::string::npos)
    {
      ext = filename.substr(p+1);
      name = filename.substr(0, p);
    }

  QDir d(directory.c_str(), "*", QDir::DefaultSort, QDir::Files);

  std::string s;

  for(int i = 0; i < d.count(); ++i)
    {
      filename = d[i].latin1();

      if( (idx = filename.find(".part_")) == std::string::npos ) continue;
      if( (filename.substr(0, idx) != name) || (ext != filename.substr(filename.find_last_of(".")+1)) ) continue;

      idx = filename.find("_", idx+6);
      idx = filename.find("_", idx+1);
      std::string::size_type  idx2 = filename.find("_", idx+1);
      
      std::string s_host = filename.substr(idx+1, idx2-idx-1);
      
      if( (s_host == host) )
	{
	  std::string s_new = directory + "/" + filename;
	  QFile f(s_new.c_str());
	  
	  if( f.size() > current_size )
	    {
	      s = s_new;
	      current_size = f.size();
	    }
	}
    }

  return s;
}


std::string Download::findFile(std::string file, unsigned long size, std::string host)
{
  std::string::size_type  idx = file.find_last_of("/");

  std::string         directory = file.substr(0, idx);
  std::string         filename  = file.substr(idx+1);
  unsigned long  current_size = 0;

  std::string            ext;
  std::string            name;
  std::string::size_type p;
  if( (p = filename.find_last_of(".")) != std::string::npos)
    {
      ext = filename.substr(p+1);
      name = filename.substr(0, p);
    }

  QDir d(directory.c_str(), "*", QDir::DefaultSort, QDir::Files);

  std::string s;

  for(int i = 0; i < d.count(); ++i)
    {
      filename = d[i].latin1();

      if( (idx = filename.find(".part_")) == std::string::npos ) continue;
      if( (filename.substr(0, idx) != name) || (ext != filename.substr(filename.find_last_of(".")+1)) ) continue;

      idx  = filename.find("_", idx+6);

      std::string::size_type  idx2;
      std::string::size_type  idx3;
      
      idx2 = filename.find("_", idx+1);
      idx3 = filename.find("_", idx2+1);
      
      std::string s_size = filename.substr(idx+1, idx2-idx-1);
      std::string s_host = filename.substr(idx2+1, idx3-idx2-1);
      
      std::strstream str;
      str << s_size << std::ends;
      unsigned long str_size;
      str >> str_size;
      
      if( (s_host == host) && (str_size == size) )
	{
	  std::string s_new = directory + "/" + filename;
	  QFile f(s_new.c_str());
	  
	  if( f.size() > current_size )
	    {
	      s = s_new;
	      current_size = f.size();
	    }
	}
    }

  return s;
}


void Download::slotUpdateStatus()
{
  static const std::string status[] = 
  {
    "Idle", "Error", "Abort", "Closed", "Completed", "Closed",
    "Connecting", "NotFound", "Timeout", "Busy", "Connected", "Exist",
    "Queued", "Push" 
  };

  // s[0] - filename
  // s[1] - status
  // s[2] - content length
  // s[3] - progress
  // s[4] - host
  // s[5] - bandwidth
  // s[6] - time
  // s[7] - client

  std::string  s[8];
  int          i;

  for(i = 0; i < 8; ++i) s[i] = "";

  s[1] = status[readState()];
  if( _replaced )
    s[1] = "Replaced";

  if( (readState()==dtBusy) || (readState()==dtClosed) || (readState()==dtTimeout) || 
      ( (readState()==dtError) && _active ) )
    {
      s[1] += " - ";
      std::strstream str;
      str << readRetryTimeout() << "s" << std::ends;
      s[1] += str.str();
      str.freeze(false);
    }

  if( readState() == dtClosedCompleted )
    {
      time_t         diff = _end_time - _start_time;
      std::strstream str;
      int            h   = diff / 3600;
      int            m   = (diff - h * 3600) / 60;
      int            sec = diff - h * 3600 - m * 60;
      str.fill('0');
      str << h << ":" << std::setw(2) << m << ":" << std::setw(2) << sec << std::ends;
      s[6].assign(std::string(str.str()));
      str.freeze(false);

      if(diff>0)
	{
	  double b = (double)(bytesRead() - _start_read) / diff / 1024;
	  std::strstream str;
	  str.setf(std::ios::fixed);
	  str << std::setprecision(2) << b << " KB/s" << std::ends;
	  s[5] = str.str();
	  str.freeze(false);
	}
    }

  if( readState() == dtConnected )
    {
      // bandwidth
      std::strstream str;
      str.setf(std::ios::fixed);
      str << std::setprecision(2) << readBandwidth() << " KB/s" << std::ends;
      s[5] = str.str();
      str.freeze(false);

      // calculate time to go
      Q_UINT32  seconds   = 0;
      Q_UINT32  toread    = bytesToRead();
      Q_UINT32  read      = bytesRead() - _start_read;
      double bandwidth   = readBandwidth();
      if(read > 0) 
	{
	  double d = static_cast<double>(toread) / static_cast<double>(read);
	  d *= static_cast<double>(time(NULL) - _start_time);
	  seconds = static_cast<Q_UINT32>(d);
	}
      int      h   = seconds / 3600;
      int      m   = (seconds - h * 3600) / 60;
      int      sec = seconds - h * 3600 - m * 60;
      std::strstream time_str;
      time_str.fill('0');
      time_str << h << ":" << std::setw(2) << m << ":" << std::setw(2) << sec << std::ends;
      s[6] = time_str.str();
      time_str.freeze(false);
    }

  // progress
  if( QtellaSub::getInstance()->_widget_config->ui_checkbox_numericprogress->isChecked() )
    {
      double p = 100;
      if(_content_length > 0) 
	p = static_cast<double>(100) * static_cast<double>(bytesRead()) / static_cast<double>(_content_length);
      if( p > 100 )
	p = 100;

      std::strstream str;
      str.setf(std::ios::fixed);
      str << std::setprecision(1) << p << "%" << std::ends;
      _item->setText(3, str.str());
      str.freeze(false);
    }
  else _item->setText(3, " ");

  if( QtellaSub::getInstance()->_widget_config->ui_checkbox_showprogressbar->isChecked() )
    {
      unsigned long l = 0;
      if( _content_length > 0 ) 
	l = static_cast<unsigned long>( 62 * static_cast<double>( bytesRead() ) / 
					static_cast<double>( _content_length ) );
      if( l > 62 ) l = 62;

      QColor color(0x20, 0x20, 0xff); // blue, downloading

      if( _segmented )
	if( _replaced )
	  color.setRgb( 0xcc, 0xcc, 0x20 );
	else
	  color.setRgb( 0x80, 0x80, 0x20 ); // yellow, segmented download

      if( bytesRead() >= _content_length )
	if( ! _segmented )
	  color.setRgb(0x20, 0xff, 0x20);  // green, download finished, not segmented
	else
	  color.setRgb( 0xcc, 0xcc, 0x20 ); // light yellow, download finished, segmented

      QImage img(64, 11, 32, 16);
      img.fill(qRgb(0xff, 0xff, 0xff));
      int x;
      int y;
      for(x = 0; x < 64; x++)
	{
	  img.setPixel(x, 0, qRgb(0xaa, 0xaa, 0xaa));
	  img.setPixel(x, 10, qRgb(0xaa, 0xaa, 0xaa));
	}
      for(y = 0; y < 11; y++)
	{
	  img.setPixel(0, y, qRgb(0xaa, 0xaa, 0xaa));
	  img.setPixel(63, y, qRgb(0xaa, 0xaa, 0xaa));
	}
      QPixmap pict;
			  
      for(y = 1; y <= 9; ++y)
	for(x = 1; x <= l; ++x) img.setPixel(x, y, color.rgb());
			  
      pict.convertFromImage(img);
      _item->setPixmap(3, pict);
    }
  else
    {
      QPixmap pict;
      _item->setPixmap(3, pict);
    }

  s[0] = _qhe.filename;
  if(_dest_file != _qhe.filename) s[0] += " [saved as: " + _dest_file + "]";

  s[4] = _qhe.addr.strIP();
  s[7] = _client;

  std::strstream str;
  str << _content_length << std::ends;
  s[2] = StringManipulation::insertDelimiter(str.str(), ',', 3);
  str.freeze(false);

  for(i = 0; i < 8; ++i) 
    if(i != 3) _item->setText(i, s[i].c_str());
}


bool Download::isActive()
{
  return _active;
}


bool Download::moveFile(std::string directory)
{
  std::string current(_filename);
  if(current.find_last_of("/") != std::string::npos) current.erase(0, current.find_last_of("/")+1);

  QDir d;
  if( d.rename(_filename.c_str(), (directory + current).c_str()) )
    {
      _f->setName( (directory+current).c_str() );
      _filename = directory + current;
    }
  else 
    if( copyFile(_filename, directory + current) )
      {
	_f->setName( (directory+current).c_str() );
	_filename = directory + current;
      }
    else return false;

  return true;
}


std::string Download::getFileName()
{
  return _filename;
}


bool Download::copyFile(std::string src, std::string dst)
{
  QFile fsrc(src.c_str());
  QFile fdst(dst.c_str());

  if( ! fsrc.open( IO_ReadOnly ) ) return false;
  if( ! fdst.open( IO_WriteOnly ) ) return false;

  int  r = 0;
  char buffer[4096];

  while( ( r = fsrc.readBlock(buffer, 4096) ) > 0 )
    if( r != fdst.writeBlock(buffer, r) ) return false;

  fsrc.close();
  fsrc.remove();

  _filename = dst;
  return true;
}


const unsigned long Download::readStart()
{
  return _start;
}


const std::string Download::getID()
{
  std::string s;
  std::strstream str;
  str << _time << "-" << _id << std::ends;
  s = str.str();
  str.freeze(false);
  return s;
}


const void Download::updateEnd( const unsigned long new_end )
{
  _end = new_end;
  _content_length = _end - _start + 1;
}
