#include "DownloadManager.h"
#include "InterruptedDownloads.h"
#include "InterruptedFile.h"
#include "WidgetConfig.h"
#include "WidgetConnections.h"
#include "WidgetDownloads.h"

#include "qtimer.h"
#include "qlistview.h"
#include "qlabel.h"
#include "qspinbox.h"
#include "qfile.h"
#include "qfiledialog.h"
#include "qmessagebox.h"
#include "qpixmap.h"
#include "qradiobutton.h"

#include "QueryHitEntry.h"
#include "QtellaSub.h"
#include "Download.h"
#include "Connections.h"
#include "StringManipulation.h"
#include "DownloadViewItem.h"
#include "InterruptedDownloads.h"
#include "InterruptedFile.h"
#include "Logging.h"
#if !defined(_WIN32)
#include "AudioFile.h"
#endif

#include <iomanip>
#include <iostream>
#include <strstream>

#undef COUT_OUTPUT

#ifndef SHARP
QMutex DownloadManager::_mutex;
#endif

DownloadManager::DownloadManager(QtellaSub* parent)
  : _parent(parent)
{
  _update_timer = new QTimer(this);
  connect(_update_timer, SIGNAL(timeout()), this, SLOT(slotUpdate()));
  _update_timer->start(1000);
}


void DownloadManager::getFreeSegments(const unsigned long filesize, 
				      std::vector< std::pair< unsigned long, unsigned long > >& segments)
{
  getUnfinishedSegments(filesize, segments);

  std::vector<Download*>::iterator  pos;
#ifndef SHARP
  _mutex.lock();
#endif
  for( pos = _downloads.begin(); pos != _downloads.end(); ++pos )
    {
      if( (*pos)->_qhe.size != filesize )
	continue;

      // for each replaced download another download exists
      if( (*pos)->_replaced )
	continue;

      unsigned long start = (*pos)->_start;
      unsigned long end   = (*pos)->_end;

      std::vector< std::pair< unsigned long, unsigned long > >::iterator p;

      for( p = segments.begin(); p != segments.end(); ++p )
	if( ( ( start >= p->first ) && ( start <= p->second ) ) ||
	    ( ( end >= p->first ) && ( end <= p->second ) ) )
	  // this download in downloading somewhere in this segment
	  // therefore remove the segment from list
	  {
	    segments.erase(p);
	    break;
	  }
    }

#ifndef SHARP
  _mutex.unlock();
#endif
}


void DownloadManager::getFinishedFiles(const unsigned long filesize, 
				       std::vector< InterruptedFile >& sorted_found)
{
  int i;

  // search the interrupted files to find already existing files of same size
  std::vector< InterruptedFile > ifiles;  // vector of all interrupted files
  std::vector< InterruptedFile > found;   // vector of files with original size equal to filesize

  InterruptedDownloads::getInstance()->getFiles( ifiles, true );
  for(i = 0; i < ifiles.size(); ++i)
    if( ifiles[i].size() == filesize ) found.push_back( ifiles[i] );

  // sort the files which have been found, sort by start offset
  for( i = 0; i < found.size(); ++i)
    if( found[i].read() > 0 )
      {
	std::vector< InterruptedFile >::iterator pos = sorted_found.begin();
	while( pos != sorted_found.end() )
	  {
	    if( found[i].start() < (*pos).start() )
	      {
		sorted_found.insert( pos, 1, found[i] );
		break;
	      }
	    ++pos;
	  }
	
	if( pos == sorted_found.end() )
	  sorted_found.push_back( found[i] );
      }
  
#ifdef COUT_OUTPUT
  cout << endl << "--- finsihed segments found in InterruptedDownloads" << endl;
  for( i = 0; i < sorted_found.size(); ++i )
    cout << sorted_found[i].start() << " - " << sorted_found[i].end() << " - " 
	 << sorted_found[i].read() << endl;
#endif

}


void DownloadManager::getUnfinishedSegments(const unsigned long filesize, 
					    std::vector< std::pair< unsigned long, unsigned long> >& segments)
{
  int i;
  int j;

  std::vector< InterruptedFile > sorted_found;
  getFinishedFiles( filesize, sorted_found );

  // in this vector all segments are stored which already have been downloaded
  std::vector< std::pair< unsigned long, unsigned long > > segments_downloaded;

  for( i = 0; i < sorted_found.size(); ++i )
    {
      unsigned long start = sorted_found[i].start();
      unsigned long end   = start + sorted_found[i].read() - 1;
      
      j = segments_downloaded.size() - 1;

      if( i == 0 )
	segments_downloaded.push_back( std::make_pair( start, end ) );
      else
	if( ( start <= ( segments_downloaded[j].second + 1 ) ) && ( end > segments_downloaded[j].second ) )
	  segments_downloaded[j].second = end;
	else
	  if( start > ( segments_downloaded[j].second + 1 ) )
	    segments_downloaded.push_back( std::make_pair( start, end ) );
    }

#ifdef COUT_OUTPUT
  cout << endl << "=== sorted segments downloaded" << endl;
  for( i = 0; i < segments_downloaded.size(); ++i )
    cout << segments_downloaded[i].first << " - " << segments_downloaded[i].second << endl;
#endif

  unsigned long s = 0;
  for( i = 0; i < segments_downloaded.size(); ++i )
    if( s < segments_downloaded[i].first )
      {
	segments.push_back( std::make_pair( s, segments_downloaded[i].first - 1 ) );
	s = segments_downloaded[i].second + 1;
      }
    else
      s = segments_downloaded[i].second + 1;

  if( ! segments_downloaded.empty() )
    if( segments_downloaded.back().second < ( filesize - 1 ) )
      segments.push_back( std::make_pair( segments_downloaded.back().second + 1, filesize - 1 ) );


  if( segments.empty() )
    {
      if( segments_downloaded.empty() )
	segments.push_back( std::make_pair( 0, filesize - 1 ) );
    }
  else
    if( segments.back().second < ( filesize - 1 ) )
      segments.push_back( std::make_pair( segments.back().second + 1, filesize - 1 ) );
  
#ifdef COUT_OUTPUT
  cout << endl << "*** segments not finished yet" << endl;
  for( i = 0; i < segments.size(); ++i )
    cout << segments[i].first << " - " << segments[i].second << endl;
#endif 
}


void DownloadManager::addDownload(std::vector<QueryHitEntry*> v)
{
  if( v.empty() ) return;

  int i;

  unsigned long filesize = v[0]->size;

  // segments that have to be downloaded
  std::vector< std::pair< unsigned long, unsigned long > > segments;

  getUnfinishedSegments( filesize, segments );

  // if more hosts are available than segments => split the largest segments
  if( ! segments.empty() )
    while( segments.size() < v.size() )
      {
		std::vector< std::pair<unsigned long, unsigned long> >::iterator max = segments.begin();
		std::vector< std::pair<unsigned long, unsigned long> >::iterator p = max;
		++p;
		while( p != segments.end() )
		  {
			if( ( p->second - p->first ) > ( max->second - max->first ) )
			  max = p;
			++p;
		  }
		
		if( ( max->second - max->first ) < 2048 ) break;  // at least 2k blocks
		
		unsigned long middle = (max->second + max->first) / 2;
		unsigned long max_e  = max->second;
		
		(*max).second = middle - 1;
		segments.push_back( std::make_pair( middle, max_e ) );
      }
  
#ifdef COUT_OUTPUT
  cout << endl << "*** downloading segments for selected files" << endl;
  for( i = 0; i < segments.size(); ++i )
    cout << segments[i].first << " - " << segments[i].second << endl;
#endif
  
#ifndef SHARP
  _mutex.lock();
#endif

  QListViewItem* p_first = NULL;

  // start downloads
  for(i = 0; i < v.size(); ++i)
    {
      if( i >= segments.size() ) break;

      QueryHitEntry q = *v[i];
      q.item = NULL;  // do not delete item when constructor is called

      QListViewItem* p = NULL;
      if( i > 0 )
		//p = _downloads[0]->_item;
		p = p_first;
      
      unsigned long start_dl = segments[i].first;
      unsigned long end_dl   = segments[i].second;

      Download *d = new Download(q, this, v[i]->item, "", NULL, "", start_dl, end_dl, true, p);

      if( i == 0 )
		p_first = d->_item;
	  
      _downloads.push_back(d);
	  
      if( numberOfDownloads(false) < _parent->_widget_config->ui_spinbox_maxdownloads->value() )
		{
		  int n = 0;
		  for(int j = 0; j < _downloads.size() - 1; ++j)
			if( (_downloads[j]->_qhe.addr == v[i]->addr) &&
				( (_downloads[j]->readState() == Download::dtConnected) ||
				  (_downloads[j]->readState() == Download::dtConnecting) )  ) ++n;
		  
		  if( n < _parent->_widget_config->ui_spinbox_dl_per_host->value() ) d->run();
		  
		}
    }

#ifndef SHARP
  _mutex.unlock();
#endif
}


void DownloadManager::addDownload(QueryHitEntry &qhe, QListViewItem* item, std::string dest_file, 
				  QListViewItem* inter, std::string dest_dir, const bool start_new)
{
  int i;

  // qhe.item is already set to NULL

  // search for double downloads

#ifndef SHARP
  _mutex.lock();
#endif

  for(std::vector<Download*>::iterator pos = _downloads.begin(); pos != _downloads.end(); ++pos)
    if( (*pos)->_qhe == qhe && (*pos)->_qhe.addr == qhe.addr && 
	( (*pos)->readState() == Download::dtConnected || (*pos)->readState() == Download::dtConnecting )  )
      {
	std::strstream str;

	str << "File: ";

	if( (*pos)->_qhe.filename.size() > 40 )
	  str << (*pos)->_qhe.filename.substr(0, 40) << "...";
	else
	  str << (*pos)->_qhe.filename;

#ifndef SHARP
	_mutex.unlock();
#endif	
	str << std::endl << std::endl << "You cannot download the same file" << std::endl
	    << "from the the same host twice." << std::ends;

	QMessageBox::information(NULL, "Download warning", str.str(), QMessageBox::Ok);
	str.freeze(false);

	return;
      }

#ifndef SHARP
  _mutex.unlock();
#endif

  Download* d;

  // maybe there already exist segmented downloads of this file
  // first search for segments which have not been downloaded yet
  std::vector< std::pair< unsigned long, unsigned long > > v;
  getFreeSegments( qhe.size, v );

#ifdef COUT_OUTPUT
  std::cout << "--- free segments: download first one" << std::endl;

  for( i = 0; i < v.size(); ++i )
    std::cout << i << ". " << v[i].first << " - " << v[i].second << std::endl;
#endif

#ifndef SHARP
  _mutex.lock();
#endif

  if( ! v.empty() )
    {
      // found at least one segment
      // start downloading it

      //qDebug("new segment");

      // search the parent item for this download
      QListViewItem* parent_item = NULL;

      for( i = 0; i < _downloads.size(); ++i )
	if( ( _downloads[i]->_qhe.size == qhe.size ) && _downloads[i]->_parent_item )
	  parent_item = _downloads[i]->_parent_item;

      bool segmented = true;
      
      if( v[0].first == 0 && v[0].second == ( qhe.size - 1) )
	segmented = false;

      d = new Download( qhe, this, item, dest_file, NULL, dest_dir, v[0].first, v[0].second, segmented, parent_item );

#ifdef COUT_OUTPUT      
      std::cout << "started download of segment: " << v[0].first << " - " << v[0].second << std::endl;
#endif
    }
  else
    {
      // no free segments found => try to replace an inactive download
#ifdef COUT_OUTPUT
      std::cout << "no free segment found" << std::endl;
#endif

      for( i = 0; i < _downloads.size(); ++i )
	if( ( _downloads[i]->readState() != Download::dtConnected ) && ( ! _downloads[i]->_replaced ) 
	    && ( _downloads[i]->readState() != Download::dtClosedCompleted ) &&
	    _downloads[i]->_qhe.size == qhe.size )
	  {
	    // replace this download
	    _downloads[i]->abortDownload(true);
	    
	    QListViewItem* parent_item = _downloads[i]->_parent_item;

	    if( ! parent_item )
	      parent_item = _downloads[i]->_item;

	    unsigned long  start = _downloads[i]->_start + _downloads[i]->bytesRead();
	    unsigned long  end   = _downloads[i]->_end;

	    //qDebug( "replaced download" );

	    d = new Download( qhe, this, item, "", NULL, "", start, end, true, parent_item );
	    break;
	  }

      if( i == _downloads.size() )
	{
	  // try to split an active download

	  //qDebug("split download");

	  int k = -1;
	  unsigned long diff = 0;

	  for( i = 0; i < _downloads.size(); ++i )
	    if( _downloads[i]->readState() == Download::dtConnected &&
		_downloads[i]->_qhe.size == qhe.size )
	      {
		unsigned long s = _downloads[i]->_start + _downloads[i]->bytesRead();
		unsigned long e = _downloads[i]->_end;

		if( (e - s ) > diff )
		  {
		    diff = e - s;
		    k = i;
		  }
	      }

	  if( k != -1 )
	    {
	      QListViewItem* parent_item = _downloads[k]->_parent_item;
	      
	      if( ! parent_item )
		parent_item = _downloads[k]->_item;

	      unsigned long s = _downloads[k]->_start + _downloads[k]->bytesRead();
	      unsigned long e = _downloads[k]->_end;

	      // split only downloads for with more than 20k are missing
	      if( ( e - s ) >= 20480 )
		{
		  unsigned long new_end = ( s + e ) / 2;

		  _downloads[k]->updateEnd( new_end );
		  _downloads[k]->_segmented = true;

#ifdef COUT_OUTPUT
		  std::cout << "splitted download" << std::endl;
#endif
		  d = new Download( qhe, this, item, "", NULL, "", new_end + 1, e, true, parent_item );
		}
	      else
		{
#ifndef SHARP
		  _mutex.unlock();
#endif
		  return;
		}
	    }
	  else
	    {
	      // start a normal download
	      if( ! start_new )
		{
#ifndef SHARP
		  _mutex.unlock();
#endif
		  return;
		}

#ifdef COUT_OUTPUT
	      std::cout << "starting new download." << std::endl;
#endif
	      d = new Download(qhe, this, item, dest_file, inter, dest_dir);
	    }
	}
    }
  
  _downloads.push_back(d);

  if( numberOfDownloads(false) < _parent->_widget_config->ui_spinbox_maxdownloads->value() )
    {
      int n = 0;
      for(int j = 0; j < _downloads.size() - 1; ++j)
        if( (_downloads[j]->_qhe.addr == qhe.addr) &&
            ( (_downloads[j]->readState() == Download::dtConnected) ||
              (_downloads[j]->readState() == Download::dtConnecting) )  ) ++n;
      
      if( n < _parent->_widget_config->ui_spinbox_dl_per_host->value() ) d->run();
    }

#ifndef SHARP
  _mutex.unlock();
#endif
}


void DownloadManager::updatePush(std::string data, QSocket* socket)
{
#ifndef SHARP
  _mutex.lock();
#endif

  std::vector<Download*>::iterator  pos = _downloads.begin();
  
  std::string s(data);
  s.erase(0, 4);                             // erase "GIV "
  std::string::size_type idx = s.find(":");  // find first ":"
  std::string s_index = s.substr(0, idx);    // s_index = index
  s.erase(0, idx+1);
  idx = s.find("/");                         // find "/"
  std::string id = s.substr(0, idx);         // id
  s.erase(0, idx+1);
  idx = s.find("\n");
  std::string file = s.substr(0, idx);       // filename

  int index;
  std::strstream str;
  str << s_index << std::ends;
  str >> index;

  for(pos; pos != _downloads.end(); ++pos)
    if( (*pos)->readState() == Download::dtPushed )
      {
	StringManipulation sid(Connections::toHex((*pos)->_qhe.id));
	StringManipulation mid(id);

	if( (sid.to_lower()==mid.to_lower()) && (index==(*pos)->_qhe.index) && (file==(*pos)->_qhe.filename) )
	  (*pos)->startPushedDownload(socket);
      }

#ifndef SHARP
  _mutex.unlock();
#endif
}


void DownloadManager::slotClearInactive()
{
#ifndef SHARP
  _mutex.lock();
#endif

  std::vector<Download*>::iterator  pos;

  for(pos = _downloads.begin(); pos != _downloads.end(); )
    if ((*pos))
      {
	if( ! (*pos)->isActive() )
	  {
	    delete *pos;
	    pos = _downloads.erase(pos);
	  }
	else
	  ++pos;
      }
    else
      ++pos;

#ifndef SHARP
  _mutex.unlock();
#endif
}


void DownloadManager::slotRemoveDownload()
{
#ifndef SHARP
  _mutex.lock();
#endif

  std::vector<Download*>::iterator  pos;

  for(pos = _downloads.begin(); pos != _downloads.end(); )
    if ((*pos))
      {
	if( (*pos)->_item->isSelected() )
	  {
	    delete *pos;
	    pos = _downloads.erase(pos);
	  }
	else
	  ++pos;
      }
    else
      ++pos;

#ifndef SHARP
  _mutex.unlock();
#endif
}


void DownloadManager::slotResumeDownload()
{
#ifndef SHARP
  _mutex.lock();
#endif

  for(int i = 0; i < _downloads.size(); i++)
    if( _downloads[i]->_item->isSelected() ) _downloads[i]->resumeDownload();
  
#ifndef SHARP
  _mutex.unlock();
#endif
}


void DownloadManager::slotAbortDownload()
{
#ifndef SHARP
  _mutex.lock();
#endif

  for(int i = 0; i < _downloads.size(); i++)
    if( _downloads[i]->_item->isSelected() ) _downloads[i]->abortDownload();

#ifndef SHARP
  _mutex.unlock();
#endif
}


void DownloadManager::abortDownload(QueryHitEntry& qhe)
{
#ifndef SHARP
  _mutex.lock();
#endif

  for(int i = 0; i < _downloads.size(); i++)
    {
      if(_downloads[i]->_qhe == qhe) _downloads[i]->abortDownload();
      break;
    }
  
#ifndef SHARP
  _mutex.unlock();
#endif
}


void DownloadManager::deleteDownload(QueryHitEntry& qhe)
{
#ifndef SHARP
  _mutex.lock();
#endif

  std::vector<Download*>::iterator pos;

  for(pos = _downloads.begin(); pos != _downloads.end(); ++pos)
    if((*pos)->_qhe == qhe) 
      {
	(*pos)->abortDownload();
	QtellaSub::getInstance()->_interrupted_downloads->deleteFromIndex( (*pos)->_interrupted_filename );
	if((*pos)->_f) if((*pos)->_f->exists()) (*pos)->_f->remove();
	delete *pos;
	_downloads.erase(pos);
	break;
      }

#ifndef SHARP
  _mutex.unlock();
#endif
}


void DownloadManager::removeDownload(QueryHitEntry& qhe)
{
#ifndef SHARP
  _mutex.lock();
#endif

  std::vector<Download*>::iterator pos;

  for(pos = _downloads.begin(); pos != _downloads.end(); ++pos)
    if((*pos)->_qhe == qhe) 
      {
	(*pos)->abortDownload();
	delete *pos;
	_downloads.erase(pos);
	break;
      }

#ifndef SHARP
  _mutex.unlock();
#endif
}


bool DownloadManager::exists(QueryHitEntry& qhe)
{
#ifndef SHARP
  _mutex.lock();
#endif

  std::vector<Download*>::iterator pos;
  
  for(pos = _downloads.begin(); pos != _downloads.end(); ++pos)
    if((*pos)->_qhe == qhe) 
      {
#ifndef SHARP
	_mutex.unlock();
#endif
	return true;
      }

#ifndef SHARP
  _mutex.unlock();
#endif

  return false;
}


void DownloadManager::slotUpdate()
{
  double bandwidth = 0;

#ifndef SHARP
  _mutex.lock();
#endif

  for(int i = 0; i < _downloads.size(); i++)
    {
      _downloads[i]->slotUpdateStatus();

      if(_downloads[i]->readState() == Download::dtConnected)
	bandwidth += _downloads[i]->readBandwidth();
      else
	if(_downloads[i]->readState() == Download::dtQueued)
	  if( numberOfDownloads(false) < _parent->_widget_config->ui_spinbox_maxdownloads->value() )
	    {
	      int n = 0;
	      for(int j = 0; j < _downloads.size(); ++j)
		if( i != j )
		  if( (_downloads[j]->_qhe.addr == _downloads[i]->_qhe.addr) &&
		      ( (_downloads[j]->readState() == Download::dtConnected) ||
			(_downloads[j]->readState() == Download::dtConnecting) )  ) ++n;

	      if( n < _parent->_widget_config->ui_spinbox_dl_per_host->value() ) _downloads[i]->run();
	    }
    }

  std::strstream str;
  str.setf(std::ios::fixed);
  str << std::setprecision(2) << bandwidth << " KB/s" << std::ends;
  
  _parent->_widget_downloads->ui_textlabel_bandwidth->setText(str.str());
  str .freeze(false);

  static time_t last_sort = 0;
  if( time(NULL) - last_sort > 1 )
    {
      _parent->_widget_downloads->ui_listview_download->sort();
      last_sort = time(NULL);
    }

#ifndef SHARP
  _mutex.unlock();
#endif
} 


int DownloadManager::numberOfDownloads( const bool lock_mutex )
{
#ifndef SHARP
  if( lock_mutex )
    _mutex.lock();
#endif

  int n = 0;
  std::vector<Download*>::iterator  pos = _downloads.begin();

  for( pos; pos != _downloads.end(); ++pos )
    if( ((*pos)->readState() == Download::dtConnected) || ((*pos)->readState() == Download::dtConnecting) ||
	((*pos)->readState() == Download::dtPushed) )
      ++n;

#ifndef SHARP
  if( lock_mutex )
    _mutex.unlock();
#endif

  return n;
}


void DownloadManager::slotDelete()
{
#ifndef SHARP
  _mutex.lock();
#endif

  std::vector<Download*>::iterator  pos;

  for(pos = _downloads.begin(); pos != _downloads.end(); )
    if( (*pos)->_item->isSelected() )
      {
	(*pos)->abortDownload();
	QtellaSub::getInstance()->_interrupted_downloads->deleteFromIndex( (*pos)->_interrupted_filename );
	if((*pos)->_f) if((*pos)->_f->exists()) (*pos)->_f->remove();
	delete *pos;
	pos = _downloads.erase(pos);
      }
    else
      ++pos;

#ifndef SHARP
  _mutex.unlock();
#endif
}


void DownloadManager::slotMoveDownloads()
{
#ifndef SHARP
  QFileDialog d(0, "Select Directory", true);

  d.setMode(QFileDialog::DirectoryOnly);
  d.setFilter("All (*)");

  if(d.exec() == QDialog::Accepted)
    {
      std::vector<std::string>     failed;
      std::string                  dirname(d.selectedFile().latin1());

      for(int i = 0; i < _downloads.size(); ++i)
		if( _downloads[i]->readState() == Download::dtClosedCompleted && _downloads[i]->_item->isSelected() )
		  if( ! _downloads[i]->moveFile(dirname) )
			failed.push_back(_downloads[i]->getFileName());
	  
      if( failed.empty() )
		QMessageBox::information(0, "Information", "All files successfully moved.", QMessageBox::Ok);
      else
		{
		  std::string t("Failed to move files:\n\n");
		  for(int i = 0; i < failed.size(); ++i) t += failed[i] + "\n";
		  QMessageBox::warning(0, "Warning", t.c_str(), QMessageBox::Ok, 0);
		}
    }
#endif
}

void DownloadManager::slotDownloadProperties()
{
#ifndef SHARP
  _mutex.lock();
#endif

  std::vector<Download*>::iterator pos;

  for(pos = _downloads.begin(); pos != _downloads.end(); )
    if( (*pos)->_item->isSelected() )
      {
	std::string s((*pos)->_f->name().latin1());
	std::string d;
	std::string f;

	d = s.substr(0, s.find_last_of("/"));
	f = s.substr(s.find_last_of("/") + 1);

	AudioFile p(0, tr("Properties", "of audio files"), false);
	p.setFile( d, f, true );
	p.setProperty();
	p.show();

	break;
      }
    else
      ++pos;

#ifndef SHARP
  _mutex.unlock();
#endif
}


void DownloadManager::downloadFinished(Download* download)
{
  if( ! download->_segmented )
    return;

  int i;

#ifdef COUT_OUTPUT
  std::cout << "One download has been finished" << std::endl;
#endif

  std::vector< std::pair< unsigned long, unsigned long > > v;
  getUnfinishedSegments( download->_qhe.size, v );

  if( v.empty() )
    {
      // all segments have been downloaded
      // we can merge them now
#ifdef COUT_OUTPUT
      std::cout << "All segments have been downloaded." << std::endl;
#endif
      
      std::vector< InterruptedFile > files;
      getFinishedFiles( download->_qhe.size, files );

      unsigned long bytes_written = 0;

      std::string filename;

      if( ! download->_parent_item ) // this is the parent item
	download->createUniqueFilename( filename );
      else
	// get destination filename of parent item
	{
#ifndef SHARP
	  _mutex.lock();
#endif
	  for( i = 0; i < _downloads.size(); ++i )
	    if( _downloads[i]->_item == download->_parent_item )
	      {
		download->createUniqueFilename( filename );
		break;
	      }
#ifndef SHARP
	  _mutex.unlock();
#endif
	}


      // remove all sub items from group parent
      QListViewItem* parent_item = download->_parent_item;
      
      if( ! parent_item )
	parent_item = download->_item;

#ifndef SHARP
      _mutex.lock();
#endif
      QListViewItem* item;
      while( ( item = parent_item->firstChild() ) )
	{
	  std::vector< Download* >::iterator pos;
	  for( pos = _downloads.begin(); pos != _downloads.end(); ++pos )
	    if( (*pos)->_item == item )
	      {
		delete *pos;
		_downloads.erase(pos);
		break;
	      }
	}

      // update the content length parent item
      for( i = 0; i < _downloads.size(); ++i )
	if( _downloads[i]->_item == parent_item )
	  {
	    _downloads[i]->_content_length = download->_qhe.size;
	    if( _downloads[i]->_f )
	      _downloads[i]->_f->setName( filename.c_str() );
	    _downloads[i]->_segmented = false; // green progress bar
	    _downloads[i]->setState( Download::dtClosedCompleted );
	    _downloads[i]->_replaced = false;
	    break;
	  }

#ifndef SHARP
      _mutex.unlock();
#endif

      QFile dest( filename.c_str() );
      
      if( ! dest.open( IO_WriteOnly ) )
	{
	  std::string error = "Cannot open file: " + filename;
	  error += " / Can't merge downloads.";
	  Logging::getInstance()->log(Logging::Error, error);
	  return;
	}

      for( i = 0; i < files.size(); ++i )
	{
#ifdef COUT_OUTPUT
	  std::cout << "processing: " << files[i].completeName() << std::endl;
#endif
	  char   buffer[10240];
	  int    bytes;
	  QFile  f( files[i].completeName().c_str() );
	  
	  f.open( IO_ReadOnly );

	  if( bytes_written > files[i].start() )
	    if( f.size() > ( bytes_written - files[i].start() ) )
	      f.at( bytes_written - files[i].start() );
	    else 
	      continue;

	  while( ( bytes = f.readBlock( buffer, 10240 ) ) > 0 )
	    {
	      dest.writeBlock( buffer, bytes );
	      bytes_written += bytes;
	    }
	}

      // delete segmented downloads
      for( i = 0; i < files.size(); ++i )
	{
	  QFile f( files[i].completeName().c_str() );
	  f.remove();
	}

    }
  else
    {
#ifdef COUT_OUTPUT
      std::cout << "restart download from this host" << std::endl;
#endif
      addDownload( download->_qhe, download->_search_item, "", NULL, "", false );
    }
}

