#include <iostream.h>
#include <qlineedit.h>
#include <qlistbox.h>
#include <qnetwork.h>
#include <qregexp.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "GWebCacheManager.h"
#include "GWebCache.h"
#include "HttpTransactor.h"
//#include "Qtella.h"
#include "QtellaSub.h"
#include "WidgetConfig.h"
#include "WidgetConnections.h"

#include "GWebCacheStaticList.h"

static int gwebCacheDebugLevel = 0;

GWebCacheManager::GWebCacheManager(Qtella *interface)
:_interface(interface), _sendUpdateTransaction(NULL), _requestHostsTransaction(NULL)
{
      qInitNetworkProtocols();

      // Randomly distribute the known caches

      srand(time(NULL));
      int n = sizeof(GWebCacheStaticList) / sizeof(GWebCacheStaticList[0]);
      const char *randomCacheSet[n];
      memcpy(randomCacheSet, GWebCacheStaticList, sizeof(GWebCacheStaticList));
      while(n) {
	    int p = rand() % n;
	    addCache(randomCacheSet[p]);
	    randomCacheSet[p] = randomCacheSet[n - 1];
	    n--;
      }

      connect(&_sendUpdate, SIGNAL(timeout()), this, SLOT(sendUpdate()));
      _sendUpdate.start(3600 * 1000);
}

void GWebCacheManager::addCache(const char *url)
{
      _list.append(new GWebCache(url));
}

GWebCache *GWebCacheManager::get()
{
      return _list.getLast();
}

void GWebCacheManager::sendUpdate()
{
      if (_sendUpdateTransaction) return; // Some Transaction in progress
      GWebCache *h = get();
      if (!h) return;
      rotateCache();
      QString query = "http://";
      query += h -> _url;
      query += "?ip=";
      query += QtellaSub::getInstance() -> _widget_config -> ui_lineedit_ip -> text();
      query += ':';
      query += QtellaSub::getInstance() -> _widget_config -> ui_lineedit_listenport -> text();
      query += "&client=QTEL&version=" VERSION ADDITIONAL_VERSION_STR;
      
      if (gwebCacheDebugLevel > 0)
	    std::cerr << "Sending update as " << query << std::endl;

      _sendUpdateTransaction = new HttpTransactor;
      connect(_sendUpdateTransaction, SIGNAL(finished(const QNetworkOperation &, const HttpTransactor &)), 
	      this, SLOT(sendingUpdateFinished(const QNetworkOperation &, const HttpTransactor &)));

      if (!_sendUpdateTransaction -> start(query)) {
	    std::cerr << "BAD QUERY " << query
		 << " while updating GWebCache with our address, may be the cache should be discarded (not implemented yet)" << std::endl;
	    delete _sendUpdateTransaction;
	    _sendUpdateTransaction = NULL;
      }
}

void GWebCacheManager::rotateCache()
{
      GWebCache *h = _list.getLast();
      if (!h) return;
      _list.removeLast();
      _list.prepend(h);
}

void GWebCacheManager::sendingUpdateFinished(const QNetworkOperation &networkStatus, const HttpTransactor &)
{
      if (gwebCacheDebugLevel > 0)
	    std::cerr << "GWebCacheManager::sendingUpdateFinished, status = " << networkStatus.state() << std::endl;
      if (networkStatus.state() != QNetworkProtocol::StDone) {
	    // Error while sending update. Too bad... For the remote cache.
      } else {
	    // Nothing special
      }
      delete _sendUpdateTransaction;
      _sendUpdateTransaction = NULL;
}

void GWebCacheManager::requestHosts()
{
      if (_requestHostsTransaction) return; // Some transaction in progress
      GWebCache *h = get();
      if (!h) return;
      rotateCache();
      
      QString query = "http://";
      query += h -> _url;
      query += "?hostfile=1";
      query += "&client=QTEL&version=" VERSION ADDITIONAL_VERSION_STR "x";

      if (gwebCacheDebugLevel > 0)
	    std::cerr << "Querying GWebCache: " << query << std::endl;

      _requestHostsTransaction = new HttpTransactor;
      connect(_requestHostsTransaction, SIGNAL(finished(const QNetworkOperation &, const HttpTransactor &)), 
	      this, SLOT(requestingHostsFinished(const QNetworkOperation &, const HttpTransactor &)));

      if (!_requestHostsTransaction -> start(query)) {
	    std::cerr << "BAD QUERY " << query
		 << " while updating GWebCache with our address, may be the cache should be discarded (not implemented yet)" << std::endl;
	    delete _requestHostsTransaction;
	    _requestHostsTransaction = NULL;
      }
}

static QRegExp extractRecord = QString("(\\d{1,3}(?:\\.\\d{1,3}){3}:\\d{1,5})[\\r\\n]+");

void GWebCacheManager::requestingHostsFinished(const QNetworkOperation &op, const HttpTransactor &t)
{
      if (gwebCacheDebugLevel > 0) {
	    std::cerr << "request completed with status " << op.state() << std::endl;
	    if (op.state() == QNetworkProtocol::StDone) {
		  std::cerr << "===\n" << t.reply() << "===" << std::endl;
	    }
      }
      int newHosts = 0;
      if (op.state() == QNetworkProtocol::StDone) {
	    // We must control tightly what we get, because if an error occured on the
	    // cache server, we get gibberish.
	    for(int pos = 0; pos >= 0; pos += extractRecord.matchedLength()) {
		  if (extractRecord.search(t.reply(), pos) != pos) {
			// Regexp not found where looked for, that is, not found at all
			break;
		  }
		  newHosts++;
		  QtellaSub::getInstance()->_widget_connections->ui_listbox_hostlist->insertItem(extractRecord.cap(1));
	    }
		  
      } else {
	    // mark cache as bad
      }

      delete _requestHostsTransaction;
      _requestHostsTransaction = NULL;

      if (op.state() != QNetworkProtocol::StDone || !newHosts) requestHosts();
}
