#include "InConnection.h"
#include "Logging.h"
#include "QtellaSub.h"
#include "Gnutella.h"
#include "Address.h"
#include "UploadManager.h"
#include "DownloadManager.h"
#include "Connections.h"
#include "StringManipulation.h"
#include "Pong.h"
#include "PongCache.h"
#include "WidgetConfig.h"

#include <qtimer.h>
#include <qsocket.h>
#include <qhostaddress.h>
#include <qlistbox.h>

#undef _DEBUG

InConnection::InConnection(int socket) : _state(Connected), _timeout(0)
{
  _v06_ok = false;
  _user_agent = "";

  _socket = new QSocket();
  _socket->setSocket(socket);

  // check whether host is blocked or not
  std::string addr( (const char*)(_socket->peerAddress()).toString() );

  for(int i = 0; i < QtellaSub::getInstance()->_widget_config->ui_listbox_blockedhosts->count(); ++i)
    {
      std::string s( (const char*)QtellaSub::getInstance()->_widget_config->ui_listbox_blockedhosts->text(i) );
      
      if( s == addr )
        {
          _state = Closed;
          delete _socket;
          _socket = NULL;
          return;
        }
    }

  // host is not blocked

  connect(_socket, SIGNAL(readyRead()), SLOT(slotReadyRead()));

  _timeout = new QTimer(this);
  connect(_timeout, SIGNAL(timeout()), SLOT(slotTimeout()));
  _timeout->start(10000);
}  


InConnection::~InConnection()
{
  if( _socket ) delete _socket;
  if( _timeout ) delete _timeout;
}


void InConnection::slotReadyRead()
{
  QtellaSub* p = QtellaSub::getInstance();


  int     r = 0;
  char    buffer[1024];

  r = _socket->readBlock(buffer, 1024);
  _rec.append(buffer, r);

#ifdef _DEBUG
  qDebug( "InConnection: bytes read=%d", r );
#endif

  if( _v06_ok )
    {
      _timeout->stop();

#ifdef _DEBUG
      StringManipulation sm( _rec );
      sm.replace( '\n', "\\n" );
      sm.replace( '\r', "\\r" );
      qDebug( "InConnection <%d>: Buffer: <%s>\n", _socket->socket(), sm.str().c_str() );
#endif
      if( _rec.find("\r\n\r\n") != std::string::npos)
        {
          std::string::size_type idx1;
          
          idx1 = _rec.find("/");
          idx1 = _rec.find_first_of( "0123456789", idx1 );  // find first number of version
          idx1 = _rec.find_first_of( " ", idx1 );           // find the space before status code
          
          std::string status = _rec.substr( idx1 + 1, 3 );
          
          if( status == "200" )
            {
              _timeout->stop();
              disconnect(_socket, 0, 0, 0);
              
              Logging::getInstance()->log(Logging::Debug, "New incoming connection. v0.6");
              
              std::string ip(_socket->peerAddress().toString().latin1());
              p->connections->addServant(Address(ip, _socket->port()), _socket, _user_agent, _v06_header);
              
              _socket = NULL;
              _state = Closed;
              return;
            }
        }
      
      delete _socket;
      _socket = NULL;
      _state = Closed;
      return;
    }

  // incoming connection old 0.4 client
  if(_rec.find("GNUTELLA CONNECT/0.4") != std::string::npos)
    {
      _timeout->stop();

      disconnect(_socket, 0, 0, 0);

      if( ! QtellaSub::getInstance()->_use_opensourcep2p )
        {
          Logging::getInstance()->log(Logging::Debug, "New incoming connection. v0.4");
          std::string ip(_socket->peerAddress().toString().latin1());
          p->connections->addServant(Address(ip, _socket->port()), _socket);
        }
      else
        delete _socket;

      _socket = NULL;
      _state = Closed;
      return;
    }

  if( _rec.find("\r\n\r\n") != std::string::npos )
    {
      // incoming v0.6 handshake ?
      if( _rec.substr(0, 16) == "GNUTELLA CONNECT" || _rec.substr(0, 18) == "OPENSOURCE CONNECT" )
        {
          if( (_rec.substr(0, 16) == "GNUTELLA CONNECT" && QtellaSub::getInstance()->_use_opensourcep2p) ||
              (_rec.substr(0, 18) == "OPENSOURCE CONNECT" && (!QtellaSub::getInstance()->_use_opensourcep2p) ) )
            {
              _timeout->stop();
              delete _socket;
              _socket = NULL;
              _state = Closed;
              return;
            }

	  _v06_header = _rec.substr(0, _rec.find("\r\n\r\n"));
#ifdef _DEBUG
	  StringManipulation sm( _rec );
	  sm.replace( '\n', "\\n" );
	  sm.replace( '\r', "\\r" );
	  qDebug( "InConnection <%d>: Initial received: <%s>", _socket->socket(), sm.str().c_str() );
#endif
	  std::string::size_type idx1;
	  std::string::size_type idx2;

	  // get handshaking version
	  idx1 = _rec.find_first_of( "0123456789", 16 );
	  idx2 = _rec.find_first_not_of( "0123456789.", idx1 );

	  std::string version = _rec.substr( idx1, idx2 - idx1 );

	  // get user agent
	  idx1 = _rec.find( "\r\nUser-Agent:", idx2 );
	  if( idx1 != std::string::npos )
	    {
	      idx1 = _rec.find_first_not_of( " ", idx1 + 13 );
	      idx2 = _rec.find( "\r\n", idx1 + 2 );

	      _user_agent = _rec.substr( idx1, idx2 - idx1 );
	    }
	  else _user_agent = "?";

#ifdef _DEBUG
	  qDebug( "InConnection <%d> v0.6: version <%s>, user agent <%s>", 
		  _socket->socket(), version.c_str(), _user_agent.c_str() );
#endif


	  // answer
	  QHostAddress ha = _socket->peerAddress();
	  std::string remote_ip = std::string("Remote-IP: ") + std::string(ha.toString().latin1());

	  std::string ok;

	  if( ! Connections::getInstance()->incomingSlotsFree() )
	    {
#ifdef _DEBUG
	      qDebug( "InConnection: busy" );
#endif
	      std::list<Pong>               ponglist;
	      PongCache::getInstance()->getPong(ponglist, 10);
	      std::list<Pong>::iterator     pos = ponglist.begin();
	      
	      std::string s;
	      if( QtellaSub::getInstance()->_use_opensourcep2p )
		s = "OPENSOURCE";
	      else
		s = "GNUTELLA";

	      s += std::string("/0.6 503 BUSY\r\n");
	      s += "User-Agent: Qtella " + std::string(VERSION) + ADDITIONAL_VERSION_STR + "\r\n" + remote_ip + "\r\n";
	      s += "X-Try: ";

	      while( pos != ponglist.end() )
		{
		  s += ((*pos)._address).toString();
		  ++pos;
		  if( pos != ponglist.end() ) s += std::string(",");
		}
	      
	      s += std::string("\r\n\r\n");
#ifdef _DEBUG
	      qDebug( "InConnection: send <%s>", s.c_str() );
#endif
	      _socket->writeBlock( s.data(), s.size() );
	      _socket->flush();

	      _timeout->stop();
	      delete _socket;
	      _socket = NULL;
	      _state = Closed;
	      return;
	    }

	  if( QtellaSub::getInstance()->_use_opensourcep2p )
	    ok = "OPENSOURCEP2P";
	  else
	    ok = "GNUTELLA";

	  ok += "/0.6 200 OK\r\nUser-Agent: Qtella " + std::string(VERSION) + ADDITIONAL_VERSION_STR + "\r\n"
	    + remote_ip + "\r\n\r\n";

	  _socket->writeBlock( ok.data(), ok.size() );
	  _socket->flush();

	  _v06_ok = true;
	  _rec.erase();
	  return;
	}

      disconnect(_socket, 0, 0, 0);

      // GET ?
      if(_rec.substr(0, 3) == "GET")
	{
#ifdef _DEBUG
	  StringManipulation sm( _rec );
	  sm.replace( '\n', "\\n" );
	  sm.replace( '\r', "\\r" );
	  qDebug( "InConnection <%d>: GET request <%s>", _socket->socket(), sm.str().c_str() );
#endif
	  _timeout->stop();
	  Logging::getInstance()->log(Logging::Debug, "New upload request.");
	  p->upload_manager->addUpload(_rec, _socket);

	  _socket = NULL;
	  _state = Closed;
	  return;
	}

      _timeout->stop();
    }

  if( _rec.find("\n\n") != std::string::npos )
    {
      disconnect(_socket, 0, 0, 0);

      if(_rec.substr(0, 3) == "GIV")
	{
	  _timeout->stop();
	  Logging::getInstance()->log(Logging::Debug, "Got GIV request.");
	  p->download_manager->updatePush(_rec, _socket);

	  _socket = NULL;
	  _state = Closed;
	  return;
	}
    }
}


void InConnection::slotTimeout()
{
#ifdef _DEBUG
  qDebug( "InConnection: <%d> Connection timed out", _socket->socket() );
#endif

  _timeout->stop();
  Logging::getInstance()->log(Logging::Error, "Incoming connection timed out!");

  if(_socket)
    {
      delete _socket;
      _socket = NULL;
    }

  _state = Closed;
}


int InConnection::state()
{
  return _state;
}
